"""Module defining Real Node types, which can be run using our DeltaPySimulator."""
from __future__ import annotations
import dill
from inspect import _empty, signature
import logging
from queue import Full
import sys
import textwrap
from threading import Event
from time import sleep
import typing
from collections import OrderedDict

from deltalanguage.data_types import (BaseDeltaType,
                                      DeltaTypeError,
                                      Optional,
                                      Void,
                                      DeltaIOError,
                                      as_delta_type,
                                      delta_type)
from deltalanguage.logging import MessageLog, make_logger
from deltalanguage._utils import QueueMessage

from .._body_templates import BodyTemplate, MethodBodyTemplate
from .abstract_node import AbstractNode, IndexProxyNode
from .node_bodies import (PyConstBody,
                          PyInteractiveBody,
                          PyMigenBody,
                          Body)
from .port_classes import InPort, OutPort

if typing.TYPE_CHECKING:
    from deltalanguage.runtime import DeltaQueue, DeltaPySimulator
    from .. import DeltaGraph


class RealNode(AbstractNode):
    """Class to represent a non-abstract node that can form part of
    :py:class:`DeltaGraph`.

    Parameters
    ----------
    graph : DeltaGraph
        Graph this node is a member of.
    bodies : typing.List[Body]
        typing.List of bodies that can represent the implementation for this node.
    outputs : BaseDeltaType
        The type of what we expect the body to output.
    name : typing.Optional[str]
        The name of the node (an index is appended to the end).
    lvl : int
        Logging level for the node.
        By default only error logs are displayed.
    is_autogenerated : bool
        True if this node is created automatically to provide an input to
        another node. For instance:

        .. code-block:: python

            with dl.DeltaGraph() as graph:
                printer(42)

        has an autogenerated node that provides 42. For strict typing this
        node should send data in the same format as printer's input.
    """

    def __init__(self,
                 graph,
                 bodies: typing.List[Body],
                 inputs: typing.OrderedDict[str,
                                            typing.Union[BaseDeltaType, Optional]] = None,
                 outputs:  typing.OrderedDict[str, BaseDeltaType] = None,
                 name: typing.Optional[str] = None,
                 lvl: int = logging.ERROR,
                 is_autogenerated: bool = False):
        self.graph = graph
        self.graph.add_node(self)  # Registering self with parent graph

        self.is_autogenerated = is_autogenerated

        self._body = None
        self.bodies = bodies

        if len(self.bodies) == 1:
            self._body = bodies[0]

        self.inputs = inputs if inputs is not None else typing.OrderedDict()
        self.outputs = outputs if outputs is not None else typing.OrderedDict()

        idx = RealNode.get_next_index()
        if name is None:
            # set my name to the next unique available name
            self._name = ("node", idx)
        else:
            self._name = (name, idx)

        # Ports in/out to this node
        # Note that in_ports are always stored in the same order as inputs
        self.in_ports: typing.List[InPort] = []
        # Note that out_ports are always stored in the same order as outputs
        self.out_ports: typing.List[OutPort] = []

        self.log = make_logger(lvl,
                               f"{self.__class__.__name__} {self.full_name}")

        # See MessageLog for detail
        self._clock = 0

        self.out_names = list(self.outputs.keys())
        for out_name in self.out_names:
            if out_name in dir(self):
                raise NameError("Invalid out name: " +
                                out_name + " for node " + self.full_name)

    def __eq__(self, other) -> bool:
        """Equality, up to isomorphism as component of .df file.
        """
        if not isinstance(other, RealNode):
            return False

        if self._name[0] != other._name[0]:
            return False

        if self.inputs != other.inputs:
            return False

        if self.outputs != other.outputs:
            return False

        # Ports not checked due to recursion, check separately if needed

        return self.bodies == other.bodies

    def select_body(self,
                    exclusions: typing.List[object] = None,
                    preferred: typing.List[object] = None,
                    override=True):
        """Select a body for this node, out of the existing list of bodies
        attached to it.
        Exclusions take priority over preferences. If multiple choices
        remain the earliest added node will be selected.

        Parameters
        ----------
        exclusions : typing.List[object]
            typing.List of keys to exclude from selection
        preferred : typing.List[object]
            typing.List of keys to be preferred for selection if available
        override : Bool
            If true, then this selection will override existing selection
            Otherwise this selection will not occur if there has already
            been a selection

        Returns
        -------
        bool
            ``True`` if a ``body`` is defined at the end of selection.

        """
        exclusions = exclusions if exclusions is not None else []
        preferred = preferred if preferred is not None else []

        if override or not self._body:

            not_excluded_list = []
            for body in self.bodies:
                # If body has no excluded tag
                if not [tag for tag in exclusions if tag in body.access_tags]:
                    not_excluded_list.append(body)

            if not_excluded_list:
                prefered_list = []
                for body in not_excluded_list:
                    # If body has a preferred tag
                    if [tag for tag in preferred if tag in body.access_tags]:
                        prefered_list.append(body)

                if prefered_list:
                    self._body = prefered_list[0]
                else:
                    self._body = not_excluded_list[0]
            else:
                self._body = None
        return bool(self._body)

    def __str__(self) -> str:

        indent_prefix = "    "

        ret = 'ports:\n'
        if self.in_ports:
            ret += textwrap.indent(
                'in:\n' + textwrap.indent(
                    ''.join(
                        [f"{in_port}\n" for in_port in self.in_ports]
                    ),
                    prefix=indent_prefix),
                prefix=indent_prefix)
        if self.out_ports:
            ret += textwrap.indent(
                'out:\n' + textwrap.indent(
                    ''.join(
                        [f"{out_port}\n" for out_port in self.out_ports]
                    ),
                    prefix=indent_prefix),
                prefix=indent_prefix)

        ret += 'bodies:\n'
        if not self.bodies:
            ret += textwrap.indent('None\n\n', prefix=indent_prefix)
        else:
            ret += textwrap.indent(
                ''.join(
                    [
                        f'{"*" if body is self._body else ""}' +
                        f'{body.__class__.__name__}\n' +
                        textwrap.indent(
                            f'tags: {next(i for i in body.access_tags if i is not body.__class__)}\n',
                            prefix=indent_prefix
                        ) for body in self.bodies
                    ]
                ),
                prefix=indent_prefix
            ) + '\n'

        ret = f"node[{self.full_name}]:\n" + textwrap.indent(
            ret, prefix=indent_prefix
        )

        return ret

    def __repr__(self):
        return self.full_name

    def __getattr__(self, item):
        if item in self.out_names:
            return IndexProxyNode(self, item)
        elif len(self.outputs) == 0:
            raise AttributeError(
                f"Cannot fetch {item} from {self.full_name}, as we don't "
                f"have multiple outputs. Suggest using the node on its own."
            )
        elif item == '__del__':
            raise AttributeError('Migen tries to get this attribute')
        else:
            raise DeltaIOError(
                f"Node {self.full_name} has out ports {self.out_names}. "
                f"The requested out port \'{item}\' is not found. "
                "Please check the node's definition."
            )

    def add_out_port(self, port_destination: InPort, index=None):
        """Creates an out-port and adds it to my out-port store.

        Parameters
        ----------
        index : typing.Optional[str]
            Index corresponding to the name of this specficic out-port
            Can be None if there is only one out-port, then the index will be
            inferred in this method.
        port_destination : InPort
            The in-port that this out-port exports to.
        """
        if len(self.outputs) == 0:
            raise DeltaIOError(
                f"Cannot make an out-port on node {self.full_name} "
                "with no outputs.\n"
                "Please either add the proper outputs to the "
                "node definition or do not try to create an "
                "output data channel from this node."
            )

        if index is None:
            if len(self.outputs) > 1:
                raise ValueError("Non-subscripted output cannot be used when "
                                 "there is more than one output from a node.")
            index = list(self.outputs.keys())[0]

        if self.is_autogenerated:
            type_out = port_destination.port_type
        else:
            type_out = self.outputs[index]

        # Create new port and insert in correct position according to outputs
        new_port = OutPort(index, type_out, port_destination, self)
        new_port_i = list(self.outputs).index(new_port.index)
        for i, curr_port in enumerate(self.out_ports):
            curr_port_i = list(self.outputs).index(curr_port.index)
            if new_port_i < curr_port_i:
                self.out_ports.insert(i, new_port)
                break
        else:
            self.out_ports.append(new_port)

        # If this port is going into a port on a different graph,
        # flatten this graph into said graph
        into_graph = port_destination.node.graph
        if into_graph is not self.graph:
            into_graph.flatten(self.graph)

    def add_in_port(self, arg_name: str, in_type: typing.Type, in_port_size: int = 0):
        """Creates an in-port and adds it to my in-port store.

        Parameters
        ----------
        arg_name : str
            Name of the argument this port supplies.
        in_type : typing.Type
            The type that is expected to be supplied for this port.
        in_port_size: int
            Maximum size of the in ports.
            If 0 then size is unlimited.

        Returns
        -------
        InPort
            The created port.
        """
        my_port = InPort(arg_name, as_delta_type(in_type), self, in_port_size)
        self.in_ports.append(my_port)
        return my_port

    def _create_upstream_ports(self,
                               required_in_ports: typing.Dict[str, typing.Type],
                               given_nodes: typing.Dict[str, AbstractNode],
                               in_port_size: int = 0):
        """Create the ports going into this node and their
        corresponding out-ports.

        Parameters
        ----------
        required_in_ports
            Dictionary that describes what in-ports we want coming in to
            this node.
        given_nodes
            The nodes that are expected to send results to out in-ports.
        in_port_size
            The maximum size of the node's in ports. If 0 then unlimited size.
        """
        for arg_name, type_wanted in required_in_ports.items():
            if not arg_name in given_nodes:
                raise DeltaIOError(
                    f"Node {self.full_name} has a mismatch between defined and "
                    f"used input ports.\nArgument \'{arg_name}\' cannot be "
                    f"found in given ports {list(given_nodes.keys())}."
                )

            in_port = self.add_in_port(arg_name, type_wanted, in_port_size)
            given_nodes[arg_name].add_out_port(in_port, None)

    def _ports_from_arguments(self,
                              required_inputs,
                              pos_in_nodes,
                              kw_in_nodes,
                              in_port_size=0):
        """Manages the creation of upstream ports by getting the given nodes
        into the correct data structure.

        Parameters
        ----------
        required_inputs
            Dictionary that describes what in-ports we want coming in to
            this node.
        pos_in_nodes
            Nodes expected to send results to this node, specified
            positionally.
        kw_in_nodes
            Nodes expected to send results to this node, specified by keyword.
        in_port_size
            The maximum size of the node's in ports. If 0 then unlimited size.
        """
        # wrap the given positional in_nodes up with the name of the param they
        # satisfy
        pos_in_nodes_dict = {
            param_name: given_node
            for (given_node, (param_name, _)) in zip(
                pos_in_nodes,
                required_inputs.items()
            )
        }

        self._create_upstream_ports(required_inputs,
                                    {**pos_in_nodes_dict, **kw_in_nodes},
                                    in_port_size=in_port_size)

    @property
    def full_name(self) -> str:
        """Unique name of this node as string
        Made unique by appending an underscore followed by unique integer to
        user-specified name
        """
        return f"{self._name[0]}_{self._name[1]}"

    @property
    def name(self) -> str:
        """Non-unique name of this node as string
        """
        return self._name[0]

    @property
    def body(self):
        if self._body:
            return self._body
        else:
            if self.bodies:
                raise ValueError("Please call select_body on node "
                                 f"{self.full_name} before body access")
            else:
                raise ValueError("No bodies available for selection on node "
                                 f"{self.full_name}.")


class PythonNode(RealNode):
    """Parent Node type for all Python constructs.

    Attributes
    ----------
    in_queues : typing.Dict[str, DeltaQueue]
        Queues providing input(s).
    out_queues : typing.Dict[str, DeltaQueue]
        Queues consumins output(s).
    sig_stop : Event
        Communication channel through which a runtime simulator or a runtime
        signals `thread_worker` to stop.
    node_key : typing.Optional[str]
        Keyword argument used for providing the node to the block, included for
        debugging purposes.
    lvl : int
        Logging level for the node.
        By default only error logs are displayed.
    """

    def __init__(self,
                 graph,
                 bodies,
                 inputs: typing.OrderedDict[str, typing.Union[BaseDeltaType, Optional]],
                 pos_in_nodes,
                 kw_in_nodes,
                 in_port_size: int = 0,
                 node_key: typing.Optional[str] = None,
                 outputs: typing.OrderedDict[str, BaseDeltaType] = None,
                 name: str = None,
                 lvl: int = logging.ERROR,
                 is_autogenerated: bool = False):
        super().__init__(graph=graph,
                         bodies=bodies,
                         inputs=inputs,
                         outputs=outputs,
                         name=name,
                         lvl=lvl,
                         is_autogenerated=is_autogenerated)

        self._is_const = False

        const_body_no = sum(isinstance(body, PyConstBody) for body in bodies)
        if const_body_no > 0:
            if const_body_no == len(bodies):
                self._is_const = True
                for in_type in self.inputs.values():
                    if isinstance(in_type, Optional):
                        raise TypeError('Optional input is not allowed '
                                        'for constant nodes')
                    elif not isinstance(in_type, BaseDeltaType):
                        raise TypeError('Unsupported data type')
            else:
                raise ValueError("For a multi-body node, if one body is const "
                                 "all bodies must be const")

        self.in_queues: typing.Dict[str, DeltaQueue] = None
        self.out_queues: typing.Dict[str, DeltaQueue] = None
        self.sig_stop: Event = None
        self.node_key = node_key
        self.in_port_size = in_port_size

        self._ports_from_arguments(self.inputs, pos_in_nodes,
                                   kw_in_nodes, self.in_port_size)

    def set_communications(self, runtime: DeltaPySimulator):
        """Get the in and out queues relating to this node, as well as the
        utility events such as ``sig_stop`` from a runtime simulator or
        a runtime and save them in the instance.

        Parameters
        ----------
        runtime : DeltaPySimulator
            API of a runtime simulator or a runtime.
        """
        self.in_queues = runtime.in_queues[self.full_name]
        self.out_queues = runtime.out_queues[self.full_name]
        self.sig_stop = runtime.sig_stop

    def check_stop(self):
        """Check the stop signal, which can be set by a runtime simulator or
        a runtime. If set, it stops the current thread.
        """
        if self.sig_stop.is_set():
            self.log.info(f"Stopped {repr(self)}.")
            sys.exit()

    def receive(self, *args: str) -> typing.Union[typing.Dict[str, typing.Any], typing.Any]:
        """Receives the node's input(s) via in ports.

        Compulsory inputs block the further execution until the data is
        provided, whereas optional inputs do not block.

        Check if the node should stop.

        Parameters
        ----------
        args : str
            Optionally filter inputs. Only the specified ones will be received.

        Returns
        -------
        typing.Union[typing.Dict[str, typing.Any], typing.Any]
            If there is one input is specified via ``args`` it is received as
            an object, overwise the input values are returned as a dictionary.
        """
        # receive all or only selected inputs
        if args:
            in_queues = {name: in_q
                         for name, in_q in self.in_queues.items()
                         if name in args}
        else:
            in_queues = self.in_queues

        values = {}
        for name, in_q in in_queues.items():
            values[name] = in_q.get()
            if not isinstance(values[name], QueueMessage):
                raise TypeError(
                    f"Queue {in_q} from port {repr(in_q._src)} contained "
                    f"an item which was not a QueueMessage: {values[name]}"
                )
            self.msg_log.add_message(self.full_name, name, values[name])

        # logical clock update
        self._clock = max([self._clock] + [v.clk for v in values.values()])

        # unpack the inner msg
        val = {k: v.msg for k, v in values.items()}

        self.check_stop()

        if all(v is None for v in val.values()):
            # let the Python GIL take a look at the other threads
            sleep(1e-9)

        # if there is just one value to return, unpack it from the dict
        if len(val) == 1 and args:
            val = list(val.values())[0]

        if val:
            self.log.info(f"<- {val}")

        return val

    def _unpack_and_send(self, ret_to_send: typing.Union[object, typing.Tuple]):
        """Unpack a tuple-based return value and then send the output using the
        normal send method.

        Parameters
        ----------
        ret_to_send : typing.Union[object, typing.Tuple]
            Either a single value to be sent or a tuple of multiple values
            to be sent.
        """
        if ret_to_send is not None:
            if len(self.outputs) > 1 and hasattr(ret_to_send, '__iter__'):
                self.send(*ret_to_send)
            else:
                self.send(ret_to_send)

    def send(self, *args, **kwargs):
        """Sends the node's output(s) via out ports.

        If sending is blocked this method will lock the execution of the node
        until unblocked or a stop signal is raised, which shall stop
        the execution of the node.

        Parameters
        ----------
        ret : typing.Union[object, typing.Tuple]
            The return value. It is implied by construction that if it is a
            single object then the node has only one out port, otherwise
            a named tuple is used, with the names of the fields matching
            the names of the out ports.
        """

        def check_stop_send(out_q: DeltaQueue, message: QueueMessage):
            while True:
                try:
                    out_q.put(message)
                    self.check_stop()
                    break
                except Full:
                    self.check_stop()
                except:
                    raise

        # Log only non-trivial output(s)
        if self.log.isEnabledFor(logging.INFO):
            if args and not all(x is None for x in args):
                self.log.info(f"-> {args}")

            if kwargs and not all(v is None for _, v in kwargs.items()):
                self.log.info(f"-> {kwargs}")

        self._clock += 1

        if len(self.outputs) < len(args) + len(kwargs):
            raise ValueError(
                f"Node {self.full_name} tried to send too many values")

        positional_indicies = []
        for index, send_val in zip(self.outputs.keys(), args):
            positional_indicies.append(index)
            if index in self.out_queues:
                out_q = self.out_queues[index]
                check_stop_send(out_q, QueueMessage(send_val, clk=self._clock))

        for index, send_val in kwargs.items():
            if index not in self.outputs:
                raise NameError(f"Node {self.full_name} tried to send value with "
                                f"invalid keyword {index}")
            if index in positional_indicies:
                raise ValueError(f"Node {self.full_name} tried to send the same "
                                 "output positionaly and by keyword.")
            if index in self.out_queues:
                out_q = self.out_queues[index]
                check_stop_send(out_q, QueueMessage(send_val, clk=self._clock))

        # let the Python GIL take a look at the other threads
        sleep(1e-9)

    def thread_worker(self, runtime: DeltaPySimulator):
        """Run a regular Python node.

        Waits for input on all the mandatory inputs.
        Then, de-queues the optional inputs and executes its body.
        The output is written to the appropriate output queues.

        Parameters
        ----------
        runtime : DeltaPySimulator
            API of a runtime simulator or a runtime.
        """
        if isinstance(self.body, PyConstBody):
            # Thread worker should never run on a const body
            raise NotImplementedError

        if isinstance(self.body, PyInteractiveBody):
            self.log.debug(f"Running... {self}")
            self.body.eval(self)

        else:
            while True:
                values = self.receive()

                # If a node keyword has been specified for debugging then add
                # the node to the arguments.
                if self.node_key:
                    # the self arg is effectively a const message,
                    # so from time 0
                    values[self.node_key] = self

                self.log.debug("Running...")
                self._unpack_and_send(self.body.eval(**values))

    def run_once(self, runtime: DeltaPySimulator):
        """Compute the value of the node and pass it to the output queues.

        The output queues are
        :py:class:`ConstQueue<deltalanguage.runtime.ConstQueue>` that store
        this value and retrive a deepcopy at each request of the receiving
        node.
        This is done purely for optimisation purposes.

        Parameters
        ----------
        runtime : DeltaPySimulator
            API of a runtime simulator or a runtime.
        """
        self._unpack_and_send(self.body.eval())

    def capnp(self, capnp_node, capnp_bodies):
        """Generate ``capnp`` form of this node.

        Parameters
        ----------
        capnp_node
            The capnp object of this node.
        capnp_bodies
            List of bodies so we can check if a body is already serialised.
        """
        capnp_node.name = self.full_name
        capnp_node.init("bodies", len(self.bodies))

        for i_bod, bod in enumerate(self.bodies):

            body_impl = bod.as_serialised

            if isinstance(bod, PyMigenBody):
                body_id = 'migen'
                impl_type = 'verilog'
            elif isinstance(bod, PyInteractiveBody):
                body_id = 'interactive'
                impl_type = 'dillImpl'
            else:
                body_id = 'python'
                impl_type = 'dillImpl'

            def get_body_impl(body, impl_type=impl_type, body_id=body_id):
                return body.__getattr__(body_id).__getattr__(impl_type)

            def set_body_impl(body, impl, impl_type=impl_type, body_id=body_id):
                body.__getattr__(body_id).__setattr__(impl_type, impl)

            for i, body in enumerate(capnp_bodies):
                if body.which() == body_id:
                    if get_body_impl(body) == body_impl:
                        capnp_node.bodies[i_bod] = i
                        break
            else:
                body = capnp_bodies.add()
                body.init(body_id)
                set_body_impl(body, body_impl)
                body.tags = dill.dumps(bod.access_tags)
                capnp_node.bodies[i_bod] = len(capnp_bodies) - 1

        # 2. save I/O ports
        self.capnp_ports(capnp_node)

    def capnp_ports(self, capnp_node):
        """Helper method, generates capnp for in/out ports of the node.

        Parameters
        ----------
        capnp_node
            The node of the interest.
        """
        in_ports = capnp_node.init("inPorts", len(self.in_ports))
        for capnp_in_port, in_port in zip(in_ports, self.in_ports):
            in_port.capnp(capnp_in_port)

        out_ports = capnp_node.init("outPorts", len(self.out_ports))
        for capnp_out_port, out_port in zip(out_ports, self.out_ports):
            out_port.capnp(capnp_out_port)

    def capnp_wiring(self, capnp_nodes, capnp_wiring):
        """Generate capnp form of this node's wires.

        Parameters
        ----------
        capnp_nodes
            List of nodes so indexes can be found.
        capnp_wiring
            List of wires so we can add our relevant wires.
        """
        for i, capnp_node in enumerate(capnp_nodes):
            if capnp_node.name == self.full_name:
                capnp_node_index = i
                break

        for i, out_port in enumerate(self.out_ports):
            capnp_wire = capnp_wiring.add()
            capnp_wire.srcNode = capnp_node_index
            capnp_wire.srcOutPort = i
            out_port.capnp_wiring(capnp_nodes, capnp_wire)

    def set_msg_log(self, msg_log: MessageLog):
        """Sets the log for messages received.

        Parameters
        ----------
        msg_log : MessageLog
            Instance of the message log.
        """
        self.msg_log = msg_log

    def add_body(self, body: typing.Union[BodyTemplate, typing.Callable]):
        """Add some body to this node, using a ``BodyTemplate`` as a recipe
        for how to construct the body to add. This will also enable checks
        about body compatibility to be made.

        Parameters
        ----------
        body : typing.Union[BodyTemplate, typing.Callable]
            ``BodyTemplate`` to add or some object for which
            ``template`` is defined, where ``template`` is a ``BodyTemplate``
        """
        if self.is_const():
            raise NotImplementedError("Adding a body to a const node "
                                      "post-creation is currently unsupported")

        template = body
        if not isinstance(template, BodyTemplate):
            template = body.template

        if template.compatible(self.node_key, self.in_port_size,
                               self.inputs, self.outputs):
            if isinstance(template, MethodBodyTemplate):
                self.bodies.append(template.construct_body(body.__self__))
            else:
                self.bodies.append(template.construct_body())
            self.select_body(override=False)

    def is_const(self):
        """Method to return state of self._is_const.

        Returns
        -------
        bool
            If the current body is constant or not.
        """
        return self._is_const


def as_node(potential_node: typing.Union[AbstractNode, object],
            graph: DeltaGraph) -> PythonNode:
    """Ensures argument is a node and if not makes it into a constant node.

    Parameters
    ----------
    potential_node : typing.Union[AbstractNode, object]
        Node that could be a node or not.
    graph : DeltaGraph
        Graph the node would be in

    Returns
    -------
    PythonNode
        Made for potential_node or potential_node as it was already a node.
    """
    if isinstance(potential_node, AbstractNode):
        return potential_node
    else:
        return PythonNode(
            graph,
            [PyConstBody(lambda: potential_node, value=potential_node)],
            {},
            [],
            {},
            outputs=typing.OrderedDict(
                [('output', delta_type(potential_node))]),
            is_autogenerated=True
        )


def get_func_inputs_outputs(
    a_func: typing.Callable,
    is_method: bool,
    node_key: typing.Optional[str] = None,
    dec_outputs: typing.List[typing.Tuple[str, typing.Type]] = None
) -> typing.Tuple[typing.OrderedDict[str, typing.Union[BaseDeltaType, Optional]],
                  typing.OrderedDict[str, BaseDeltaType]]:
    """Helper function to extract input and output types of a node function.

    Parameters
    ----------
    a_func : typing.Callable
        The function to analyse.
    is_method : bool
        Flag to specify if function is a class method.
    node_key : typing.Optional[str]
        Keyword argument used for providing the node to the block, included for
        some logic purposes.
    dec_outputs : typing.List[typing.Tuple[str, typing.Type]]
        outputs list if specified in a decorator

    Returns
    -------
    typing.OrderedDict[str, typing.Union[BaseDeltaType, Optional]]
        Types of the in parameters.
    typing.OrderedDict[str, BaseDeltaType]]
        Type of the output the node to be made.

    Raises
    ------
    TypeError
        Raised if either the input or output types aren't specified in the
        function signature.
    """
    dec_outputs = dec_outputs if dec_outputs is not None else []

    in_annotation = signature(a_func).parameters
    out_annotation = signature(a_func).return_annotation

    if out_annotation == Void:
        out_annotation = _empty

    inputs = typing.OrderedDict()
    for i, (arg_name, arg_param) in enumerate(in_annotation.items()):

        # first argument should always be 'self' for a method
        if i == 0 and is_method:
            continue

        if arg_param.annotation == _empty:
            raise TypeError(
                "Must specify the type of argument " +
                f"'{arg_name}' as annotation in " +
                f"function '{a_func.__name__}'"
            )
        inputs[arg_name] = arg_param.annotation

    if out_annotation == _empty or dec_outputs:
        outputs = dec_outputs
    else:
        outputs = [('output', out_annotation)]

    inputs = inputs_as_delta_types(inputs, node_key)
    outputs = outputs_as_delta_types(typing.OrderedDict(outputs))

    return inputs, outputs


def inputs_as_delta_types(
    inputs: typing.OrderedDict[str, typing.Type],
    node_key: typing.Optional[str] = None
) -> typing.OrderedDict[str, typing.Union[BaseDeltaType, Optional]]:
    """Take ``inputs`` and convert to DeltaTypes, raising appropriate
    errors, skips and removes ``node_key``.

    Parameters
    ----------
    inputs : typing.OrderedDict[str, typing.Type]
        Dictionary of inputs that need to become DeltaTypes
    node_key : typing.Optional[str], optional
        Node key for node param if available, by default None

    Returns
    -------
    typing.Dict[str, typing.Union[BaseDeltaType, Optional]]
        typing.Dictionary of inputs as DeltaTypes
    """
    for in_name, in_type in inputs.items():

        if node_key == in_name and in_type == PythonNode:
            inputs.pop(node_key)
            continue

        delta_type_in = as_delta_type(in_type)
        if not isinstance(delta_type_in, (BaseDeltaType, Optional)):
            raise DeltaTypeError(f"Unsupported in type = {in_type}")

        inputs[in_name] = delta_type_in
    return inputs


def outputs_as_delta_types(
    outputs: typing.OrderedDict[str, typing.Type],
) -> typing.OrderedDict[str, BaseDeltaType]:
    """Take ``outputs`` and convert to DeltaTypes, raising appropriate
    errors.

    Parameters
    ----------
    outputs : OrderedDict[str, Type]
        Dictionary of output types that need to become DeltaTypes

    Returns
    -------
    Dict[str, BaseDeltaType]
        Dictionary of outputs as DeltaTypes
    """
    for out_name, out_type in outputs.items():

        delta_type_out = as_delta_type(out_type)
        if not isinstance(delta_type_out, BaseDeltaType):
            raise DeltaTypeError(f"Unsupported out type = {out_type}")

        outputs[out_name] = delta_type_out
    return outputs
