import torch

from torch_grid_utils import fftfreq_grid


def test_fftfreq_grid_2d_basic():
    grid = fftfreq_grid(
        image_shape=(4, 4),
        rfft=False,
    )
    assert tuple(grid.shape) == (4, 4, 2)  # (image_h, image_w, freq_hw)
    expected = torch.tensor([[[0.0000, 0.0000],
                              [0.0000, 0.2500],
                              [0.0000, -0.5000],
                              [0.0000, -0.2500]],

                             [[0.2500, 0.0000],
                              [0.2500, 0.2500],
                              [0.2500, -0.5000],
                              [0.2500, -0.2500]],

                             [[-0.5000, 0.0000],
                              [-0.5000, 0.2500],
                              [-0.5000, -0.5000],
                              [-0.5000, -0.2500]],

                             [[-0.2500, 0.0000],
                              [-0.2500, 0.2500],
                              [-0.2500, -0.5000],
                              [-0.2500, -0.2500]]])
    assert torch.allclose(grid, expected)


def test_fftfreq_grid_2d_rfft():
    grid = fftfreq_grid(
        image_shape=(4, 4),
        rfft=True,
    )
    assert tuple(grid.shape) == (4, 3, 2)  # (image_h, image_w_rfft, freq_hw)
    expected = torch.tensor([[[0.0000, 0.0000],
                              [0.0000, 0.2500],
                              [0.0000, 0.5000]],

                             [[0.2500, 0.0000],
                              [0.2500, 0.2500],
                              [0.2500, 0.5000]],

                             [[-0.5000, 0.0000],
                              [-0.5000, 0.2500],
                              [-0.5000, 0.5000]],

                             [[-0.2500, 0.0000],
                              [-0.2500, 0.2500],
                              [-0.2500, 0.5000]]])
    assert torch.allclose(grid, expected)


def test_fftfreq_grid_2d_with_spacing():
    grid = fftfreq_grid(
        image_shape=(4, 4),
        rfft=False,
        spacing=2,
    )
    assert tuple(grid.shape) == (4, 4, 2)  # (image_h, image_w, freq_hw)
    expected = torch.tensor([[[0.0000, 0.0000],
                              [0.0000, 0.1250],
                              [0.0000, -0.2500],
                              [0.0000, -0.1250]],

                             [[0.1250, 0.0000],
                              [0.1250, 0.1250],
                              [0.1250, -0.2500],
                              [0.1250, -0.1250]],

                             [[-0.2500, 0.0000],
                              [-0.2500, 0.1250],
                              [-0.2500, -0.2500],
                              [-0.2500, -0.1250]],

                             [[-0.1250, 0.0000],
                              [-0.1250, 0.1250],
                              [-0.1250, -0.2500],
                              [-0.1250, -0.1250]]])
    assert torch.allclose(grid, expected)


def test_fftfreq_grid_2d_with_norm():
    grid = fftfreq_grid(
        image_shape=(4, 4),
        rfft=False,
        norm=True,
    )
    assert tuple(grid.shape) == (4, 4)  # (image_h, image_w, freq_hw)
    expected = torch.tensor([[0.0000, 0.2500, 0.5000, 0.2500],
                             [0.2500, 0.3536, 0.5590, 0.3536],
                             [0.5000, 0.5590, 0.7071, 0.5590],
                             [0.2500, 0.3536, 0.5590, 0.3536]])
    assert torch.allclose(grid, expected, atol=1e-4)


def test_fftfreq_grid_2d_with_fftshift():
    grid = fftfreq_grid(
        image_shape=(4, 4),
        rfft=False,
        fftshift=True,
    )
    assert tuple(grid.shape) == (4, 4, 2)
    expected = torch.tensor([[[-0.5000, -0.5000],
                              [-0.5000, -0.2500],
                              [-0.5000, 0.0000],
                              [-0.5000, 0.2500]],

                             [[-0.2500, -0.5000],
                              [-0.2500, -0.2500],
                              [-0.2500, 0.0000],
                              [-0.2500, 0.2500]],

                             [[0.0000, -0.5000],
                              [0.0000, -0.2500],
                              [0.0000, 0.0000],
                              [0.0000, 0.2500]],

                             [[0.2500, -0.5000],
                              [0.2500, -0.2500],
                              [0.2500, 0.0000],
                              [0.2500, 0.2500]]])
    assert torch.allclose(grid, expected, atol=1e-4)


def test_fftfreq_grid_3d_basic():
    grid = fftfreq_grid(
        image_shape=(4, 4, 4),
        rfft=False,
    )
    assert tuple(grid.shape) == (4, 4, 4, 3)  # (image_d, image_h, image_w, freq_dhw)
    expected = torch.tensor([[[[0.0000, 0.0000, 0.0000],
                               [0.0000, 0.0000, 0.2500],
                               [0.0000, 0.0000, -0.5000],
                               [0.0000, 0.0000, -0.2500]],

                              [[0.0000, 0.2500, 0.0000],
                               [0.0000, 0.2500, 0.2500],
                               [0.0000, 0.2500, -0.5000],
                               [0.0000, 0.2500, -0.2500]],

                              [[0.0000, -0.5000, 0.0000],
                               [0.0000, -0.5000, 0.2500],
                               [0.0000, -0.5000, -0.5000],
                               [0.0000, -0.5000, -0.2500]],

                              [[0.0000, -0.2500, 0.0000],
                               [0.0000, -0.2500, 0.2500],
                               [0.0000, -0.2500, -0.5000],
                               [0.0000, -0.2500, -0.2500]]],

                             [[[0.2500, 0.0000, 0.0000],
                               [0.2500, 0.0000, 0.2500],
                               [0.2500, 0.0000, -0.5000],
                               [0.2500, 0.0000, -0.2500]],

                              [[0.2500, 0.2500, 0.0000],
                               [0.2500, 0.2500, 0.2500],
                               [0.2500, 0.2500, -0.5000],
                               [0.2500, 0.2500, -0.2500]],

                              [[0.2500, -0.5000, 0.0000],
                               [0.2500, -0.5000, 0.2500],
                               [0.2500, -0.5000, -0.5000],
                               [0.2500, -0.5000, -0.2500]],

                              [[0.2500, -0.2500, 0.0000],
                               [0.2500, -0.2500, 0.2500],
                               [0.2500, -0.2500, -0.5000],
                               [0.2500, -0.2500, -0.2500]]],

                             [[[-0.5000, 0.0000, 0.0000],
                               [-0.5000, 0.0000, 0.2500],
                               [-0.5000, 0.0000, -0.5000],
                               [-0.5000, 0.0000, -0.2500]],

                              [[-0.5000, 0.2500, 0.0000],
                               [-0.5000, 0.2500, 0.2500],
                               [-0.5000, 0.2500, -0.5000],
                               [-0.5000, 0.2500, -0.2500]],

                              [[-0.5000, -0.5000, 0.0000],
                               [-0.5000, -0.5000, 0.2500],
                               [-0.5000, -0.5000, -0.5000],
                               [-0.5000, -0.5000, -0.2500]],

                              [[-0.5000, -0.2500, 0.0000],
                               [-0.5000, -0.2500, 0.2500],
                               [-0.5000, -0.2500, -0.5000],
                               [-0.5000, -0.2500, -0.2500]]],

                             [[[-0.2500, 0.0000, 0.0000],
                               [-0.2500, 0.0000, 0.2500],
                               [-0.2500, 0.0000, -0.5000],
                               [-0.2500, 0.0000, -0.2500]],

                              [[-0.2500, 0.2500, 0.0000],
                               [-0.2500, 0.2500, 0.2500],
                               [-0.2500, 0.2500, -0.5000],
                               [-0.2500, 0.2500, -0.2500]],

                              [[-0.2500, -0.5000, 0.0000],
                               [-0.2500, -0.5000, 0.2500],
                               [-0.2500, -0.5000, -0.5000],
                               [-0.2500, -0.5000, -0.2500]],

                              [[-0.2500, -0.2500, 0.0000],
                               [-0.2500, -0.2500, 0.2500],
                               [-0.2500, -0.2500, -0.5000],
                               [-0.2500, -0.2500, -0.2500]]]])
    assert torch.allclose(grid, expected)


def test_fftfreq_grid_3d_rfft():
    grid = fftfreq_grid(
        image_shape=(4, 4, 4),
        rfft=True,
    )
    assert tuple(grid.shape) == (4, 4, 3, 3)  # (image_d, image_h, image_w_rfft, freq_dhw)
    expected = torch.tensor([[[[0.0000, 0.0000, 0.0000],
                               [0.0000, 0.0000, 0.2500],
                               [0.0000, 0.0000, 0.5000]],

                              [[0.0000, 0.2500, 0.0000],
                               [0.0000, 0.2500, 0.2500],
                               [0.0000, 0.2500, 0.5000]],

                              [[0.0000, -0.5000, 0.0000],
                               [0.0000, -0.5000, 0.2500],
                               [0.0000, -0.5000, 0.5000]],

                              [[0.0000, -0.2500, 0.0000],
                               [0.0000, -0.2500, 0.2500],
                               [0.0000, -0.2500, 0.5000]]],

                             [[[0.2500, 0.0000, 0.0000],
                               [0.2500, 0.0000, 0.2500],
                               [0.2500, 0.0000, 0.5000]],

                              [[0.2500, 0.2500, 0.0000],
                               [0.2500, 0.2500, 0.2500],
                               [0.2500, 0.2500, 0.5000]],

                              [[0.2500, -0.5000, 0.0000],
                               [0.2500, -0.5000, 0.2500],
                               [0.2500, -0.5000, 0.5000]],

                              [[0.2500, -0.2500, 0.0000],
                               [0.2500, -0.2500, 0.2500],
                               [0.2500, -0.2500, 0.5000]]],

                             [[[-0.5000, 0.0000, 0.0000],
                               [-0.5000, 0.0000, 0.2500],
                               [-0.5000, 0.0000, 0.5000]],

                              [[-0.5000, 0.2500, 0.0000],
                               [-0.5000, 0.2500, 0.2500],
                               [-0.5000, 0.2500, 0.5000]],

                              [[-0.5000, -0.5000, 0.0000],
                               [-0.5000, -0.5000, 0.2500],
                               [-0.5000, -0.5000, 0.5000]],

                              [[-0.5000, -0.2500, 0.0000],
                               [-0.5000, -0.2500, 0.2500],
                               [-0.5000, -0.2500, 0.5000]]],

                             [[[-0.2500, 0.0000, 0.0000],
                               [-0.2500, 0.0000, 0.2500],
                               [-0.2500, 0.0000, 0.5000]],

                              [[-0.2500, 0.2500, 0.0000],
                               [-0.2500, 0.2500, 0.2500],
                               [-0.2500, 0.2500, 0.5000]],

                              [[-0.2500, -0.5000, 0.0000],
                               [-0.2500, -0.5000, 0.2500],
                               [-0.2500, -0.5000, 0.5000]],

                              [[-0.2500, -0.2500, 0.0000],
                               [-0.2500, -0.2500, 0.2500],
                               [-0.2500, -0.2500, 0.5000]]]])
    assert torch.allclose(grid, expected)


def test_fftfreq_grid_3d_with_spacing():
    grid = fftfreq_grid(
        image_shape=(4, 4, 4),
        rfft=False,
        spacing=2,
    )
    assert tuple(grid.shape) == (4, 4, 4, 3)  # (image_h, image_w, freq_hw)
    expected = torch.tensor([[[[0.0000, 0.0000, 0.0000],
                               [0.0000, 0.0000, 0.1250],
                               [0.0000, 0.0000, -0.2500],
                               [0.0000, 0.0000, -0.1250]],

                              [[0.0000, 0.1250, 0.0000],
                               [0.0000, 0.1250, 0.1250],
                               [0.0000, 0.1250, -0.2500],
                               [0.0000, 0.1250, -0.1250]],

                              [[0.0000, -0.2500, 0.0000],
                               [0.0000, -0.2500, 0.1250],
                               [0.0000, -0.2500, -0.2500],
                               [0.0000, -0.2500, -0.1250]],

                              [[0.0000, -0.1250, 0.0000],
                               [0.0000, -0.1250, 0.1250],
                               [0.0000, -0.1250, -0.2500],
                               [0.0000, -0.1250, -0.1250]]],

                             [[[0.1250, 0.0000, 0.0000],
                               [0.1250, 0.0000, 0.1250],
                               [0.1250, 0.0000, -0.2500],
                               [0.1250, 0.0000, -0.1250]],

                              [[0.1250, 0.1250, 0.0000],
                               [0.1250, 0.1250, 0.1250],
                               [0.1250, 0.1250, -0.2500],
                               [0.1250, 0.1250, -0.1250]],

                              [[0.1250, -0.2500, 0.0000],
                               [0.1250, -0.2500, 0.1250],
                               [0.1250, -0.2500, -0.2500],
                               [0.1250, -0.2500, -0.1250]],

                              [[0.1250, -0.1250, 0.0000],
                               [0.1250, -0.1250, 0.1250],
                               [0.1250, -0.1250, -0.2500],
                               [0.1250, -0.1250, -0.1250]]],

                             [[[-0.2500, 0.0000, 0.0000],
                               [-0.2500, 0.0000, 0.1250],
                               [-0.2500, 0.0000, -0.2500],
                               [-0.2500, 0.0000, -0.1250]],

                              [[-0.2500, 0.1250, 0.0000],
                               [-0.2500, 0.1250, 0.1250],
                               [-0.2500, 0.1250, -0.2500],
                               [-0.2500, 0.1250, -0.1250]],

                              [[-0.2500, -0.2500, 0.0000],
                               [-0.2500, -0.2500, 0.1250],
                               [-0.2500, -0.2500, -0.2500],
                               [-0.2500, -0.2500, -0.1250]],

                              [[-0.2500, -0.1250, 0.0000],
                               [-0.2500, -0.1250, 0.1250],
                               [-0.2500, -0.1250, -0.2500],
                               [-0.2500, -0.1250, -0.1250]]],

                             [[[-0.1250, 0.0000, 0.0000],
                               [-0.1250, 0.0000, 0.1250],
                               [-0.1250, 0.0000, -0.2500],
                               [-0.1250, 0.0000, -0.1250]],

                              [[-0.1250, 0.1250, 0.0000],
                               [-0.1250, 0.1250, 0.1250],
                               [-0.1250, 0.1250, -0.2500],
                               [-0.1250, 0.1250, -0.1250]],

                              [[-0.1250, -0.2500, 0.0000],
                               [-0.1250, -0.2500, 0.1250],
                               [-0.1250, -0.2500, -0.2500],
                               [-0.1250, -0.2500, -0.1250]],

                              [[-0.1250, -0.1250, 0.0000],
                               [-0.1250, -0.1250, 0.1250],
                               [-0.1250, -0.1250, -0.2500],
                               [-0.1250, -0.1250, -0.1250]]]])
    assert torch.allclose(grid, expected)


def test_fftfreq_grid_3d_with_norm():
    grid = fftfreq_grid(
        image_shape=(4, 4, 4),
        rfft=False,
        norm=True,
    )
    assert tuple(grid.shape) == (4, 4, 4)  # (image_h, image_w, freq_hw)
    expected = torch.tensor([[[0.0000, 0.2500, 0.5000, 0.2500],
                              [0.2500, 0.3536, 0.5590, 0.3536],
                              [0.5000, 0.5590, 0.7071, 0.5590],
                              [0.2500, 0.3536, 0.5590, 0.3536]],

                             [[0.2500, 0.3536, 0.5590, 0.3536],
                              [0.3536, 0.4330, 0.6124, 0.4330],
                              [0.5590, 0.6124, 0.7500, 0.6124],
                              [0.3536, 0.4330, 0.6124, 0.4330]],

                             [[0.5000, 0.5590, 0.7071, 0.5590],
                              [0.5590, 0.6124, 0.7500, 0.6124],
                              [0.7071, 0.7500, 0.8660, 0.7500],
                              [0.5590, 0.6124, 0.7500, 0.6124]],

                             [[0.2500, 0.3536, 0.5590, 0.3536],
                              [0.3536, 0.4330, 0.6124, 0.4330],
                              [0.5590, 0.6124, 0.7500, 0.6124],
                              [0.3536, 0.4330, 0.6124, 0.4330]]])
    assert torch.allclose(grid, expected, atol=1e-4)


def test_fftfreq_grid_3d_with_fftshift():
    grid = fftfreq_grid(
        image_shape=(4, 4, 4),
        rfft=False,
        fftshift=True,
    )
    assert tuple(grid.shape) == (4, 4, 4, 3)
    expected = torch.tensor([[[[-0.5000, -0.5000, -0.5000],
                               [-0.5000, -0.5000, -0.2500],
                               [-0.5000, -0.5000, 0.0000],
                               [-0.5000, -0.5000, 0.2500]],

                              [[-0.5000, -0.2500, -0.5000],
                               [-0.5000, -0.2500, -0.2500],
                               [-0.5000, -0.2500, 0.0000],
                               [-0.5000, -0.2500, 0.2500]],

                              [[-0.5000, 0.0000, -0.5000],
                               [-0.5000, 0.0000, -0.2500],
                               [-0.5000, 0.0000, 0.0000],
                               [-0.5000, 0.0000, 0.2500]],

                              [[-0.5000, 0.2500, -0.5000],
                               [-0.5000, 0.2500, -0.2500],
                               [-0.5000, 0.2500, 0.0000],
                               [-0.5000, 0.2500, 0.2500]]],

                             [[[-0.2500, -0.5000, -0.5000],
                               [-0.2500, -0.5000, -0.2500],
                               [-0.2500, -0.5000, 0.0000],
                               [-0.2500, -0.5000, 0.2500]],

                              [[-0.2500, -0.2500, -0.5000],
                               [-0.2500, -0.2500, -0.2500],
                               [-0.2500, -0.2500, 0.0000],
                               [-0.2500, -0.2500, 0.2500]],

                              [[-0.2500, 0.0000, -0.5000],
                               [-0.2500, 0.0000, -0.2500],
                               [-0.2500, 0.0000, 0.0000],
                               [-0.2500, 0.0000, 0.2500]],

                              [[-0.2500, 0.2500, -0.5000],
                               [-0.2500, 0.2500, -0.2500],
                               [-0.2500, 0.2500, 0.0000],
                               [-0.2500, 0.2500, 0.2500]]],

                             [[[0.0000, -0.5000, -0.5000],
                               [0.0000, -0.5000, -0.2500],
                               [0.0000, -0.5000, 0.0000],
                               [0.0000, -0.5000, 0.2500]],

                              [[0.0000, -0.2500, -0.5000],
                               [0.0000, -0.2500, -0.2500],
                               [0.0000, -0.2500, 0.0000],
                               [0.0000, -0.2500, 0.2500]],

                              [[0.0000, 0.0000, -0.5000],
                               [0.0000, 0.0000, -0.2500],
                               [0.0000, 0.0000, 0.0000],
                               [0.0000, 0.0000, 0.2500]],

                              [[0.0000, 0.2500, -0.5000],
                               [0.0000, 0.2500, -0.2500],
                               [0.0000, 0.2500, 0.0000],
                               [0.0000, 0.2500, 0.2500]]],

                             [[[0.2500, -0.5000, -0.5000],
                               [0.2500, -0.5000, -0.2500],
                               [0.2500, -0.5000, 0.0000],
                               [0.2500, -0.5000, 0.2500]],

                              [[0.2500, -0.2500, -0.5000],
                               [0.2500, -0.2500, -0.2500],
                               [0.2500, -0.2500, 0.0000],
                               [0.2500, -0.2500, 0.2500]],

                              [[0.2500, 0.0000, -0.5000],
                               [0.2500, 0.0000, -0.2500],
                               [0.2500, 0.0000, 0.0000],
                               [0.2500, 0.0000, 0.2500]],

                              [[0.2500, 0.2500, -0.5000],
                               [0.2500, 0.2500, -0.2500],
                               [0.2500, 0.2500, 0.0000],
                               [0.2500, 0.2500, 0.2500]]]])
    assert torch.allclose(grid, expected, atol=1e-4)
