# SPDX-FileCopyrightText: 2020 Diego Elio Pettenò
#
# SPDX-License-Identifier: MIT

import unittest

from pysdrc import decoder


class SIRCDecoderTest(unittest.TestCase):
    def test_simple(self):
        pulses = [
            2442,
            560,
            623,
            575,
            1221,
            577,
            617,
            582,
            622,
            577,
            1219,
            579,
            625,
            574,
            620,
            578,
            1218,
            580,
            624,
            575,
            619,
            581,
            623,
            575,
            618,
        ]
        self.assertEqual(decoder.decode_sirc(pulses), (18, 1))

    def test_ideal(self):
        pulses = [
            2400,
            600,
            600,
            600,
            1200,
            600,
            600,
            600,
            600,
            600,
            1200,
            600,
            600,
            600,
            600,
            600,
            1200,
            600,
            600,
            600,
            600,
            600,
            600,
            600,
            600,
        ]
        self.assertEqual(decoder.decode_sirc(pulses), (18, 1))

    def test_bad_header(self):
        pulses = [
            200000,
            600,
            600,
            600,
            1200,
            600,
            600,
            600,
            600,
            600,
            1200,
            600,
            600,
            600,
            600,
            600,
            1200,
            600,
            600,
            600,
            600,
            600,
            600,
            600,
            600,
        ]
        with self.assertRaises(decoder.DecodeException):
            decoder.decode_sirc(pulses)

    def test_short(self):
        pulses = [
            2400,
            600,
            600,
        ]
        with self.assertRaises(decoder.DecodeException):
            decoder.decode_sirc(pulses)

    def test_bad_length(self):
        pulses = [
            2400,
            600,
            600,
            600,
            1200,
            600,
            600,
            600,
            600,
            600,
            1200,
            600,
            600,
            600,
            600,
            600,
            1200,
            600,
            600,
            600,
            600,
            600,
        ]
        with self.assertRaises(decoder.DecodeException):
            decoder.decode_sirc(pulses)


class NECDecoderTest(unittest.TestCase):
    def test_simple(self):
        pulses = [
            9155,
            4505,
            604,
            536,
            596,
            548,
            595,
            541,
            601,
            539,
            604,
            536,
            606,
            534,
            598,
            542,
            600,
            1650,
            603,
            1648,
            596,
            1653,
            600,
            1650,
            603,
            1647,
            596,
            1654,
            600,
            1649,
            604,
            1646,
            597,
            543,
            599,
            1651,
            601,
            538,
            605,
            1646,
            597,
            542,
            600,
            540,
            603,
            537,
            604,
            536,
            608,
            533,
            597,
            543,
            600,
            1650,
            603,
            537,
            605,
            1645,
            599,
            1651,
            603,
            1648,
            604,
            1644,
            600,
            1650,
            602,
        ]

        self.assertEqual(decoder.decode_nec(pulses), (128, 5))

    def test_invalid(self):
        pulses = [
            9155,
            4505,
            604,
            536,
            596,
            548,
            595,
            541,
            601,
            539,
            604,
            536,
            606,
            534,
            598,
            542,
            600,
            1650,
            603,
            1648,
            596,
            1653,
            600,
            1650,
            603,
            1647,
            596,
            1654,
            600,
            1649,
            604,
            1646,
            597,
            543,
            599,
            1651,
            601,
            538,
            605,
            1646,
            597,
            542,
            600,
            540,
            603,
            537,
            604,
            536,
            608,
            533,
            597,
            543,
            600,
            1650,
            603,
            537,
            605,
            1645,
            599,
            1651,
            603,
            1648,
            604,
            600,
            600,
            1650,
            602,
        ]

        with self.assertRaises(decoder.NECDecodeException):
            decoder.decode_nec(pulses)
