"use strict";
/**
 *  Copyright 2020 Amazon.com, Inc. or its affiliates. All Rights Reserved.
 *
 *  Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance
 *  with the License. A copy of the License is located at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  or in the 'license' file accompanying this file. This file is distributed on an 'AS IS' BASIS, WITHOUT WARRANTIES
 *  OR CONDITIONS OF ANY KIND, express or implied. See the License for the specific language governing permissions
 *  and limitations under the License.
 */
Object.defineProperty(exports, "__esModule", { value: true });
exports.KinesisStreamsToKinesisFirehoseToS3 = void 0;
const kdfToS3 = require("@aws-solutions-constructs/aws-kinesisfirehose-s3");
const core_1 = require("@aws-cdk/core");
const defaults = require("@aws-solutions-constructs/core");
const iam = require("@aws-cdk/aws-iam");
const core_2 = require("@aws-solutions-constructs/core");
class KinesisStreamsToKinesisFirehoseToS3 extends core_1.Construct {
    /**
     * @param scope - represents the scope for all the resources.
     * @param id - this is a a scope-unique id.
     * @param props - user provided props for the construct.
     * @summary Constructs a new instance of the KinesisStreamsToKinesisFirehoseToS3 class.
     * @since 1.68.0
     * @access public
     */
    constructor(scope, id, props) {
        super(scope, id);
        // Setup the Kinesis Stream
        this.kinesisStream = defaults.buildKinesisStream(this, {
            existingStreamObj: props.existingStreamObj,
            kinesisStreamProps: props.kinesisStreamProps
        });
        const kinesisStreamsRole = new iam.Role(scope, 'KinesisStreamsRole', {
            assumedBy: new iam.ServicePrincipal('firehose.amazonaws.com'),
            inlinePolicies: {
                KinesisStreamsRoleRolePolicy: new iam.PolicyDocument({
                    statements: [new iam.PolicyStatement({
                            actions: [
                                "kinesis:DescribeStream",
                                "kinesis:GetShardIterator",
                                "kinesis:GetRecords",
                                "kinesis:ListShards"
                            ],
                            resources: [this.kinesisStream.streamArn]
                        })]
                })
            }
        });
        // This Construct requires that the deliveryStreamType be overriden regardless of what is specified in the user props
        if (props.kinesisFirehoseProps) {
            if (props.kinesisFirehoseProps.deliveryStreamType !== undefined) {
                defaults.printWarning('Overriding deliveryStreamType type to be KinesisStreamAsSource');
            }
            if (props.kinesisFirehoseProps.kinesisStreamSourceConfiguration !== undefined) {
                defaults.printWarning('Overriding kinesisStreamSourceConfiguration');
            }
        }
        const _kinesisFirehoseProps = {
            deliveryStreamType: 'KinesisStreamAsSource',
            kinesisStreamSourceConfiguration: {
                kinesisStreamArn: this.kinesisStream.streamArn,
                roleArn: kinesisStreamsRole.roleArn
            }
        };
        const kdfToS3Construct = new kdfToS3.KinesisFirehoseToS3(this, 'KinesisFirehoseToS3', {
            kinesisFirehoseProps: core_2.overrideProps(props.kinesisFirehoseProps, _kinesisFirehoseProps),
            existingBucketObj: props.existingBucketObj,
            bucketProps: props.bucketProps
        });
        this.kinesisFirehose = kdfToS3Construct.kinesisFirehose;
        this.kinesisFirehoseRole = kdfToS3Construct.kinesisFirehoseRole;
        this.kinesisFirehoseLogGroup = kdfToS3Construct.kinesisFirehoseLogGroup;
        if (props.createCloudWatchAlarms === undefined || props.createCloudWatchAlarms) {
            // Deploy best practices CW Alarms for Kinesis Stream
            this.cloudwatchAlarms = defaults.buildKinesisStreamCWAlarms(this);
        }
    }
}
exports.KinesisStreamsToKinesisFirehoseToS3 = KinesisStreamsToKinesisFirehoseToS3;
//# sourceMappingURL=data:application/json;base64,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