# BSD 2-Clause License
#
# Copyright (c) 2021, Hewlett Packard Enterprise
# All rights reserved.
#
# Redistribution and use in source and binary forms, with or without
# modification, are permitted provided that the following conditions are met:
#
# 1. Redistributions of source code must retain the above copyright notice, this
#    list of conditions and the following disclaimer.
#
# 2. Redistributions in binary form must reproduce the above copyright notice,
#    this list of conditions and the following disclaimer in the documentation
#    and/or other materials provided with the distribution.
#
# THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS"
# AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
# IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE
# DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT HOLDER OR CONTRIBUTORS BE LIABLE
# FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
# DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR
# SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER
# CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY,
# OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
# OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.

import os
import os.path as osp
from itertools import product

from ...error import SSConfigError
from ...utils import get_logger
from .step import Step

logger = get_logger(__name__)


class SbatchStep(Step):
    def __init__(self, name, cwd, batch_settings):
        """Initialize a Slurm Sbatch step

        :param name: name of the entity to launch
        :type name: str
        :param cwd: path to launch dir
        :type cwd: str
        :param batch_settings: batch settings for entity
        :type batch_settings: BatchSettings
        """
        super().__init__(name, cwd)
        self.batch_settings = batch_settings
        self.step_cmds = []
        self.managed = True

    def get_launch_cmd(self):
        """Get the launch command for the batch

        :return: launch command for the batch
        :rtype: list[str]
        """
        script = self._write_script()
        return [self.batch_settings.batch_cmd, "--parsable", script]

    def add_to_batch(self, step):
        """Add a job step to this batch

        :param step: a job step instance e.g. SrunStep
        :type step: Step
        """
        launch_cmd = ["cd", step.cwd, ";"]
        launch_cmd += step.get_launch_cmd()
        self.step_cmds.append(launch_cmd)
        logger.debug(f"Added step command to batch for {step.name}")

    def _write_script(self):
        """Write the batch script

        :return: batch script path after writing
        :rtype: str
        """
        batch_script = self.get_step_file(ending=".sh")
        output, error = self.get_output_files()
        with open(batch_script, "w") as f:
            f.write("#!/bin/bash\n\n")
            f.write(f"#SBATCH --output={output}\n")
            f.write(f"#SBATCH --error={error}\n")
            f.write(f"#SBATCH --job-name={self.name}\n")

            # add additional sbatch options
            for opt in self.batch_settings.format_batch_args():
                f.write(f"#SBATCH {opt}\n")

            for cmd in self.batch_settings._preamble:
                f.write(f"{cmd}\n")

            for i, cmd in enumerate(self.step_cmds):
                f.write("\n")
                f.write(f"{' '.join((cmd))} &\n")
                if i == len(self.step_cmds) - 1:
                    f.write("\n")
                    f.write("wait\n")
        return batch_script


class SrunStep(Step):
    def __init__(self, name, cwd, run_settings):
        """Initialize a srun job step

        :param name: name of the entity to be launched
        :type name: str
        :param cwd: path to launch dir
        :type cwd: str
        :param run_settings: run settings for entity
        :type run_settings: RunSettings
        """
        super().__init__(name, cwd)
        self.run_settings = run_settings
        self.alloc = None
        self.managed = True
        if not self.run_settings.in_batch:
            self._set_alloc()

    def get_launch_cmd(self):
        """Get the command to launch this step

        :return: launch command
        :rtype: list[str]
        """
        srun = self.run_settings.run_command
        output, error = self.get_output_files()

        srun_cmd = [srun, "--output", output, "--error", error, "--job-name", self.name]

        if self.alloc:
            srun_cmd += ["--jobid", str(self.alloc)]

        if self.run_settings.env_vars:
            env_var_str = self.run_settings.format_env_vars()
            srun_cmd += ["--export", env_var_str]

        srun_cmd += self.run_settings.format_run_args()
        srun_cmd += self._build_exe()
        return srun_cmd

    def _set_alloc(self):
        """Set the id of the allocation

        :raises SSConfigError: allocation not listed or found
        """
        if self.run_settings.alloc:
            self.alloc = str(self.run_settings.alloc)
        else:
            if "SLURM_JOB_ID" in os.environ:
                self.alloc = os.environ["SLURM_JOB_ID"]
                logger.debug(
                    f"Running on allocation {self.alloc} gleaned from user environment"
                )
            else:
                raise SSConfigError(
                    "No allocation specified or found and not running in batch"
                )

    def _build_exe(self):
        """Build the executable for this step

        :return: executable list
        :rtype: list[str]
        """
        exe = self.run_settings.exe
        args = self.run_settings.exe_args
        if self.run_settings.mpmd:
            cmd = self._make_mpmd(exe, args)
            mp_cmd = ["--multi-prog", cmd]
            return mp_cmd
        else:
            cmd = exe + args
            return cmd

    def _make_mpmd(self, executable, exe_args):
        """Build Slurm multi-prog (MPMD) executable

        Launch multiple programs on seperate CPUs on the same node using the
        slurm --multi-prog feature.
        """
        mpmd_file = self.get_step_file(ending=".mpmd")
        launch_args = list(product(executable, exe_args))
        with open(mpmd_file, "w+") as f:
            proc_num = 0
            for exe, args in launch_args:
                e_args = " ".join(args)
                f.write(" ".join((str(proc_num), exe, e_args, "\n")))
                proc_num += 1
        return mpmd_file
