"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.EdgeFunction = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const path = require("path");
const iam = require("@aws-cdk/aws-iam");
const lambda = require("@aws-cdk/aws-lambda");
// hack, as this is not exported by the Lambda module
const function_hash_1 = require("@aws-cdk/aws-lambda/lib/function-hash");
const ssm = require("@aws-cdk/aws-ssm");
const core_1 = require("@aws-cdk/core");
/**
 * A Lambda@Edge function.
 *
 * Convenience resource for requesting a Lambda function in the 'us-east-1' region for use with Lambda@Edge.
 * Implements several restrictions enforced by Lambda@Edge.
 *
 * Note that this construct requires that the 'us-east-1' region has been bootstrapped.
 * See https://docs.aws.amazon.com/cdk/latest/guide/bootstrapping.html or 'cdk bootstrap --help' for options.
 *
 * @stability stable
 * @resource AWS::Lambda::Function
 */
class EdgeFunction extends core_1.Resource {
    /**
     * @stability stable
     */
    constructor(scope, id, props) {
        super(scope, id);
        /**
         * Whether or not this Lambda function was bound to a VPC.
         *
         * If this is is `false`, trying to access the `connections` object will fail.
         *
         * @stability stable
         */
        this.isBoundToVpc = false;
        // Create a simple Function if we're already in us-east-1; otherwise create a cross-region stack.
        const regionIsUsEast1 = !core_1.Token.isUnresolved(this.stack.region) && this.stack.region === 'us-east-1';
        const { edgeFunction, edgeArn } = regionIsUsEast1
            ? this.createInRegionFunction(props)
            : this.createCrossRegionFunction(id, props);
        this.edgeArn = edgeArn;
        this.functionArn = edgeArn;
        this._edgeFunction = edgeFunction;
        this.functionName = this._edgeFunction.functionName;
        this.grantPrincipal = this._edgeFunction.role;
        this.permissionsNode = this._edgeFunction.permissionsNode;
        this.version = lambda.extractQualifierFromArn(this.functionArn);
        this.node.defaultChild = this._edgeFunction;
    }
    /**
     * The underlying AWS Lambda function.
     *
     * @stability stable
     */
    get lambda() {
        return this._edgeFunction;
    }
    /**
     * Convenience method to make `EdgeFunction` conform to the same interface as `Function`.
     *
     * @stability stable
     */
    get currentVersion() {
        return this;
    }
    /**
     * Defines an alias for this version.
     *
     * @stability stable
     */
    addAlias(aliasName, options = {}) {
        return new lambda.Alias(this._edgeFunction, `Alias${aliasName}`, {
            aliasName,
            version: this._edgeFunction.currentVersion,
            ...options,
        });
    }
    /**
     * Not supported.
     *
     * Connections are only applicable to VPC-enabled functions.
     *
     * @stability stable
     */
    get connections() {
        throw new Error('Lambda@Edge does not support connections');
    }
    /**
     * The `$LATEST` version of this function.
     *
     * Note that this is reference to a non-specific AWS Lambda version, which
     * means the function this version refers to can return different results in
     * different invocations.
     *
     * To obtain a reference to an explicit version which references the current
     * function configuration, use `lambdaFunction.currentVersion` instead.
     *
     * @stability stable
     */
    get latestVersion() {
        throw new Error('$LATEST function version cannot be used for Lambda@Edge');
    }
    /**
     * Adds an event source that maps to this AWS Lambda function.
     *
     * @stability stable
     */
    addEventSourceMapping(id, options) {
        return this.lambda.addEventSourceMapping(id, options);
    }
    /**
     * Adds a permission to the Lambda resource policy.
     *
     * @stability stable
     */
    addPermission(id, permission) {
        return this.lambda.addPermission(id, permission);
    }
    /**
     * Adds a statement to the IAM role assumed by the instance.
     *
     * @stability stable
     */
    addToRolePolicy(statement) {
        return this.lambda.addToRolePolicy(statement);
    }
    /**
     * Grant the given identity permissions to invoke this Lambda.
     *
     * @stability stable
     */
    grantInvoke(identity) {
        return this.lambda.grantInvoke(identity);
    }
    /**
     * Return the given named metric for this Lambda Return the given named metric for this Function.
     *
     * Return the given named metric for this Lambda Return the given named metric for this Function.
     *
     * @stability stable
     */
    metric(metricName, props) {
        return this.lambda.metric(metricName, { ...props, region: EdgeFunction.EDGE_REGION });
    }
    /**
     * Metric for the Duration of this Lambda How long execution of this Lambda takes.
     *
     * Average over 5 minutes
     * Metric for the Duration of this Lambda How long execution of this Lambda takes.
     *
     * Average over 5 minutes
     *
     * @stability stable
     */
    metricDuration(props) {
        return this.lambda.metricDuration({ ...props, region: EdgeFunction.EDGE_REGION });
    }
    /**
     * How many invocations of this Lambda fail.
     *
     * Sum over 5 minutes
     *
     * @stability stable
     */
    metricErrors(props) {
        return this.lambda.metricErrors({ ...props, region: EdgeFunction.EDGE_REGION });
    }
    /**
     * Metric for the number of invocations of this Lambda How often this Lambda is invoked.
     *
     * Sum over 5 minutes
     * Metric for the number of invocations of this Lambda How often this Lambda is invoked.
     *
     * Sum over 5 minutes
     *
     * @stability stable
     */
    metricInvocations(props) {
        return this.lambda.metricInvocations({ ...props, region: EdgeFunction.EDGE_REGION });
    }
    /**
     * Metric for the number of throttled invocations of this Lambda How often this Lambda is throttled.
     *
     * Sum over 5 minutes
     * Metric for the number of throttled invocations of this Lambda How often this Lambda is throttled.
     *
     * Sum over 5 minutes
     *
     * @stability stable
     */
    metricThrottles(props) {
        return this.lambda.metricThrottles({ ...props, region: EdgeFunction.EDGE_REGION });
    }
    /**
     * Adds an event source to this function.
     *
     * @stability stable
     */
    addEventSource(source) {
        return this.lambda.addEventSource(source);
    }
    /**
     * Configures options for asynchronous invocation.
     *
     * @stability stable
     */
    configureAsyncInvoke(options) {
        return this.lambda.configureAsyncInvoke(options);
    }
    /** Create a function in-region */
    createInRegionFunction(props) {
        const edgeFunction = new lambda.Function(this, 'Fn', props);
        addEdgeLambdaToRoleTrustStatement(edgeFunction.role);
        return { edgeFunction, edgeArn: edgeFunction.currentVersion.edgeArn };
    }
    /** Create a support stack and function in us-east-1, and a SSM reader in-region */
    createCrossRegionFunction(id, props) {
        const parameterNamePrefix = '/cdk/EdgeFunctionArn';
        if (core_1.Token.isUnresolved(this.env.region)) {
            throw new Error('stacks which use EdgeFunctions must have an explicitly set region');
        }
        const parameterName = `${parameterNamePrefix}/${this.env.region}/${this.node.path}`;
        const functionStack = this.edgeStack(props.stackId);
        const edgeFunction = new lambda.Function(functionStack, id, props);
        addEdgeLambdaToRoleTrustStatement(edgeFunction.role);
        // Store the current version's ARN to be retrieved by the cross region reader below.
        new ssm.StringParameter(edgeFunction, 'Parameter', {
            parameterName,
            stringValue: edgeFunction.currentVersion.edgeArn,
        });
        const edgeArn = this.createCrossRegionArnReader(parameterNamePrefix, parameterName, edgeFunction);
        return { edgeFunction, edgeArn };
    }
    createCrossRegionArnReader(parameterNamePrefix, parameterName, edgeFunction) {
        // Prefix of the parameter ARN that applies to all EdgeFunctions.
        // This is necessary because the `CustomResourceProvider` is a singleton, and the `policyStatement`
        // must work for multiple EdgeFunctions.
        const parameterArnPrefix = this.stack.formatArn({
            service: 'ssm',
            region: EdgeFunction.EDGE_REGION,
            resource: 'parameter',
            resourceName: parameterNamePrefix + '/*',
            sep: '',
        });
        const resourceType = 'Custom::CrossRegionStringParameterReader';
        const serviceToken = core_1.CustomResourceProvider.getOrCreate(this, resourceType, {
            codeDirectory: path.join(__dirname, 'edge-function'),
            runtime: core_1.CustomResourceProviderRuntime.NODEJS_12_X,
            policyStatements: [{
                    Effect: 'Allow',
                    Resource: parameterArnPrefix,
                    Action: ['ssm:GetParameter'],
                }],
        });
        const resource = new core_1.CustomResource(this, 'ArnReader', {
            resourceType: resourceType,
            serviceToken,
            properties: {
                Region: EdgeFunction.EDGE_REGION,
                ParameterName: parameterName,
                // This is used to determine when the function has changed, to refresh the ARN from the custom resource.
                RefreshToken: function_hash_1.calculateFunctionHash(edgeFunction),
            },
        });
        return resource.getAttString('FunctionArn');
    }
    edgeStack(stackId) {
        const stage = core_1.Stage.of(this);
        if (!stage) {
            throw new Error('stacks which use EdgeFunctions must be part of a CDK app or stage');
        }
        const edgeStackId = stackId !== null && stackId !== void 0 ? stackId : `edge-lambda-stack-${this.stack.node.addr}`;
        let edgeStack = stage.node.tryFindChild(edgeStackId);
        if (!edgeStack) {
            edgeStack = new core_1.Stack(stage, edgeStackId, {
                env: {
                    region: EdgeFunction.EDGE_REGION,
                    account: core_1.Stack.of(this).account,
                },
            });
        }
        this.stack.addDependency(edgeStack);
        return edgeStack;
    }
}
exports.EdgeFunction = EdgeFunction;
_a = JSII_RTTI_SYMBOL_1;
EdgeFunction[_a] = { fqn: "@aws-cdk/aws-cloudfront.experimental.EdgeFunction", version: "1.102.0" };
EdgeFunction.EDGE_REGION = 'us-east-1';
function addEdgeLambdaToRoleTrustStatement(role) {
    if (role instanceof iam.Role && role.assumeRolePolicy) {
        const statement = new iam.PolicyStatement();
        const edgeLambdaServicePrincipal = new iam.ServicePrincipal('edgelambda.amazonaws.com');
        statement.addPrincipals(edgeLambdaServicePrincipal);
        statement.addActions(edgeLambdaServicePrincipal.assumeRoleAction);
        role.assumeRolePolicy.addStatements(statement);
    }
}
//# sourceMappingURL=data:application/json;base64,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