"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.ClientAttributes = exports.DateTimeAttribute = exports.BooleanAttribute = exports.NumberAttribute = exports.StringAttribute = void 0;
const core_1 = require("@aws-cdk/core");
const attr_names_1 = require("./private/attr-names");
/**
 * The String custom attribute type.
 *
 * @stability stable
 */
class StringAttribute {
    /**
     * @stability stable
     */
    constructor(props = {}) {
        if (props.minLen && !core_1.Token.isUnresolved(props.minLen) && props.minLen < 0) {
            throw new Error(`minLen cannot be less than 0 (value: ${props.minLen}).`);
        }
        if (props.maxLen && !core_1.Token.isUnresolved(props.maxLen) && props.maxLen > 2048) {
            throw new Error(`maxLen cannot be greater than 2048 (value: ${props.maxLen}).`);
        }
        this.minLen = props === null || props === void 0 ? void 0 : props.minLen;
        this.maxLen = props === null || props === void 0 ? void 0 : props.maxLen;
        this.mutable = props === null || props === void 0 ? void 0 : props.mutable;
    }
    /**
     * Bind this custom attribute type to the values as expected by CloudFormation.
     *
     * @stability stable
     */
    bind() {
        let stringConstraints;
        if (this.minLen || this.maxLen) {
            stringConstraints = {
                minLen: this.minLen,
                maxLen: this.maxLen,
            };
        }
        return {
            dataType: 'String',
            stringConstraints,
            mutable: this.mutable,
        };
    }
}
exports.StringAttribute = StringAttribute;
/**
 * The Number custom attribute type.
 *
 * @stability stable
 */
class NumberAttribute {
    /**
     * @stability stable
     */
    constructor(props = {}) {
        this.min = props === null || props === void 0 ? void 0 : props.min;
        this.max = props === null || props === void 0 ? void 0 : props.max;
        this.mutable = props === null || props === void 0 ? void 0 : props.mutable;
    }
    /**
     * Bind this custom attribute type to the values as expected by CloudFormation.
     *
     * @stability stable
     */
    bind() {
        let numberConstraints;
        if (this.min || this.max) {
            numberConstraints = {
                min: this.min,
                max: this.max,
            };
        }
        return {
            dataType: 'Number',
            numberConstraints,
            mutable: this.mutable,
        };
    }
}
exports.NumberAttribute = NumberAttribute;
/**
 * The Boolean custom attribute type.
 *
 * @stability stable
 */
class BooleanAttribute {
    /**
     * @stability stable
     */
    constructor(props = {}) {
        this.mutable = props === null || props === void 0 ? void 0 : props.mutable;
    }
    /**
     * Bind this custom attribute type to the values as expected by CloudFormation.
     *
     * @stability stable
     */
    bind() {
        return {
            dataType: 'Boolean',
            mutable: this.mutable,
        };
    }
}
exports.BooleanAttribute = BooleanAttribute;
/**
 * The DateTime custom attribute type.
 *
 * @stability stable
 */
class DateTimeAttribute {
    /**
     * @stability stable
     */
    constructor(props = {}) {
        this.mutable = props === null || props === void 0 ? void 0 : props.mutable;
    }
    /**
     * Bind this custom attribute type to the values as expected by CloudFormation.
     *
     * @stability stable
     */
    bind() {
        return {
            dataType: 'DateTime',
            mutable: this.mutable,
        };
    }
}
exports.DateTimeAttribute = DateTimeAttribute;
/**
 * A set of attributes, useful to set Read and Write attributes.
 *
 * @stability stable
 */
class ClientAttributes {
    /**
     * Creates a ClientAttributes with the specified attributes.
     *
     * @default - a ClientAttributes object without any attributes
     * @stability stable
     */
    constructor() {
        this.attributesSet = new Set();
    }
    /**
     * Creates a custom ClientAttributes with the specified attributes.
     *
     * @param attributes a list of standard attributes to add to the set.
     * @stability stable
     */
    withStandardAttributes(attributes) {
        let attributesSet = new Set(this.attributesSet);
        // iterate through key-values in the `StandardAttributeNames` constant
        // to get the value for all attributes
        for (const attributeKey in attr_names_1.StandardAttributeNames) {
            if (attributes[attributeKey] === true) {
                const attributeName = attr_names_1.StandardAttributeNames[attributeKey];
                attributesSet.add(attributeName);
            }
        }
        let aux = new ClientAttributes();
        aux.attributesSet = attributesSet;
        return aux;
    }
    /**
     * Creates a custom ClientAttributes with the specified attributes.
     *
     * @param attributes a list of custom attributes to add to the set.
     * @stability stable
     */
    withCustomAttributes(...attributes) {
        let attributesSet = new Set(this.attributesSet);
        for (let attribute of attributes) {
            // custom attributes MUST begin with `custom:`, so add the string if not present
            if (!attribute.startsWith('custom:')) {
                attribute = 'custom:' + attribute;
            }
            attributesSet.add(attribute);
        }
        let aux = new ClientAttributes();
        aux.attributesSet = attributesSet;
        return aux;
    }
    /**
     * The list of attributes represented by this ClientAttributes.
     *
     * @stability stable
     */
    attributes() {
        // sorting is unnecessary but it simplify testing
        return Array.from(this.attributesSet).sort();
    }
}
exports.ClientAttributes = ClientAttributes;
//# sourceMappingURL=data:application/json;base64,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