# -*- coding: utf-8 -*-
from setuptools import setup

packages = \
['betterletter']

package_data = \
{'': ['*'], 'betterletter': ['resources/*', 'resources/dicts/*']}

install_requires = \
['pyperclip>=1.8.2,<2.0.0']

entry_points = \
{'console_scripts': ['betterletter = betterletter.__main__:main']}

setup_kwargs = {
    'name': 'betterletter',
    'version': '1.0.3',
    'description': 'Substitute alternative spellings of native characters (e.g. German umlauts [ae, oe, ue] etc. [ss]) with their correct versions (ä, ö, ü, ß).',
    'long_description': '# betterletter\n\nIn a given text, replaces alternative spellings of native characters with their proper spellings.\n\nFor example, German native characters and their corresponding alternative spellings (e.g. when no proper keyboard layout is at hand, or ASCII is used) are:\n\n| Native Character | Alternative Spelling |\n| :--------------: | :------------------: |\n|       Ä/ä        |        Ae/ae         |\n|       Ö/ö        |        Oe/oe         |\n|       Ü/ü        |        Ue/ue         |\n|       ẞ/ß        |        SS/ss         |\n\nThese pairings are recorded [here](https://github.com/alexpovel/betterletter/blob/master/betterletter/resources/languages.json).\n\nGoing from left to right is simple: replace all native characters with their alternative spellings, minding case.\nThat use case is also supported by this tool (`reverse` flag).\n\nThe other direction is much less straightforward: there exist countless words for which alternative spellings occur somewhere as a pattern, yet replacing them with the corresponding native character would be wrong:\n\n| Character | Correct Spelling  | Wrong Spelling |\n| --------- | ----------------- | -------------- |\n| *Ä*       | **Ae**rodynamik   | Ärodynamik     |\n| *Ä*       | Isr**ae**l        | Isräl          |\n| *Ä*       | Schuf**ae**intrag | Schufäintrag   |\n| *Ö*       | K**oe**ffizient   | Köffizient     |\n| *Ö*       | Domin**oe**ffekt  | Dominöffekt    |\n| *Ö*       | P**oet**          | Pöt            |\n| *Ü*       | Abente**ue**r     | Abenteür       |\n| *Ü*       | Ma**ue**r         | Maür           |\n| *Ü*       | Ste**ue**rung     | Steürung       |\n| *ß*       | Me**ss**gerät     | Meßgerät       |\n| *ß*       | Me**ss**e         | Meße           |\n| *ß*       | Abschlu**ss**     | Abschluß       |\n\njust to name a few, pretty common examples.\n\nAs such, this tool is based on a dictionary lookup, see also the [containing directory](https://github.com/alexpovel/betterletter/blob/master/betterletter/resources/dicts/).\n\n## Examples\n\nSee also the [tests](https://github.com/alexpovel/betterletter/blob/master/tests/).\n\n### de\n\nThe input:\n\n> Ueberhaupt braeuchte es mal einen Teststring.\n> Saetze ohne Bedeutung, aber mit vielen Umlauten.\n> DRPFA-Angehoerige gehoeren haeufig nicht dazu.\n> Bindestrich-Woerter spraechen Baende ueber Fehler.\n> Doppelgaenger-Doppelgaenger sind doppelt droelfzig.\n> Oder Uemlaeuten? Auslaeuten? Leute gaebe es, wuerde man meinen.\n> Ueble Nachrede ist naechtens nicht erlaubt.\n> Erlaube man dieses, waere es schoen uebertrieben.\n> Busse muesste geloest werden, bevor Gruesse zum Gruss kommen.\n> Busse sind Geraete, die womoeglich schnell fuehren.\n> Voegel sind aehnlich zu Oel.\n> Hierfuer ist fuer den droegen Poebel zu beachten, dass Anmassungen zu Gehoerverlust fuehren koennen.\n> Stroemelschnoesseldaemel!\n\nis turned into:\n\n> Überhaupt bräuchte es mal einen Teststring.\n> Sätze ohne Bedeutung, aber mit vielen Umlauten.\n> DRPFA-Angehörige gehören häufig nicht dazu.\n> Bindestrich-Wörter sprächen Bände über Fehler.\n> Doppelgänger-Doppelgänger sind doppelt droelfzig.\n> Oder Uemlaeuten? Auslaeuten? Leute gäbe es, würde man meinen.\n> Üble Nachrede ist nächtens nicht erlaubt.\n> Erlaube man dieses, wäre es schön übertrieben.\n> Buße müsste gelöst werden, bevor Grüße zum Gruß kommen.\n> Buße sind Geräte, die womöglich schnell führen.\n> Vögel sind ähnlich zu Öl.\n> Hierfür ist für den drögen Pöbel zu beachten, dass Anmaßungen zu Gehörverlust führen können.\n> Stroemelschnoesseldaemel!\n\n---\n\nNote that some corrections are out of scope for this little script, e.g.:\n\n> Busse\n\nIn German, *Busse* and *Buße* are two words of vastly different meaning (*busses* and *penance*, respectively).\nUnfortunately, they map to the same alternative spelling of *Busse*.\nThe tool sees *Busse* (meaning *just that*, with no intent of changing it), notices *Buße* is a legal substitution, and therefore makes it.\nThe tool has no awareness of context.\n\nTurning substitutions like these off would mean the tool would no longer emit *Buße*, ever.\nThis could be as undesirable as the current behaviour.\nThere seems to be no easy resolve.\n\n## Installation\n\n```shell\npip install betterletter\n```\n\n## Usage\n\nThe package will install a Python script of the same name, so instead of `python -m betterletter`, you can simply invoke that directly, if the Python script directory is on your `$PATH`:\n\n```bash\n$ betterletter -h\nusage: betterletter [-h] [-c] [-f] [-r] [-g] [-d] [--debug] {de}\n\nTool to replace alternative spellings of native characters (e.g. German umlauts [ä, ö, ü] etc. [ß]) with the proper native characters. For example, this\nproblem occurs when no proper keyboard layout was available. This program is dictionary-based to check if replacements are valid words. By default, reads\nfrom STDIN and writes to STDOUT.\n\npositional arguments:\n  {de}             Text language to work with, in ISO 639-1 format.\n\noptions:\n  -h, --help       show this help message and exit\n  -c, --clipboard  Read from and write back to clipboard instead of STDIN/STDOUT.\n  -f, --force-all  Force substitutions and return the text version with the maximum number of substitutions, even if they are illegal words (useful for\n                   names).\n  -r, --reverse    Reverse mode, where all native characters are simply replaced by their alternative spellings.\n  -g, --gui        Stop and open a GUI prompt for confirmation before finishing.\n  -d, --diff       Print a diff view of the substitutions to stderr.\n  --debug          Output detailed logging information.\n```\n\n### Usage Examples\n\nNormal usage:\n\n```bash\n$ echo "Hoeflich fragen waere angebracht!" | betterletter de\nHöflich fragen wäre angebracht!\n```\n\nReverse it:\n\n```bash\n$ echo "Höflich fragen wäre angebracht!" | betterletter --reverse de\nHoeflich fragen waere angebracht!\n```\n\nA diff view, useful for longer text and to confirm correctness.\nThe diff is written to STDERR so won\'t interfere with further redirection.\n\n```bash\n$ "Hoeflich fragen waere angebracht!" | python -m betterletter --diff de\n- Hoeflich fragen waere angebracht!\n?  ^^              ^^\n+ Höflich fragen wäre angebracht!\n?  ^              ^\nHöflich fragen wäre angebracht!\n```\n\nThe tool may be coerced into working with names:\n\n```bash\n$ echo "Sehr geehrte Frau Huebenstetter, ..." | betterletter de  # A name won\'t be in the dictionary\nSehr geehrte Frau Huebenstetter, ...\n$ echo "Sehr geehrte Frau Huebenstetter, ..." | betterletter --force de\nSehr geehrte Frau Hübenstetter, ...\n```\n\nClipboard-based workflows are also possible:\n\n```bash\nbetterletter --clipboard de  # Nothing happens: clipboard is read and written to silently.\n```\n\n## Development\n\n### Prerequisites\n\nThis project uses [poetry](https://python-poetry.org/) for dependency management.\nRefer to the [poetry config file](https://github.com/alexpovel/betterletter/blob/master/pyproject.toml) for more info (e.g. the required Python modules to install if you don\'t want to deal with `poetry`).\n\nUsing poetry, from the project root, run:\n\n```bash\n# Installs virtual environment according to lock file (if available in repo),\n# otherwise pyproject.toml:\npoetry install\n# Run command within that environment:\npoetry run python -m betterletter -h\n```\n\nDevelopment tasks are all run through `poetry`, within the context of the virtual environment.\nThe latter is created through\n\n```bash\npoetry install\n```\n\nand then accessed through either `poetry run <command>` or `poetry shell`.\n\nRun `make` (without arguments) for more available commands related to development.\n\n## AutoHotKey\n\nThis tool can be integrated with [AutoHotKey](https://www.autohotkey.com/), allowing you to use it at the touch of a button.\nThis can be used to setup a keyboard shortcut to run this tool in-place, quickly replacing what you need without leaving your text editing environment.\n\nThe AutoHotKey file is [here](https://github.com/alexpovel/betterletter/blob/master/betterletter.ahk).\n\nFollow [this guide](https://www.autohotkey.com/docs/FAQ.htm#Startup) to have the script launch on boot automatically.\n',
    'author': 'Alex Povel',
    'author_email': 'python@alexpovel.de',
    'maintainer': None,
    'maintainer_email': None,
    'url': 'https://github.com/alexpovel/betterletter/',
    'packages': packages,
    'package_data': package_data,
    'install_requires': install_requires,
    'entry_points': entry_points,
    'python_requires': '>=3.9,<4.0',
}


setup(**setup_kwargs)
