import hgvs.parser
import hgvs.assemblymapper
import hgvs.dataproviders.uta
import pandas as pd
import re


class Hgvs_variant():
    """Hgvs_variant takes in a standard or informal chgvs or gchgvs specification, then project variants between sequences. 
    """
    def __init__(self, assembly_name='GRCh37'):
        self.hp = hgvs.parser.Parser()
        self.hdp = hgvs.dataproviders.uta.connect()
        self.am = hgvs.assemblymapper.AssemblyMapper(
            self.hdp,
            assembly_name=assembly_name,
            alt_aln_method='splign',
            replace_reference=True)

        self.nm_to_hugo_dict = {
            'NM_000020.2': 'ACVRL1',
            'NM_000038.5': 'APC',
            'NM_000044.3': 'AR',
            'NM_000051.3': 'ATM',
            'NM_000057.2': 'BLM',
            'NM_000059.3': 'BRCA2',
            'NM_000061.2': 'BTK',
            'NM_000073.2': 'CD3G',
            'NM_000075.3': 'CDK4',
            'NM_000077.4': 'CDKN2A',
            'NM_000102.3': 'CYP17A1',
            'NM_000106.5': 'CYP2D6',
            'NM_000110.3': 'DPYD',
            'NM_000121.3': 'EPOR',
            'NM_000122.1': 'ERCC3',
            'NM_000127.2': 'EXT1',
            'NM_000135.2': 'FANCA',
            'NM_000136.2': 'FANCC',
            'NM_000141.4': 'FGFR2',
            'NM_000142.4': 'FGFR3',
            'NM_000143.3': 'FH',
            'NM_000179.2': 'MSH6',
            'NM_000197.1': 'HSD17B3',
            'NM_000198.3': 'HSD3B2',
            'NM_000206.2': 'IL2RG',
            'NM_000211.3': 'ITGB2',
            'NM_000215.3': 'JAK3',
            'NM_000222.2': 'KIT',
            'NM_000233.3': 'LHCGR',
            'NM_000238.3': 'KCNH2',
            'NM_000245.2': 'MET',
            'NM_000248.3': 'MITF',
            'NM_000249.3': 'MLH1',
            'NM_000251.2': 'MSH2',
            'NM_000254.2': 'MTR',
            'NM_000264.3': 'PTCH1',
            'NM_000267.3': 'NF1',
            'NM_000314.4': 'PTEN',
            'NM_000321.2': 'RB1',
            'NM_000342.3': 'SLC4A1',
            'NM_000346.3': 'SOX9',
            'NM_000348.3': 'SRD5A2',
            'NM_000367.2': 'TPMT',
            'NM_000368.4': 'TSC1',
            'NM_000369.2': 'TSHR',
            'NM_000372.4': 'TYR',
            'NM_000373.3': 'UMPS',
            'NM_000380.3': 'XPA',
            'NM_000392.3': 'ABCC2',
            'NM_000400.3': 'ERCC2',
            'NM_000401.3': 'EXT2',
            'NM_000406.2': 'GNRHR',
            'NM_000417.2': 'IL2RA',
            'NM_000435.2': 'NOTCH3',
            'NM_000455.4': 'STK11',
            'NM_000459.3': 'TEK',
            'NM_000463.2': 'UGT1A1',
            'NM_000465.2': 'BARD1',
            'NM_000489.3': 'ATRX',
            'NM_000491.3': 'C1QB',
            'NM_000516.4': 'GNAS',
            'NM_000534.4': 'PMS1',
            'NM_000535.5': 'PMS2',
            'NM_000545.5': 'HNF1A',
            'NM_000546.5': 'TP53',
            'NM_000548.3': 'TSC2',
            'NM_000551.3': 'VHL',
            'NM_000566.3': 'FCGR1A',
            'NM_000569.6': 'FCGR3A',
            'NM_000600.3': 'IL6',
            'NM_000601.4': 'HGF',
            'NM_000618.3': 'IGF1',
            'NM_000626.2': 'CD79B',
            'NM_000629.2': 'IFNAR1',
            'NM_000633.2': 'BCL2',
            'NM_000640.2': 'IL13RA2',
            'NM_000732.4': 'CD3D',
            'NM_000733.3': 'CD3E',
            'NM_000754.3': 'COMT',
            'NM_000770.3': 'CYP2C8',
            'NM_000777.3': 'CYP3A5',
            'NM_000778.3': 'CYP4A11',
            'NM_000782.4': 'CYP24A1',
            'NM_000788.2': 'DCK',
            'NM_000791.3': 'DHFR',
            'NM_000833.3': 'GRIN2A',
            'NM_000840.2': 'GRM3',
            'NM_000852.3': 'GSTP1',
            'NM_000875.3': 'IGF1R',
            'NM_000878.3': 'IL2RB',
            'NM_000927.4': 'ABCB1',
            'NM_000964.3': 'RARA',
            'NM_000965.3': 'RARB',
            'NM_000966.5': 'RARG',
            'NM_000969.3': 'RPL5',
            'NM_000983.3': 'RPL22',
            'NM_001001391.1': 'CD44',
            'NM_001002295.1': 'GATA3',
            'NM_001003931.2': 'PARP3',
            'NM_001005236.3': 'OR1L1',
            'NM_001005503.1': 'OR11G2',
            'NM_001007792.1': 'NTRK1',
            'NM_001008781.2': 'FAT3',
            'NM_001008895.1': 'CUL4A',
            'NM_001010874.4': 'TECRL',
            'NM_001012288.1': 'CRLF2',
            'NM_001013699.2': 'H3F3C',
            'NM_001014432.1': 'AKT1',
            'NM_001014796.1': 'DDR2',
            'NM_001015877.1': 'PHF6',
            'NM_001017963.2': 'HSP90AA1',
            'NM_001018077.1': 'NR3C1',
            'NM_001024465.1': 'SOD2',
            'NM_001025071.1': 'RPS14',
            'NM_001025159.2': 'CD74',
            'NM_001025366.2': 'VEGFA',
            'NM_001031684.2': 'SRSF7',
            'NM_001031698.2': 'PRPF40B',
            'NM_001033.3': 'RRM1',
            'NM_001033082.2': 'MYCL1',
            'NM_001039590.2': 'USP9X',
            'NM_001040275.1': 'ESR2',
            'NM_001042603.1': 'KDM5A',
            'NM_001042749.1': 'STAG2',
            'NM_001050.2': 'SSTR2',
            'NM_001051.2': 'SSTR3',
            'NM_001058.3': 'TACR1',
            'NM_001067.3': 'TOP2A',
            'NM_001068.2': 'TOP2B',
            'NM_001070.4': 'TUBG1',
            'NM_001071.2': 'TYMS',
            'NM_001072.3': 'UGT1A6',
            'NM_001078177.1': 'SLC29A1',
            'NM_001079675.2': 'ETV4',
            'NM_001098482.1': 'CRTC1',
            'NM_001100119.1': 'XRCC3',
            'NM_001102416.2': 'KNG1',
            'NM_001114101.1': 'C1QC',
            'NM_001114122.2': 'CHEK1',
            'NM_001114309.1': 'ELF3',
            'NM_001122742.1': 'ESR1',
            'NM_001127208.2': 'TET2',
            'NM_001127598.1': 'IGF2',
            'NM_001128166.1': 'PAK3',
            'NM_001128425.1': 'MUTYH',
            'NM_001128844.1': 'SMARCA4',
            'NM_001130173.1': 'MYB',
            'NM_001130823.1': 'DNMT1',
            'NM_001135052.2': 'SYK',
            'NM_001135937.2': 'SMAD2',
            'NM_001139.2': 'ALOX12B',
            'NM_001141970.1': 'DAXX',
            'NM_001142548.1': 'RAD54L',
            'NM_001145260.1': 'NCOA4',
            'NM_001145306.1': 'CDK6',
            'NM_001145358.1': 'SIN3A',
            'NM_001145785.1': 'MEF2B',
            'NM_001146.3': 'ANGPT1',
            'NM_001147.2': 'ANGPT2',
            'NM_001160133.1': 'KCNQ5',
            'NM_001164665.1': 'KIAA1549',
            'NM_001165.4': 'BIRC3',
            'NM_001165962.1': 'ELAC2',
            'NM_001172133.1': 'HCK',
            'NM_001172560.1': 'SSTR5',
            'NM_001177314.1': 'RRAS2',
            'NM_001184.3': 'ATR',
            'NM_001188.3': 'BAK1',
            'NM_001190737.1': 'NFIB',
            'NM_001197104.1': 'MLL',
            'NM_001198.3': 'PRDM1',
            'NM_001199874.1': 'B4GALT3',
            'NM_001202559.1': 'CHURC1-FNTB',
            'NM_001204401.1': 'XIAP',
            'NM_001204747.1': 'RFC1',
            'NM_001204830.1': 'LIPT1',
            'NM_001228.4': 'CASP8',
            'NM_001236.3': 'CBR3',
            'NM_001238.2': 'CCNE1',
            'NM_001242314.1': 'MAP3K13',
            'NM_001242758.1': 'HLA-A',
            'NM_001243.3': 'TNFRSF8',
            'NM_001243835.1': 'STAT4',
            'NM_001244753.1': 'FCGR3B',
            'NM_001256163.1': 'BIRC2',
            'NM_001256849.1': 'POLD1',
            'NM_001260.1': 'CDK8',
            'NM_001262.2': 'CDKN2C',
            'NM_001265.4': 'CDX2',
            'NM_001267547.1': 'ARFRP1',
            'NM_001270.2': 'CHD1',
            'NM_001271.3': 'CHD2',
            'NM_001273.2': 'CHD4',
            'NM_001278.3': 'CHUK',
            'NM_001429.3': 'EP300',
            'NM_001527.3': 'HDAC2',
            'NM_001530.3': 'HIF1A',
            'NM_001556.2': 'IKBKB',
            'NM_001616.3': 'ACVR2A',
            'NM_001618.3': 'PARP1',
            'NM_001626.4': 'AKT2',
            'NM_001654.4': 'ARAF',
            'NM_001657.2': 'AREG',
            'NM_001664.2': 'RHOA',
            'NM_001699.4': 'AXL',
            'NM_001706.4': 'BCL6',
            'NM_001731.2': 'BTG1',
            'NM_001733.4': 'C1R',
            'NM_001754.4': 'RUNX1',
            'NM_001755.2': 'CBFB',
            'NM_001757.2': 'CBR1',
            'NM_001759.3': 'CCND2',
            'NM_001760.3': 'CCND3',
            'NM_001771.3': 'CD22',
            'NM_001772.3': 'CD33',
            'NM_001783.3': 'CD79A',
            'NM_001785.2': 'CDA',
            'NM_001790.3': 'CDC25C',
            'NM_001791.3': 'CDC42',
            'NM_001798.3': 'CDK2',
            'NM_001803.2': 'CD52',
            'NM_001859.3': 'SLC31A1',
            'NM_001903.2': 'CTNNA1',
            'NM_001904.3': 'CTNNB1',
            'NM_001946.2': 'DUSP6',
            'NM_001957.3': 'EDNRA',
            'NM_001967.3': 'EIF4A2',
            'NM_001982.3': 'ERBB3',
            'NM_001983.3': 'ERCC1',
            'NM_001987.4': 'ETV6',
            'NM_002007.2': 'FGF4',
            'NM_002009.3': 'FGF7',
            'NM_002017.4': 'FLI1',
            'NM_002019.4': 'FLT1',
            'NM_002020.4': 'FLT4',
            'NM_002027.2': 'FNTA',
            'NM_002028.3': 'FNTB',
            'NM_002037.5': 'FYN',
            'NM_002049.3': 'GATA1',
            'NM_002067.2': 'GNA11',
            'NM_002072.3': 'GNAQ',
            'NM_002093.3': 'GSK3B',
            'NM_002107.4': 'H3F3A',
            'NM_002154.3': 'HSPA4',
            'NM_002168.2': 'IDH2',
            'NM_002185.3': 'IL7R',
            'NM_002192.2': 'INHBA',
            'NM_002227.2': 'JAK1',
            'NM_002228.3': 'JUN',
            'NM_002253.2': 'KDR',
            'NM_002310.5': 'LIFR',
            'NM_002314.3': 'LIMK1',
            'NM_002315.2': 'LMO1',
            'NM_002350.3': 'LYN',
            'NM_002354.2': 'EPCAM',
            'NM_002386.3': 'MC1R',
            'NM_002392.4': 'MDM2',
            'NM_002393.4': 'MDM4',
            'NM_002439.4': 'MSH3',
            'NM_002440.3': 'MSH4',
            'NM_002454.2': 'MTRR',
            'NM_002456.5': 'MUC1',
            'NM_002460.3': 'IRF4',
            'NM_002467.4': 'MYC',
            'NM_002468.4': 'MYD88',
            'NM_002485.4': 'NBN',
            'NM_002520.6': 'NPM1',
            'NM_002524.4': 'NRAS',
            'NM_002530.3': 'NTRK3',
            'NM_002585.3': 'PBX1',
            'NM_002609.3': 'PDGFRB',
            'NM_002610.3': 'PDK1',
            'NM_002639.4': 'SERPINB5',
            'NM_002643.3': 'PIGF',
            'NM_002645.2': 'PIK3C2A',
            'NM_002646.3': 'PIK3C2B',
            'NM_002647.2': 'PIK3C3',
            'NM_002649.2': 'PIK3CG',
            'NM_002655.2': 'PLAG1',
            'NM_002701.4': 'POU5F1',
            'NM_002737.2': 'PRKCA',
            'NM_002738.6': 'PRKCB',
            'NM_002739.3': 'PRKCG',
            'NM_002745.4': 'MAPK1',
            'NM_002746.2': 'MAPK3',
            'NM_002755.3': 'MAP2K1',
            'NM_002773.3': 'PRSS8',
            'NM_002793.3': 'PSMB1',
            'NM_002794.4': 'PSMB2',
            'NM_002797.3': 'PSMB5',
            'NM_002834.3': 'PTPN11',
            'NM_002839.3': 'PTPRD',
            'NM_002872.3': 'RAC2',
            'NM_002875.4': 'RAD51',
            'NM_002880.3': 'RAF1',
            'NM_002908.2': 'REL',
            'NM_002941.3': 'ROBO1',
            'NM_002942.4': 'ROBO2',
            'NM_002944.2': 'ROS1',
            'NM_002945.3': 'RPA1',
            'NM_002957.4': 'RXRA',
            'NM_003000.2': 'SDHB',
            'NM_003001.3': 'SDHC',
            'NM_003002.2': 'SDHD',
            'NM_003010.2': 'MAP2K4',
            'NM_003016.4': 'SRSF2',
            'NM_003058.3': 'SLC22A2',
            'NM_003061.2': 'SLIT1',
            'NM_003069.3': 'SMARCA1',
            'NM_003073.3': 'SMARCB1',
            'NM_003076.4': 'SMARCD1',
            'NM_003106.3': 'SOX2',
            'NM_003161.2': 'RPS6KB1',
            'NM_003200.3': 'TCF3',
            'NM_003242.5': 'TGFBR2',
            'NM_003286.2': 'TOP1',
            'NM_003317.3': 'NKX2-1',
            'NM_003377.4': 'VEGFB',
            'NM_003379.4': 'EZR',
            'NM_003400.3': 'XPO1',
            'NM_003482.3': 'MLL2',
            'NM_003496.3': 'TRRAP',
            'NM_003502.3': 'AXIN1',
            'NM_003537.3': 'HIST1H3B',
            'NM_003563.3': 'SPOP',
            'NM_003588.3': 'CUL4B',
            'NM_003701.3': 'TNFSF11',
            'NM_003738.4': 'PTCH2',
            'NM_003743.4': 'NCOA1',
            'NM_003745.1': 'SOCS1',
            'NM_003749.2': 'IRS2',
            'NM_003750.2': 'EIF3A',
            'NM_003786.3': 'ABCC3',
            'NM_003820.2': 'TNFRSF14',
            'NM_003847.1': 'PEX11A',
            'NM_003879.5': 'CFLAR',
            'NM_003883.3': 'HDAC3',
            'NM_003902.3': 'FUBP1',
            'NM_004001.4': 'FCGR2B',
            'NM_004044.6': 'ATIC',
            'NM_004048.2': 'B2M',
            'NM_004049.3': 'BCL2A1',
            'NM_004050.4': 'BCL2L2',
            'NM_004064.3': 'CDKN1B',
            'NM_004119.2': 'FLT3',
            'NM_004187.3': 'KDM5C',
            'NM_004217.3': 'AURKB',
            'NM_004235.4': 'KLF4',
            'NM_004289.6': 'NFE2L3',
            'NM_004304.4': 'ALK',
            'NM_004327.3': 'BCR',
            'NM_004329.2': 'BMPR1A',
            'NM_004333.4': 'BRAF',
            'NM_004360.3': 'CDH1',
            'NM_004364.3': 'CEBPA',
            'NM_004380.2': 'CREBBP',
            'NM_004430.2': 'EGR3',
            'NM_004431.3': 'EPHA2',
            'NM_004439.5': 'EPHA5',
            'NM_004441.4': 'EPHB1',
            'NM_004445.3': 'EPHB6',
            'NM_004448.2': 'ERBB2',
            'NM_004449.4': 'ERG',
            'NM_004454.2': 'ETV5',
            'NM_004456.4': 'EZH2',
            'NM_004465.1': 'FGF10',
            'NM_004491.4': 'ARHGAP35',
            'NM_004496.3': 'FOXA1',
            'NM_004521.2': 'KIF5B',
            'NM_004525.2': 'LRP2',
            'NM_004557.3': 'NOTCH4',
            'NM_004570.4': 'PIK3C2G',
            'NM_004606.3': 'TAF1',
            'NM_004628.4': 'XPC',
            'NM_004629.1': 'FANCG',
            'NM_004630.3': 'SF1',
            'NM_004655.3': 'AXIN2',
            'NM_004656.2': 'BAP1',
            'NM_004787.1': 'SLIT2',
            'NM_004859.3': 'CLTC',
            'NM_004946.2': 'DOCK2',
            'NM_004956.4': 'ETV1',
            'NM_004957.4': 'FPGS',
            'NM_004958.3': 'MTOR',
            'NM_004960.3': 'FUS',
            'NM_004964.2': 'HDAC1',
            'NM_004972.3': 'JAK2',
            'NM_005018.2': 'PDCD1',
            'NM_005027.2': 'PIK3R2',
            'NM_005030.3': 'PLK1',
            'NM_005080.3': 'XBP1',
            'NM_005089.3': 'ZRSR2',
            'NM_005117.2': 'FGF19',
            'NM_005120.2': 'MED12',
            'NM_005157.4': 'ABL1',
            'NM_005158.4': 'ABL2',
            'NM_005171.4': 'ATF1',
            'NM_005188.3': 'CBL',
            'NM_005191.3': 'CD80',
            'NM_005207.3': 'CRKL',
            'NM_005211.3': 'CSF1R',
            'NM_005214.4': 'CTLA4',
            'NM_005228.3': 'EGFR',
            'NM_005233.5': 'EPHA3',
            'NM_005235.2': 'ERBB4',
            'NM_005241.3': 'MECOM',
            'NM_005247.2': 'FGF3',
            'NM_005319.3': 'HIST1H1C',
            'NM_005343.2': 'HRAS',
            'NM_005356.3': 'LCK',
            'NM_005359.5': 'SMAD4',
            'NM_005373.2': 'MPL',
            'NM_005378.4': 'MYCN',
            'NM_005417.3': 'SRC',
            'NM_005433.3': 'YES1',
            'NM_005436.4': 'CCDC6',
            'NM_005445.3': 'SMC3',
            'NM_005456.3': 'MAPK8IP1',
            'NM_005465.4': 'AKT3',
            'NM_005475.2': 'SH2B3',
            'NM_005484.3': 'PARP2',
            'NM_005590.3': 'MRE11A',
            'NM_005614.3': 'RHEB',
            'NM_005631.4': 'SMO',
            'NM_005656.3': 'TMPRSS2',
            'NM_005724.5': 'TSPAN3',
            'NM_005732.3': 'RAD50',
            'NM_005800.4': 'USPL1',
            'NM_005845.3': 'ABCC4',
            'NM_005877.4': 'SF3A1',
            'NM_005896.2': 'IDH1',
            'NM_005902.3': 'SMAD3',
            'NM_005921.1': 'MAP3K1',
            'NM_005957.4': 'MTHFR',
            'NM_005973.4': 'PRCC',
            'NM_006009.3': 'TUBA1A',
            'NM_006015.4': 'ARID1A',
            'NM_006037.3': 'HDAC4',
            'NM_006044.2': 'HDAC6',
            'NM_006045.1': 'ATP9A',
            'NM_006060.4': 'IKZF1',
            'NM_006070.5': 'TFG',
            'NM_006080.2': 'SEMA3A',
            'NM_006096.3': 'NDRG1',
            'NM_006164.4': 'NFE2L2',
            'NM_006167.3': 'NKX3-1',
            'NM_006180.3': 'NTRK2',
            'NM_006196.3': 'PCBP1',
            'NM_006197.3': 'PCM1',
            'NM_006206.4': 'PDGFRA',
            'NM_006218.2': 'PIK3CA',
            'NM_006219.2': 'PIK3CB',
            'NM_006231.2': 'POLE',
            'NM_006265.2': 'RAD21',
            'NM_006290.3': 'TNFAIP3',
            'NM_006297.2': 'XRCC1',
            'NM_006306.2': 'SMC1A',
            'NM_006311.3': 'NCOR1',
            'NM_006379.3': 'SEMA3C',
            'NM_006424.2': 'SLC34A2',
            'NM_006437.3': 'PARP4',
            'NM_006446.4': 'SLCO1B1',
            'NM_006521.4': 'TFE3',
            'NM_006526.2': 'ZNF217',
            'NM_006540.2': 'NCOA2',
            'NM_006565.3': 'CTCF',
            'NM_006572.4': 'GNA13',
            'NM_006573.4': 'TNFSF13B',
            'NM_006642.3': 'SDCCAG8',
            'NM_006758.2': 'U2AF1',
            'NM_006766.3': 'KAT6A',
            'NM_006813.2': 'PNRC1',
            'NM_006904.6': 'PRKDC',
            'NM_006908.4': 'RAC1',
            'NM_006917.4': 'RXRG',
            'NM_006924.4': 'SRSF1',
            'NM_006941.3': 'SOX10',
            'NM_007013.3': 'WWP1',
            'NM_007120.2': 'UGT1A4',
            'NM_007146.2': 'VEZF1',
            'NM_007194.3': 'CHEK2',
            'NM_007279.2': 'U2AF2',
            'NM_007294.3': 'BRCA1',
            'NM_012092.3': 'ICOS',
            'NM_012099.1': 'CD3EAP',
            'NM_012289.3': 'KEAP1',
            'NM_012431.2': 'SEMA3E',
            'NM_012433.2': 'SF3B1',
            'NM_012448.3': 'STAT5B',
            'NM_013953.3': 'PAX8',
            'NM_013986.3': 'EWSR1',
            'NM_013993.2': 'DDR1',
            'NM_014002.3': 'IKBKE',
            'NM_014143.3': 'CD274',
            'NM_014159.6': 'SETD2',
            'NM_014225.5': 'PPP2R1A',
            'NM_014381.2': 'MLH3',
            'NM_014390.2': 'SND1',
            'NM_014616.2': 'ATP11B',
            'NM_014669.4': 'NUP93',
            'NM_014727.1': 'MLL4',
            'NM_014800.10': 'ELMO1',
            'NM_014868.4': 'RNF10',
            'NM_014903.4': 'NAV3',
            'NM_014953.3': 'DIS3',
            'NM_015001.2': 'SPEN',
            'NM_015074.3': 'KIF1B',
            'NM_015125.3': 'CIC',
            'NM_015247.2': 'CYLD',
            'NM_015253.1': 'WSCD1',
            'NM_015338.5': 'ASXL1',
            'NM_015355.2': 'SUZ12',
            'NM_015508.4': 'TIPARP',
            'NM_015559.2': 'SETBP1',
            'NM_015906.3': 'TRIM33',
            'NM_015991.2': 'C1QA',
            'NM_016169.3': 'SUFU',
            'NM_016261.3': 'TUBD1',
            'NM_016262.4': 'TUBE1',
            'NM_016302.3': 'CRBN',
            'NM_016491.3': 'MRPL37',
            'NM_016507.2': 'CDK12',
            'NM_016569.3': 'TBX3',
            'NM_016734.1': 'PAX5',
            'NM_017449.3': 'EPHB2',
            'NM_017460.5': 'CYP3A4',
            'NM_017519.2': 'ARID1B',
            'NM_017539.1': 'DNAH3',
            'NM_017617.3': 'NOTCH1',
            'NM_017709.3': 'FAM46C',
            'NM_017745.5': 'BCOR',
            'NM_017763.4': 'RNF43',
            'NM_017778.2': 'WHSC1L1',
            'NM_017841.2': 'SDHAF2',
            'NM_017849.3': 'TMEM127',
            'NM_018062.3': 'FANCL',
            'NM_018131.4': 'CEP55',
            'NM_018193.2': 'FANCI',
            'NM_018313.4': 'PBRM1',
            'NM_018486.2': 'HDAC8',
            'NM_018557.2': 'LRP1B',
            'NM_019063.3': 'EML4',
            'NM_019075.2': 'UGT1A10',
            'NM_019076.4': 'UGT1A8',
            'NM_019077.2': 'UGT1A7',
            'NM_019078.1': 'UGT1A5',
            'NM_019093.2': 'UGT1A3',
            'NM_020193.3': 'C11orf30',
            'NM_020328.3': 'ACVR1B',
            'NM_020341.3': 'PAK7',
            'NM_020529.2': 'NFKBIA',
            'NM_020638.2': 'FGF23',
            'NM_020761.2': 'RPTOR',
            'NM_020856.2': 'TSHZ3',
            'NM_020975.4': 'RET',
            'NM_020996.1': 'FGF6',
            'NM_021027.2': 'UGT1A9',
            'NM_021032.4': 'FGF12',
            'NM_021063.3': 'HIST1H2BD',
            'NM_021133.3': 'RNASEL',
            'NM_021140.2': 'KDM6A',
            'NM_021151.3': 'CROT',
            'NM_021181.3': 'SLAMF7',
            'NM_021642.3': 'FCGR2A',
            'NM_021922.2': 'FANCE',
            'NM_021946.4': 'BCORL1',
            'NM_021960.4': 'MCL1',
            'NM_021976.4': 'RXRB',
            'NM_022107.1': 'GPSM3',
            'NM_022304.2': 'HRH2',
            'NM_022454.3': 'SOX17',
            'NM_022455.4': 'NSD1',
            'NM_022552.4': 'DNMT3A',
            'NM_022725.3': 'FANCF',
            'NM_023067.3': 'FOXL2',
            'NM_023110.2': 'FGFR1',
            'NM_024052.4': 'GID4',
            'NM_024083.3': 'ASPSCR1',
            'NM_024408.3': 'NOTCH2',
            'NM_024426.4': 'WT1',
            'NM_024529.4': 'CDC73',
            'NM_024665.4': 'TBL1XR1',
            'NM_024675.3': 'PALB2',
            'NM_024800.4': 'NEK11',
            'NM_025069.1': 'ZNF703',
            'NM_030662.3': 'MAP2K2',
            'NM_030964.3': 'SPRY4',
            'NM_031226.2': 'CYP19A1',
            'NM_031308.1': 'EPPK1',
            'NM_032043.2': 'BRIP1',
            'NM_032199.2': 'ARID5B',
            'NM_032271.2': 'TRAF7',
            'NM_032415.4': 'CARD11',
            'NM_032427.1': 'MAML2',
            'NM_032482.2': 'DOT1L',
            'NM_032583.3': 'ABCC11',
            'NM_032611.1': 'PTP4A3',
            'NM_032638.4': 'GATA2',
            'NM_032777.9': 'GPR124',
            'NM_032834.3': 'ALG10',
            'NM_032855.2': 'HSH2D',
            'NM_032876.4': 'AJUBA',
            'NM_033084.3': 'FANCD2',
            'NM_033102.2': 'SLC45A3',
            'NM_033238.2': 'PML',
            'NM_033360.2': 'KRAS',
            'NM_033632.3': 'FBXW7',
            'NM_052958.2': 'C8orf34',
            'NM_053056.2': 'CCND1',
            'NM_058216.1': 'RAD51C',
            'NM_058243.2': 'BRD4',
            'NM_078467.2': 'CDKN1A',
            'NM_078487.2': 'CDKN2B',
            'NM_080491.2': 'GAB2',
            'NM_130446.2': 'KLHL6',
            'NM_130803.2': 'MEN1',
            'NM_133509.3': 'RAD51B',
            'NM_133629.2': 'RAD51D',
            'NM_134424.2': 'RAD52',
            'NM_134442.3': 'CREB1',
            'NM_138295.3': 'PKD1L1',
            'NM_138554.4': 'TLR4',
            'NM_138578.1': 'BCL2L1',
            'NM_138621.4': 'BCL2L11',
            'NM_138712.3': 'PPARG',
            'NM_138715.2': 'MSR1',
            'NM_139049.1': 'MAPK8',
            'NM_139215.2': 'TAF15',
            'NM_139241.2': 'FGD4',
            'NM_139276.2': 'STAT3',
            'NM_144997.5': 'FLCN',
            'NM_145003.3': 'TSNARE1',
            'NM_145740.3': 'GSTA1',
            'NM_148961.3': 'OTOS',
            'NM_152424.3': 'FAM123B',
            'NM_152641.2': 'ARID2',
            'NM_152756.3': 'RICTOR',
            'NM_152866.2': 'MS4A1',
            'NM_153000.4': 'APCDD1',
            'NM_153649.3': 'TPM3',
            'NM_153675.2': 'FOXA2',
            'NM_170606.2': 'MLL3',
            'NM_170662.3': 'CBLB',
            'NM_172165.3': 'MSH5',
            'NM_172173.2': 'CAMK2G',
            'NM_173200.2': 'NR4A3',
            'NM_173354.3': 'SIK1',
            'NM_173485.5': 'TSHZ2',
            'NM_175061.3': 'JAZF1',
            'NM_175635.2': 'RUNX1T1',
            'NM_175741.1': 'C15orf55',
            'NM_175918.3': 'CRIPAK',
            'NM_175929.2': 'FGF14',
            'NM_177417.2': 'KLC3',
            'NM_177559.2': 'CSNK2A1',
            'NM_178014.2': 'TUBB',
            'NM_181523.2': 'PIK3R1',
            'NM_181830.2': 'NF2',
            'NM_181882.2': 'PRX',
            'NM_182729.2': 'TXNRD1',
            'NM_194255.2': 'SLC19A1',
            'NM_197957.3': 'MAX',
            'NM_198239.1': 'WISP3',
            'NM_198253.2': 'TERT',
            'NM_198433.1': 'AURKA',
            'NM_198578.3': 'LRRK2',
            'NM_199420.3': 'POLQ',
            'NM_201442.2': 'C1S',
            'NM_202758.3': 'ENOSF1',
            'NM_203394.2': 'E2F7',
            'NM_206866.1': 'BACH1',
            'NM_206907.3': 'PRKAA1',
            'NM_207585.1': 'IFNAR2',
            'NM_212471.1': 'PRKAR1A',
            'NM_213647.1': 'FGFR4',
            'NM_213649.1': 'SFXN4',
            'NR_002785.2': 'GNAS-AS1',
            'NR_002819.2': 'MALAT1',
            'NR_004053.2': 'SOX2-OT',
            'NR_029683.1': 'MIR142',
            'NR_038892.1': 'CBR3-AS1',
            'NR_038982.1': 'LOC100507346',
            'NR_039881.2': 'MIR4728',
            'NR_045122.1': 'LOC100506321',
            'NR_047648.1': 'FCGR2C'
        }

    def parse(
            self,
            hgvs_spec: str,
    ):
        """parse the input chgvs or ghgvs variant and project to other sequences

        Args:
            hgvs_spec (str): standard hgvs nonmenclature of chgvs and gchgvs are accepted. 
                for compromize, variant like "EGFR:c.2573T>G" using default transcription or "NC_000007.13:g.55259515T>G" are also accepted.

        Raises:
            ValueError: unsupport variant specification
        """
        if re.findall(r'c\.\d+', hgvs_spec):
            self.var_c = self.hp.parse_hgvs_variant(self.norm_c(hgvs_spec))
            self.var_g = self.am.c_to_g(self.var_c)
        elif re.findall(r'g\.\d+', hgvs_spec):
            self.var_g = self.hp.parse_hgvs_variant(self.norm_g(hgvs_spec))
            nms = self.am.relevant_transcripts(self.var_g)
            nm_in_list = [
                nm for nm in nms if nm in self.nm_to_hugo_dict.keys()
            ]
            nm = nm_in_list[0] if len(nm_in_list) > 0 else nms[0]
            self.var_c = self.am.g_to_c(self.var_g, nm)
        else:
            raise ValueError(
                'unsupported variant description, please specify "c." or "g.".'
            )

        self.var_p = self.am.c_to_p(self.var_c)

    def get_g(self) -> hgvs.sequencevariant.SequenceVariant:
        """get ghgvs

        Returns:
            [type]: [description]
        """
        return str(self.var_g)

    def get_c(self) -> hgvs.sequencevariant.SequenceVariant:
        """get chgvs

        Returns:
            hgvs.sequencevariant.SequenceVariant: [description]
        """
        return str(self.var_c)

    def get_p(self) -> hgvs.sequencevariant.SequenceVariant:
        """get phgvs

        Returns:
            hgvs.sequencevariant.SequenceVariant: [description]
        """
        return str(self.var_p)

    def get_hugo(self) -> str:
        """get hugo gene name for default transcript

        Returns:
            str: [description]
        """

        return self.nm_to_hugo_dict[
            self.var_c.ac] if self.var_c.ac in self.nm_to_hugo_dict.keys(
            ) else 'undefined'

    def validate(self):
        pass

    def normalize(self):
        pass

    def _chrid_to_NC(self, chrid: str) -> str:
        chrid_to_NC_dict = {
            '1': 'NC_000001.10',
            '2': 'NC_000002.11',
            '3': 'NC_000003.11',
            '4': 'NC_000004.11',
            '5': 'NC_000005.9',
            '6': 'NC_000006.11',
            '7': 'NC_000007.13',
            '8': 'NC_000008.10',
            '9': 'NC_000009.11',
            '10': 'NC_000010.10',
            '11': 'NC_000011.9',
            '12': 'NC_000012.11',
            '13': 'NC_000013.10',
            '14': 'NC_000014.8',
            '15': 'NC_000015.9',
            '16': 'NC_000016.9',
            '17': 'NC_000017.10',
            '18': 'NC_000018.9',
            '19': 'NC_000019.9',
            '20': 'NC_000020.10',
            '21': 'NC_000021.8',
            '22': 'NC_000022.10',
            'X': 'NC_000023.10',
            'Y': 'NC_000024.9'
        }
        if chrid in chrid_to_NC_dict.keys():
            return chrid_to_NC_dict[chrid]
        else:
            raise ValueError('unsupported chr id, please set 1-22 or X,Y')

    def norm_g(self, ghgvs: str):
        if not re.findall('NC', ghgvs):
            splits = re.findall(r'^([Cc]hr)?([1-9XY]+):g\.\d+', ghgvs)
            if splits:
                NC = self._chrid_to_NC(splits[0][1])
            else:
                raise ValueError('unsupported chr specification.')

            ghgvs = NC + ':' + ghgvs.split(':')[1]

        return ghgvs

    def _hugo_to_nm(self, hugo: str) -> str:
        hugo_to_nm_dict = {
            'ACVRL1': 'NM_000020.2',
            'APC': 'NM_000038.5',
            'AR': 'NM_000044.3',
            'ATM': 'NM_000051.3',
            'BLM': 'NM_000057.2',
            'BRCA2': 'NM_000059.3',
            'BTK': 'NM_000061.2',
            'CD3G': 'NM_000073.2',
            'CDK4': 'NM_000075.3',
            'CDKN2A': 'NM_000077.4',
            'CYP17A1': 'NM_000102.3',
            'CYP2D6': 'NM_000106.5',
            'DPYD': 'NM_000110.3',
            'EPOR': 'NM_000121.3',
            'ERCC3': 'NM_000122.1',
            'EXT1': 'NM_000127.2',
            'FANCA': 'NM_000135.2',
            'FANCC': 'NM_000136.2',
            'FGFR2': 'NM_000141.4',
            'FGFR3': 'NM_000142.4',
            'FH': 'NM_000143.3',
            'MSH6': 'NM_000179.2',
            'HSD17B3': 'NM_000197.1',
            'HSD3B2': 'NM_000198.3',
            'IL2RG': 'NM_000206.2',
            'ITGB2': 'NM_000211.3',
            'JAK3': 'NM_000215.3',
            'KIT': 'NM_000222.2',
            'LHCGR': 'NM_000233.3',
            'KCNH2': 'NM_000238.3',
            'MET': 'NM_000245.2',
            'MITF': 'NM_000248.3',
            'MLH1': 'NM_000249.3',
            'MSH2': 'NM_000251.2',
            'MTR': 'NM_000254.2',
            'PTCH1': 'NM_000264.3',
            'NF1': 'NM_000267.3',
            'PTEN': 'NM_000314.4',
            'RB1': 'NM_000321.2',
            'SLC4A1': 'NM_000342.3',
            'SOX9': 'NM_000346.3',
            'SRD5A2': 'NM_000348.3',
            'TPMT': 'NM_000367.2',
            'TSC1': 'NM_000368.4',
            'TSHR': 'NM_000369.2',
            'TYR': 'NM_000372.4',
            'UMPS': 'NM_000373.3',
            'XPA': 'NM_000380.3',
            'ABCC2': 'NM_000392.3',
            'ERCC2': 'NM_000400.3',
            'EXT2': 'NM_000401.3',
            'GNRHR': 'NM_000406.2',
            'IL2RA': 'NM_000417.2',
            'NOTCH3': 'NM_000435.2',
            'STK11': 'NM_000455.4',
            'TEK': 'NM_000459.3',
            'UGT1A1': 'NM_000463.2',
            'BARD1': 'NM_000465.2',
            'ATRX': 'NM_000489.3',
            'C1QB': 'NM_000491.3',
            'GNAS': 'NM_000516.4',
            'PMS1': 'NM_000534.4',
            'PMS2': 'NM_000535.5',
            'HNF1A': 'NM_000545.5',
            'TP53': 'NM_000546.5',
            'TSC2': 'NM_000548.3',
            'VHL': 'NM_000551.3',
            'FCGR1A': 'NM_000566.3',
            'FCGR3A': 'NM_000569.6',
            'IL6': 'NM_000600.3',
            'HGF': 'NM_000601.4',
            'IGF1': 'NM_000618.3',
            'CD79B': 'NM_000626.2',
            'IFNAR1': 'NM_000629.2',
            'BCL2': 'NM_000633.2',
            'IL13RA2': 'NM_000640.2',
            'CD3D': 'NM_000732.4',
            'CD3E': 'NM_000733.3',
            'COMT': 'NM_000754.3',
            'CYP2C8': 'NM_000770.3',
            'CYP3A5': 'NM_000777.3',
            'CYP4A11': 'NM_000778.3',
            'CYP24A1': 'NM_000782.4',
            'DCK': 'NM_000788.2',
            'DHFR': 'NM_000791.3',
            'GRIN2A': 'NM_000833.3',
            'GRM3': 'NM_000840.2',
            'GSTP1': 'NM_000852.3',
            'IGF1R': 'NM_000875.3',
            'IL2RB': 'NM_000878.3',
            'ABCB1': 'NM_000927.4',
            'RARA': 'NM_000964.3',
            'RARB': 'NM_000965.3',
            'RARG': 'NM_000966.5',
            'RPL5': 'NM_000969.3',
            'RPL22': 'NM_000983.3',
            'CD44': 'NM_001001391.1',
            'GATA3': 'NM_001002295.1',
            'PARP3': 'NM_001003931.2',
            'OR1L1': 'NM_001005236.3',
            'OR11G2': 'NM_001005503.1',
            'NTRK1': 'NM_001007792.1',
            'FAT3': 'NM_001008781.2',
            'CUL4A': 'NM_001008895.1',
            'TECRL': 'NM_001010874.4',
            'CRLF2': 'NM_001012288.1',
            'H3F3C': 'NM_001013699.2',
            'AKT1': 'NM_001014432.1',
            'DDR2': 'NM_001014796.1',
            'PHF6': 'NM_001015877.1',
            'HSP90AA1': 'NM_001017963.2',
            'NR3C1': 'NM_001018077.1',
            'SOD2': 'NM_001024465.1',
            'RPS14': 'NM_001025071.1',
            'CD74': 'NM_001025159.2',
            'VEGFA': 'NM_001025366.2',
            'SRSF7': 'NM_001031684.2',
            'PRPF40B': 'NM_001031698.2',
            'RRM1': 'NM_001033.3',
            'MYCL1': 'NM_001033082.2',
            'USP9X': 'NM_001039590.2',
            'ESR2': 'NM_001040275.1',
            'KDM5A': 'NM_001042603.1',
            'STAG2': 'NM_001042749.1',
            'SSTR2': 'NM_001050.2',
            'SSTR3': 'NM_001051.2',
            'TACR1': 'NM_001058.3',
            'TOP2A': 'NM_001067.3',
            'TOP2B': 'NM_001068.2',
            'TUBG1': 'NM_001070.4',
            'TYMS': 'NM_001071.2',
            'UGT1A6': 'NM_001072.3',
            'SLC29A1': 'NM_001078177.1',
            'ETV4': 'NM_001079675.2',
            'CRTC1': 'NM_001098482.1',
            'XRCC3': 'NM_001100119.1',
            'KNG1': 'NM_001102416.2',
            'C1QC': 'NM_001114101.1',
            'CHEK1': 'NM_001114122.2',
            'ELF3': 'NM_001114309.1',
            'ESR1': 'NM_001122742.1',
            'TET2': 'NM_001127208.2',
            'IGF2': 'NM_001127598.1',
            'PAK3': 'NM_001128166.1',
            'MUTYH': 'NM_001128425.1',
            'SMARCA4': 'NM_001128844.1',
            'MYB': 'NM_001130173.1',
            'DNMT1': 'NM_001130823.1',
            'SYK': 'NM_001135052.2',
            'SMAD2': 'NM_001135937.2',
            'ALOX12B': 'NM_001139.2',
            'DAXX': 'NM_001141970.1',
            'RAD54L': 'NM_001142548.1',
            'NCOA4': 'NM_001145260.1',
            'CDK6': 'NM_001145306.1',
            'SIN3A': 'NM_001145358.1',
            'MEF2B': 'NM_001145785.1',
            'ANGPT1': 'NM_001146.3',
            'ANGPT2': 'NM_001147.2',
            'KCNQ5': 'NM_001160133.1',
            'KIAA1549': 'NM_001164665.1',
            'BIRC3': 'NM_001165.4',
            'ELAC2': 'NM_001165962.1',
            'HCK': 'NM_001172133.1',
            'SSTR5': 'NM_001172560.1',
            'RRAS2': 'NM_001177314.1',
            'ATR': 'NM_001184.3',
            'BAK1': 'NM_001188.3',
            'NFIB': 'NM_001190737.1',
            'MLL': 'NM_001197104.1',
            'PRDM1': 'NM_001198.3',
            'B4GALT3': 'NM_001199874.1',
            'CHURC1-FNTB': 'NM_001202559.1',
            'XIAP': 'NM_001204401.1',
            'RFC1': 'NM_001204747.1',
            'LIPT1': 'NM_001204830.1',
            'CASP8': 'NM_001228.4',
            'CBR3': 'NM_001236.3',
            'CCNE1': 'NM_001238.2',
            'MAP3K13': 'NM_001242314.1',
            'HLA-A': 'NM_001242758.1',
            'TNFRSF8': 'NM_001243.3',
            'STAT4': 'NM_001243835.1',
            'FCGR3B': 'NM_001244753.1',
            'BIRC2': 'NM_001256163.1',
            'POLD1': 'NM_001256849.1',
            'CDK8': 'NM_001260.1',
            'CDKN2C': 'NM_001262.2',
            'CDX2': 'NM_001265.4',
            'ARFRP1': 'NM_001267547.1',
            'CHD1': 'NM_001270.2',
            'CHD2': 'NM_001271.3',
            'CHD4': 'NM_001273.2',
            'CHUK': 'NM_001278.3',
            'EP300': 'NM_001429.3',
            'HDAC2': 'NM_001527.3',
            'HIF1A': 'NM_001530.3',
            'IKBKB': 'NM_001556.2',
            'ACVR2A': 'NM_001616.3',
            'PARP1': 'NM_001618.3',
            'AKT2': 'NM_001626.4',
            'ARAF': 'NM_001654.4',
            'AREG': 'NM_001657.2',
            'RHOA': 'NM_001664.2',
            'AXL': 'NM_001699.4',
            'BCL6': 'NM_001706.4',
            'BTG1': 'NM_001731.2',
            'C1R': 'NM_001733.4',
            'RUNX1': 'NM_001754.4',
            'CBFB': 'NM_001755.2',
            'CBR1': 'NM_001757.2',
            'CCND2': 'NM_001759.3',
            'CCND3': 'NM_001760.3',
            'CD22': 'NM_001771.3',
            'CD33': 'NM_001772.3',
            'CD79A': 'NM_001783.3',
            'CDA': 'NM_001785.2',
            'CDC25C': 'NM_001790.3',
            'CDC42': 'NM_001791.3',
            'CDK2': 'NM_001798.3',
            'CD52': 'NM_001803.2',
            'SLC31A1': 'NM_001859.3',
            'CTNNA1': 'NM_001903.2',
            'CTNNB1': 'NM_001904.3',
            'DUSP6': 'NM_001946.2',
            'EDNRA': 'NM_001957.3',
            'EIF4A2': 'NM_001967.3',
            'ERBB3': 'NM_001982.3',
            'ERCC1': 'NM_001983.3',
            'ETV6': 'NM_001987.4',
            'FGF4': 'NM_002007.2',
            'FGF7': 'NM_002009.3',
            'FLI1': 'NM_002017.4',
            'FLT1': 'NM_002019.4',
            'FLT4': 'NM_002020.4',
            'FNTA': 'NM_002027.2',
            'FNTB': 'NM_002028.3',
            'FYN': 'NM_002037.5',
            'GATA1': 'NM_002049.3',
            'GNA11': 'NM_002067.2',
            'GNAQ': 'NM_002072.3',
            'GSK3B': 'NM_002093.3',
            'H3F3A': 'NM_002107.4',
            'HSPA4': 'NM_002154.3',
            'IDH2': 'NM_002168.2',
            'IL7R': 'NM_002185.3',
            'INHBA': 'NM_002192.2',
            'JAK1': 'NM_002227.2',
            'JUN': 'NM_002228.3',
            'KDR': 'NM_002253.2',
            'LIFR': 'NM_002310.5',
            'LIMK1': 'NM_002314.3',
            'LMO1': 'NM_002315.2',
            'LYN': 'NM_002350.3',
            'EPCAM': 'NM_002354.2',
            'MC1R': 'NM_002386.3',
            'MDM2': 'NM_002392.4',
            'MDM4': 'NM_002393.4',
            'MSH3': 'NM_002439.4',
            'MSH4': 'NM_002440.3',
            'MTRR': 'NM_002454.2',
            'MUC1': 'NM_002456.5',
            'IRF4': 'NM_002460.3',
            'MYC': 'NM_002467.4',
            'MYD88': 'NM_002468.4',
            'NBN': 'NM_002485.4',
            'NPM1': 'NM_002520.6',
            'NRAS': 'NM_002524.4',
            'NTRK3': 'NM_002530.3',
            'PBX1': 'NM_002585.3',
            'PDGFRB': 'NM_002609.3',
            'PDK1': 'NM_002610.3',
            'SERPINB5': 'NM_002639.4',
            'PIGF': 'NM_002643.3',
            'PIK3C2A': 'NM_002645.2',
            'PIK3C2B': 'NM_002646.3',
            'PIK3C3': 'NM_002647.2',
            'PIK3CG': 'NM_002649.2',
            'PLAG1': 'NM_002655.2',
            'POU5F1': 'NM_002701.4',
            'PRKCA': 'NM_002737.2',
            'PRKCB': 'NM_002738.6',
            'PRKCG': 'NM_002739.3',
            'MAPK1': 'NM_002745.4',
            'MAPK3': 'NM_002746.2',
            'MAP2K1': 'NM_002755.3',
            'PRSS8': 'NM_002773.3',
            'PSMB1': 'NM_002793.3',
            'PSMB2': 'NM_002794.4',
            'PSMB5': 'NM_002797.3',
            'PTPN11': 'NM_002834.3',
            'PTPRD': 'NM_002839.3',
            'RAC2': 'NM_002872.3',
            'RAD51': 'NM_002875.4',
            'RAF1': 'NM_002880.3',
            'REL': 'NM_002908.2',
            'ROBO1': 'NM_002941.3',
            'ROBO2': 'NM_002942.4',
            'ROS1': 'NM_002944.2',
            'RPA1': 'NM_002945.3',
            'RXRA': 'NM_002957.4',
            'SDHB': 'NM_003000.2',
            'SDHC': 'NM_003001.3',
            'SDHD': 'NM_003002.2',
            'MAP2K4': 'NM_003010.2',
            'SRSF2': 'NM_003016.4',
            'SLC22A2': 'NM_003058.3',
            'SLIT1': 'NM_003061.2',
            'SMARCA1': 'NM_003069.3',
            'SMARCB1': 'NM_003073.3',
            'SMARCD1': 'NM_003076.4',
            'SOX2': 'NM_003106.3',
            'RPS6KB1': 'NM_003161.2',
            'TCF3': 'NM_003200.3',
            'TGFBR2': 'NM_003242.5',
            'TOP1': 'NM_003286.2',
            'NKX2-1': 'NM_003317.3',
            'VEGFB': 'NM_003377.4',
            'EZR': 'NM_003379.4',
            'XPO1': 'NM_003400.3',
            'MLL2': 'NM_003482.3',
            'TRRAP': 'NM_003496.3',
            'AXIN1': 'NM_003502.3',
            'HIST1H3B': 'NM_003537.3',
            'SPOP': 'NM_003563.3',
            'CUL4B': 'NM_003588.3',
            'TNFSF11': 'NM_003701.3',
            'PTCH2': 'NM_003738.4',
            'NCOA1': 'NM_003743.4',
            'SOCS1': 'NM_003745.1',
            'IRS2': 'NM_003749.2',
            'EIF3A': 'NM_003750.2',
            'ABCC3': 'NM_003786.3',
            'TNFRSF14': 'NM_003820.2',
            'PEX11A': 'NM_003847.1',
            'CFLAR': 'NM_003879.5',
            'HDAC3': 'NM_003883.3',
            'FUBP1': 'NM_003902.3',
            'FCGR2B': 'NM_004001.4',
            'ATIC': 'NM_004044.6',
            'B2M': 'NM_004048.2',
            'BCL2A1': 'NM_004049.3',
            'BCL2L2': 'NM_004050.4',
            'CDKN1B': 'NM_004064.3',
            'FLT3': 'NM_004119.2',
            'KDM5C': 'NM_004187.3',
            'AURKB': 'NM_004217.3',
            'KLF4': 'NM_004235.4',
            'NFE2L3': 'NM_004289.6',
            'ALK': 'NM_004304.4',
            'BCR': 'NM_004327.3',
            'BMPR1A': 'NM_004329.2',
            'BRAF': 'NM_004333.4',
            'CDH1': 'NM_004360.3',
            'CEBPA': 'NM_004364.3',
            'CREBBP': 'NM_004380.2',
            'EGR3': 'NM_004430.2',
            'EPHA2': 'NM_004431.3',
            'EPHA5': 'NM_004439.5',
            'EPHB1': 'NM_004441.4',
            'EPHB6': 'NM_004445.3',
            'ERBB2': 'NM_004448.2',
            'ERG': 'NM_004449.4',
            'ETV5': 'NM_004454.2',
            'EZH2': 'NM_004456.4',
            'FGF10': 'NM_004465.1',
            'ARHGAP35': 'NM_004491.4',
            'FOXA1': 'NM_004496.3',
            'KIF5B': 'NM_004521.2',
            'LRP2': 'NM_004525.2',
            'NOTCH4': 'NM_004557.3',
            'PIK3C2G': 'NM_004570.4',
            'TAF1': 'NM_004606.3',
            'XPC': 'NM_004628.4',
            'FANCG': 'NM_004629.1',
            'SF1': 'NM_004630.3',
            'AXIN2': 'NM_004655.3',
            'BAP1': 'NM_004656.2',
            'SLIT2': 'NM_004787.1',
            'CLTC': 'NM_004859.3',
            'DOCK2': 'NM_004946.2',
            'ETV1': 'NM_004956.4',
            'FPGS': 'NM_004957.4',
            'MTOR': 'NM_004958.3',
            'FUS': 'NM_004960.3',
            'HDAC1': 'NM_004964.2',
            'JAK2': 'NM_004972.3',
            'PDCD1': 'NM_005018.2',
            'PIK3R2': 'NM_005027.2',
            'PLK1': 'NM_005030.3',
            'XBP1': 'NM_005080.3',
            'ZRSR2': 'NM_005089.3',
            'FGF19': 'NM_005117.2',
            'MED12': 'NM_005120.2',
            'ABL1': 'NM_005157.4',
            'ABL2': 'NM_005158.4',
            'ATF1': 'NM_005171.4',
            'CBL': 'NM_005188.3',
            'CD80': 'NM_005191.3',
            'CRKL': 'NM_005207.3',
            'CSF1R': 'NM_005211.3',
            'CTLA4': 'NM_005214.4',
            'EGFR': 'NM_005228.3',
            'EPHA3': 'NM_005233.5',
            'ERBB4': 'NM_005235.2',
            'MECOM': 'NM_005241.3',
            'FGF3': 'NM_005247.2',
            'HIST1H1C': 'NM_005319.3',
            'HRAS': 'NM_005343.2',
            'LCK': 'NM_005356.3',
            'SMAD4': 'NM_005359.5',
            'MPL': 'NM_005373.2',
            'MYCN': 'NM_005378.4',
            'SRC': 'NM_005417.3',
            'YES1': 'NM_005433.3',
            'CCDC6': 'NM_005436.4',
            'SMC3': 'NM_005445.3',
            'MAPK8IP1': 'NM_005456.3',
            'AKT3': 'NM_005465.4',
            'SH2B3': 'NM_005475.2',
            'PARP2': 'NM_005484.3',
            'MRE11A': 'NM_005590.3',
            'RHEB': 'NM_005614.3',
            'SMO': 'NM_005631.4',
            'TMPRSS2': 'NM_005656.3',
            'TSPAN3': 'NM_005724.5',
            'RAD50': 'NM_005732.3',
            'USPL1': 'NM_005800.4',
            'ABCC4': 'NM_005845.3',
            'SF3A1': 'NM_005877.4',
            'IDH1': 'NM_005896.2',
            'SMAD3': 'NM_005902.3',
            'MAP3K1': 'NM_005921.1',
            'MTHFR': 'NM_005957.4',
            'PRCC': 'NM_005973.4',
            'TUBA1A': 'NM_006009.3',
            'ARID1A': 'NM_006015.4',
            'HDAC4': 'NM_006037.3',
            'HDAC6': 'NM_006044.2',
            'ATP9A': 'NM_006045.1',
            'IKZF1': 'NM_006060.4',
            'TFG': 'NM_006070.5',
            'SEMA3A': 'NM_006080.2',
            'NDRG1': 'NM_006096.3',
            'NFE2L2': 'NM_006164.4',
            'NKX3-1': 'NM_006167.3',
            'NTRK2': 'NM_006180.3',
            'PCBP1': 'NM_006196.3',
            'PCM1': 'NM_006197.3',
            'PDGFRA': 'NM_006206.4',
            'PIK3CA': 'NM_006218.2',
            'PIK3CB': 'NM_006219.2',
            'POLE': 'NM_006231.2',
            'RAD21': 'NM_006265.2',
            'TNFAIP3': 'NM_006290.3',
            'XRCC1': 'NM_006297.2',
            'SMC1A': 'NM_006306.2',
            'NCOR1': 'NM_006311.3',
            'SEMA3C': 'NM_006379.3',
            'SLC34A2': 'NM_006424.2',
            'PARP4': 'NM_006437.3',
            'SLCO1B1': 'NM_006446.4',
            'TFE3': 'NM_006521.4',
            'ZNF217': 'NM_006526.2',
            'NCOA2': 'NM_006540.2',
            'CTCF': 'NM_006565.3',
            'GNA13': 'NM_006572.4',
            'TNFSF13B': 'NM_006573.4',
            'SDCCAG8': 'NM_006642.3',
            'U2AF1': 'NM_006758.2',
            'KAT6A': 'NM_006766.3',
            'PNRC1': 'NM_006813.2',
            'PRKDC': 'NM_006904.6',
            'RAC1': 'NM_006908.4',
            'RXRG': 'NM_006917.4',
            'SRSF1': 'NM_006924.4',
            'SOX10': 'NM_006941.3',
            'WWP1': 'NM_007013.3',
            'UGT1A4': 'NM_007120.2',
            'VEZF1': 'NM_007146.2',
            'CHEK2': 'NM_007194.3',
            'U2AF2': 'NM_007279.2',
            'BRCA1': 'NM_007294.3',
            'ICOS': 'NM_012092.3',
            'CD3EAP': 'NM_012099.1',
            'KEAP1': 'NM_012289.3',
            'SEMA3E': 'NM_012431.2',
            'SF3B1': 'NM_012433.2',
            'STAT5B': 'NM_012448.3',
            'PAX8': 'NM_013953.3',
            'EWSR1': 'NM_013986.3',
            'DDR1': 'NM_013993.2',
            'IKBKE': 'NM_014002.3',
            'CD274': 'NM_014143.3',
            'SETD2': 'NM_014159.6',
            'PPP2R1A': 'NM_014225.5',
            'MLH3': 'NM_014381.2',
            'SND1': 'NM_014390.2',
            'ATP11B': 'NM_014616.2',
            'NUP93': 'NM_014669.4',
            'MLL4': 'NM_014727.1',
            'ELMO1': 'NM_014800.10',
            'RNF10': 'NM_014868.4',
            'NAV3': 'NM_014903.4',
            'DIS3': 'NM_014953.3',
            'SPEN': 'NM_015001.2',
            'KIF1B': 'NM_015074.3',
            'CIC': 'NM_015125.3',
            'CYLD': 'NM_015247.2',
            'WSCD1': 'NM_015253.1',
            'ASXL1': 'NM_015338.5',
            'SUZ12': 'NM_015355.2',
            'TIPARP': 'NM_015508.4',
            'SETBP1': 'NM_015559.2',
            'TRIM33': 'NM_015906.3',
            'C1QA': 'NM_015991.2',
            'SUFU': 'NM_016169.3',
            'TUBD1': 'NM_016261.3',
            'TUBE1': 'NM_016262.4',
            'CRBN': 'NM_016302.3',
            'MRPL37': 'NM_016491.3',
            'CDK12': 'NM_016507.2',
            'TBX3': 'NM_016569.3',
            'PAX5': 'NM_016734.1',
            'EPHB2': 'NM_017449.3',
            'CYP3A4': 'NM_017460.5',
            'ARID1B': 'NM_017519.2',
            'DNAH3': 'NM_017539.1',
            'NOTCH1': 'NM_017617.3',
            'FAM46C': 'NM_017709.3',
            'BCOR': 'NM_017745.5',
            'RNF43': 'NM_017763.4',
            'WHSC1L1': 'NM_017778.2',
            'SDHAF2': 'NM_017841.2',
            'TMEM127': 'NM_017849.3',
            'FANCL': 'NM_018062.3',
            'CEP55': 'NM_018131.4',
            'FANCI': 'NM_018193.2',
            'PBRM1': 'NM_018313.4',
            'HDAC8': 'NM_018486.2',
            'LRP1B': 'NM_018557.2',
            'EML4': 'NM_019063.3',
            'UGT1A10': 'NM_019075.2',
            'UGT1A8': 'NM_019076.4',
            'UGT1A7': 'NM_019077.2',
            'UGT1A5': 'NM_019078.1',
            'UGT1A3': 'NM_019093.2',
            'C11orf30': 'NM_020193.3',
            'ACVR1B': 'NM_020328.3',
            'PAK7': 'NM_020341.3',
            'NFKBIA': 'NM_020529.2',
            'FGF23': 'NM_020638.2',
            'RPTOR': 'NM_020761.2',
            'TSHZ3': 'NM_020856.2',
            'RET': 'NM_020975.4',
            'FGF6': 'NM_020996.1',
            'UGT1A9': 'NM_021027.2',
            'FGF12': 'NM_021032.4',
            'HIST1H2BD': 'NM_021063.3',
            'RNASEL': 'NM_021133.3',
            'KDM6A': 'NM_021140.2',
            'CROT': 'NM_021151.3',
            'SLAMF7': 'NM_021181.3',
            'FCGR2A': 'NM_021642.3',
            'FANCE': 'NM_021922.2',
            'BCORL1': 'NM_021946.4',
            'MCL1': 'NM_021960.4',
            'RXRB': 'NM_021976.4',
            'GPSM3': 'NM_022107.1',
            'HRH2': 'NM_022304.2',
            'SOX17': 'NM_022454.3',
            'NSD1': 'NM_022455.4',
            'DNMT3A': 'NM_022552.4',
            'FANCF': 'NM_022725.3',
            'FOXL2': 'NM_023067.3',
            'FGFR1': 'NM_023110.2',
            'GID4': 'NM_024052.4',
            'ASPSCR1': 'NM_024083.3',
            'NOTCH2': 'NM_024408.3',
            'WT1': 'NM_024426.4',
            'CDC73': 'NM_024529.4',
            'TBL1XR1': 'NM_024665.4',
            'PALB2': 'NM_024675.3',
            'NEK11': 'NM_024800.4',
            'ZNF703': 'NM_025069.1',
            'MAP2K2': 'NM_030662.3',
            'SPRY4': 'NM_030964.3',
            'CYP19A1': 'NM_031226.2',
            'EPPK1': 'NM_031308.1',
            'BRIP1': 'NM_032043.2',
            'ARID5B': 'NM_032199.2',
            'TRAF7': 'NM_032271.2',
            'CARD11': 'NM_032415.4',
            'MAML2': 'NM_032427.1',
            'DOT1L': 'NM_032482.2',
            'ABCC11': 'NM_032583.3',
            'PTP4A3': 'NM_032611.1',
            'GATA2': 'NM_032638.4',
            'GPR124': 'NM_032777.9',
            'ALG10': 'NM_032834.3',
            'HSH2D': 'NM_032855.2',
            'AJUBA': 'NM_032876.4',
            'FANCD2': 'NM_033084.3',
            'SLC45A3': 'NM_033102.2',
            'PML': 'NM_033238.2',
            'KRAS': 'NM_033360.2',
            'FBXW7': 'NM_033632.3',
            'C8orf34': 'NM_052958.2',
            'CCND1': 'NM_053056.2',
            'RAD51C': 'NM_058216.1',
            'BRD4': 'NM_058243.2',
            'CDKN1A': 'NM_078467.2',
            'CDKN2B': 'NM_078487.2',
            'GAB2': 'NM_080491.2',
            'KLHL6': 'NM_130446.2',
            'MEN1': 'NM_130803.2',
            'RAD51B': 'NM_133509.3',
            'RAD51D': 'NM_133629.2',
            'RAD52': 'NM_134424.2',
            'CREB1': 'NM_134442.3',
            'PKD1L1': 'NM_138295.3',
            'TLR4': 'NM_138554.4',
            'BCL2L1': 'NM_138578.1',
            'BCL2L11': 'NM_138621.4',
            'PPARG': 'NM_138712.3',
            'MSR1': 'NM_138715.2',
            'MAPK8': 'NM_139049.1',
            'TAF15': 'NM_139215.2',
            'FGD4': 'NM_139241.2',
            'STAT3': 'NM_139276.2',
            'FLCN': 'NM_144997.5',
            'TSNARE1': 'NM_145003.3',
            'GSTA1': 'NM_145740.3',
            'OTOS': 'NM_148961.3',
            'FAM123B': 'NM_152424.3',
            'ARID2': 'NM_152641.2',
            'RICTOR': 'NM_152756.3',
            'MS4A1': 'NM_152866.2',
            'APCDD1': 'NM_153000.4',
            'TPM3': 'NM_153649.3',
            'FOXA2': 'NM_153675.2',
            'MLL3': 'NM_170606.2',
            'CBLB': 'NM_170662.3',
            'MSH5': 'NM_172165.3',
            'CAMK2G': 'NM_172173.2',
            'NR4A3': 'NM_173200.2',
            'SIK1': 'NM_173354.3',
            'TSHZ2': 'NM_173485.5',
            'JAZF1': 'NM_175061.3',
            'RUNX1T1': 'NM_175635.2',
            'C15orf55': 'NM_175741.1',
            'CRIPAK': 'NM_175918.3',
            'FGF14': 'NM_175929.2',
            'KLC3': 'NM_177417.2',
            'CSNK2A1': 'NM_177559.2',
            'TUBB': 'NM_178014.2',
            'PIK3R1': 'NM_181523.2',
            'NF2': 'NM_181830.2',
            'PRX': 'NM_181882.2',
            'TXNRD1': 'NM_182729.2',
            'SLC19A1': 'NM_194255.2',
            'MAX': 'NM_197957.3',
            'WISP3': 'NM_198239.1',
            'TERT': 'NM_198253.2',
            'AURKA': 'NM_198433.1',
            'LRRK2': 'NM_198578.3',
            'POLQ': 'NM_199420.3',
            'C1S': 'NM_201442.2',
            'ENOSF1': 'NM_202758.3',
            'E2F7': 'NM_203394.2',
            'BACH1': 'NM_206866.1',
            'PRKAA1': 'NM_206907.3',
            'IFNAR2': 'NM_207585.1',
            'PRKAR1A': 'NM_212471.1',
            'FGFR4': 'NM_213647.1',
            'SFXN4': 'NM_213649.1',
            'GNAS-AS1': 'NR_002785.2',
            'MALAT1': 'NR_002819.2',
            'SOX2-OT': 'NR_004053.2',
            'MIR142': 'NR_029683.1',
            'CBR3-AS1': 'NR_038892.1',
            'LOC100507346': 'NR_038982.1',
            'MIR4728': 'NR_039881.2',
            'LOC100506321': 'NR_045122.1',
            'FCGR2C': 'NR_047648.1'
        }
        if hugo in hugo_to_nm_dict.keys():
            return hugo_to_nm_dict[hugo]
        else:
            raise ValueError('unsupported Hugo specification.')

    def norm_c(self, chgvs: str):
        if not re.findall('NM|NR', chgvs):
            splits = chgvs.split(':')
            if splits:
                NM = self._hugo_to_nm(splits[0])
            else:
                raise ValueError('unsupported chr specification.')

            chgvs = NM + ':' + splits[1]

        return chgvs
