# AUTOGENERATED! DO NOT EDIT! File to edit: nbs/core.ipynb (unless otherwise specified).

__all__ = ['StatsForecast']

# Cell
import inspect
import logging
from functools import partial
from os import cpu_count

import numpy as np
import pandas as pd

# Internal Cell
logging.basicConfig(
    format='%(asctime)s %(name)s %(levelname)s: %(message)s',
    datefmt='%Y-%m-%d %H:%M:%S',
)
logger = logging.getLogger(__name__)

# Internal Cell
class GroupedArray:

    def __init__(self, data, indptr):
        self.data = data
        self.indptr = indptr
        self.n_groups = self.indptr.size - 1

    def __getitem__(self, idx):
        if isinstance(idx, int):
            return self.data[self.indptr[idx] : self.indptr[idx + 1]]
        elif isinstance(idx, slice):
            idx = slice(idx.start, idx.stop + 1, idx.step)
            new_indptr = self.indptr[idx].copy()
            new_data = self.data[new_indptr[0] : new_indptr[-1]].copy()
            new_indptr -= new_indptr[0]
            return GroupedArray(new_data, new_indptr)
        raise ValueError(f'idx must be either int or slice, got {type(idx)}')

    def __len__(self):
        return self.n_groups

    def __repr__(self):
        return f'GroupedArray(n_data={self.data.size:,}, n_groups={self.n_groups:,})'

    def __eq__(self, other):
        if not hasattr(other, 'data') or not hasattr(other, 'indptr'):
            return False
        return np.allclose(self.data, other.data) and np.array_equal(self.indptr, other.indptr)

    def compute_forecasts(self, h, func, xreg=None, level=None, *args):
        has_level = 'level' in inspect.signature(func).parameters and level is not None
        if has_level:
            out = np.full((h * self.n_groups, 2 * len(level) + 1), np.nan, dtype=np.float32)
            func = partial(func, level=level)
        else:
            out = np.full(h * self.n_groups, np.nan, dtype=np.float32)
        xr = None
        keys = None
        for i, grp in enumerate(self):
            if xreg is not None:
                xr = xreg[i]
            res = func(grp, h, xr, *args)
            if has_level:
                if keys is None:
                    keys = list(res.keys())
                for j, key in enumerate(keys):
                    out[h * i : h * (i + 1), j] = res[key]
            else:
                out[h * i : h * (i + 1)] = res
        return out, keys

    def split(self, n_chunks):
        return [self[x[0] : x[-1] + 1] for x in np.array_split(range(self.n_groups), n_chunks) if x.size]

# Internal Cell
def _grouped_array_from_df(df):
    df = df.set_index('ds', append=True)
    if not df.index.is_monotonic_increasing:
        df = df.sort_index()
    data = df.values.astype(np.float32)
    indices_sizes = df.index.get_level_values('unique_id').value_counts(sort=False)
    indices = indices_sizes.index
    sizes = indices_sizes.values
    cum_sizes = sizes.cumsum()
    dates = df.index.get_level_values('ds')[cum_sizes - 1]
    indptr = np.append(0, cum_sizes).astype(np.int32)
    return GroupedArray(data, indptr), indices, dates

# Internal Cell
def _build_forecast_name(model, *args) -> str:
    model_name = f'{model.__name__}'
    func_params = inspect.signature(model).parameters
    func_args = list(func_params.items())[3:]  # remove input array, horizon and xreg
    changed_params = [
        f'{name}-{value}'
        for value, (name, arg) in zip(args, func_args)
        if arg.default != value
    ]
    if changed_params:
        model_name += '_' + '_'.join(changed_params)
    return model_name

# Internal Cell
def _as_tuple(x):
    if isinstance(x, tuple):
        return x
    return (x,)

# Internal Cell
def _get_n_jobs(n_groups, n_jobs, ray_address):
    if ray_address is not None:
        logger.info(
            'Using ray address,'
            'using available resources insted of `n_jobs`'
        )
        try:
            import ray
        except ModuleNotFoundError as e:
            msg = (
                '{e}. To use a ray cluster you have to install '
                'ray. Please run `pip install ray`. '
            )
            raise ModuleNotFoundError(msg) from e
        if not ray.is_initialized():
            ray.init(ray_address, ignore_reinit_error=True)
        actual_n_jobs = int(ray.available_resources()['CPU'])
    else:
        if n_jobs == -1 or (n_jobs is None):
            actual_n_jobs = cpu_count()
        else:
            actual_n_jobs = n_jobs
    return min(n_groups, actual_n_jobs)

# Cell
class StatsForecast:

    def __init__(self, df, models, freq, n_jobs=1, ray_address=None):
        self.ga, self.uids, self.last_dates = _grouped_array_from_df(df)
        self.models = models
        self.freq = pd.tseries.frequencies.to_offset(freq)
        self.n_jobs = _get_n_jobs(len(self.ga), n_jobs, ray_address)
        self.ray_address = ray_address

    def forecast(self, h, xreg=None, level=None):
        if xreg is not None:
            expected_shape = (h * len(self.ga), self.ga.data.shape[1])
            if xreg.shape != expected_shape:
                raise ValueError(f'Expected xreg to have shape {expected_shape}, but got {xreg.shape}')
            xreg, _, _ = _grouped_array_from_df(xreg)
        if self.n_jobs == 1:
            fcsts = self._sequential_forecast(h, xreg, level)
        else:
            fcsts = self._data_parallel_forecast(h, xreg, level)
        if issubclass(self.last_dates.dtype.type, np.integer):
            last_date_f = lambda x: np.arange(x + 1, x + 1 + h, dtype=self.last_dates.dtype)
        else:
            last_date_f = lambda x: pd.date_range(x + self.freq, periods=h, freq=self.freq)
        if len(np.unique(self.last_dates)) == 1:
            dates = np.tile(last_date_f(self.last_dates[0]), len(self.ga))
        else:
            dates = np.hstack([
                last_date_f(last_date)
                for last_date in self.last_dates
            ])
        idx = pd.Index(np.repeat(self.uids, h), name='unique_id')
        return pd.DataFrame({'ds': dates, **fcsts}, index=idx)

    def _sequential_forecast(self, h, xreg, level):
        fcsts = {}
        logger.info('Computing forecasts')
        for model_args in self.models:
            model, *args = _as_tuple(model_args)
            model_name = _build_forecast_name(model, *args)
            values, keys = self.ga.compute_forecasts(h, model, xreg, level, *args)
            if keys is not None:
                for j, key in enumerate(keys):
                    fcsts[f'{model_name}_{key}'] = values[:, j]
            else:
                fcsts[model_name] = values
            logger.info(f'Computed forecasts for {model_name}.')
        return fcsts

    def _data_parallel_forecast(self, h, xreg, level):
        fcsts = {}
        logger.info('Computing forecasts')
        gas = self.ga.split(self.n_jobs)
        if xreg is not None:
            xregs = xreg.split(self.n_jobs)
        else:
            from itertools import repeat

            xregs = repeat(None)

        if self.ray_address is not None:
            try:
                from ray.util.multiprocessing import Pool
            except ModuleNotFoundError as e:
                msg = (
                    f'{e}. To use a ray cluster you have to install '
                    'ray. Please run `pip install ray`. '
                )
                raise ModuleNotFoundError(msg) from e
            kwargs = dict(ray_address=self.ray_address)
        else:
            from multiprocessing import Pool
            kwargs = dict()

        with Pool(self.n_jobs, **kwargs) as executor:
            for model_args in self.models:
                model, *args = _as_tuple(model_args)
                model_name = _build_forecast_name(model, *args)
                futures = []
                for ga, xr in zip(gas, xregs):
                    future = executor.apply_async(ga.compute_forecasts, (h, model, xr, level, *args,))
                    futures.append(future)
                values, keys = list(zip(*[f.get() for f in futures]))
                keys = keys[0]
                if keys is not None:
                    values = np.vstack(values)
                    for j, key in enumerate(keys):
                        fcsts[f'{model_name}_{key}'] = values[:, j]
                else:
                    values = np.hstack(values)
                    fcsts[model_name] = values
                logger.info(f'Computed forecasts for {model_name}.')
        return fcsts