import * as iam from '@aws-cdk/aws-iam';
import { RemovalPolicy, Resource } from '@aws-cdk/core';
import { Construct } from 'constructs';
import { IKey } from './key';
/**
 * A KMS Key alias.
 *
 * An alias can be used in all places that expect a key.
 *
 * @stability stable
 */
export interface IAlias extends IKey {
    /**
     * The name of the alias.
     *
     * @stability stable
     * @attribute true
     */
    readonly aliasName: string;
    /**
     * The Key to which the Alias refers.
     *
     * @stability stable
     * @attribute true
     */
    readonly aliasTargetKey: IKey;
}
/**
 * Construction properties for a KMS Key Alias object.
 *
 * @stability stable
 */
export interface AliasProps {
    /**
     * The name of the alias.
     *
     * The name must start with alias followed by a
     * forward slash, such as alias/. You can't specify aliases that begin with
     * alias/AWS. These aliases are reserved.
     *
     * @stability stable
     */
    readonly aliasName: string;
    /**
     * The ID of the key for which you are creating the alias.
     *
     * Specify the key's
     * globally unique identifier or Amazon Resource Name (ARN). You can't
     * specify another alias.
     *
     * @stability stable
     */
    readonly targetKey: IKey;
    /**
     * Policy to apply when the alias is removed from this stack.
     *
     * @default - The alias will be deleted
     * @stability stable
     */
    readonly removalPolicy?: RemovalPolicy;
}
declare abstract class AliasBase extends Resource implements IAlias {
    abstract readonly aliasName: string;
    abstract readonly aliasTargetKey: IKey;
    /**
     * The ARN of the key.
     *
     * @stability stable
     */
    get keyArn(): string;
    /**
     * The ID of the key (the part that looks something like: 1234abcd-12ab-34cd-56ef-1234567890ab).
     *
     * @stability stable
     */
    get keyId(): string;
    /**
     * Defines a new alias for the key.
     *
     * @stability stable
     */
    addAlias(alias: string): Alias;
    /**
     * Adds a statement to the KMS key resource policy.
     *
     * @stability stable
     */
    addToResourcePolicy(statement: iam.PolicyStatement, allowNoOp?: boolean): iam.AddToResourcePolicyResult;
    /**
     * Grant the indicated permissions on this key to the given principal.
     *
     * @stability stable
     */
    grant(grantee: iam.IGrantable, ...actions: string[]): iam.Grant;
    /**
     * Grant decryption permissions using this key to the given principal.
     *
     * @stability stable
     */
    grantDecrypt(grantee: iam.IGrantable): iam.Grant;
    /**
     * Grant encryption permissions using this key to the given principal.
     *
     * @stability stable
     */
    grantEncrypt(grantee: iam.IGrantable): iam.Grant;
    /**
     * Grant encryption and decryption permissions using this key to the given principal.
     *
     * @stability stable
     */
    grantEncryptDecrypt(grantee: iam.IGrantable): iam.Grant;
}
/**
 * Properties of a reference to an existing KMS Alias.
 *
 * @stability stable
 */
export interface AliasAttributes {
    /**
     * Specifies the alias name.
     *
     * This value must begin with alias/ followed by a name (i.e. alias/ExampleAlias)
     *
     * @stability stable
     */
    readonly aliasName: string;
    /**
     * The customer master key (CMK) to which the Alias refers.
     *
     * @stability stable
     */
    readonly aliasTargetKey: IKey;
}
/**
 * Defines a display name for a customer master key (CMK) in AWS Key Management Service (AWS KMS).
 *
 * Using an alias to refer to a key can help you simplify key
 * management. For example, when rotating keys, you can just update the alias
 * mapping instead of tracking and changing key IDs. For more information, see
 * Working with Aliases in the AWS Key Management Service Developer Guide.
 *
 * You can also add an alias for a key by calling `key.addAlias(alias)`.
 *
 * @stability stable
 * @resource AWS::KMS::Alias
 */
export declare class Alias extends AliasBase {
    /**
     * Import an existing KMS Alias defined outside the CDK app.
     *
     * @param scope The parent creating construct (usually `this`).
     * @param id The construct's name.
     * @param attrs the properties of the referenced KMS Alias.
     * @stability stable
     */
    static fromAliasAttributes(scope: Construct, id: string, attrs: AliasAttributes): IAlias;
    /**
     * Import an existing KMS Alias defined outside the CDK app, by the alias name.
     *
     * This method should be used
     * instead of 'fromAliasAttributes' when the underlying KMS Key ARN is not available.
     * This Alias will not have a direct reference to the KMS Key, so addAlias and grant* methods are not supported.
     *
     * @param scope The parent creating construct (usually `this`).
     * @param id The construct's name.
     * @param aliasName The full name of the KMS Alias (e.g., 'alias/aws/s3', 'alias/myKeyAlias').
     * @stability stable
     */
    static fromAliasName(scope: Construct, id: string, aliasName: string): IAlias;
    /**
     * The name of the alias.
     *
     * @stability stable
     */
    readonly aliasName: string;
    /**
     * The Key to which the Alias refers.
     *
     * @stability stable
     */
    readonly aliasTargetKey: IKey;
    /**
     * @stability stable
     */
    constructor(scope: Construct, id: string, props: AliasProps);
    /**
     * @stability stable
     */
    protected generatePhysicalName(): string;
}
export {};
