"use strict";
var _a, _b, _c, _d, _e, _f, _g, _h, _j, _k, _l, _m, _o, _p, _q, _r;
Object.defineProperty(exports, "__esModule", { value: true });
exports.PodConnections = exports.PodConnectionsIsolation = exports.PodScheduling = exports.Topology = exports.Node = exports.NamedNode = exports.TaintedNode = exports.LabeledNode = exports.Pods = exports.NodeTaintQuery = exports.TaintEffect = exports.LabelExpression = exports.NodeLabelQuery = exports.DnsPolicy = exports.FsGroupChangePolicy = exports.RestartPolicy = exports.PodSecurityContext = exports.PodDns = exports.Pod = exports.LabelSelector = exports.AbstractPod = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const cdk8s_1 = require("cdk8s");
const constructs_1 = require("constructs");
const base = require("./base");
const container = require("./container");
const k8s = require("./imports/k8s");
const networkpolicy = require("./network-policy");
const utils_1 = require("./utils");
class AbstractPod extends base.Resource {
    constructor(scope, id, props = {}) {
        super(scope, id);
        this._containers = [];
        this._initContainers = [];
        this._hostAliases = [];
        this._volumes = new Map();
        this.restartPolicy = props.restartPolicy ?? RestartPolicy.ALWAYS;
        this.serviceAccount = props.serviceAccount;
        this.securityContext = new PodSecurityContext(props.securityContext);
        this.dns = new PodDns(props.dns);
        this.dockerRegistryAuth = props.dockerRegistryAuth;
        this.automountServiceAccountToken = props.automountServiceAccountToken ?? false;
        this.isolate = props.isolate ?? false;
        this.hostNetwork = props.hostNetwork ?? false;
        this.terminationGracePeriod = props.terminationGracePeriod ?? cdk8s_1.Duration.seconds(30);
        if (props.containers) {
            props.containers.forEach(c => this.addContainer(c));
        }
        if (props.volumes) {
            props.volumes.forEach(v => this.addVolume(v));
        }
        if (props.initContainers) {
            props.initContainers.forEach(c => this.addInitContainer(c));
        }
        if (props.hostAliases) {
            props.hostAliases.forEach(c => this.addHostAlias(c));
        }
    }
    get containers() {
        return [...this._containers];
    }
    get initContainers() {
        return [...this._initContainers];
    }
    get volumes() {
        return Array.from(this._volumes.values());
    }
    get hostAliases() {
        return [...this._hostAliases];
    }
    /**
     * @see IPodSelector.toPodSelectorConfig()
     */
    toPodSelectorConfig() {
        const podAddress = this.podMetadata.getLabel(Pod.ADDRESS_LABEL);
        if (!podAddress) {
            // shouldn't happen because we add this label automatically in both pods and workloads.
            throw new Error(`Unable to create a label selector since ${Pod.ADDRESS_LABEL} label is missing`);
        }
        return {
            labelSelector: LabelSelector.of({ labels: { [Pod.ADDRESS_LABEL]: podAddress } }),
            namespaces: this.metadata.namespace ? {
                names: [this.metadata.namespace],
            } : undefined,
        };
    }
    /**
     * @see INetworkPolicyPeer.toNetworkPolicyPeerConfig()
     */
    toNetworkPolicyPeerConfig() {
        return { podSelector: this.toPodSelectorConfig() };
    }
    /**
     * @see INetworkPolicyPeer.toPodSelector()
     */
    toPodSelector() {
        return this;
    }
    addContainer(cont) {
        const impl = new container.Container(cont);
        this.attachContainer(impl);
        return impl;
    }
    attachContainer(cont) {
        this._containers.push(cont);
    }
    addInitContainer(cont) {
        // https://kubernetes.io/docs/concepts/workloads/pods/init-containers/#differences-from-regular-containers
        if (cont.readiness) {
            throw new Error('Init containers must not have a readiness probe');
        }
        if (cont.liveness) {
            throw new Error('Init containers must not have a liveness probe');
        }
        if (cont.startup) {
            throw new Error('Init containers must not have a startup probe');
        }
        const impl = new container.Container({
            ...cont,
            name: cont.name ?? `init-${this._initContainers.length}`,
        });
        this._initContainers.push(impl);
        return impl;
    }
    addHostAlias(hostAlias) {
        this._hostAliases.push(hostAlias);
    }
    addVolume(vol) {
        const existingVolume = this._volumes.get(vol.name);
        if (existingVolume) {
            throw new Error(`Volume with name ${vol.name} already exists`);
        }
        this._volumes.set(vol.name, vol);
    }
    /**
     * @see ISubect.toSubjectConfiguration()
     */
    toSubjectConfiguration() {
        if (!this.serviceAccount && !this.automountServiceAccountToken) {
            throw new Error(`${this.name} cannot be converted to a role binding subject:`
                + ' You must either assign a service account to it, or use \'automountServiceAccountToken: true\'');
        }
        // 'default' is assumed to be the name of the default service account
        // in the cluster.
        const serviceAccountName = this.serviceAccount?.name ?? 'default';
        return {
            kind: 'ServiceAccount',
            name: serviceAccountName,
            apiGroup: '',
        };
    }
    /**
     * @internal
     */
    _toPodSpec() {
        if (this.containers.length === 0) {
            throw new Error('PodSpec must have at least 1 container');
        }
        const volumes = new Map();
        const containers = [];
        const initContainers = [];
        for (const cont of this.containers) {
            // check if restartPolicy is defined for containers
            if (cont.restartPolicy) {
                throw new Error(`Invalid container spec: ${cont.name} has non-empty restartPolicy field. The field can only be specified for initContainers`);
            }
            // automatically add volume from the container mount
            // to this pod so thats its available to the container.
            for (const mount of cont.mounts) {
                addVolume(mount.volume);
            }
            containers.push(cont._toKube());
        }
        for (const cont of this.initContainers) {
            // automatically add volume from the container mount
            // to this pod so thats its available to the container.
            for (const mount of cont.mounts) {
                addVolume(mount.volume);
            }
            initContainers.push(cont._toKube());
        }
        for (const vol of this.volumes) {
            addVolume(vol);
        }
        function addVolume(vol) {
            const existingVolume = volumes.get(vol.name);
            // its ok to call this function twice on the same volume, but its not ok to
            // call it twice on a different volume with the same name.
            if (existingVolume && existingVolume !== vol) {
                throw new Error(`Invalid mount configuration. At least two different volumes have the same name: ${vol.name}`);
            }
            volumes.set(vol.name, vol);
        }
        const dns = this.dns._toKube();
        return {
            restartPolicy: this.restartPolicy,
            serviceAccountName: this.serviceAccount?.name,
            containers: containers,
            securityContext: utils_1.undefinedIfEmpty(this.securityContext._toKube()),
            initContainers: utils_1.undefinedIfEmpty(initContainers),
            hostAliases: utils_1.undefinedIfEmpty(this.hostAliases),
            volumes: utils_1.undefinedIfEmpty(Array.from(volumes.values()).map(v => v._toKube())),
            dnsPolicy: dns.policy,
            dnsConfig: utils_1.undefinedIfEmpty(dns.config),
            hostname: dns.hostname,
            subdomain: dns.subdomain,
            setHostnameAsFqdn: dns.hostnameAsFQDN,
            imagePullSecrets: this.dockerRegistryAuth ? [{ name: this.dockerRegistryAuth.name }] : undefined,
            automountServiceAccountToken: this.automountServiceAccountToken,
            hostNetwork: this.hostNetwork,
            terminationGracePeriodSeconds: this.terminationGracePeriod?.toSeconds(),
        };
    }
}
exports.AbstractPod = AbstractPod;
_a = JSII_RTTI_SYMBOL_1;
AbstractPod[_a] = { fqn: "cdk8s-plus-28.AbstractPod", version: "2.3.13" };
/**
 * Match a resource by labels.
 */
class LabelSelector {
    constructor(expressions, labels) {
        this.expressions = expressions;
        this.labels = labels;
    }
    static of(options = {}) {
        return new LabelSelector(options.expressions ?? [], options.labels ?? {});
    }
    isEmpty() {
        return this.expressions.length === 0 && Object.keys(this.labels).length === 0;
    }
    /**
     * @internal
     */
    _toKube() {
        if (this.isEmpty()) {
            return {};
        }
        return {
            matchExpressions: utils_1.undefinedIfEmpty(this.expressions.map(q => ({ key: q.key, operator: q.operator, values: q.values }))),
            matchLabels: utils_1.undefinedIfEmpty(this.labels),
        };
    }
}
exports.LabelSelector = LabelSelector;
_b = JSII_RTTI_SYMBOL_1;
LabelSelector[_b] = { fqn: "cdk8s-plus-28.LabelSelector", version: "2.3.13" };
/**
 * Pod is a collection of containers that can run on a host. This resource is
 * created by clients and scheduled onto hosts.
 */
class Pod extends AbstractPod {
    constructor(scope, id, props = {}) {
        super(scope, id, props);
        this.resourceType = 'pods';
        this.apiObject = new k8s.KubePod(this, 'Resource', {
            metadata: props.metadata,
            spec: cdk8s_1.Lazy.any({ produce: () => this._toKube() }),
        });
        this.metadata.addLabel(Pod.ADDRESS_LABEL, cdk8s_1.Names.toLabelValue(this));
        this.scheduling = new PodScheduling(this);
        this.connections = new PodConnections(this);
        if (this.isolate) {
            this.connections.isolate();
        }
    }
    get podMetadata() {
        return this.metadata;
    }
    /**
     * @internal
     */
    _toKube() {
        const scheduling = this.scheduling._toKube();
        return {
            ...this._toPodSpec(),
            affinity: scheduling.affinity,
            nodeName: scheduling.nodeName,
            tolerations: scheduling.tolerations,
        };
    }
}
exports.Pod = Pod;
_c = JSII_RTTI_SYMBOL_1;
Pod[_c] = { fqn: "cdk8s-plus-28.Pod", version: "2.3.13" };
/**
 * This label is autoamtically added by cdk8s to any pod. It provides
 * a unique and stable identifier for the pod.
 */
Pod.ADDRESS_LABEL = 'cdk8s.io/metadata.addr';
/**
 * Holds dns settings of the pod.
 */
class PodDns {
    constructor(props = {}) {
        this.hostname = props.hostname;
        this.subdomain = props.subdomain;
        this.policy = props.policy ?? DnsPolicy.CLUSTER_FIRST;
        this.hostnameAsFQDN = props.hostnameAsFQDN ?? false;
        this._nameservers = props.nameservers ?? [];
        this._searches = props.searches ?? [];
        this._options = props.options ?? [];
    }
    /**
     * Nameservers defined for this pod.
     */
    get nameservers() {
        return [...this._nameservers];
    }
    /**
     * Search domains defined for this pod.
     */
    get searches() {
        return [...this._searches];
    }
    /**
     * Custom dns options defined for this pod.
     */
    get options() {
        return [...this._options];
    }
    /**
     * Add a nameserver.
     */
    addNameserver(...nameservers) {
        this._nameservers.push(...nameservers);
    }
    /**
     * Add a search domain.
     */
    addSearch(...searches) {
        this._searches.push(...searches);
    }
    /**
     * Add a custom option.
     */
    addOption(...options) {
        this._options.push(...options);
    }
    /**
     * @internal
     */
    _toKube() {
        if (this.policy === DnsPolicy.NONE && this.nameservers.length === 0) {
            throw new Error('When dns policy is set to NONE, at least one nameserver is required');
        }
        if (this.nameservers.length > 3) {
            throw new Error('There can be at most 3 nameservers specified');
        }
        if (this.searches.length > 6) {
            throw new Error('There can be at most 6 search domains specified');
        }
        return {
            hostname: this.hostname,
            subdomain: this.subdomain,
            hostnameAsFQDN: this.hostnameAsFQDN,
            policy: this.policy,
            config: {
                nameservers: utils_1.undefinedIfEmpty(this.nameservers),
                searches: utils_1.undefinedIfEmpty(this.searches),
                options: utils_1.undefinedIfEmpty(this.options),
            },
        };
    }
}
exports.PodDns = PodDns;
_d = JSII_RTTI_SYMBOL_1;
PodDns[_d] = { fqn: "cdk8s-plus-28.PodDns", version: "2.3.13" };
/**
 * Holds pod-level security attributes and common container settings.
 */
class PodSecurityContext {
    constructor(props = {}) {
        this._sysctls = [];
        this.ensureNonRoot = props.ensureNonRoot ?? true;
        this.fsGroupChangePolicy = props.fsGroupChangePolicy ?? FsGroupChangePolicy.ALWAYS;
        this.user = props.user;
        this.group = props.group;
        this.fsGroup = props.fsGroup;
        for (const sysctl of props.sysctls ?? []) {
            this._sysctls.push(sysctl);
        }
    }
    get sysctls() {
        return [...this._sysctls];
    }
    /**
     * @internal
     */
    _toKube() {
        return {
            runAsGroup: this.group,
            runAsUser: this.user,
            fsGroup: this.fsGroup,
            runAsNonRoot: this.ensureNonRoot,
            fsGroupChangePolicy: this.fsGroupChangePolicy,
            sysctls: utils_1.undefinedIfEmpty(this._sysctls),
        };
    }
}
exports.PodSecurityContext = PodSecurityContext;
_e = JSII_RTTI_SYMBOL_1;
PodSecurityContext[_e] = { fqn: "cdk8s-plus-28.PodSecurityContext", version: "2.3.13" };
/**
 * Restart policy for all containers within the pod.
 */
var RestartPolicy;
(function (RestartPolicy) {
    /**
     * Always restart the pod after it exits.
     */
    RestartPolicy["ALWAYS"] = "Always";
    /**
     * Only restart if the pod exits with a non-zero exit code.
     */
    RestartPolicy["ON_FAILURE"] = "OnFailure";
    /**
     * Never restart the pod.
     */
    RestartPolicy["NEVER"] = "Never";
})(RestartPolicy = exports.RestartPolicy || (exports.RestartPolicy = {}));
var FsGroupChangePolicy;
(function (FsGroupChangePolicy) {
    /**
     * Only change permissions and ownership if permission and ownership of root directory does
     * not match with expected permissions of the volume.
     * This could help shorten the time it takes to change ownership and permission of a volume
     */
    FsGroupChangePolicy["ON_ROOT_MISMATCH"] = "OnRootMismatch";
    /**
     * Always change permission and ownership of the volume when volume is mounted.
     */
    FsGroupChangePolicy["ALWAYS"] = "Always";
})(FsGroupChangePolicy = exports.FsGroupChangePolicy || (exports.FsGroupChangePolicy = {}));
/**
 * Pod DNS policies.
 */
var DnsPolicy;
(function (DnsPolicy) {
    /**
     * Any DNS query that does not match the configured cluster domain suffix,
     * such as "www.kubernetes.io", is forwarded to the
     * upstream nameserver inherited from the node.
     * Cluster administrators may have extra stub-domain and upstream DNS servers configured.
     */
    DnsPolicy["CLUSTER_FIRST"] = "ClusterFirst";
    /**
     * For Pods running with hostNetwork, you should
     * explicitly set its DNS policy "ClusterFirstWithHostNet".
     */
    DnsPolicy["CLUSTER_FIRST_WITH_HOST_NET"] = "ClusterFirstWithHostNet";
    /**
     * The Pod inherits the name resolution configuration
     * from the node that the pods run on.
     */
    DnsPolicy["DEFAULT"] = "Default";
    /**
     * It allows a Pod to ignore DNS settings from the Kubernetes environment.
     * All DNS settings are supposed to be provided using the dnsConfig
     * field in the Pod Spec.
     */
    DnsPolicy["NONE"] = "None";
})(DnsPolicy = exports.DnsPolicy || (exports.DnsPolicy = {}));
/**
 * Represents a query that can be performed against nodes with labels.
 */
class NodeLabelQuery {
    constructor(key, operator, values) {
        this.key = key;
        this.operator = operator;
        this.values = values;
    }
    /**
     * Requires value of label `key` to equal `value`.
     */
    static is(key, value) {
        return NodeLabelQuery.in(key, [value]);
    }
    /**
     * Requires value of label `key` to be one of `values`.
     */
    static in(key, values) {
        return new NodeLabelQuery(key, 'In', values);
    }
    /**
     * Requires value of label `key` to be none of `values`.
     */
    static notIn(key, values) {
        return new NodeLabelQuery(key, 'NotIn', values);
    }
    /**
     * Requires label `key` to exist.
     */
    static exists(key) {
        return new NodeLabelQuery(key, 'Exists', undefined);
    }
    /**
     * Requires label `key` to not exist.
     */
    static doesNotExist(key) {
        return new NodeLabelQuery(key, 'DoesNotExist', undefined);
    }
    /**
     * Requires value of label `key` to greater than all elements in `values`.
     */
    static gt(key, values) {
        return new NodeLabelQuery(key, 'Gt', values);
    }
    /**
     * Requires value of label `key` to less than all elements in `values`.
     */
    static lt(key, values) {
        return new NodeLabelQuery(key, 'Lt', values);
    }
    /**
     * @internal
     */
    _toKube() {
        return {
            key: this.key,
            operator: this.operator,
            values: this.values,
        };
    }
}
exports.NodeLabelQuery = NodeLabelQuery;
_f = JSII_RTTI_SYMBOL_1;
NodeLabelQuery[_f] = { fqn: "cdk8s-plus-28.NodeLabelQuery", version: "2.3.13" };
/**
 * Represents a query that can be performed against resources with labels.
 */
class LabelExpression {
    constructor(key, operator, values) {
        this.key = key;
        this.operator = operator;
        this.values = values;
    }
    /**
     * Requires value of label `key` to be one of `values`.
     */
    static in(key, values) {
        return new LabelExpression(key, 'In', values);
    }
    /**
     * Requires value of label `key` to be none of `values`.
     */
    static notIn(key, values) {
        return new LabelExpression(key, 'NotIn', values);
    }
    /**
     * Requires label `key` to exist.
     */
    static exists(key) {
        return new LabelExpression(key, 'Exists', undefined);
    }
    /**
     * Requires label `key` to not exist.
     */
    static doesNotExist(key) {
        return new LabelExpression(key, 'DoesNotExist', undefined);
    }
}
exports.LabelExpression = LabelExpression;
_g = JSII_RTTI_SYMBOL_1;
LabelExpression[_g] = { fqn: "cdk8s-plus-28.LabelExpression", version: "2.3.13" };
/**
 * Taint effects.
 */
var TaintEffect;
(function (TaintEffect) {
    /**
     * This means that no pod will be able to schedule
     * onto the node unless it has a matching toleration.
     */
    TaintEffect["NO_SCHEDULE"] = "NoSchedule";
    /**
     * This is a "preference" or "soft" version of `NO_SCHEDULE` -- the system
     * will try to avoid placing a pod that does not tolerate the taint on the node,
     * but it is not required
     */
    TaintEffect["PREFER_NO_SCHEDULE"] = "PreferNoSchedule";
    /**
     * This affects pods that are already running on the node as follows:
     *
     * - Pods that do not tolerate the taint are evicted immediately.
     * - Pods that tolerate the taint without specifying `duration` remain bound forever.
     * - Pods that tolerate the taint with a specified `duration` remain bound for
     *   the specified amount of time.
     */
    TaintEffect["NO_EXECUTE"] = "NoExecute";
})(TaintEffect = exports.TaintEffect || (exports.TaintEffect = {}));
/**
 * Taint queries that can be perfomed against nodes.
 */
class NodeTaintQuery {
    constructor(operator, key, value, effect, evictAfter) {
        this.operator = operator;
        this.key = key;
        this.value = value;
        this.effect = effect;
        this.evictAfter = evictAfter;
        if (evictAfter && effect !== TaintEffect.NO_EXECUTE) {
            throw new Error('Only \'NO_EXECUTE\' effects can specify \'evictAfter\'');
        }
    }
    /**
     * Matches a taint with a specific key and value.
     */
    static is(key, value, options = {}) {
        return new NodeTaintQuery('Equal', key, value, options.effect, options.evictAfter);
    }
    /**
     * Matches a tain with any value of a specific key.
     */
    static exists(key, options = {}) {
        return new NodeTaintQuery('Exists', key, undefined, options.effect, options.evictAfter);
    }
    /**
     * Matches any taint.
     */
    static any() {
        return new NodeTaintQuery('Exists');
    }
    /**
     * @internal
     */
    _toKube() {
        return {
            effect: this.effect,
            key: this.key,
            operator: this.operator,
            tolerationSeconds: this.evictAfter?.toSeconds(),
            value: this.value,
        };
    }
}
exports.NodeTaintQuery = NodeTaintQuery;
_h = JSII_RTTI_SYMBOL_1;
NodeTaintQuery[_h] = { fqn: "cdk8s-plus-28.NodeTaintQuery", version: "2.3.13" };
/**
 * Represents a group of pods.
 */
class Pods extends constructs_1.Construct {
    constructor(scope, id, expressions, labels, namespaces) {
        super(scope, id);
        this.expressions = expressions;
        this.labels = labels;
        this.namespaces = namespaces;
    }
    /**
     * Select pods in the cluster with various selectors.
     */
    static select(scope, id, options) {
        return new Pods(scope, id, options.expressions, options.labels, options.namespaces);
    }
    /**
     * Select all pods.
     */
    static all(scope, id, options = {}) {
        return Pods.select(scope, id, { namespaces: options.namespaces });
    }
    /**
     * @see IPodSelector.toPodSelectorConfig()
     */
    toPodSelectorConfig() {
        return {
            labelSelector: LabelSelector.of({ expressions: this.expressions, labels: this.labels }),
            namespaces: this.namespaces?.toNamespaceSelectorConfig(),
        };
    }
    /**
     * @see INetworkPolicyPeer.toNetworkPolicyPeerConfig()
     */
    toNetworkPolicyPeerConfig() {
        return { podSelector: this.toPodSelectorConfig() };
    }
    /**
     * @see INetworkPolicyPeer.toPodSelector()
     */
    toPodSelector() {
        return this;
    }
}
exports.Pods = Pods;
_j = JSII_RTTI_SYMBOL_1;
Pods[_j] = { fqn: "cdk8s-plus-28.Pods", version: "2.3.13" };
/**
 * A node that is matched by label selectors.
 */
class LabeledNode {
    constructor(labelSelector) {
        this.labelSelector = labelSelector;
    }
    ;
}
exports.LabeledNode = LabeledNode;
_k = JSII_RTTI_SYMBOL_1;
LabeledNode[_k] = { fqn: "cdk8s-plus-28.LabeledNode", version: "2.3.13" };
/**
 * A node that is matched by taint selectors.
 */
class TaintedNode {
    constructor(taintSelector) {
        this.taintSelector = taintSelector;
    }
    ;
}
exports.TaintedNode = TaintedNode;
_l = JSII_RTTI_SYMBOL_1;
TaintedNode[_l] = { fqn: "cdk8s-plus-28.TaintedNode", version: "2.3.13" };
/**
 * A node that is matched by its name.
 */
class NamedNode {
    constructor(name) {
        this.name = name;
    }
    ;
}
exports.NamedNode = NamedNode;
_m = JSII_RTTI_SYMBOL_1;
NamedNode[_m] = { fqn: "cdk8s-plus-28.NamedNode", version: "2.3.13" };
/**
 * Represents a node in the cluster.
 */
class Node {
    /**
     * Match a node by its labels.
     */
    static labeled(...labelSelector) {
        return new LabeledNode(labelSelector);
    }
    /**
     * Match a node by its name.
     */
    static named(nodeName) {
        return new NamedNode(nodeName);
    }
    /**
     * Match a node by its taints.
     */
    static tainted(...taintSelector) {
        return new TaintedNode(taintSelector);
    }
}
exports.Node = Node;
_o = JSII_RTTI_SYMBOL_1;
Node[_o] = { fqn: "cdk8s-plus-28.Node", version: "2.3.13" };
/**
 * Available topology domains.
 */
class Topology {
    constructor(key) {
        this.key = key;
    }
    /**
     * Custom key for the node label that the system uses to denote the topology domain.
     */
    static custom(key) {
        return new Topology(key);
    }
    ;
}
exports.Topology = Topology;
_p = JSII_RTTI_SYMBOL_1;
Topology[_p] = { fqn: "cdk8s-plus-28.Topology", version: "2.3.13" };
/**
 * A hostname represents a single node in the cluster.
 *
 * @see https://kubernetes.io/docs/reference/labels-annotations-taints/#kubernetesiohostname
 */
Topology.HOSTNAME = new Topology('kubernetes.io/hostname');
/**
 * A zone represents a logical failure domain. It is common for Kubernetes clusters to
 * span multiple zones for increased availability. While the exact definition of a zone is
 * left to infrastructure implementations, common properties of a zone include very low
 * network latency within a zone, no-cost network traffic within a zone, and failure
 * independence from other zones. For example, nodes within a zone might share a network
 * switch, but nodes in different zones should not.
 *
 * @see https://kubernetes.io/docs/reference/labels-annotations-taints/#topologykubernetesiozone
 */
Topology.ZONE = new Topology('topology.kubernetes.io/zone');
/**
 * A region represents a larger domain, made up of one or more zones. It is uncommon
 * for Kubernetes clusters to span multiple regions. While the exact definition of a
 * zone or region is left to infrastructure implementations, common properties of a region
 * include higher network latency between them than within them, non-zero cost for network
 * traffic between them, and failure independence from other zones or regions.
 *
 * For example, nodes within a region might share power infrastructure (e.g. a UPS or generator), but
 * nodes in different regions typically would not.
 *
 * @see https://kubernetes.io/docs/reference/labels-annotations-taints/#topologykubernetesioregion
 */
Topology.REGION = new Topology('topology.kubernetes.io/region');
/**
 * Controls the pod scheduling strategy.
 */
class PodScheduling {
    constructor(instance) {
        this.instance = instance;
        this._nodeAffinityPreferred = [];
        this._nodeAffinityRequired = [];
        this._podAffinityPreferred = [];
        this._podAffinityRequired = [];
        this._podAntiAffinityPreferred = [];
        this._podAntiAffinityRequired = [];
        this._tolerations = [];
    }
    /**
     * Assign this pod a specific node by name.
     *
     * The scheduler ignores the Pod, and the kubelet on the named node
     * tries to place the Pod on that node. Overrules any affinity rules of the pod.
     *
     * Some limitations of static assignment are:
     *
     * - If the named node does not exist, the Pod will not run, and in some
     *   cases may be automatically deleted.
     * - If the named node does not have the resources to accommodate the Pod,
     *   the Pod will fail and its reason will indicate why, for example OutOfmemory or OutOfcpu.
     * - Node names in cloud environments are not always predictable or stable.
     *
     * Will throw is the pod is already assigned to named node.
     *
     * Under the hood, this method utilizes the `nodeName` property.
     */
    assign(node) {
        if (this._nodeName) {
            // disallow overriding an static node assignment
            throw new Error(`Cannot assign ${this.instance.podMetadata.name} to node ${node.name}. It is already assigned to node ${this._nodeName}`);
        }
        else {
            this._nodeName = node.name;
        }
    }
    /**
     * Allow this pod to tolerate taints matching these tolerations.
     *
     * You can put multiple taints on the same node and multiple tolerations on the same pod.
     * The way Kubernetes processes multiple taints and tolerations is like a filter: start with
     * all of a node's taints, then ignore the ones for which the pod has a matching toleration;
     * the remaining un-ignored taints have the indicated effects on the pod. In particular:
     *
     * - if there is at least one un-ignored taint with effect NoSchedule then Kubernetes will
     *   not schedule the pod onto that node
     * - if there is no un-ignored taint with effect NoSchedule but there is at least one un-ignored
     *   taint with effect PreferNoSchedule then Kubernetes will try to not schedule the pod onto the node
     * - if there is at least one un-ignored taint with effect NoExecute then the pod will be evicted from
     *   the node (if it is already running on the node), and will not be scheduled onto the node (if it is
     *   not yet running on the node).
     *
     * Under the hood, this method utilizes the `tolerations` property.
     *
     * @see https://kubernetes.io/docs/concepts/scheduling-eviction/taint-and-toleration/
     */
    tolerate(node) {
        for (const query of node.taintSelector) {
            this._tolerations.push(query._toKube());
        }
    }
    /**
     * Attract this pod to a node matched by selectors.
     * You can select a node by using `Node.labeled()`.
     *
     * Attracting to multiple nodes (i.e invoking this method multiple times) acts as
     * an OR condition, meaning the pod will be assigned to either one of the nodes.
     *
     * Under the hood, this method utilizes the `nodeAffinity` property.
     *
     * @see https://kubernetes.io/docs/concepts/scheduling-eviction/assign-pod-node/#node-affinity
     */
    attract(node, options = {}) {
        const term = this.createNodeAffinityTerm(node);
        if (options.weight) {
            this.validateWeight(options.weight);
            this._nodeAffinityPreferred.push({ weight: options.weight, preference: term });
        }
        else {
            this._nodeAffinityRequired.push(term);
        }
    }
    /**
     * Co-locate this pod with a scheduling selection.
     *
     * A selection can be one of:
     *
     * - An instance of a `Pod`.
     * - An instance of a `Workload` (e.g `Deployment`, `StatefulSet`).
     * - An un-managed pod that can be selected via `Pods.select()`.
     *
     * Co-locating with multiple selections ((i.e invoking this method multiple times)) acts as
     * an AND condition. meaning the pod will be assigned to a node that satisfies all
     * selections (i.e runs at least one pod that satisifies each selection).
     *
     * Under the hood, this method utilizes the `podAffinity` property.
     *
     * @see https://kubernetes.io/docs/concepts/scheduling-eviction/assign-pod-node/#inter-pod-affinity-and-anti-affinity
     */
    colocate(selector, options = {}) {
        const topology = options.topology ?? Topology.HOSTNAME;
        const term = this.createPodAffinityTerm(topology, selector);
        if (options.weight) {
            this.validateWeight(options.weight);
            this._podAffinityPreferred.push({ weight: options.weight, podAffinityTerm: term });
        }
        else {
            this._podAffinityRequired.push(term);
        }
    }
    /**
     * Seperate this pod from a scheduling selection.
     *
     * A selection can be one of:
     *
     * - An instance of a `Pod`.
     * - An instance of a `Workload` (e.g `Deployment`, `StatefulSet`).
     * - An un-managed pod that can be selected via `Pods.select()`.
     *
     * Seperating from multiple selections acts as an AND condition. meaning the pod
     * will not be assigned to a node that satisfies all selections (i.e runs at least one pod that satisifies each selection).
     *
     * Under the hood, this method utilizes the `podAntiAffinity` property.
     *
     * @see https://kubernetes.io/docs/concepts/scheduling-eviction/assign-pod-node/#inter-pod-affinity-and-anti-affinity
     */
    separate(selector, options = {}) {
        const topology = options.topology ?? Topology.HOSTNAME;
        const term = this.createPodAffinityTerm(topology, selector);
        if (options.weight) {
            this.validateWeight(options.weight);
            this._podAntiAffinityPreferred.push({ weight: options.weight, podAffinityTerm: term });
        }
        else {
            this._podAntiAffinityRequired.push(term);
        }
    }
    createPodAffinityTerm(topology, selector) {
        const config = selector.toPodSelectorConfig();
        return {
            topologyKey: topology.key,
            labelSelector: config.labelSelector._toKube(),
            namespaceSelector: config.namespaces?.labelSelector?._toKube(),
            namespaces: config.namespaces?.names,
        };
    }
    createNodeAffinityTerm(node) {
        return { matchExpressions: node.labelSelector.map(s => s._toKube()) };
    }
    validateWeight(weight) {
        if (weight < 1 || weight > 100) {
            // https://kubernetes.io/docs/concepts/scheduling-eviction/assign-pod-node/#node-affinity-weight
            throw new Error(`Invalid affinity weight: ${weight}. Must be in range 1-100`);
        }
    }
    /**
     * @internal
     */
    _toKube() {
        const atLeastOne = (...arrays) => {
            return arrays.flat().length > 0;
        };
        const hasNodeAffinity = atLeastOne(this._nodeAffinityPreferred, this._nodeAffinityRequired);
        const hasPodAffinity = atLeastOne(this._podAffinityPreferred, this._podAffinityRequired);
        const hasPodAntiAffinty = atLeastOne(this._podAntiAffinityPreferred, this._podAntiAffinityRequired);
        const hasAffinity = hasNodeAffinity || hasPodAffinity || hasPodAntiAffinty;
        return {
            affinity: hasAffinity ? {
                nodeAffinity: hasNodeAffinity ? {
                    preferredDuringSchedulingIgnoredDuringExecution: utils_1.undefinedIfEmpty(this._nodeAffinityPreferred),
                    requiredDuringSchedulingIgnoredDuringExecution: this._nodeAffinityRequired.length > 0 ? {
                        nodeSelectorTerms: this._nodeAffinityRequired,
                    } : undefined,
                } : undefined,
                podAffinity: hasPodAffinity ? {
                    preferredDuringSchedulingIgnoredDuringExecution: utils_1.undefinedIfEmpty(this._podAffinityPreferred),
                    requiredDuringSchedulingIgnoredDuringExecution: utils_1.undefinedIfEmpty(this._podAffinityRequired),
                } : undefined,
                podAntiAffinity: hasPodAntiAffinty ? {
                    preferredDuringSchedulingIgnoredDuringExecution: utils_1.undefinedIfEmpty(this._podAntiAffinityPreferred),
                    requiredDuringSchedulingIgnoredDuringExecution: utils_1.undefinedIfEmpty(this._podAntiAffinityRequired),
                } : undefined,
            } : undefined,
            nodeName: this._nodeName,
            tolerations: utils_1.undefinedIfEmpty(this._tolerations),
        };
    }
}
exports.PodScheduling = PodScheduling;
_q = JSII_RTTI_SYMBOL_1;
PodScheduling[_q] = { fqn: "cdk8s-plus-28.PodScheduling", version: "2.3.13" };
/**
 * Isolation determines which policies are created
 * when allowing connections from a a pod / workload to peers.
 */
var PodConnectionsIsolation;
(function (PodConnectionsIsolation) {
    /**
     * Only creates network policies that select the pod.
     */
    PodConnectionsIsolation["POD"] = "POD";
    /**
     * Only creates network policies that select the peer.
     */
    PodConnectionsIsolation["PEER"] = "PEER";
})(PodConnectionsIsolation = exports.PodConnectionsIsolation || (exports.PodConnectionsIsolation = {}));
/**
 * Controls network isolation rules for inter-pod communication.
 */
class PodConnections {
    constructor(instance) {
        this.instance = instance;
    }
    /**
     * Allow network traffic from this pod to the peer.
     *
     * By default, this will create an egress network policy for this pod, and an ingress
     * network policy for the peer. This is required if both sides are already isolated.
     * Use `options.isolation` to control this behavior.
     *
     * @example
     *
     * // create only an egress policy that selects the 'web' pod to allow outgoing traffic
     * // to the 'redis' pod. this requires the 'redis' pod to not be isolated for ingress.
     * web.connections.allowTo(redis, { isolation: Isolation.POD })
     *
     * // create only an ingress policy that selects the 'redis' peer to allow incoming traffic
     * // from the 'web' pod. this requires the 'web' pod to not be isolated for egress.
     * web.connections.allowTo(redis, { isolation: Isolation.PEER })
     *
     */
    allowTo(peer, options = {}) {
        return this.allow('Egress', peer, { ports: this.extractPorts(peer), ...options });
    }
    /**
     * Allow network traffic from the peer to this pod.
     *
     * By default, this will create an ingress network policy for this pod, and an egress
     * network policy for the peer. This is required if both sides are already isolated.
     * Use `options.isolation` to control this behavior.
     *
     * @example
     *
     * // create only an egress policy that selects the 'web' pod to allow outgoing traffic
     * // to the 'redis' pod. this requires the 'redis' pod to not be isolated for ingress.
     * redis.connections.allowFrom(web, { isolation: Isolation.PEER })
     *
     * // create only an ingress policy that selects the 'redis' peer to allow incoming traffic
     * // from the 'web' pod. this requires the 'web' pod to not be isolated for egress.
     * redis.connections.allowFrom(web, { isolation: Isolation.POD })
     *
     */
    allowFrom(peer, options = {}) {
        return this.allow('Ingress', peer, { ports: this.extractPorts(this.instance), ...options });
    }
    allow(direction, peer, options = {}) {
        const config = peer.toNetworkPolicyPeerConfig();
        networkpolicy.validatePeerConfig(config);
        const peerAddress = utils_1.address(peer);
        if (!options.isolation || options.isolation === PodConnectionsIsolation.POD) {
            const src = new networkpolicy.NetworkPolicy(this.instance, `Allow${direction}${peerAddress}`, {
                selector: this.instance,
                // the policy must be defined in the namespace of the pod
                // so it can select it.
                metadata: { namespace: this.instance.metadata.namespace },
            });
            switch (direction) {
                case 'Egress':
                    src.addEgressRule(peer, options.ports);
                    break;
                case 'Ingress':
                    src.addIngressRule(peer, options.ports);
            }
        }
        if (!options.isolation || options.isolation === PodConnectionsIsolation.PEER) {
            if (config.ipBlock) {
                // for an ip block we don't need to create the opposite policies
                return;
            }
            const podSelector = peer.toPodSelector();
            if (!podSelector) {
                throw new Error(`Unable to create policies for peer '${peer.node.addr}' since its not a pod selector`);
            }
            const oppositeDirection = direction === 'Egress' ? 'Ingress' : 'Egress';
            const podSelectorConfig = podSelector.toPodSelectorConfig();
            let namespaces;
            if (!podSelectorConfig.namespaces) {
                // if the peer doesn't specify namespaces, we assume the same namespace.
                namespaces = [this.instance.metadata.namespace];
            }
            else {
                // a peer cannot specify namespaces by labels because
                // we won't be able to extract the names of those namespaces.
                if (podSelectorConfig.namespaces.labelSelector && !podSelectorConfig.namespaces.labelSelector.isEmpty()) {
                    throw new Error(`Unable to create an ${oppositeDirection} policy for peer '${peer.node.path}' (pod=${this.instance.name}). Peer must specify namespaces only by name`);
                }
                // a peer must specify namespaces by name.
                if (!podSelectorConfig.namespaces.names) {
                    throw new Error(`Unable to create an ${oppositeDirection} policy for peer '${peer.node.path}' (pod=${this.instance.name}). Peer must specify namespace names`);
                }
                namespaces = podSelectorConfig.namespaces.names;
            }
            for (const name of namespaces) {
                switch (direction) {
                    case 'Egress':
                        new networkpolicy.NetworkPolicy(this.instance, `AllowIngress${name}${peerAddress}`, {
                            selector: podSelector,
                            metadata: { namespace: name },
                            ingress: { rules: [{ peer: this.instance, ports: options.ports }] },
                        });
                        break;
                    case 'Ingress':
                        new networkpolicy.NetworkPolicy(this.instance, `AllowEgress${name}${peerAddress}`, {
                            selector: podSelector,
                            metadata: { namespace: name },
                            egress: { rules: [{ peer: this.instance, ports: options.ports }] },
                        });
                        break;
                    default:
                        throw new Error(`Unsupported direction: ${direction}`);
                }
            }
        }
    }
    extractPorts(selector) {
        return container.extractContainerPorts(selector).map(n => networkpolicy.NetworkPolicyPort.tcp(n.number));
    }
    /**
     * Sets the default network policy for Pod/Workload to have all egress and ingress connections as disabled
     */
    isolate() {
        new networkpolicy.NetworkPolicy(this.instance, 'DefaultDenyAll', {
            selector: this.instance,
            // the policy must be defined in the namespace of the pod
            // so it can select it.
            metadata: { namespace: this.instance.metadata.namespace },
            egress: {
                default: networkpolicy.NetworkPolicyTrafficDefault.DENY,
            },
            ingress: {
                default: networkpolicy.NetworkPolicyTrafficDefault.DENY,
            },
        });
    }
}
exports.PodConnections = PodConnections;
_r = JSII_RTTI_SYMBOL_1;
PodConnections[_r] = { fqn: "cdk8s-plus-28.PodConnections", version: "2.3.13" };
//# sourceMappingURL=data:application/json;base64,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