"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.CronJob = exports.ConcurrencyPolicy = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const cdk8s_1 = require("cdk8s");
const k8s = require("./imports/k8s");
const pod_1 = require("./pod");
const workload = require("./workload");
/**
 * Concurrency policy for CronJobs.
 */
var ConcurrencyPolicy;
(function (ConcurrencyPolicy) {
    /**
     * This policy allows to run job concurrently.
     */
    ConcurrencyPolicy["ALLOW"] = "Allow";
    /**
     * This policy does not allow to run job concurrently. It does not let a new job to be scheduled if the previous one is not finished yet.
     */
    ConcurrencyPolicy["FORBID"] = "Forbid";
    /**
     * This policy replaces the currently running job if a new job is being scheduled.
     */
    ConcurrencyPolicy["REPLACE"] = "Replace";
})(ConcurrencyPolicy = exports.ConcurrencyPolicy || (exports.ConcurrencyPolicy = {}));
/**
 * A CronJob is responsible for creating a Job and scheduling it based on provided cron schedule. This helps running Jobs in a recurring manner.
 */
class CronJob extends workload.Workload {
    constructor(scope, id, props) {
        super(scope, id, {
            restartPolicy: pod_1.RestartPolicy.NEVER,
            select: false,
            ...props,
        });
        /**
         * Represents the resource type.
         */
        this.resourceType = 'cronjob';
        this.apiObject = new k8s.KubeCronJob(this, 'Resource', {
            metadata: props.metadata,
            spec: cdk8s_1.Lazy.any({ produce: () => this._toKube() }),
        });
        if (props.startingDeadline != undefined && props.startingDeadline.toSeconds() < 10) {
            throw new Error(`The 'startingDeadline' property cannot be less than 10 seconds since the Kubernetes CronJobController checks things every 10 seconds and hence the CronJob may not be scheduled. The value passed is: ${props.startingDeadline}`);
        }
        if (props.ttlAfterFinished != undefined && (props.successfulJobsRetained != undefined || props.failedJobsRetained != undefined)) {
            throw new Error('The \'ttlAfterFinished\' property cannot be set if \'successfulJobsRetained\' property or \'failedJobsRetained\' property is set. This would cause the retention of jobs to not work properly since it would delete the job based on its value.');
        }
        if (this.isolate) {
            this.connections.isolate();
        }
        this.schedule = props.schedule;
        this.timeZone = props.timeZone;
        this.concurrencyPolicy = props.concurrencyPolicy ?? ConcurrencyPolicy.FORBID;
        this.startingDeadline = props.startingDeadline ?? cdk8s_1.Duration.seconds(10);
        this.suspend = props.suspend ?? false;
        this.successfulJobsRetained = props.successfulJobsRetained ?? 3;
        this.failedJobsRetained = props.failedJobsRetained ?? 1;
        this.jobProperties = {
            activeDeadline: props.activeDeadline,
            backoffLimit: props.backoffLimit,
            ttlAfterFinished: props.ttlAfterFinished,
        };
    }
    /**
     * @internal
     */
    _toKube() {
        return {
            concurrencyPolicy: this.concurrencyPolicy,
            failedJobsHistoryLimit: this.failedJobsRetained,
            jobTemplate: {
                spec: this._toJobSpec(),
            },
            schedule: this.schedule.expressionString,
            startingDeadlineSeconds: this.startingDeadline.toSeconds(),
            successfulJobsHistoryLimit: this.successfulJobsRetained,
            suspend: this.suspend,
            timeZone: this.timeZone,
        };
    }
    /**
     * Returns the job spec.
     */
    _toJobSpec() {
        return {
            template: {
                metadata: this.podMetadata.toJson(),
                spec: this._toPodSpec(),
            },
            activeDeadlineSeconds: this.jobProperties.activeDeadline?.toSeconds(),
            backoffLimit: this.jobProperties.backoffLimit,
            ttlSecondsAfterFinished: this.jobProperties.ttlAfterFinished?.toSeconds(),
        };
    }
}
exports.CronJob = CronJob;
_a = JSII_RTTI_SYMBOL_1;
CronJob[_a] = { fqn: "cdk8s-plus-28.CronJob", version: "2.3.13" };
//# sourceMappingURL=data:application/json;base64,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