"use strict";
var _a, _b, _c, _d, _e, _f;
Object.defineProperty(exports, "__esModule", { value: true });
exports.Env = exports.extractContainerPorts = exports.EnvFrom = exports.Cpu = exports.MountPropagation = exports.Container = exports.ContainerRestartPolicy = exports.ImagePullPolicy = exports.EnvValue = exports.ResourceFieldPaths = exports.EnvFieldPaths = exports.ContainerSecurityContext = exports.Protocol = exports.Capability = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const cdk8s_1 = require("cdk8s");
const k8s = require("./imports/k8s");
const probe = require("./probe");
const utils_1 = require("./utils");
/**
 * Capability - complete list of POSIX capabilities
 */
var Capability;
(function (Capability) {
    /**
     * ALL
     */
    Capability["ALL"] = "ALL";
    /**
     * CAP_AUDIT_CONTROL
     */
    Capability["AUDIT_CONTROL"] = "AUDIT_CONTROL";
    /**
     * CAP_AUDIT_READ
     */
    Capability["AUDIT_READ"] = "AUDIT_READ";
    /**
     * CAP_AUDIT_WRITE
     */
    Capability["AUDIT_WRITE"] = "AUDIT_WRITE";
    /**
     * CAP_BLOCK_SUSPEND
     */
    Capability["BLOCK_SUSPEND"] = "BLOCK_SUSPEND";
    /**
     * CAP_BPF
     */
    Capability["BPF"] = "BPF";
    /**
     * CAP_CHECKPOINT_RESTORE
     */
    Capability["CHECKPOINT_RESTORE"] = "CHECKPOINT_RESTORE";
    /**
     * CAP_CHOWN
     */
    Capability["CHOWN"] = "CHOWN";
    /**
     * CAP_DAC_OVERRIDE
     */
    Capability["DAC_OVERRIDE"] = "DAC_OVERRIDE";
    /**
     * CAP_DAC_READ_SEARCH
     */
    Capability["DAC_READ_SEARCH"] = "DAC_READ_SEARCH";
    /**
     * CAP_FOWNER
     */
    Capability["FOWNER"] = "FOWNER";
    /**
     * CAP_FSETID
     */
    Capability["FSETID"] = "FSETID";
    /**
     * CAP_IPC_LOCK
     */
    Capability["IPC_LOCK"] = "IPC_LOCK";
    /**
     * CAP_IPC_OWNER
     */
    Capability["IPC_OWNER"] = "IPC_OWNER";
    /**
     * CAP_KILL
     */
    Capability["KILL"] = "KILL";
    /**
     * CAP_LEASE
     */
    Capability["LEASE"] = "LEASE";
    /**
     * CAP_LINUX_IMMUTABLE
     */
    Capability["LINUX_IMMUTABLE"] = "LINUX_IMMUTABLE";
    /**
     * CAP_MAC_ADMIN
     */
    Capability["MAC_ADMIN"] = "MAC_ADMIN";
    /**
     * CAP_MAC_OVERRIDE
     */
    Capability["MAC_OVERRIDE"] = "MAC_OVERRIDE";
    /**
     * CAP_MKNOD
     */
    Capability["MKNOD"] = "MKNOD";
    /**
     * CAP_NET_ADMIN
     */
    Capability["NET_ADMIN"] = "NET_ADMIN";
    /**
     * CAP_NET_BIND_SERVICE
     */
    Capability["NET_BIND_SERVICE"] = "NET_BIND_SERVICE";
    /**
     * CAP_NET_BROADCAST
     */
    Capability["NET_BROADCAST"] = "NET_BROADCAST";
    /**
     * CAP_NET_RAW
     */
    Capability["NET_RAW"] = "NET_RAW";
    /**
     * CAP_PERFMON
     */
    Capability["PERFMON"] = "PERFMON";
    /**
     * CAP_SETGID
     */
    Capability["SETGID"] = "SETGID";
    /**
     * CAP_SETFCAP
     */
    Capability["SETFCAP"] = "SETFCAP";
    /**
     * CAP_SETPCAP
     */
    Capability["SETPCAP"] = "SETPCAP";
    /**
     * CAP_SETUID
     */
    Capability["SETUID"] = "SETUID";
    /**
     * CAP_SYS_ADMIN
     */
    Capability["SYS_ADMIN"] = "SYS_ADMIN";
    /**
     * CAP_SYS_BOOT
     */
    Capability["SYS_BOOT"] = "SYS_BOOT";
    /**
     * CAP_SYS_CHROOT
     */
    Capability["SYS_CHROOT"] = "SYS_CHROOT";
    /**
     * CAP_SYS_MODULE
     */
    Capability["SYS_MODULE"] = "SYS_MODULE";
    /**
     * CAP_SYS_NICE
     */
    Capability["SYS_NICE"] = "SYS_NICE";
    /**
     * CAP_SYS_PACCT
     */
    Capability["SYS_PACCT"] = "SYS_PACCT";
    /**
     * CAP_SYS_PTRACE
     */
    Capability["SYS_PTRACE"] = "SYS_PTRACE";
    /**
     * CAP_SYS_RAWIO
     */
    Capability["SYS_RAWIO"] = "SYS_RAWIO";
    /**
     * CAP_SYS_RESOURCE
     */
    Capability["SYS_RESOURCE"] = "SYS_RESOURCE";
    /**
     * CAP_SYS_TIME
     */
    Capability["SYS_TIME"] = "SYS_TIME";
    /**
     * CAP_SYS_TTY_CONFIG
     */
    Capability["SYS_TTY_CONFIG"] = "SYS_TTY_CONFIG";
    /**
     * CAP_SYSLOG
     */
    Capability["SYSLOG"] = "SYSLOG";
    /**
     * CAP_WAKE_ALARM
     */
    Capability["WAKE_ALARM"] = "WAKE_ALARM";
})(Capability = exports.Capability || (exports.Capability = {}));
/**
 * Network protocols.
 */
var Protocol;
(function (Protocol) {
    /**
     * TCP.
     */
    Protocol["TCP"] = "TCP";
    /**
     * UDP.
     */
    Protocol["UDP"] = "UDP";
    /**
     * SCTP.
     */
    Protocol["SCTP"] = "SCTP";
})(Protocol = exports.Protocol || (exports.Protocol = {}));
/**
 * Container security attributes and settings.
 */
class ContainerSecurityContext {
    constructor(props = {}) {
        this.ensureNonRoot = props.ensureNonRoot ?? true;
        this.privileged = props.privileged ?? false;
        this.readOnlyRootFilesystem = props.readOnlyRootFilesystem ?? true;
        this.user = props.user;
        this.group = props.group;
        this.allowPrivilegeEscalation = props.allowPrivilegeEscalation ?? false;
        this.capabilities = props.capabilities;
    }
    /**
     * @internal
     */
    _toKube() {
        return {
            runAsGroup: this.group,
            runAsUser: this.user,
            runAsNonRoot: this.ensureNonRoot,
            privileged: this.privileged,
            readOnlyRootFilesystem: this.readOnlyRootFilesystem,
            allowPrivilegeEscalation: this.allowPrivilegeEscalation,
            capabilities: this.capabilities,
        };
    }
}
exports.ContainerSecurityContext = ContainerSecurityContext;
_a = JSII_RTTI_SYMBOL_1;
ContainerSecurityContext[_a] = { fqn: "cdk8s-plus-28.ContainerSecurityContext", version: "2.3.13" };
var EnvFieldPaths;
(function (EnvFieldPaths) {
    /**
     * The name of the pod.
     */
    EnvFieldPaths["POD_NAME"] = "metadata.name";
    /**
     * The namespace of the pod.
     */
    EnvFieldPaths["POD_NAMESPACE"] = "metadata.namespace";
    /**
     * The uid of the pod.
     */
    EnvFieldPaths["POD_UID"] = "metadata.uid";
    /**
     * The labels of the pod.
     */
    EnvFieldPaths["POD_LABEL"] = "metadata.labels";
    /**
     * The annotations of the pod.
     */
    EnvFieldPaths["POD_ANNOTATION"] = "metadata.annotations";
    /**
     * The ipAddress of the pod.
     */
    EnvFieldPaths["POD_IP"] = "status.podIP";
    /**
     * The service account name of the pod.
     */
    EnvFieldPaths["SERVICE_ACCOUNT_NAME"] = "spec.serviceAccountName";
    /**
     * The name of the node.
     */
    EnvFieldPaths["NODE_NAME"] = "spec.nodeName";
    /**
     * The ipAddress of the node.
     */
    EnvFieldPaths["NODE_IP"] = "status.hostIP";
    /**
     * The ipAddresess of the pod.
     */
    EnvFieldPaths["POD_IPS"] = "status.podIPs";
})(EnvFieldPaths = exports.EnvFieldPaths || (exports.EnvFieldPaths = {}));
var ResourceFieldPaths;
(function (ResourceFieldPaths) {
    /**
     * CPU limit of the container.
     */
    ResourceFieldPaths["CPU_LIMIT"] = "limits.cpu";
    /**
     * Memory limit of the container.
     */
    ResourceFieldPaths["MEMORY_LIMIT"] = "limits.memory";
    /**
     * CPU request of the container.
     */
    ResourceFieldPaths["CPU_REQUEST"] = "requests.cpu";
    /**
     * Memory request of the container.
     */
    ResourceFieldPaths["MEMORY_REQUEST"] = "requests.memory";
    /**
     * Ephemeral storage limit of the container.
     */
    ResourceFieldPaths["STORAGE_LIMIT"] = "limits.ephemeral-storage";
    /**
     * Ephemeral storage request of the container.
     */
    ResourceFieldPaths["STORAGE_REQUEST"] = "requests.ephemeral-storage";
})(ResourceFieldPaths = exports.ResourceFieldPaths || (exports.ResourceFieldPaths = {}));
/**
 * Utility class for creating reading env values from various sources.
 */
class EnvValue {
    constructor(value, valueFrom) {
        this.value = value;
        this.valueFrom = valueFrom;
    }
    /**
     * Create a value by reading a specific key inside a config map.
     *
     * @param configMap - The config map.
     * @param key - The key to extract the value from.
     * @param options - Additional options.
     */
    static fromConfigMap(configMap, key, options = {}) {
        const source = {
            configMapKeyRef: {
                name: configMap.name,
                key,
                optional: options.optional,
            },
        };
        return new EnvValue(undefined, source);
    }
    /**
     * Defines an environment value from a secret JSON value.
     *
     * @param secretValue The secret value (secrent + key)
     * @param options Additional options
     */
    static fromSecretValue(secretValue, options = {}) {
        const source = {
            secretKeyRef: {
                name: secretValue.secret.name,
                key: secretValue.key,
                optional: options.optional,
            },
        };
        return new EnvValue(undefined, source);
    }
    /**
     * Create a value from the given argument.
     *
     * @param value - The value.
     */
    static fromValue(value) {
        return new EnvValue(value);
    }
    /**
     *
     * Create a value from a field reference.
     *
     * @param fieldPath: The field reference.
     * @param options: Additional options.
     */
    static fromFieldRef(fieldPath, options = {}) {
        let needsKey = false;
        if (fieldPath === EnvFieldPaths.POD_LABEL || fieldPath === EnvFieldPaths.POD_ANNOTATION) {
            needsKey = true;
        }
        if (needsKey && options.key === undefined) {
            throw new Error(`${fieldPath} requires a key`);
        }
        const source = {
            fieldRef: {
                fieldPath: (`${fieldPath}` + (needsKey ? `['${options.key}']` : '')),
                ...(options.apiVersion ? { apiVersion: options.apiVersion } : {}),
            },
        };
        return new EnvValue(undefined, source);
    }
    /**
     * Create a value from a resource.
     *
     * @param resource: Resource to select the value from.
     * @param options: Additional options.
     */
    static fromResource(resource, options = {}) {
        const source = {
            resourceFieldRef: {
                resource: `${resource}`,
                ...(options.divisor ? { divisor: k8s.IntOrString.fromString(options.divisor) } : {}),
                ...(options.container ? { containerName: options.container.name } : {}),
            },
        };
        return new EnvValue(undefined, source);
    }
    /**
     * Create a value from a key in the current process environment.
     *
     * @param key - The key to read.
     * @param options - Additional options.
     */
    static fromProcess(key, options = {}) {
        const value = process.env[key];
        if (options.required && !value) {
            throw new Error(`Missing ${key} env variable`);
        }
        return EnvValue.fromValue(value);
    }
}
exports.EnvValue = EnvValue;
_b = JSII_RTTI_SYMBOL_1;
EnvValue[_b] = { fqn: "cdk8s-plus-28.EnvValue", version: "2.3.13" };
var ImagePullPolicy;
(function (ImagePullPolicy) {
    /**
     * Every time the kubelet launches a container, the kubelet queries the container image registry
     * to resolve the name to an image digest. If the kubelet has a container image with that exact
     * digest cached locally, the kubelet uses its cached image; otherwise, the kubelet downloads
     * (pulls) the image with the resolved digest, and uses that image to launch the container.
     *
     * Default is Always if ImagePullPolicy is omitted and either the image tag is :latest or
     * the image tag is omitted.
     */
    ImagePullPolicy["ALWAYS"] = "Always";
    /**
     * The image is pulled only if it is not already present locally.
     *
     * Default is IfNotPresent if ImagePullPolicy is omitted and the image tag is present but
     * not :latest
     */
    ImagePullPolicy["IF_NOT_PRESENT"] = "IfNotPresent";
    /**
     * The image is assumed to exist locally. No attempt is made to pull the image.
     */
    ImagePullPolicy["NEVER"] = "Never";
})(ImagePullPolicy = exports.ImagePullPolicy || (exports.ImagePullPolicy = {}));
/**
 * RestartPolicy defines the restart behavior of individual containers in a pod.
 * This field may only be set for init containers, and the only allowed value is "Always".
 * For non-init containers or when this field is not specified,
 * the restart behavior is defined by the Pod's restart policy and the container type.
 * Setting the RestartPolicy as "Always" for the init container will have the following effect:
 * this init container will be continually restarted on exit until all regular containers have terminated.
 * Once all regular containers have completed, all init containers with restartPolicy "Always" will be shut down.
 * This lifecycle differs from normal init containers and is often referred to as a "sidecar" container.
 *
 * @see https://kubernetes.io/docs/concepts/workloads/pods/sidecar-containers/
 */
var ContainerRestartPolicy;
(function (ContainerRestartPolicy) {
    /**
     * If an init container is created with its restartPolicy set to Always,
     * it will start and remain running during the entire life of the Pod.
     * For regular containers, this is ignored by Kubernetes.
     */
    ContainerRestartPolicy["ALWAYS"] = "Always";
})(ContainerRestartPolicy = exports.ContainerRestartPolicy || (exports.ContainerRestartPolicy = {}));
/**
 * A single application container that you want to run within a pod.
 */
class Container {
    constructor(props) {
        this._ports = [];
        if (props instanceof Container) {
            throw new Error('Attempted to construct a container from a Container object.');
        }
        const defaultResourceSpec = {
            cpu: {
                request: Cpu.millis(1000),
                limit: Cpu.millis(1500),
            },
            memory: {
                request: cdk8s_1.Size.mebibytes(512),
                limit: cdk8s_1.Size.mebibytes(2048),
            },
        };
        if (props.port && props.portNumber) {
            throw new Error('Either \'port\' or \'portNumber\' is allowed. Use \'portNumber\' since \'port\' is deprecated');
        }
        const portNumber = props.portNumber ?? props.port;
        const defaultProbeConfiguration = probe.Probe.fromTcpSocket({ port: portNumber });
        this.name = props.name ?? 'main';
        this.image = props.image;
        this.portNumber = portNumber;
        this._command = props.command;
        this._args = props.args;
        this._readiness = props.readiness;
        this._liveness = props.liveness;
        this._startup = props.startup ?? (this.portNumber ? defaultProbeConfiguration : undefined);
        this._lifecycle = props.lifecycle;
        this.resources = props.resources ?? defaultResourceSpec;
        this.workingDir = props.workingDir;
        this.mounts = props.volumeMounts ?? [];
        this.imagePullPolicy = props.imagePullPolicy ?? ImagePullPolicy.ALWAYS;
        this.securityContext = new ContainerSecurityContext(props.securityContext);
        this.env = new Env(props.envFrom ?? [], props.envVariables ?? {});
        this.restartPolicy = props.restartPolicy;
        if (this.portNumber) {
            this.addPort({
                number: this.portNumber,
            });
        }
        for (const port of props.ports ?? []) {
            this.addPort(port);
        }
    }
    /**
     * @deprecated - use `portNumber`.
     */
    get port() {
        return this.portNumber;
    }
    /**
     * Ports exposed by this containers.
     * Returns a copy, use `addPort` to modify.
     */
    get ports() {
        return [...this._ports];
    }
    /**
     * Entrypoint array (the command to execute when the container starts).
     * @returns a copy of the entrypoint array, cannot be modified
     */
    get command() {
        return this._command ? [...this._command] : undefined;
    }
    /**
     * Arguments to the entrypoint.
     *
     * @returns a copy of the arguments array, cannot be modified.
     */
    get args() {
        return this._args ? [...this._args] : undefined;
    }
    /**
     * Mount a volume to a specific path so that it is accessible by the container.
     * Every pod that is configured to use this container will autmoatically have access to the volume.
     *
     * @param path - The desired path in the container.
     * @param storage - The storage to mount.
     */
    mount(path, storage, options = {}) {
        this.mounts.push({ path, volume: storage.asVolume(), ...options });
    }
    /**
     * Add a port to expose from this container.
     */
    addPort(port) {
        const names = this._ports.map(p => p.name).filter(x => x);
        const numberProtocols = this._ports.map(p => `${p.number}/${p.protocol || Protocol.TCP}`);
        if (port.name && names.includes(port.name)) {
            throw new Error(`Port with name ${port.name} already exists`);
        }
        const protocol = `${port.number}/${port.protocol || Protocol.TCP}`;
        if (numberProtocols.includes(protocol)) {
            throw new Error(`Port with number ${port.number} and protocol ${port.protocol || Protocol.TCP} already exists`);
        }
        this._ports.push(port);
    }
    /**
     * @internal
     */
    _toKube() {
        const volumeMounts = [];
        for (const mount of this.mounts) {
            volumeMounts.push({
                name: mount.volume.name,
                mountPath: mount.path,
                readOnly: mount.readOnly,
                mountPropagation: mount.propagation,
                subPath: mount.subPath,
                subPathExpr: mount.subPathExpr,
            });
        }
        const ports = new Array();
        for (const port of this.ports) {
            ports.push({
                containerPort: port.number,
                protocol: port.protocol?.toString(),
                name: port.name,
                hostPort: port.hostPort,
                hostIp: port.hostIp,
            });
        }
        // Resource requests and limits
        const cpuLimit = this.resources?.cpu?.limit?.amount;
        const cpuRequest = this.resources?.cpu?.request?.amount;
        const memoryLimit = this.resources?.memory?.limit;
        const memoryRequest = this.resources?.memory?.request;
        const ephemeralStorageLimit = this.resources?.ephemeralStorage?.limit;
        const ephemeralStorageRequest = this.resources?.ephemeralStorage?.request;
        const limits = {};
        const requests = {};
        if (cpuLimit) {
            limits.cpu = k8s.Quantity.fromString(cpuLimit);
        }
        if (memoryLimit) {
            limits.memory = k8s.Quantity.fromString(memoryLimit.toMebibytes().toString() + 'Mi');
        }
        if (ephemeralStorageLimit) {
            limits['ephemeral-storage'] = k8s.Quantity.fromString(ephemeralStorageLimit.toGibibytes().toString() + 'Gi');
        }
        if (cpuRequest) {
            requests.cpu = k8s.Quantity.fromString(cpuRequest);
        }
        if (memoryRequest) {
            requests.memory = k8s.Quantity.fromString(memoryRequest.toMebibytes().toString() + 'Mi');
        }
        if (ephemeralStorageRequest) {
            requests['ephemeral-storage'] = k8s.Quantity.fromString(ephemeralStorageRequest.toGibibytes().toString() + 'Gi');
        }
        let resourceRequirements = undefined;
        if (Object.keys(limits).length > 0 || Object.keys(requests).length > 0) {
            resourceRequirements = {
                limits: utils_1.undefinedIfEmpty(limits),
                requests: utils_1.undefinedIfEmpty(requests),
            };
        }
        const env = this.env._toKube();
        return {
            name: this.name,
            image: this.image,
            imagePullPolicy: this.imagePullPolicy,
            ports: utils_1.undefinedIfEmpty(ports),
            volumeMounts: utils_1.undefinedIfEmpty(volumeMounts),
            command: this.command,
            args: this.args,
            workingDir: this.workingDir,
            env: env.variables,
            envFrom: env.from,
            readinessProbe: this._readiness?._toKube(this),
            livenessProbe: this._liveness?._toKube(this),
            startupProbe: this._startup?._toKube(this),
            lifecycle: this._lifecycle ? {
                postStart: this._lifecycle.postStart?._toKube(this),
                preStop: this._lifecycle.preStop?._toKube(this),
            } : undefined,
            resources: resourceRequirements,
            restartPolicy: this.restartPolicy,
            securityContext: this.securityContext._toKube(),
        };
    }
}
exports.Container = Container;
_c = JSII_RTTI_SYMBOL_1;
Container[_c] = { fqn: "cdk8s-plus-28.Container", version: "2.3.13" };
var MountPropagation;
(function (MountPropagation) {
    /**
     * This volume mount will not receive any subsequent mounts that are mounted
     * to this volume or any of its subdirectories by the host. In similar
     * fashion, no mounts created by the Container will be visible on the host.
     *
     * This is the default mode.
     *
     * This mode is equal to `private` mount propagation as described in the Linux
     * kernel documentation
     */
    MountPropagation["NONE"] = "None";
    /**
     * This volume mount will receive all subsequent mounts that are mounted to
     * this volume or any of its subdirectories.
     *
     * In other words, if the host mounts anything inside the volume mount, the
     * Container will see it mounted there.
     *
     * Similarly, if any Pod with Bidirectional mount propagation to the same
     * volume mounts anything there, the Container with HostToContainer mount
     * propagation will see it.
     *
     * This mode is equal to `rslave` mount propagation as described in the Linux
     * kernel documentation
     */
    MountPropagation["HOST_TO_CONTAINER"] = "HostToContainer";
    /**
     * This volume mount behaves the same the HostToContainer mount. In addition,
     * all volume mounts created by the Container will be propagated back to the
     * host and to all Containers of all Pods that use the same volume
     *
     * A typical use case for this mode is a Pod with a FlexVolume or CSI driver
     * or a Pod that needs to mount something on the host using a hostPath volume.
     *
     * This mode is equal to `rshared` mount propagation as described in the Linux
     * kernel documentation
     *
     * Caution: Bidirectional mount propagation can be dangerous. It can damage
     * the host operating system and therefore it is allowed only in privileged
     * Containers. Familiarity with Linux kernel behavior is strongly recommended.
     * In addition, any volume mounts created by Containers in Pods must be
     * destroyed (unmounted) by the Containers on termination.
     *
     */
    MountPropagation["BIDIRECTIONAL"] = "Bidirectional";
})(MountPropagation = exports.MountPropagation || (exports.MountPropagation = {}));
/**
 * Represents the amount of CPU.
 * The amount can be passed as millis or units.
 */
class Cpu {
    constructor(amount) {
        this.amount = amount;
    }
    static millis(amount) {
        return new Cpu(amount + 'm');
    }
    static units(amount) {
        return new Cpu(amount.toString());
    }
}
exports.Cpu = Cpu;
_d = JSII_RTTI_SYMBOL_1;
Cpu[_d] = { fqn: "cdk8s-plus-28.Cpu", version: "2.3.13" };
/**
 * A collection of env variables defined in other resources.
 */
class EnvFrom {
    constructor(configMap, prefix, sec) {
        this.configMap = configMap;
        this.prefix = prefix;
        this.sec = sec;
    }
    ;
    /**
     * @internal
     */
    _toKube() {
        return {
            configMapRef: this.configMap ? {
                name: this.configMap.name,
            } : undefined,
            secretRef: this.sec ? {
                name: this.sec.name,
            } : undefined,
            prefix: this.prefix,
        };
    }
}
exports.EnvFrom = EnvFrom;
_e = JSII_RTTI_SYMBOL_1;
EnvFrom[_e] = { fqn: "cdk8s-plus-28.EnvFrom", version: "2.3.13" };
function extractContainerPorts(selector) {
    if (!selector) {
        return [];
    }
    // we don't use instanceof intentionally since it can create
    // cyclic import problems.
    const containers = selector.containers ?? [];
    return containers.flatMap(c => c.ports);
}
exports.extractContainerPorts = extractContainerPorts;
/**
 * Container environment variables.
 */
class Env {
    constructor(sources, variables) {
        this._sources = sources;
        this._variables = variables;
    }
    /**
     * Selects a ConfigMap to populate the environment variables with.
     * The contents of the target ConfigMap's Data field will represent
     * the key-value pairs as environment variables.
     */
    static fromConfigMap(configMap, prefix) {
        return new EnvFrom(configMap, prefix, undefined);
    }
    /**
     * Selects a Secret to populate the environment variables with.
     * The contents of the target Secret's Data field will represent
     * the key-value pairs as environment variables.
     */
    static fromSecret(secr) {
        return new EnvFrom(undefined, undefined, secr);
    }
    /**
     * Add a single variable by name and value.
     * The variable value can come from various dynamic sources such a secrets of config maps.
     * Use `EnvValue.fromXXX` to select sources.
     */
    addVariable(name, value) {
        this._variables[name] = value;
    }
    /**
     * The environment variables for this container.
     * Returns a copy. To add environment variables use `container.env.addVariable()`.
     */
    get variables() {
        return { ...this._variables };
    }
    /**
     * Add a collection of variables by copying from another source.
     * Use `Env.fromXXX` functions to select sources.
     */
    copyFrom(from) {
        this._sources.push(from);
    }
    /**
     * The list of sources used to populate the container environment,
     * in addition to the `variables`.
     *
     * Returns a copy. To add a source use `container.env.copyFrom()`.
     */
    get sources() {
        return [...this._sources];
    }
    renderEnv(env) {
        const result = new Array();
        for (const [name, v] of Object.entries(env)) {
            result.push({
                name,
                value: v.value,
                valueFrom: v.valueFrom,
            });
        }
        return result;
    }
    /**
     * @internal
     */
    _toKube() {
        return {
            from: utils_1.undefinedIfEmpty(this._sources.map(s => s._toKube())),
            variables: utils_1.undefinedIfEmpty(this.renderEnv(this._variables)),
        };
    }
}
exports.Env = Env;
_f = JSII_RTTI_SYMBOL_1;
Env[_f] = { fqn: "cdk8s-plus-28.Env", version: "2.3.13" };
//# sourceMappingURL=data:application/json;base64,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