import os
import uuid

from django.conf import settings
from django.contrib.contenttypes.fields import GenericForeignKey, GenericRelation
from django.contrib.contenttypes.models import ContentType
from django.contrib.sites.models import Site
from django.core.files.storage import FileSystemStorage
from django.core.serializers.json import DjangoJSONEncoder
from django.core.signing import Signer
from django.dispatch import Signal
from django.db import models
from django.urls import reverse
from django.utils.text import slugify

from djmoney.models.fields import MoneyField
from treebeard.mp_tree import MP_Node
from versatileimagefield.fields import VersatileImageField

from .managers import DocumentManager
from .util import (
    bumpy_case_words,
    UploadImageAttachmentTo,
    UploadSecureAttachmentTo,
)

from .shortcuts import get_current_site


class ModelBase(models.Model):
    email_classes = {}

    class Meta:
        abstract = True

    # DRY-rest-permissions
    @staticmethod
    def has_read_permission(request):
        return True

    def has_object_read_permission(self, request):
        return True

    @staticmethod
    def has_write_permission(request):
        return True

    def has_object_write_permission(self, request):
        return True

    @property
    def content_type_name(self):
        return str(self._meta)

    @property
    def informal_description(self):
        """
        Return the class name as a string of lowercase words. Used in construction
        of comment email body.
        """
        return " ".join([w.lower() for w in bumpy_case_words(self.__class__.__name__)])

    def clone(self):
        """
        Returns: a new model instance with all of its direct field values set
        to those of this model.
        """
        ModelClass = self._meta.model
        model_fields = ModelClass._meta.get_fields()

        duplicate = ModelClass()

        for field in model_fields:
            field_name = field.name

            if (isinstance(field, GenericRelation)
                or field.one_to_many
                or field.auto_created
                or field.many_to_many
                or field.one_to_one
            ):
                continue

            record_value = getattr(self, field_name)
            setattr(duplicate, field_name, record_value)

        duplicate.save()
        return duplicate

    def notification_subscribers(self, event_name=None, digest=None):
        from notificationcenter.models import default_center as notification_center
        notification_name = self.notification_name_for_event(event_name)
        if digest is None:
            return notification_center.subscribers(notification_name, sender=self, digest=True) | \
                   notification_center.subscribers(notification_name, sender=self, digest=False)

        return notification_center.subscribers(notification_name, sender=self, digest=digest)

    @property
    def all_subscribers(self):
        return self.notification_subscribers()

    @property
    def alert_subscribers(self):
        return self.notification_subscribers(digest=False)

    @property
    def digest_subscribers(self):
        return self.notification_subscribers(digest=True)

    @property
    def notification_mail_classes(self):
        return {}

    def notification_name_for_event(self, event_name=None):
        content_type = ContentType.objects.get_for_model(self)
        name_components = [content_type.model]
        if event_name is not None:
            if isinstance(event_name, str):
                name_components.append(event_name)
            else:
                raise TypeError('event_name expected to be of type string. Got %s' % type(event_name))

        return ':'.join(name_components)

    def send_notification_email(self, event_type, **kwargs):
        from notificationcenter.models import default_center as notification_center

        request = kwargs['request']

        EmailClass = self.notification_mail_classes.get(event_type)
        if not EmailClass:
            return

        email = EmailClass(self, request=request)
        email.render()

        notification_context = {
            'creator': request.user.id,
            #'headers': email.extra_headers,
        }

        # Let the NotificationCenter handle the actual sending of the email
        # TODO: introduce an argument to control the notify_immediately behavior
        notification_name = self.notification_name_for_event(event_type)
        notification_center.post_notification(
            notification_name,
            self,
            email.subject,
            text=email.body,
            html=email.html,
            context=notification_context,
            notify_immediately=False
        )

    def notify_users(self, event_type, **kwargs):
        """
        Perform email notification in response to the given event type. Subclasses
        may customize the behavior of this method (if necessary) by overriding
        one or more of the following related methods:
          - send_notification_email
        """
        self.send_notification_email(event_type, **kwargs)


class Place(ModelBase):
    place_id = models.CharField(max_length=255, unique=True)
    info = models.JSONField()

    def __str__(self):
        return self.place_id


class Region(MP_Node):
    name = models.CharField(max_length=150)
    slug = models.SlugField(max_length=150, blank=False, default=None)
    code = models.CharField(max_length=4, blank=True)
    place = models.ForeignKey(Place, blank=True, null=True, on_delete=models.SET_NULL)

    node_order_by = ['name']

    def __str__(self):
        return self.name

    @property
    def full_name(self):
        name_components = [d.name for d in self.get_ancestors()]
        name_components.append(self.name)
        return ' - '.join(name_components)

    @property
    def root_name(self):
        return self.get_root().name


class Tag(MP_Node):
    name = models.CharField(max_length=50)
    type = models.CharField(max_length=20, blank=True, null=True)
    purpose = models.CharField(max_length=255, blank=True, null=True)
    position = models.SmallIntegerField(default=0)

    node_order_by = ['position']

    class Meta:
        unique_together = ('name', 'type',)

    def __str__(self):
        return self.name

    @property
    def full_name(self):
        name_components = [d.name for d in self.get_ancestors()]
        name_components.append(self.name)
        return ' - '.join(name_components)

    @property
    def root_name(self):
        return self.get_root().name


class Document(ModelBase):
    date_created = models.DateTimeField(auto_now_add=True, editable=False)
    date_modified = models.DateTimeField(auto_now=True, editable=False)

    created_by = models.ForeignKey(
        settings.AUTH_USER_MODEL,
        related_name='%(class)s_created_by',
        on_delete=models.PROTECT,
    )

    uuid = models.UUIDField(default=uuid.uuid4, editable=False, unique=True)
    site = models.ForeignKey(Site, on_delete=models.PROTECT)

    objects = DocumentManager()

    class Meta:
        abstract = True

    # DRY-rest-permissions
    def has_object_read_permission(self, request):
        return self.site == get_current_site(request)

    def has_object_write_permission(self, request):
        return self.site == get_current_site(request)

    @staticmethod
    def merge_strategy():
        return {}

    def get_media_attachment_directory(self, attachment):
        model_name = slugify(self._meta.verbose_name)
        return os.path.join(model_name, str(self.uuid))

    def get_detail_url_name(self):
        if hasattr(self, 'detail_url_name'):
            return getattr(self, 'detail_url_name')
        # TODO: Let url name be derived from the class name of document
        raise NotImplementedError

    def get_namespaced_detail_url_name(self, url_info):
        url_name = self.get_detail_url_name()
        namespace = url_info.kwargs.get('namespace')
        if namespace:
            url_name = '{}:{}'.format(namespace, url_name)
        return url_name

    def get_detail_url_info(self, url_info):
        namespace = url_info.kwargs.get('namespace')

        url_kwargs = {}
        if namespace == 'public_client':
            url_kwargs['uuid'] = self.uuid
        elif namespace == 'admin_client':
            url_kwargs['pk'] = self.id

        return {'kwargs': url_kwargs}

    def get_absolute_url(self, url_info):
        try:
            detail_url_name = self.get_namespaced_detail_url_name(url_info)
        except NotImplementedError:
            return None

        detail_url_info = self.get_detail_url_info(url_info)
        url = reverse(detail_url_name, kwargs=detail_url_info['kwargs'])

        url_prefix = detail_url_info.get('prefix')
        if url_prefix:
            url = url_prefix + url

        return url


def get_attachment_file_storage():
    return FileSystemStorage(location=settings.SECURE_MEDIA_ROOT)

class Attachment(Document):
    content_type = models.ForeignKey(ContentType, on_delete=models.CASCADE)
    object_id = models.PositiveIntegerField()
    object = GenericForeignKey('content_type', 'object_id')
    file = models.FileField(
        upload_to=UploadSecureAttachmentTo('attachments'),
        max_length=255,
        storage=get_attachment_file_storage,
    )

    signer = Signer(sep='/', salt='appkit.Attachment')

    def __str__(self):
        return self.file.name

    def basename(self):
        return os.path.basename(self.file.name)

    def media_root(self):
        if not hasattr(self.object, 'get_media_attachment_directory'):
            return ''
        return self.object.get_media_attachment_directory(self)


class ImageAttachment(Document):
    content_type = models.ForeignKey(ContentType, on_delete=models.CASCADE)
    object_id = models.PositiveIntegerField()
    object = GenericForeignKey('content_type', 'object_id')

    image = VersatileImageField(max_length=255, upload_to=UploadImageAttachmentTo())
    width = models.IntegerField()
    height = models.IntegerField()

    position = models.IntegerField(default=0)
    label = models.CharField(max_length=30, blank=True)

    class Meta:
        ordering = ('position',)

    def __str__(self):
        description = super().__str__()

        if self.label:
            description += ': {}'.format(self.label)

        return description


    @property
    def media_url(self):
        media_root = ''
        if hasattr(self.object, 'media_root'):
            media_root = self.object.media_root()

        return u"{0}{1}".format(
            media_root,
            self.get_absolute_url()
        )


class Thing(models.Model):
    attributes = models.JSONField(default=dict, encoder=DjangoJSONEncoder)
    name = models.CharField(max_length=100)
    description = models.TextField(blank=True)

    attachments = GenericRelation(Attachment)
    feature_image = models.OneToOneField(ImageAttachment, blank=True, null=True, on_delete=models.SET_NULL)
    images = GenericRelation(ImageAttachment)
    place = models.ForeignKey(Place, blank=True, null=True, on_delete=models.SET_NULL)
    tags = models.ManyToManyField(Tag, blank=True)

    class Meta:
        abstract = True

    def __str__(self):
        return self.name


class Product(Thing):
    archived = models.BooleanField(default=False)
    price = MoneyField(max_digits=14, decimal_places=2, blank=True, null=True, default_currency=None)
    sku = models.CharField(max_length=64, blank=True, null=True)

    class Meta:
        abstract = True


class Note(Document):
    class Meta:
        ordering = ('-date_created',)

    subject = models.CharField(max_length=255)
    text = models.TextField(blank=True, default='')

    attachments = GenericRelation(Attachment)
    notes = models.ManyToManyField('self', blank=True, related_name='notes', symmetrical=True)

    def __str__(self):
        return self.subject


class Arrangement(Document, Thing):
    type = models.CharField(max_length=20, blank=False, null=True)

    class Meta:
        unique_together = ('site', 'type',)

    @property
    def item_count(self):
        return self.items.count()


class ArrangementItem(Document):
    content_type = models.ForeignKey(ContentType, on_delete=models.CASCADE)
    object_id = models.PositiveIntegerField()
    represented_object = GenericForeignKey('content_type', 'object_id')

    position = models.SmallIntegerField(default=0)
    arrangement = models.ForeignKey(Arrangement, on_delete=models.CASCADE, related_name='items')

    def __str__(self):
        return str(self.represented_object)

    class Meta:
        ordering = ('position',)
        unique_together = ('arrangement', 'content_type', 'object_id',)

    # def get_detail_url_name(self):
    #     detail_url_name_template = 'core.{}.item.read'
    #     return detail_url_name_template.format(self.arrangement.type)
    #
    # def get_detail_url_info(self, url_info):
    #     detail_url_info = super().get_detail_url_info(url_info)
    #
    #     namespace = url_info.kwargs.get('client_namespace')
    #     if namespace == 'admin_client':
    #         detail_url_info['kwargs']['arrangement_id'] = self.arrangement.pk
    #     else:
    #         detail_url_info['kwargs']['arrangement_uuid'] = self.arrangement.uuid
    #
    #     return detail_url_info


# ------------------------------------------------------------------------------
document_event = Signal(providing_args=["type"])
def document_event_signal_receiver(sender, type, **kwargs):
    """
    Send notification email to users subscribed and/or concerned with the event
    of "type" that occurred on the given sender (Document)
    """
    sender.notify_users(type, **kwargs)
document_event.connect(
    document_event_signal_receiver,
    dispatch_uid='document_event_signal_receiver'
)
