# -*- coding: utf-8 -*-
from setuptools import setup

packages = \
['openapi_tester']

package_data = \
{'': ['*']}

install_requires = \
['django>=2.2,<4.0',
 'djangorestframework',
 'inflection',
 'openapi-spec-validator',
 'prance',
 'pyYAML']

setup_kwargs = {
    'name': 'drf-openapi-tester',
    'version': '1.3.9',
    'description': 'Test utility for validating OpenAPI response documentation',
    'long_description': '<a href="https://pypi.org/project/drf-openapi-tester/">\n    <img src="https://img.shields.io/pypi/v/drf-openapi-tester.svg" alt="Package version">\n</a>\n<a href="https://codecov.io/gh/snok/drf-openapi-tester">\n    <img src="https://codecov.io/gh/snok/drf-openapi-tester/branch/master/graph/badge.svg" alt="Code coverage">\n</a>\n<a href="https://pypi.org/project/drf-openapi-tester/">\n    <img src="https://img.shields.io/badge/python-3.6%2B-blue" alt="Supported Python versions">\n</a>\n<a href="https://pypi.python.org/pypi/drf-openapi-tester">\n    <img src="https://img.shields.io/badge/django%20versions-2.2%2B-blue" alt="Supported Django versions">\n</a>\n<a href="http://mypy-lang.org/">\n    <img src="http://www.mypy-lang.org/static/mypy_badge.svg" alt="Checked with mypy">\n</a>\n\n# DRF OpenAPI Tester\n\nThis is a test utility to validate DRF Test Responses against OpenAPI 2 and 3 schema. It has built-in support for:\n\n- OpenAPI 2/3 yaml or json schema files.\n- OpenAPI 2 schemas created with [drf-yasg](https://github.com/axnsan12/drf-yasg).\n- OpenAPI 3 schemas created with [drf-spectacular](https://github.com/tfranzel/drf-spectacular).\n\n## Installation\n\n```shell script\npip install drf-openapi-tester\n```\n\n## Usage\n\nFirst instantiate one or more instances of SchemaTester:\n\n```python\nfrom openapi_tester import SchemaTester\n\nschema_tester = SchemaTester()\n\n\n```\n\nIf you are using either [drf-yasg](https://github.com/axnsan12/drf-yasg)\nor [drf-spectacular](https://github.com/tfranzel/drf-spectacular) this will be auto-detected, and the schema will be\nloaded by the SchemaTester automatically. If you are using schema files though, you will need to pass the file path to\nthe tester:\n\n```python\nfrom openapi_tester import SchemaTester\n\n# path should be a string\nschema_tester = SchemaTester(schema_file_path="./schemas/publishedSpecs.yaml")\n\n\n```\n\nOnce you instantiate a tester, you can use it to test responses:\n\n```python\nfrom openapi_tester.schema_tester import SchemaTester\n\n# you need to create at least one instance of SchemaTester.\n# you can pass kwargs to it\nschema_tester = SchemaTester()\n\n\ndef test_response_documentation(client):\n    response = client.get(\'api/v1/test/1\')\n    assert response.status_code == 200\n    schema_tester.validate_response(response=response)\n```\n\nIf you are using the Django testing framework, you can create a base APITestCase that incorporates schema validation:\n\n```python\nfrom openapi_tester.schema_tester import SchemaTester\nfrom rest_framework.test import APITestCase\nfrom rest_framework.response import Response\n\nschema_tester = SchemaTester()\n\n\nclass BaseAPITestCase(APITestCase):\n    """ Base test class for api views including schema validation """\n\n    @staticmethod\n    def assertResponse(response: Response, **kwargs) -> None:\n        """ helper to run validate_response and pass kwargs to it """\n        schema_tester.validate_response(response=response, **kwargs)\n```\n\nThen use it in a test file:\n\n```python\nfrom shared.testing import BaseAPITestCase\n\n\nclass MyAPITests(BaseAPITestCase):\n    def test_some_view(self):\n        response = self.client.get("...")\n        self.assertResponse(response)\n```\n\n## Options\n\nYou can pass options either globally, when instantiating a `SchemaTester`, or locally, when\ninvoking `validate_response`:\n\n```python\nfrom openapi_tester import SchemaTester, is_camel_case\nfrom tests.utils import my_uuid_4_validator\n\nschema_test_with_case_validation = SchemaTester(\n    case_tester=is_camel_case,\n    ignore_case=["IP"],\n    validators=[my_uuid_4_validator]\n)\n\n```\n\nOr\n\n```python\nfrom openapi_tester import SchemaTester, is_camel_case\nfrom tests.utils import my_uuid_4_validator\n\nschema_tester = SchemaTester()\n\n\ndef my_test(client):\n    response = client.get(\'api/v1/test/1\')\n    assert response.status_code == 200\n    schema_tester.validate_response(\n        response=response,\n        case_tester=is_camel_case,\n        ignore_case=["IP"],\n        validators=[my_uuid_4_validator]\n    )\n```\n\n### case_tester\n\nThe case tester argument takes a callable that is used to validate the key casings of both schemas and responses. If\nnothing is passed, case validation is skipped.\n\nThe library currently has 4 built-in case testers:\n\n- `is_pascal_case`\n- `is_snake_case`\n- `is_camel_case`\n- `is_kebab_case`\n\nYou can of course pass your own custom case tester.\n\n### ignore_case\n\nList of keys to ignore when testing key casing. This setting only applies when case_tester is not `None`.\n\n### validators\n\nList of custom validators. A validator is a function that receives two parameters: schema_section and data, and returns\neither an error message or `None`, e.g.:\n\n```python\nfrom typing import Any, Optional\nfrom uuid import UUID\n\n\ndef my_uuid_4_validator(schema_section: dict, data: Any) -> Optional[str]:\n    schema_format = schema_section.get("format")\n    if schema_format == "uuid4":\n        try:\n            result = UUID(data, version=4)\n            if not str(result) == str(data):\n                return f"Expected uuid4, but received {data}"\n        except ValueError:\n            return f"Expected uuid4, but received {data}"\n    return None\n```\n\n### field_key_map\n\nYou can pass an optional dictionary that maps custom url parameter names into values, for cases where this cannot be\ninferred by the DRF `EndpointEnumerator`. A concrete use case for this option is when\nthe [django i18n locale prefixes](https://docs.djangoproject.com/en/3.1/topics/i18n/translation/#language-prefix-in-url-patterns).\n\n```python\nfrom openapi_tester import SchemaTester\n\nschema_tester = SchemaTester(field_key_map={\n  "language": "en",\n})\n```\n\n## Schema Validation\n\nWhen the SchemaTester loads a schema, it runs it through\n[OpenAPI Spec validator](https://github.com/p1c2u/openapi-spec-validator) which validates that the schema passes without\nspecification compliance issues. In case of issues with the schema itself, the validator will raise the appropriate\nerror.\n\n## Known Issues\n\n* We are using [prance](https://github.com/jfinkhaeuser/prance) as a schema resolver, and it has some issues with the\n  resolution of (very) complex OpenAPI 2.0 schemas. If you encounter\n  issues, [please document them here](https://github.com/snok/drf-openapi-tester/issues/205).\n\n## Contributing\n\nContributions are welcome. Please see the [contributing guide](CONTRIBUTING.md)\n',
    'author': 'Sondre Lillebø Gundersen',
    'author_email': 'sondrelg@live.no',
    'maintainer': "Na'aman Hirschfeld",
    'maintainer_email': 'nhirschfeld@gmail.com',
    'url': 'https://github.com/snok/drf-openapi-tester',
    'packages': packages,
    'package_data': package_data,
    'install_requires': install_requires,
    'python_requires': '>=3.6.1,<4.0.0',
}


setup(**setup_kwargs)
