"""Handles the creation of saved_queries."""

import click
from spyctl.commands.apply import handle_apply_saved_query

import spyctl.config.configs as cfg
import spyctl.spyctl_lib as lib
from spyctl import cli
from spyctl.api.saved_queries import get_saved_query
from spyctl.resources import saved_queries


@click.command("saved-query", cls=lib.CustomCommand, epilog=lib.SUB_EPILOG)
@click.help_option("-h", "--help", hidden=True)
@click.option(
    "-o",
    "--output",
    default=lib.OUTPUT_DEFAULT,
    type=click.Choice(lib.OUTPUT_CHOICES, case_sensitive=False),
)
@click.option(
    "-a",
    "--apply",
    is_flag=True,
    help="Apply the saved query during creation.",
)
@click.option(
    "-n",
    "--name",
    help="The name of the saved query.",
)
@click.option(
    "-q",
    "--query",
    help="The query to be saved.",
)
@click.option(
    "-d",
    "--description",
    help="A description of the saved query.",
)
@click.option(
    "-s",
    "--schema",
    help="The schema of the saved query.",
)
@click.option(
    "-y",
    "--yes",
    is_flag=True,
    help="Automatically answer yes to all prompts.",
)
def create_saved_query(output, **kwargs):
    """Create a saved query."""
    yes_option = kwargs.pop("yes")
    if yes_option:
        cli.set_yes_option()
    handle_create_saved_query(output, **kwargs)


def handle_create_saved_query(output, **kwargs):
    """Handle the creation of a saved query."""
    should_apply = kwargs.pop("apply")
    if output == lib.OUTPUT_DEFAULT:
        output = lib.OUTPUT_YAML
    data = {
        "name": kwargs.pop("name", ""),
        "query": kwargs.pop("query", ""),
        "schema": kwargs.pop("schema", ""),
        "description": kwargs.pop("description", ""),
    }
    saved_query_resrc = saved_queries.data_to_yaml(data)
    if should_apply:
        ctx = cfg.get_current_context()
        uid = handle_apply_saved_query(saved_query_resrc)
        sq = get_saved_query(*ctx.get_api_data(), uid)
        model = saved_queries.data_to_yaml(sq)
    else:
        model = saved_query_resrc
    cli.show(model, output)
