import importlib
import json
import logging
import sys
from importlib.util import module_from_spec, spec_from_file_location
import datetime
from pathlib import Path
from shutil import copytree, rmtree
from tempfile import TemporaryDirectory
from typing import Union

import lxml.etree as ET

from pyfmu.builder import PyfmuProject
from pyfmu.resources import Resources
from pyfmu.types import AnyPath
from pyfmu.fmi2.types import Fmi2SlaveLike


# from pyfmu.builder.utils import DisplayablePath

logger = logging.getLogger(__name__)


class PyfmuArchive:
    """Object representation of exported Python FMU.
    """

    def __init__(
        self,
        root: AnyPath,
        resources_dir: AnyPath,
        slave_configuration_path: AnyPath,
        binaries_dir: AnyPath,
        wrapper_win64: AnyPath,
        wrapper_linux64: AnyPath,
        slave_script_path: AnyPath,
        model_description: str,
        model_description_path: AnyPath,
        slave_script: AnyPath,
        slave_class: AnyPath,
    ):
        """Creates an object representation of the exported Python FMU.

        Arguments:
            model_description {str} -- The model description of the exported FMU.
        """
        self.model_description = model_description

        self.slave_script = slave_script
        self.slave_class = slave_class
        self.slave_configuration = None

        # paths
        self.root = Path(root)
        self.resources_dir = Path(resources_dir)
        self.slave_configuration_path = Path(slave_configuration_path)
        self.slave_script_path = Path(slave_script_path)
        self.model_description_path = Path(model_description_path)
        self.binaries_dir = Path(binaries_dir)
        self.wrapper_win64 = Path(wrapper_win64)
        self.wrapper_linux64 = Path(wrapper_linux64)


def extract_model_description(slave: Fmi2SlaveLike) -> bytes:
    """Extract model description from an instance of a FMI2 slave.

    Scalar variables are generated by iterating over the slaves variables attribute.
    For variables which must define a start value, such as inputs, exact outputs, this is
    determined by accessing the attributes of the slave.
    """

    # 2.2.1 p.29) Structure

    data_time_obj = datetime.datetime.now()
    date_str_xsd = datetime.datetime.strftime(data_time_obj, "%Y-%m-%dT%H:%M:%SZ")

    fmd = ET.Element("fmiModelDescription")
    fmd.set("fmiVersion", "2.0")
    fmd.set("modelName", slave.model_name)
    fmd.set("guid", slave.guid)
    fmd.set("author", slave.author)
    fmd.set("generationDateAndTime", date_str_xsd)
    fmd.set("variableNamingConvention", "flat")
    fmd.set("generationTool", "pyfmu")

    #
    cs = ET.SubElement(fmd, "CoSimulation")
    cs.set("modelIdentifier", "pyfmu")
    cs.set("needsExecutionTool", "true")
    cs.set("canNotUseMemoryManagementFunctions", "false")
    cs.set("canHandleVariableCommunicationStepSize", "true")

    # 2.2.4 p.42) Log categories:
    cs = ET.SubElement(fmd, "LogCategories")
    for ac in slave.log_categories:
        c = ET.SubElement(cs, "Category")
        c.set("name", ac)

    # 2.2.7 p.47) ModelVariables
    mvs = ET.SubElement(fmd, "ModelVariables")

    variable_index = 0

    type_to_fmitype = {
        "real": "Real",
        "integer": "Integer",
        "boolean": "Boolean",
        "string": "String",
    }

    for var in slave.variables:
        var.variability
        value_reference = str(var.value_reference)

        idx_comment = ET.Comment(f'Index of variable = "{variable_index + 1}"')
        mvs.append(idx_comment)
        sv = ET.SubElement(mvs, "ScalarVariable")
        sv.set("name", var.name)
        sv.set("valueReference", value_reference)
        sv.set("variability", var.variability)
        sv.set("causality", var.causality)

        if var.description:
            sv.set("description", var.description)

        if var.initial:
            i = var.initial
            sv.set("initial", i)

        val = ET.SubElement(sv, type_to_fmitype[var.data_type])

        # 2.2.7. p.48) start values
        if var.initial in {"exact", "approx"} or var.causality == "input":
            start = str(
                getattr(slave, var.name)
            ).lower()  # lower maps from python uppercase True to fmi2 true.
            val.set("start", start)

        variable_index += 1
    ms = ET.SubElement(fmd, "ModelStructure")

    # 2.2.8) For each output we must declare 'Outputs' and 'InitialUnknowns'
    outputs = [
        (idx + 1, o) for idx, o in enumerate(slave.variables) if o.causality == "output"
    ]

    if outputs:
        os = ET.SubElement(ms, "Outputs")
        for idx, o in outputs:
            ET.SubElement(os, "Unknown", {"index": str(idx), "dependencies": ""})

        os = ET.SubElement(ms, "InitialUnknowns")
        for idx, o in outputs:
            ET.SubElement(os, "Unknown", {"index": str(idx), "dependencies": ""})

    try:
        # FMI requires encoding to be encoded as UTF-8 and contain a header:
        #
        # See 2.2 p.28
        md: bytes = ET.tostring(
            fmd, pretty_print=True, encoding="utf-8", xml_declaration=True
        )

    except Exception as e:
        raise RuntimeError(
            f"Failed to parse model description. Write resulted in error: {e}"
        ) from e

    return md


def export_project(
    project_or_path: Union[PyfmuProject, AnyPath],
    output_path: AnyPath,
    compress: bool,
    overwrite=True,
) -> PyfmuArchive:

    if compress:
        raise NotImplementedError()

    output_path = Path(output_path)
    assert isinstance(output_path, Path)

    if overwrite and output_path.is_dir():
        logger.debug(f"Erasing existing directory {output_path}")
        rmtree(path=output_path)

    if not isinstance(project_or_path, PyfmuProject):
        project = PyfmuProject.from_existing(project_or_path)
    else:
        project: PyfmuProject = project_or_path

    logger.debug(
        "Creating temporary directory for archive used to store files until the archive is finalized"
    )
    with TemporaryDirectory() as tmpdir:

        # create directories
        tmpdir = Path(tmpdir)

        # copy resources
        archive_resources_path = tmpdir / "resources"
        logger.debug(
            f"Copying projects resource directory {project.resources_dir} to archive resources {archive_resources_path}"
        )
        copytree(
            src=project.resources_dir, dst=archive_resources_path,
        )

        # slave configuration
        config_path = tmpdir / "resources" / "slave_configuration.json"
        logger.debug(f"Writing slave configuration to {config_path}")
        with open(config_path, "w") as config:
            json.dump(
                obj={
                    "slave_class": project.slave_class,
                    "slave_script": project.slave_script,
                },
                fp=config,
            )

        # copy-binaries
        binaries_dir = Resources.get().binaries_dir
        archive_binaries_dir = tmpdir / "binaries"
        logger.debug(
            f"copying binaries form {binaries_dir} to archive binaries {archive_binaries_dir}"
        )
        copytree(src=binaries_dir, dst=archive_binaries_dir)

        # Instantiate slave
        """
        Extract model description by creating an instance of the main class
        https://docs.python.org/3/library/importlib.html?highlight=import_module#importing-a-source-file-directly
        """
        # try:
        module = project.slave_script_path.stem
        logger.debug(
            f"Importing module {module} defined by {project.slave_script} which defines slave class {project.slave_class}"
        )
        sys.path.append(project.slave_script_path.parent.__fspath__())
        spec = spec_from_file_location(module, project.slave_script_path)
        module = module_from_spec(spec)
        spec.loader.exec_module(module)  # TODO consider supressing if false positive.
        sys.path.pop()  # TODO may mess up in case of exception

        logger.debug("Module loaded, creating instance of slave")
        slave = getattr(module, project.slave_class)()

        # except Exception as e:
        #     raise RuntimeError(
        #         "Extracting model description from slave failed, an exception was raised during loading and instantiation of the slave"
        #     ) from e

        # extract model description
        archive_md_path = tmpdir / "modelDescription.xml"
        logger.debug(
            f"Slave instantiated, extracting model description and writing to {archive_md_path}"
        )
        model_description = extract_model_description(slave)
        with open(archive_md_path, "wb") as md:
            md.write(model_description)

        logger.debug(
            f"Copying temporary archive {tmpdir} to output path {output_path} "
        )
        copytree(tmpdir, output_path)

        return PyfmuArchive(
            root=output_path,
            resources_dir=output_path / "resources",
            slave_configuration_path=output_path / "resources" / "configuration",
            binaries_dir=output_path / "binaries",
            wrapper_linux64=output_path / "binaries" / "linux64",
            wrapper_win64=output_path / "binaries" / "win64",
            slave_script_path=output_path / "resources" / project.slave_script,
            model_description=model_description.decode("utf-8"),
            model_description_path=output_path / "modelDescription.xml",
            slave_script=project.slave_script,
            slave_class=project.slave_class,
        )
