"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.VirtualGatewayListener = void 0;
const cdk = require("@aws-cdk/core");
const utils_1 = require("./private/utils");
const shared_interfaces_1 = require("./shared-interfaces");
/**
 * (experimental) Represents the properties needed to define listeners for a VirtualGateway.
 *
 * @experimental
 */
class VirtualGatewayListener {
    /**
     * (experimental) Returns an HTTP Listener for a VirtualGateway.
     *
     * @experimental
     */
    static httpGatewayListener(props = {}) {
        return new HttpGatewayListener(props);
    }
    /**
     * (experimental) Returns an HTTP2 Listener for a VirtualGateway.
     *
     * @experimental
     */
    static http2GatewayListener(props = {}) {
        return new Http2GatewayListener(props);
    }
    /**
     * (experimental) Returns a GRPC Listener for a VirtualGateway.
     *
     * @experimental
     */
    static grpcGatewayListener(props = {}) {
        return new GrpcGatewayListener(props);
    }
    /**
     * @experimental
     */
    renderHealthCheck(hc) {
        if (hc.protocol === shared_interfaces_1.Protocol.TCP) {
            throw new Error('TCP health checks are not permitted for gateway listeners');
        }
        if (hc.protocol === shared_interfaces_1.Protocol.GRPC && hc.path) {
            throw new Error('The path property cannot be set with Protocol.GRPC');
        }
        const protocol = hc.protocol ? hc.protocol : this.protocol;
        const healthCheck = {
            healthyThreshold: hc.healthyThreshold || 2,
            intervalMillis: (hc.interval || cdk.Duration.seconds(5)).toMilliseconds(),
            path: hc.path || ((protocol === shared_interfaces_1.Protocol.HTTP || protocol === shared_interfaces_1.Protocol.HTTP2) ? '/' : undefined),
            port: hc.port || this.port,
            protocol: hc.protocol || this.protocol,
            timeoutMillis: (hc.timeout || cdk.Duration.seconds(2)).toMilliseconds(),
            unhealthyThreshold: hc.unhealthyThreshold || 2,
        };
        utils_1.validateHealthChecks(healthCheck);
        return healthCheck;
    }
}
exports.VirtualGatewayListener = VirtualGatewayListener;
/**
 * Represents the properties needed to define an HTTP Listener for a VirtualGateway
 */
class HttpGatewayListener extends VirtualGatewayListener {
    constructor(props = {}) {
        super();
        /**
         * Protocol the listener implements
         */
        this.protocol = shared_interfaces_1.Protocol.HTTP;
        this.port = props.port ? props.port : 8080;
        this.healthCheck = props.healthCheck;
    }
    /**
     * Called when the GatewayListener type is initialized. Can be used to enforce
     * mutual exclusivity
     */
    bind(_scope) {
        return {
            listener: {
                portMapping: {
                    port: this.port,
                    protocol: this.protocol,
                },
                healthCheck: this.healthCheck ? this.renderHealthCheck(this.healthCheck) : undefined,
            },
        };
    }
}
/**
* Represents the properties needed to define an HTTP2 Listener for a VirtualGateway
*/
class Http2GatewayListener extends HttpGatewayListener {
    constructor(props = {}) {
        super(props);
        this.protocol = shared_interfaces_1.Protocol.HTTP2;
    }
}
/**
 * Represents the properties needed to define a GRPC Listener for Virtual Gateway
 */
class GrpcGatewayListener extends VirtualGatewayListener {
    constructor(props = {}) {
        super();
        /**
         * Protocol the listener implements
         */
        this.protocol = shared_interfaces_1.Protocol.GRPC;
        this.port = props.port ? props.port : 8080;
        this.healthCheck = props.healthCheck;
    }
    /**
     * Called when the GatewayListener type is initialized. Can be used to enforce
     * mutual exclusivity
     */
    bind(_scope) {
        return {
            listener: {
                portMapping: {
                    port: this.port,
                    protocol: shared_interfaces_1.Protocol.GRPC,
                },
                healthCheck: this.healthCheck ? this.renderHealthCheck(this.healthCheck) : undefined,
            },
        };
    }
}
//# sourceMappingURL=data:application/json;base64,eyJ2ZXJzaW9uIjozLCJmaWxlIjoidmlydHVhbC1nYXRld2F5LWxpc3RlbmVyLmpzIiwic291cmNlUm9vdCI6IiIsInNvdXJjZXMiOlsidmlydHVhbC1nYXRld2F5LWxpc3RlbmVyLnRzIl0sIm5hbWVzIjpbXSwibWFwcGluZ3MiOiI7OztBQUFBLHFDQUFxQztBQUVyQywyQ0FBdUQ7QUFDdkQsMkRBQTREOzs7Ozs7QUFxRDVELE1BQXNCLHNCQUFzQjs7Ozs7O0lBSW5DLE1BQU0sQ0FBQyxtQkFBbUIsQ0FBQyxRQUFrQyxFQUFFO1FBQ3BFLE9BQU8sSUFBSSxtQkFBbUIsQ0FBQyxLQUFLLENBQUMsQ0FBQztJQUN4QyxDQUFDOzs7Ozs7SUFLTSxNQUFNLENBQUMsb0JBQW9CLENBQUMsUUFBa0MsRUFBRTtRQUNyRSxPQUFPLElBQUksb0JBQW9CLENBQUMsS0FBSyxDQUFDLENBQUM7SUFDekMsQ0FBQzs7Ozs7O0lBS00sTUFBTSxDQUFDLG1CQUFtQixDQUFDLFFBQWtDLEVBQUU7UUFDcEUsT0FBTyxJQUFJLG1CQUFtQixDQUFDLEtBQUssQ0FBQyxDQUFDO0lBQ3hDLENBQUM7Ozs7SUF1QlMsaUJBQWlCLENBQUMsRUFBZTtRQUN6QyxJQUFJLEVBQUUsQ0FBQyxRQUFRLEtBQUssNEJBQVEsQ0FBQyxHQUFHLEVBQUU7WUFDaEMsTUFBTSxJQUFJLEtBQUssQ0FBQywyREFBMkQsQ0FBQyxDQUFDO1NBQzlFO1FBRUQsSUFBSSxFQUFFLENBQUMsUUFBUSxLQUFLLDRCQUFRLENBQUMsSUFBSSxJQUFJLEVBQUUsQ0FBQyxJQUFJLEVBQUU7WUFDNUMsTUFBTSxJQUFJLEtBQUssQ0FBQyxvREFBb0QsQ0FBQyxDQUFDO1NBQ3ZFO1FBRUQsTUFBTSxRQUFRLEdBQUcsRUFBRSxDQUFDLFFBQVEsQ0FBQSxDQUFDLENBQUMsRUFBRSxDQUFDLFFBQVEsQ0FBQyxDQUFDLENBQUMsSUFBSSxDQUFDLFFBQVEsQ0FBQztRQUUxRCxNQUFNLFdBQVcsR0FBOEQ7WUFDN0UsZ0JBQWdCLEVBQUUsRUFBRSxDQUFDLGdCQUFnQixJQUFJLENBQUM7WUFDMUMsY0FBYyxFQUFFLENBQUMsRUFBRSxDQUFDLFFBQVEsSUFBSSxHQUFHLENBQUMsUUFBUSxDQUFDLE9BQU8sQ0FBQyxDQUFDLENBQUMsQ0FBQyxDQUFDLGNBQWMsRUFBRTtZQUN6RSxJQUFJLEVBQUUsRUFBRSxDQUFDLElBQUksSUFBSSxDQUFDLENBQUMsUUFBUSxLQUFLLDRCQUFRLENBQUMsSUFBSSxJQUFJLFFBQVEsS0FBSyw0QkFBUSxDQUFDLEtBQUssQ0FBQyxDQUFDLENBQUMsQ0FBQyxHQUFHLENBQUMsQ0FBQyxDQUFDLFNBQVMsQ0FBQztZQUNoRyxJQUFJLEVBQUUsRUFBRSxDQUFDLElBQUksSUFBSSxJQUFJLENBQUMsSUFBSTtZQUMxQixRQUFRLEVBQUUsRUFBRSxDQUFDLFFBQVEsSUFBSSxJQUFJLENBQUMsUUFBUTtZQUN0QyxhQUFhLEVBQUUsQ0FBQyxFQUFFLENBQUMsT0FBTyxJQUFJLEdBQUcsQ0FBQyxRQUFRLENBQUMsT0FBTyxDQUFDLENBQUMsQ0FBQyxDQUFDLENBQUMsY0FBYyxFQUFFO1lBQ3ZFLGtCQUFrQixFQUFFLEVBQUUsQ0FBQyxrQkFBa0IsSUFBSSxDQUFDO1NBQy9DLENBQUM7UUFFRiw0QkFBb0IsQ0FBQyxXQUFXLENBQUMsQ0FBQztRQUVsQyxPQUFPLFdBQVcsQ0FBQztJQUNyQixDQUFDO0NBQ0Y7QUFwRUQsd0RBb0VDO0FBRUQ7O0dBRUc7QUFDSCxNQUFNLG1CQUFvQixTQUFRLHNCQUFzQjtJQW9CdEQsWUFBWSxRQUFrQyxFQUFFO1FBQzlDLEtBQUssRUFBRSxDQUFDO1FBTlY7O1dBRUc7UUFDTyxhQUFRLEdBQWEsNEJBQVEsQ0FBQyxJQUFJLENBQUM7UUFJM0MsSUFBSSxDQUFDLElBQUksR0FBRyxLQUFLLENBQUMsSUFBSSxDQUFDLENBQUMsQ0FBQyxLQUFLLENBQUMsSUFBSSxDQUFDLENBQUMsQ0FBQyxJQUFJLENBQUM7UUFDM0MsSUFBSSxDQUFDLFdBQVcsR0FBRyxLQUFLLENBQUMsV0FBVyxDQUFDO0lBQ3ZDLENBQUM7SUFFRDs7O09BR0c7SUFDSSxJQUFJLENBQUMsTUFBcUI7UUFDL0IsT0FBTztZQUNMLFFBQVEsRUFBRTtnQkFDUixXQUFXLEVBQUU7b0JBQ1gsSUFBSSxFQUFFLElBQUksQ0FBQyxJQUFJO29CQUNmLFFBQVEsRUFBRSxJQUFJLENBQUMsUUFBUTtpQkFDeEI7Z0JBQ0QsV0FBVyxFQUFFLElBQUksQ0FBQyxXQUFXLENBQUMsQ0FBQyxDQUFDLElBQUksQ0FBQyxpQkFBaUIsQ0FBQyxJQUFJLENBQUMsV0FBVyxDQUFDLENBQUEsQ0FBQyxDQUFDLFNBQVM7YUFDcEY7U0FDRixDQUFDO0lBQ0osQ0FBQztDQUNGO0FBRUQ7O0VBRUU7QUFDRixNQUFNLG9CQUFxQixTQUFRLG1CQUFtQjtJQUNwRCxZQUFZLFFBQWtDLEVBQUU7UUFDOUMsS0FBSyxDQUFDLEtBQUssQ0FBQyxDQUFDO1FBQ2IsSUFBSSxDQUFDLFFBQVEsR0FBRyw0QkFBUSxDQUFDLEtBQUssQ0FBQztJQUNqQyxDQUFDO0NBQ0Y7QUFFRDs7R0FFRztBQUNILE1BQU0sbUJBQW9CLFNBQVEsc0JBQXNCO0lBb0J0RCxZQUFZLFFBQWtDLEVBQUU7UUFDOUMsS0FBSyxFQUFFLENBQUM7UUFOVjs7V0FFRztRQUNPLGFBQVEsR0FBYSw0QkFBUSxDQUFDLElBQUksQ0FBQztRQUkzQyxJQUFJLENBQUMsSUFBSSxHQUFHLEtBQUssQ0FBQyxJQUFJLENBQUMsQ0FBQyxDQUFDLEtBQUssQ0FBQyxJQUFJLENBQUMsQ0FBQyxDQUFDLElBQUksQ0FBQztRQUMzQyxJQUFJLENBQUMsV0FBVyxHQUFHLEtBQUssQ0FBQyxXQUFXLENBQUM7SUFDdkMsQ0FBQztJQUVEOzs7T0FHRztJQUNJLElBQUksQ0FBQyxNQUFxQjtRQUMvQixPQUFPO1lBQ0wsUUFBUSxFQUFFO2dCQUNSLFdBQVcsRUFBRTtvQkFDWCxJQUFJLEVBQUUsSUFBSSxDQUFDLElBQUk7b0JBQ2YsUUFBUSxFQUFFLDRCQUFRLENBQUMsSUFBSTtpQkFDeEI7Z0JBQ0QsV0FBVyxFQUFFLElBQUksQ0FBQyxXQUFXLENBQUEsQ0FBQyxDQUFDLElBQUksQ0FBQyxpQkFBaUIsQ0FBQyxJQUFJLENBQUMsV0FBVyxDQUFDLENBQUEsQ0FBQyxDQUFDLFNBQVM7YUFDbkY7U0FDRixDQUFDO0lBQ0osQ0FBQztDQUNGIiwic291cmNlc0NvbnRlbnQiOlsiaW1wb3J0ICogYXMgY2RrIGZyb20gJ0Bhd3MtY2RrL2NvcmUnO1xuaW1wb3J0IHsgQ2ZuVmlydHVhbEdhdGV3YXkgfSBmcm9tICcuL2FwcG1lc2guZ2VuZXJhdGVkJztcbmltcG9ydCB7IHZhbGlkYXRlSGVhbHRoQ2hlY2tzIH0gZnJvbSAnLi9wcml2YXRlL3V0aWxzJztcbmltcG9ydCB7IEhlYWx0aENoZWNrLCBQcm90b2NvbCB9IGZyb20gJy4vc2hhcmVkLWludGVyZmFjZXMnO1xuXG4gICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgIFxuZXhwb3J0IGludGVyZmFjZSBIdHRwR2F0ZXdheUxpc3RlbmVyUHJvcHMge1xuICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgIFxuICByZWFkb25seSBwb3J0PzogbnVtYmVyXG5cbiAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgXG4gIHJlYWRvbmx5IGhlYWx0aENoZWNrPzogSGVhbHRoQ2hlY2s7XG59XG5cbiAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgXG5leHBvcnQgaW50ZXJmYWNlIEdycGNHYXRld2F5TGlzdGVuZXJQcm9wcyB7XG4gICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgXG4gIHJlYWRvbmx5IHBvcnQ/OiBudW1iZXJcblxuICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICBcbiAgcmVhZG9ubHkgaGVhbHRoQ2hlY2s/OiBIZWFsdGhDaGVjaztcbn1cblxuICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgXG5leHBvcnQgaW50ZXJmYWNlIFZpcnR1YWxHYXRld2F5TGlzdGVuZXJDb25maWcge1xuICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgXG4gIHJlYWRvbmx5IGxpc3RlbmVyOiBDZm5WaXJ0dWFsR2F0ZXdheS5WaXJ0dWFsR2F0ZXdheUxpc3RlbmVyUHJvcGVydHksXG59XG5cbiAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgIFxuZXhwb3J0IGFic3RyYWN0IGNsYXNzIFZpcnR1YWxHYXRld2F5TGlzdGVuZXIge1xuICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICBcbiAgcHVibGljIHN0YXRpYyBodHRwR2F0ZXdheUxpc3RlbmVyKHByb3BzOiBIdHRwR2F0ZXdheUxpc3RlbmVyUHJvcHMgPSB7fSk6IFZpcnR1YWxHYXRld2F5TGlzdGVuZXIge1xuICAgIHJldHVybiBuZXcgSHR0cEdhdGV3YXlMaXN0ZW5lcihwcm9wcyk7XG4gIH1cblxuICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgXG4gIHB1YmxpYyBzdGF0aWMgaHR0cDJHYXRld2F5TGlzdGVuZXIocHJvcHM6IEh0dHBHYXRld2F5TGlzdGVuZXJQcm9wcyA9IHt9KTogVmlydHVhbEdhdGV3YXlMaXN0ZW5lciB7XG4gICAgcmV0dXJuIG5ldyBIdHRwMkdhdGV3YXlMaXN0ZW5lcihwcm9wcyk7XG4gIH1cblxuICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgIFxuICBwdWJsaWMgc3RhdGljIGdycGNHYXRld2F5TGlzdGVuZXIocHJvcHM6IEdycGNHYXRld2F5TGlzdGVuZXJQcm9wcyA9IHt9KTogVmlydHVhbEdhdGV3YXlMaXN0ZW5lciB7XG4gICAgcmV0dXJuIG5ldyBHcnBjR2F0ZXdheUxpc3RlbmVyKHByb3BzKTtcbiAgfVxuXG4gICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgXG4gIHByb3RlY3RlZCBhYnN0cmFjdCBwcm90b2NvbDogUHJvdG9jb2w7XG5cbiAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgXG4gIHByb3RlY3RlZCBhYnN0cmFjdCBwb3J0OiBudW1iZXI7XG5cbiAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgXG4gIHByb3RlY3RlZCBhYnN0cmFjdCBoZWFsdGhDaGVjaz86IEhlYWx0aENoZWNrO1xuXG4gICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgIFxuICBwdWJsaWMgYWJzdHJhY3QgYmluZChzY29wZTogY2RrLkNvbnN0cnVjdCk6IFZpcnR1YWxHYXRld2F5TGlzdGVuZXJDb25maWc7XG5cbiAgcHJvdGVjdGVkIHJlbmRlckhlYWx0aENoZWNrKGhjOiBIZWFsdGhDaGVjayk6IENmblZpcnR1YWxHYXRld2F5LlZpcnR1YWxHYXRld2F5SGVhbHRoQ2hlY2tQb2xpY3lQcm9wZXJ0eSB8IHVuZGVmaW5lZCB7XG4gICAgaWYgKGhjLnByb3RvY29sID09PSBQcm90b2NvbC5UQ1ApIHtcbiAgICAgIHRocm93IG5ldyBFcnJvcignVENQIGhlYWx0aCBjaGVja3MgYXJlIG5vdCBwZXJtaXR0ZWQgZm9yIGdhdGV3YXkgbGlzdGVuZXJzJyk7XG4gICAgfVxuXG4gICAgaWYgKGhjLnByb3RvY29sID09PSBQcm90b2NvbC5HUlBDICYmIGhjLnBhdGgpIHtcbiAgICAgIHRocm93IG5ldyBFcnJvcignVGhlIHBhdGggcHJvcGVydHkgY2Fubm90IGJlIHNldCB3aXRoIFByb3RvY29sLkdSUEMnKTtcbiAgICB9XG5cbiAgICBjb25zdCBwcm90b2NvbCA9IGhjLnByb3RvY29sPyBoYy5wcm90b2NvbCA6IHRoaXMucHJvdG9jb2w7XG5cbiAgICBjb25zdCBoZWFsdGhDaGVjazogQ2ZuVmlydHVhbEdhdGV3YXkuVmlydHVhbEdhdGV3YXlIZWFsdGhDaGVja1BvbGljeVByb3BlcnR5ID0ge1xuICAgICAgaGVhbHRoeVRocmVzaG9sZDogaGMuaGVhbHRoeVRocmVzaG9sZCB8fCAyLFxuICAgICAgaW50ZXJ2YWxNaWxsaXM6IChoYy5pbnRlcnZhbCB8fCBjZGsuRHVyYXRpb24uc2Vjb25kcyg1KSkudG9NaWxsaXNlY29uZHMoKSwgLy8gbWluXG4gICAgICBwYXRoOiBoYy5wYXRoIHx8ICgocHJvdG9jb2wgPT09IFByb3RvY29sLkhUVFAgfHwgcHJvdG9jb2wgPT09IFByb3RvY29sLkhUVFAyKSA/ICcvJyA6IHVuZGVmaW5lZCksXG4gICAgICBwb3J0OiBoYy5wb3J0IHx8IHRoaXMucG9ydCxcbiAgICAgIHByb3RvY29sOiBoYy5wcm90b2NvbCB8fCB0aGlzLnByb3RvY29sLFxuICAgICAgdGltZW91dE1pbGxpczogKGhjLnRpbWVvdXQgfHwgY2RrLkR1cmF0aW9uLnNlY29uZHMoMikpLnRvTWlsbGlzZWNvbmRzKCksXG4gICAgICB1bmhlYWx0aHlUaHJlc2hvbGQ6IGhjLnVuaGVhbHRoeVRocmVzaG9sZCB8fCAyLFxuICAgIH07XG5cbiAgICB2YWxpZGF0ZUhlYWx0aENoZWNrcyhoZWFsdGhDaGVjayk7XG5cbiAgICByZXR1cm4gaGVhbHRoQ2hlY2s7XG4gIH1cbn1cblxuLyoqXG4gKiBSZXByZXNlbnRzIHRoZSBwcm9wZXJ0aWVzIG5lZWRlZCB0byBkZWZpbmUgYW4gSFRUUCBMaXN0ZW5lciBmb3IgYSBWaXJ0dWFsR2F0ZXdheVxuICovXG5jbGFzcyBIdHRwR2F0ZXdheUxpc3RlbmVyIGV4dGVuZHMgVmlydHVhbEdhdGV3YXlMaXN0ZW5lciB7XG4gIC8qKlxuICAgKiBQb3J0IHRvIGxpc3RlbiBmb3IgY29ubmVjdGlvbnMgb25cbiAgICpcbiAgICogQGRlZmF1bHQgLSA4MDgwXG4gICAqL1xuICByZWFkb25seSBwb3J0OiBudW1iZXI7XG5cbiAgLyoqXG4gICAqIEhlYWx0aCBjaGVja2luZyBzdHJhdGVneSB1cHN0cmVhbSBub2RlcyBzaG91bGQgdXNlIHdoZW4gY29tbXVuaWNhdGluZyB3aXRoIHRoZSBsaXN0ZW5lclxuICAgKlxuICAgKiBAZGVmYXVsdCAtIG5vIGhlYWx0aGNoZWNrXG4gICAqL1xuICByZWFkb25seSBoZWFsdGhDaGVjaz86IEhlYWx0aENoZWNrO1xuXG4gIC8qKlxuICAgKiBQcm90b2NvbCB0aGUgbGlzdGVuZXIgaW1wbGVtZW50c1xuICAgKi9cbiAgcHJvdGVjdGVkIHByb3RvY29sOiBQcm90b2NvbCA9IFByb3RvY29sLkhUVFA7XG5cbiAgY29uc3RydWN0b3IocHJvcHM6IEh0dHBHYXRld2F5TGlzdGVuZXJQcm9wcyA9IHt9KSB7XG4gICAgc3VwZXIoKTtcbiAgICB0aGlzLnBvcnQgPSBwcm9wcy5wb3J0ID8gcHJvcHMucG9ydCA6IDgwODA7XG4gICAgdGhpcy5oZWFsdGhDaGVjayA9IHByb3BzLmhlYWx0aENoZWNrO1xuICB9XG5cbiAgLyoqXG4gICAqIENhbGxlZCB3aGVuIHRoZSBHYXRld2F5TGlzdGVuZXIgdHlwZSBpcyBpbml0aWFsaXplZC4gQ2FuIGJlIHVzZWQgdG8gZW5mb3JjZVxuICAgKiBtdXR1YWwgZXhjbHVzaXZpdHlcbiAgICovXG4gIHB1YmxpYyBiaW5kKF9zY29wZTogY2RrLkNvbnN0cnVjdCk6IFZpcnR1YWxHYXRld2F5TGlzdGVuZXJDb25maWcge1xuICAgIHJldHVybiB7XG4gICAgICBsaXN0ZW5lcjoge1xuICAgICAgICBwb3J0TWFwcGluZzoge1xuICAgICAgICAgIHBvcnQ6IHRoaXMucG9ydCxcbiAgICAgICAgICBwcm90b2NvbDogdGhpcy5wcm90b2NvbCxcbiAgICAgICAgfSxcbiAgICAgICAgaGVhbHRoQ2hlY2s6IHRoaXMuaGVhbHRoQ2hlY2sgPyB0aGlzLnJlbmRlckhlYWx0aENoZWNrKHRoaXMuaGVhbHRoQ2hlY2spOiB1bmRlZmluZWQsXG4gICAgICB9LFxuICAgIH07XG4gIH1cbn1cblxuLyoqXG4qIFJlcHJlc2VudHMgdGhlIHByb3BlcnRpZXMgbmVlZGVkIHRvIGRlZmluZSBhbiBIVFRQMiBMaXN0ZW5lciBmb3IgYSBWaXJ0dWFsR2F0ZXdheVxuKi9cbmNsYXNzIEh0dHAyR2F0ZXdheUxpc3RlbmVyIGV4dGVuZHMgSHR0cEdhdGV3YXlMaXN0ZW5lciB7XG4gIGNvbnN0cnVjdG9yKHByb3BzOiBIdHRwR2F0ZXdheUxpc3RlbmVyUHJvcHMgPSB7fSkge1xuICAgIHN1cGVyKHByb3BzKTtcbiAgICB0aGlzLnByb3RvY29sID0gUHJvdG9jb2wuSFRUUDI7XG4gIH1cbn1cblxuLyoqXG4gKiBSZXByZXNlbnRzIHRoZSBwcm9wZXJ0aWVzIG5lZWRlZCB0byBkZWZpbmUgYSBHUlBDIExpc3RlbmVyIGZvciBWaXJ0dWFsIEdhdGV3YXlcbiAqL1xuY2xhc3MgR3JwY0dhdGV3YXlMaXN0ZW5lciBleHRlbmRzIFZpcnR1YWxHYXRld2F5TGlzdGVuZXIge1xuICAvKipcbiAgICogUG9ydCB0byBsaXN0ZW4gZm9yIGNvbm5lY3Rpb25zIG9uXG4gICAqXG4gICAqIEBkZWZhdWx0IC0gODA4MFxuICAgKi9cbiAgcmVhZG9ubHkgcG9ydDogbnVtYmVyO1xuXG4gIC8qKlxuICAgKiBIZWFsdGggY2hlY2tpbmcgc3RyYXRlZ3kgdXBzdHJlYW0gbm9kZXMgc2hvdWxkIHVzZSB3aGVuIGNvbW11bmljYXRpbmcgd2l0aCB0aGUgbGlzdGVuZXJcbiAgICpcbiAgICogQGRlZmF1bHQgLSBubyBoZWFsdGhjaGVja1xuICAgKi9cbiAgcmVhZG9ubHkgaGVhbHRoQ2hlY2s/OiBIZWFsdGhDaGVjaztcblxuICAvKipcbiAgICogUHJvdG9jb2wgdGhlIGxpc3RlbmVyIGltcGxlbWVudHNcbiAgICovXG4gIHByb3RlY3RlZCBwcm90b2NvbDogUHJvdG9jb2wgPSBQcm90b2NvbC5HUlBDO1xuXG4gIGNvbnN0cnVjdG9yKHByb3BzOiBIdHRwR2F0ZXdheUxpc3RlbmVyUHJvcHMgPSB7fSkge1xuICAgIHN1cGVyKCk7XG4gICAgdGhpcy5wb3J0ID0gcHJvcHMucG9ydCA/IHByb3BzLnBvcnQgOiA4MDgwO1xuICAgIHRoaXMuaGVhbHRoQ2hlY2sgPSBwcm9wcy5oZWFsdGhDaGVjaztcbiAgfVxuXG4gIC8qKlxuICAgKiBDYWxsZWQgd2hlbiB0aGUgR2F0ZXdheUxpc3RlbmVyIHR5cGUgaXMgaW5pdGlhbGl6ZWQuIENhbiBiZSB1c2VkIHRvIGVuZm9yY2VcbiAgICogbXV0dWFsIGV4Y2x1c2l2aXR5XG4gICAqL1xuICBwdWJsaWMgYmluZChfc2NvcGU6IGNkay5Db25zdHJ1Y3QpOiBWaXJ0dWFsR2F0ZXdheUxpc3RlbmVyQ29uZmlnIHtcbiAgICByZXR1cm4ge1xuICAgICAgbGlzdGVuZXI6IHtcbiAgICAgICAgcG9ydE1hcHBpbmc6IHtcbiAgICAgICAgICBwb3J0OiB0aGlzLnBvcnQsXG4gICAgICAgICAgcHJvdG9jb2w6IFByb3RvY29sLkdSUEMsXG4gICAgICAgIH0sXG4gICAgICAgIGhlYWx0aENoZWNrOiB0aGlzLmhlYWx0aENoZWNrPyB0aGlzLnJlbmRlckhlYWx0aENoZWNrKHRoaXMuaGVhbHRoQ2hlY2spOiB1bmRlZmluZWQsXG4gICAgICB9LFxuICAgIH07XG4gIH1cbn0iXX0=