"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.Route = exports.RouteType = void 0;
const cdk = require("@aws-cdk/core");
const appmesh_generated_1 = require("./appmesh.generated");
const virtual_router_1 = require("./virtual-router");
/**
 * (experimental) Type of route.
 *
 * @experimental
 */
var RouteType;
(function (RouteType) {
    RouteType["HTTP"] = "http";
    RouteType["TCP"] = "tcp";
})(RouteType = exports.RouteType || (exports.RouteType = {}));
/**
 * (experimental) Route represents a new or existing route attached to a VirtualRouter and Mesh.
 *
 * @see https://docs.aws.amazon.com/app-mesh/latest/userguide/routes.html
 * @experimental
 */
class Route extends cdk.Resource {
    /**
     * @experimental
     */
    constructor(scope, id, props) {
        super(scope, id, {
            physicalName: props.routeName || cdk.Lazy.stringValue({ produce: () => cdk.Names.uniqueId(this) }),
        });
        this.weightedTargets = new Array();
        this.virtualRouter = props.virtualRouter;
        const routeType = props.routeType !== undefined ? props.routeType :
            props.prefix !== undefined ? RouteType.HTTP :
                RouteType.TCP;
        if (routeType === RouteType.HTTP) {
            this.httpRoute = this.renderHttpRoute(props);
        }
        else {
            this.tcpRoute = this.renderTcpRoute(props);
        }
        const route = new appmesh_generated_1.CfnRoute(this, 'Resource', {
            routeName: this.physicalName,
            meshName: this.virtualRouter.mesh.meshName,
            virtualRouterName: this.virtualRouter.virtualRouterName,
            spec: {
                tcpRoute: this.tcpRoute,
                httpRoute: this.httpRoute,
            },
        });
        this.routeName = this.getResourceNameAttribute(route.attrRouteName);
        this.routeArn = this.getResourceArnAttribute(route.ref, {
            service: 'appmesh',
            resource: `mesh/${props.mesh.meshName}/virtualRouter/${props.virtualRouter.virtualRouterName}/route`,
            resourceName: this.physicalName,
        });
    }
    /**
     * (experimental) Import an existing Route given an ARN.
     *
     * @experimental
     */
    static fromRouteArn(scope, id, routeArn) {
        return new class extends cdk.Resource {
            constructor() {
                super(...arguments);
                this.routeArn = routeArn;
                this.virtualRouter = virtual_router_1.VirtualRouter.fromVirtualRouterArn(this, 'VirtualRouter', routeArn);
                this.routeName = cdk.Fn.select(4, cdk.Fn.split('/', cdk.Stack.of(scope).parseArn(routeArn).resourceName));
            }
        }(scope, id);
    }
    /**
     * (experimental) Import an existing Route given attributes.
     *
     * @experimental
     */
    static fromRouteAttributes(scope, id, attrs) {
        return new class extends cdk.Resource {
            constructor() {
                super(...arguments);
                this.routeName = attrs.routeName;
                this.virtualRouter = attrs.virtualRouter;
                this.routeArn = cdk.Stack.of(this).formatArn({
                    service: 'appmesh',
                    resource: `mesh/${attrs.virtualRouter.mesh.meshName}/virtualRouter/${attrs.virtualRouter.virtualRouterName}/route`,
                    resourceName: this.routeName,
                });
            }
        }(scope, id);
    }
    /**
     * Utility method to add weighted route targets to an existing route
     */
    renderWeightedTargets(props) {
        for (const t of props) {
            this.weightedTargets.push({
                virtualNode: t.virtualNode.virtualNodeName,
                weight: t.weight || 1,
            });
        }
        return this.weightedTargets;
    }
    renderHttpRoute(props) {
        return {
            match: {
                prefix: props.prefix || '/',
            },
            action: {
                weightedTargets: this.renderWeightedTargets(props.routeTargets),
            },
        };
    }
    renderTcpRoute(props) {
        return {
            action: {
                weightedTargets: this.renderWeightedTargets(props.routeTargets),
            },
        };
    }
}
exports.Route = Route;
//# sourceMappingURL=data:application/json;base64,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