"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.Provider = void 0;
const lambda = require("../../../aws-lambda"); // Automatically re-written from '@aws-cdk/aws-lambda'
const core_1 = require("../../../core"); // Automatically re-written from '@aws-cdk/core'
const path = require("path");
const consts = require("./runtime/consts");
const util_1 = require("./util");
const waiter_state_machine_1 = require("./waiter-state-machine");
const RUNTIME_HANDLER_PATH = path.join(__dirname, 'runtime');
const FRAMEWORK_HANDLER_TIMEOUT = core_1.Duration.minutes(15); // keep it simple for now
/**
 * Defines an AWS CloudFormation custom resource provider.
 */
class Provider extends core_1.Construct {
    constructor(scope, id, props) {
        super(scope, id);
        if (!props.isCompleteHandler && (props.queryInterval || props.totalTimeout)) {
            throw new Error('"queryInterval" and "totalTimeout" can only be configured if "isCompleteHandler" is specified. '
                + 'Otherwise, they have no meaning');
        }
        this.onEventHandler = props.onEventHandler;
        this.isCompleteHandler = props.isCompleteHandler;
        const onEventFunction = this.createFunction(consts.FRAMEWORK_ON_EVENT_HANDLER_NAME);
        if (this.isCompleteHandler) {
            const isCompleteFunction = this.createFunction(consts.FRAMEWORK_IS_COMPLETE_HANDLER_NAME);
            const timeoutFunction = this.createFunction(consts.FRAMEWORK_ON_TIMEOUT_HANDLER_NAME);
            const retry = util_1.calculateRetryPolicy(props);
            const waiterStateMachine = new waiter_state_machine_1.WaiterStateMachine(this, 'waiter-state-machine', {
                isCompleteHandler: isCompleteFunction,
                timeoutHandler: timeoutFunction,
                backoffRate: retry.backoffRate,
                interval: retry.interval,
                maxAttempts: retry.maxAttempts,
            });
            // the on-event entrypoint is going to start the execution of the waiter
            onEventFunction.addEnvironment(consts.WAITER_STATE_MACHINE_ARN_ENV, waiterStateMachine.stateMachineArn);
            waiterStateMachine.grantStartExecution(onEventFunction);
        }
        this.entrypoint = onEventFunction;
        this.serviceToken = this.entrypoint.functionArn;
    }
    /**
     * Called by `CustomResource` which uses this provider.
     * @deprecated use `provider.serviceToken` instead
     */
    bind(_) {
        return {
            serviceToken: this.entrypoint.functionArn,
        };
    }
    createFunction(entrypoint) {
        const fn = new lambda.Function(this, `framework-${entrypoint}`, {
            code: lambda.Code.fromAsset(RUNTIME_HANDLER_PATH),
            runtime: lambda.Runtime.NODEJS_10_X,
            handler: `framework.${entrypoint}`,
            timeout: FRAMEWORK_HANDLER_TIMEOUT,
        });
        fn.addEnvironment(consts.USER_ON_EVENT_FUNCTION_ARN_ENV, this.onEventHandler.functionArn);
        this.onEventHandler.grantInvoke(fn);
        if (this.isCompleteHandler) {
            fn.addEnvironment(consts.USER_IS_COMPLETE_FUNCTION_ARN_ENV, this.isCompleteHandler.functionArn);
            this.isCompleteHandler.grantInvoke(fn);
        }
        return fn;
    }
}
exports.Provider = Provider;
//# sourceMappingURL=data:application/json;base64,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