"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.Resource = void 0;
const construct_compat_1 = require("./construct-compat");
const lazy_1 = require("./lazy");
const physical_name_generator_1 = require("./private/physical-name-generator");
const stack_1 = require("./stack");
const token_1 = require("./token");
/**
 * A construct which represents an AWS resource.
 */
class Resource extends construct_compat_1.Construct {
    constructor(scope, id, props = {}) {
        super(scope, id);
        this.stack = stack_1.Stack.of(this);
        let physicalName = props.physicalName;
        if (props.physicalName && physical_name_generator_1.isGeneratedWhenNeededMarker(props.physicalName)) {
            // auto-generate only if cross-env is required
            this._physicalName = undefined;
            this._allowCrossEnvironment = true;
            physicalName = lazy_1.Lazy.stringValue({ produce: () => this._physicalName });
        }
        else if (props.physicalName && !token_1.Token.isUnresolved(props.physicalName)) {
            // concrete value specified by the user
            this._physicalName = props.physicalName;
            this._allowCrossEnvironment = true;
        }
        else {
            // either undefined (deploy-time) or has tokens, which means we can't use for cross-env
            this._physicalName = props.physicalName;
            this._allowCrossEnvironment = false;
        }
        if (physicalName === undefined) {
            physicalName = token_1.Token.asString(undefined);
        }
        this.physicalName = physicalName;
    }
    /**
     * Called when this resource is referenced across environments
     * (account/region) to order to request that a physical name will be generated
     * for this resource during synthesis, so the resource can be referenced
     * through it's absolute name/arn.
     *
     * @internal
     */
    _enableCrossEnvironment() {
        if (!this._allowCrossEnvironment) {
            // error out - a deploy-time name cannot be used across environments
            throw new Error(`Cannot use resource '${this.node.path}' in a cross-environment fashion, ` +
                "the resource's physical name must be explicit set or use `PhysicalName.GENERATE_IF_NEEDED`");
        }
        if (!this._physicalName) {
            this._physicalName = this.generatePhysicalName();
        }
    }
    generatePhysicalName() {
        return physical_name_generator_1.generatePhysicalName(this);
    }
    /**
     * Returns an environment-sensitive token that should be used for the
     * resource's "name" attribute (e.g. `bucket.bucketName`).
     *
     * Normally, this token will resolve to `nameAttr`, but if the resource is
     * referenced across environments, it will be resolved to `this.physicalName`,
     * which will be a concrete name.
     *
     * @param nameAttr The CFN attribute which resolves to the resource's name.
     * Commonly this is the resource's `ref`.
     * @experimental
     */
    getResourceNameAttribute(nameAttr) {
        return lazy_1.Lazy.stringValue({
            produce: (context) => {
                const consumingStack = stack_1.Stack.of(context.scope);
                if (this.stack.environment !== consumingStack.environment) {
                    this._enableCrossEnvironment();
                    return this.physicalName;
                }
                else {
                    return nameAttr;
                }
            },
        });
    }
    /**
     * Returns an environment-sensitive token that should be used for the
     * resource's "ARN" attribute (e.g. `bucket.bucketArn`).
     *
     * Normally, this token will resolve to `arnAttr`, but if the resource is
     * referenced across environments, `arnComponents` will be used to synthesize
     * a concrete ARN with the resource's physical name. Make sure to reference
     * `this.physicalName` in `arnComponents`.
     *
     * @param arnAttr The CFN attribute which resolves to the ARN of the resource.
     * Commonly it will be called "Arn" (e.g. `resource.attrArn`), but sometimes
     * it's the CFN resource's `ref`.
     * @param arnComponents The format of the ARN of this resource. You must
     * reference `this.physicalName` somewhere within the ARN in order for
     * cross-environment references to work.
     *
     * @experimental
     */
    getResourceArnAttribute(arnAttr, arnComponents) {
        return token_1.Token.asString({
            resolve: (context) => {
                const consumingStack = stack_1.Stack.of(context.scope);
                if (this.stack.environment !== consumingStack.environment) {
                    this._enableCrossEnvironment();
                    return this.stack.formatArn(arnComponents);
                }
                else {
                    return arnAttr;
                }
            },
        });
    }
}
exports.Resource = Resource;
//# sourceMappingURL=data:application/json;base64,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