"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.generateSchema = exports.bump = exports.update = exports.SCHEMAS = void 0;
const fs = require("fs");
const path = require("path");
const semver = require("semver");
// eslint-disable-next-line import/no-extraneous-dependencies
const tjs = require("typescript-json-schema");
function log(message) {
    // tslint:disable-next-line:no-console
    console.log(message);
}
/**
 * Where schemas are committed.
 */
const SCHEMA_DIR = path.resolve(__dirname, '../schema');
const SCHEMA_DEFINITIONS = {
    'assets': { rootTypeName: 'AssetManifest' },
    'cloud-assembly': { rootTypeName: 'AssemblyManifest' },
};
exports.SCHEMAS = Object.keys(SCHEMA_DEFINITIONS);
function update() {
    for (const s of exports.SCHEMAS) {
        generateSchema(s);
    }
    bump();
}
exports.update = update;
function bump() {
    const versionFile = path.join(SCHEMA_DIR, 'cloud-assembly.version.json');
    // eslint-disable-next-line @typescript-eslint/no-require-imports
    const metadata = require(versionFile);
    const oldVersion = metadata.version;
    const newVersion = semver.inc(oldVersion, 'major');
    log(`Updating schema version: ${oldVersion} -> ${newVersion}`);
    fs.writeFileSync(versionFile, JSON.stringify({ version: newVersion }));
}
exports.bump = bump;
/**
 * Generates a schema from typescript types.
 * @returns JSON schema
 * @param schemaName the schema to generate
 * @param shouldBump writes a new version of the schema and bumps the major version
 */
function generateSchema(schemaName, saveToFile = true) {
    const spec = SCHEMA_DEFINITIONS[schemaName];
    const out = saveToFile ? path.join(SCHEMA_DIR, `${schemaName}.schema.json`) : '';
    const settings = {
        required: true,
        ref: true,
        topRef: true,
        noExtraProps: false,
        out,
    };
    const compilerOptions = {
        strictNullChecks: true,
    };
    const program = tjs.getProgramFromFiles([path.join(__dirname, '../lib/index.d.ts')], compilerOptions);
    const schema = tjs.generateSchema(program, spec.rootTypeName, settings);
    augmentDescription(schema);
    addAnyMetadataEntry(schema);
    if (out) {
        log(`Generating schema to ${out}`);
        fs.writeFileSync(out, JSON.stringify(schema, null, 4));
    }
    return schema;
}
exports.generateSchema = generateSchema;
/**
 * Remove 'default' from the schema since its generated
 * from the tsdocs, which are not necessarily actual values,
 * but rather descriptive behavior.
 *
 * To keep this inforamtion in the schema, we append it to the
 * 'description' of the property.
 */
function augmentDescription(schema) {
    function _recurse(o) {
        for (const prop in o) {
            if (prop === 'description' && typeof o[prop] === 'string') {
                const description = o[prop];
                const defaultValue = o.default;
                if (!defaultValue) {
                    // property doesn't have a default value
                    // skip
                    continue;
                }
                const descriptionWithDefault = `${description} (Default ${defaultValue})`;
                delete o.default;
                o[prop] = descriptionWithDefault;
            }
            else if (typeof o[prop] === 'object') {
                _recurse(o[prop]);
            }
        }
    }
    _recurse(schema);
}
/**
 * Patch the properties of MetadataEntry to allow
 * specifying any free form data. This is needed since source
 * code doesn't allow this in order to enforce stricter jsii
 * compatibility checks.
 */
function addAnyMetadataEntry(schema) {
    var _a;
    (_a = schema.definitions.MetadataEntry) === null || _a === void 0 ? void 0 : _a.properties.data.anyOf.push({ description: 'Free form data.' });
}
//# sourceMappingURL=data:application/json;base64,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