"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.renderJsonPath = exports.renderList = exports.State = void 0;
const cdk = require("../../../core"); // Automatically re-written from '@aws-cdk/core'
const types_1 = require("../types");
/**
 * Base class for all other state classes
 */
class State extends cdk.Construct {
    constructor(scope, id, props) {
        super(scope, id);
        this.branches = [];
        this.retries = [];
        this.catches = [];
        this.choices = [];
        this.prefixes = [];
        /**
         * States with references to this state.
         *
         * Used for finding complete connected graph that a state is part of.
         */
        this.incomingStates = [];
        this.startState = this;
        this.comment = props.comment;
        this.inputPath = props.inputPath;
        this.parameters = props.parameters;
        this.outputPath = props.outputPath;
        this.resultPath = props.resultPath;
    }
    /**
     * Add a prefix to the stateId of all States found in a construct tree
     */
    static prefixStates(root, prefix) {
        const queue = [root];
        while (queue.length > 0) {
            const el = queue.splice(0, 1)[0];
            if (isPrefixable(el)) {
                el.addPrefix(prefix);
            }
            queue.push(...el.node.children);
        }
    }
    /**
     * Find the set of states reachable through transitions from the given start state.
     * This does not retrieve states from within sub-graphs, such as states within a Parallel state's branch.
     */
    static findReachableStates(start, options = {}) {
        const visited = new Set();
        const ret = new Set();
        const queue = [start];
        while (queue.length > 0) {
            const state = queue.splice(0, 1)[0];
            if (visited.has(state)) {
                continue;
            }
            visited.add(state);
            const outgoing = state.outgoingTransitions(options);
            queue.push(...outgoing);
            ret.add(state);
        }
        return Array.from(ret);
    }
    /**
     * Find the set of end states states reachable through transitions from the given start state
     */
    static findReachableEndStates(start, options = {}) {
        const visited = new Set();
        const ret = new Set();
        const queue = [start];
        while (queue.length > 0) {
            const state = queue.splice(0, 1)[0];
            if (visited.has(state)) {
                continue;
            }
            visited.add(state);
            const outgoing = state.outgoingTransitions(options);
            if (outgoing.length > 0) {
                // We can continue
                queue.push(...outgoing);
            }
            else {
                // Terminal state
                ret.add(state);
            }
        }
        return Array.from(ret);
    }
    /**
     * Return only the states that allow chaining from an array of states
     */
    static filterNextables(states) {
        return states.filter(isNextable);
    }
    get id() {
        return this.node.id;
    }
    /**
     * Tokenized string that evaluates to the state's ID
     */
    get stateId() {
        return this.prefixes.concat(this.id).join('');
    }
    /**
     * Add a prefix to the stateId of this state
     */
    addPrefix(x) {
        if (x !== '') {
            this.prefixes.splice(0, 0, x);
        }
    }
    /**
     * Register this state as part of the given graph
     *
     * Don't call this. It will be called automatically when you work
     * with states normally.
     */
    bindToGraph(graph) {
        if (this.containingGraph === graph) {
            return;
        }
        if (this.containingGraph) {
            // tslint:disable-next-line:max-line-length
            throw new Error(`Trying to use state '${this.stateId}' in ${graph}, but is already in ${this.containingGraph}. Every state can only be used in one graph.`);
        }
        this.containingGraph = graph;
        this.whenBoundToGraph(graph);
        for (const incoming of this.incomingStates) {
            incoming.bindToGraph(graph);
        }
        for (const outgoing of this.outgoingTransitions({ includeErrorHandlers: true })) {
            outgoing.bindToGraph(graph);
        }
        for (const branch of this.branches) {
            branch.registerSuperGraph(this.containingGraph);
        }
        if (!!this.iteration) {
            this.iteration.registerSuperGraph(this.containingGraph);
        }
    }
    /**
     * Add a retrier to the retry list of this state
     * @internal
     */
    _addRetry(props = {}) {
        this.retries.push({
            ...props,
            errors: props.errors ? props.errors : [types_1.Errors.ALL],
        });
    }
    /**
     * Add an error handler to the catch list of this state
     * @internal
     */
    _addCatch(handler, props = {}) {
        this.catches.push({
            next: handler,
            props: {
                errors: props.errors ? props.errors : [types_1.Errors.ALL],
                resultPath: props.resultPath,
            },
        });
        handler.addIncoming(this);
        if (this.containingGraph) {
            handler.bindToGraph(this.containingGraph);
        }
    }
    /**
     * Make the indicated state the default transition of this state
     */
    makeNext(next) {
        // Can't be called 'setNext' because of JSII
        if (this._next) {
            throw new Error(`State '${this.id}' already has a next state`);
        }
        this._next = next;
        next.addIncoming(this);
        if (this.containingGraph) {
            next.bindToGraph(this.containingGraph);
        }
    }
    /**
     * Add a choice branch to this state
     */
    addChoice(condition, next) {
        this.choices.push({ condition, next });
        next.startState.addIncoming(this);
        if (this.containingGraph) {
            next.startState.bindToGraph(this.containingGraph);
        }
    }
    /**
     * Add a paralle branch to this state
     */
    addBranch(branch) {
        this.branches.push(branch);
        if (this.containingGraph) {
            branch.registerSuperGraph(this.containingGraph);
        }
    }
    /**
     * Add a map iterator to this state
     */
    addIterator(iteration) {
        this.iteration = iteration;
        if (this.containingGraph) {
            iteration.registerSuperGraph(this.containingGraph);
        }
    }
    /**
     * Make the indicated state the default choice transition of this state
     */
    makeDefault(def) {
        // Can't be called 'setDefault' because of JSII
        if (this.defaultChoice) {
            throw new Error(`Choice '${this.id}' already has a default next state`);
        }
        this.defaultChoice = def;
    }
    /**
     * Render the default next state in ASL JSON format
     */
    renderNextEnd() {
        if (this._next) {
            return { Next: this._next.stateId };
        }
        else {
            return { End: true };
        }
    }
    /**
     * Render the choices in ASL JSON format
     */
    renderChoices() {
        return {
            Choices: renderList(this.choices, renderChoice),
            Default: this.defaultChoice ? this.defaultChoice.stateId : undefined,
        };
    }
    /**
     * Render InputPath/Parameters/OutputPath in ASL JSON format
     */
    renderInputOutput() {
        return {
            InputPath: renderJsonPath(this.inputPath),
            Parameters: this.parameters,
            OutputPath: renderJsonPath(this.outputPath),
        };
    }
    /**
     * Render parallel branches in ASL JSON format
     */
    renderBranches() {
        return {
            Branches: this.branches.map(b => b.toGraphJson()),
        };
    }
    /**
     * Render map iterator in ASL JSON format
     */
    renderIterator() {
        if (!this.iteration) {
            throw new Error('Iterator must not be undefined !');
        }
        return {
            Iterator: this.iteration.toGraphJson(),
        };
    }
    /**
     * Render error recovery options in ASL JSON format
     */
    renderRetryCatch() {
        return {
            Retry: renderList(this.retries, renderRetry),
            Catch: renderList(this.catches, renderCatch),
        };
    }
    /**
     * Called whenever this state is bound to a graph
     *
     * Can be overridden by subclasses.
     */
    whenBoundToGraph(graph) {
        graph.registerState(this);
    }
    /**
     * Add a state to the incoming list
     */
    addIncoming(source) {
        this.incomingStates.push(source);
    }
    /**
     * Return all states this state can transition to
     */
    outgoingTransitions(options) {
        const ret = new Array();
        if (this._next) {
            ret.push(this._next);
        }
        if (this.defaultChoice) {
            ret.push(this.defaultChoice);
        }
        for (const c of this.choices) {
            ret.push(c.next);
        }
        if (options.includeErrorHandlers) {
            for (const c of this.catches) {
                ret.push(c.next);
            }
        }
        return ret;
    }
}
exports.State = State;
/**
 * Render a choice transition
 */
function renderChoice(c) {
    return {
        ...c.condition.renderCondition(),
        Next: c.next.stateId,
    };
}
/**
 * Render a Retry object to ASL
 */
function renderRetry(retry) {
    return {
        ErrorEquals: retry.errors,
        IntervalSeconds: retry.interval && retry.interval.toSeconds(),
        MaxAttempts: retry.maxAttempts,
        BackoffRate: retry.backoffRate,
    };
}
/**
 * Render a Catch object to ASL
 */
function renderCatch(c) {
    return {
        ErrorEquals: c.props.errors,
        ResultPath: renderJsonPath(c.props.resultPath),
        Next: c.next.stateId,
    };
}
/**
 * Render a list or return undefined for an empty list
 */
function renderList(xs, fn) {
    if (xs.length === 0) {
        return undefined;
    }
    return xs.map(fn);
}
exports.renderList = renderList;
/**
 * Render JSON path, respecting the special value DISCARD
 */
function renderJsonPath(jsonPath) {
    if (jsonPath === undefined) {
        return undefined;
    }
    if (jsonPath === types_1.DISCARD) {
        return null;
    }
    if (!jsonPath.startsWith('$')) {
        throw new Error(`Expected JSON path to start with '$', got: ${jsonPath}`);
    }
    return jsonPath;
}
exports.renderJsonPath = renderJsonPath;
/**
 * Whether an object is a Prefixable
 */
function isPrefixable(x) {
    return typeof (x) === 'object' && x.addPrefix;
}
/**
 * Whether an object is INextable
 */
function isNextable(x) {
    return typeof (x) === 'object' && x.next;
}
//# sourceMappingURL=data:application/json;base64,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