"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.InputType = exports.TaskInput = void 0;
const fields_1 = require("./fields");
/**
 * Type union for task classes that accept multiple types of payload
 */
class TaskInput {
    /**
     *
     * @param type type of task input
     * @param value payload for the corresponding input type.
     * It can be a JSON-encoded object, context, data, etc.
     */
    constructor(type, value) {
        this.type = type;
        this.value = value;
    }
    /**
     * Use a literal string as task input
     *
     * This might be a JSON-encoded object, or just a text.
     */
    static fromText(text) {
        return new TaskInput(InputType.TEXT, text);
    }
    /**
     * Use an object as task input
     *
     * This object may contain Data and Context fields
     * as object values, if desired.
     */
    static fromObject(obj) {
        return new TaskInput(InputType.OBJECT, obj);
    }
    /**
     * Use a part of the execution data as task input
     *
     * Use this when you want to use a subobject or string from
     * the current state machine execution as complete payload
     * to a task.
     */
    static fromDataAt(path) {
        return new TaskInput(InputType.TEXT, fields_1.Data.stringAt(path));
    }
    /**
     * Use a part of the task context as task input
     *
     * Use this when you want to use a subobject or string from
     * the current task context as complete payload
     * to a task.
     */
    static fromContextAt(path) {
        return new TaskInput(InputType.TEXT, fields_1.Context.stringAt(path));
    }
}
exports.TaskInput = TaskInput;
/**
 * The type of task input
 */
var InputType;
(function (InputType) {
    /**
     * Use a literal string
     * This might be a JSON-encoded object, or just text.
     * valid JSON text: standalone, quote-delimited strings; objects; arrays; numbers; Boolean values; and null.
     *
     * example: `literal string`
     * example: {"json": "encoded"}
     */
    InputType[InputType["TEXT"] = 0] = "TEXT";
    /**
     * Use an object which may contain Data and Context fields
     * as object values, if desired.
     *
     * example:
     * {
     *  literal: 'literal',
     *  SomeInput: sfn.Data.stringAt('$.someField')
     * }
     *
     * @see https://docs.aws.amazon.com/step-functions/latest/dg/concepts-state-machine-data.html
     * @see https://docs.aws.amazon.com/step-functions/latest/dg/input-output-contextobject.html
     */
    InputType[InputType["OBJECT"] = 1] = "OBJECT";
})(InputType = exports.InputType || (exports.InputType = {}));
//# sourceMappingURL=data:application/json;base64,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