"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.FieldUtils = exports.Context = exports.Data = void 0;
const core_1 = require("../../core"); // Automatically re-written from '@aws-cdk/core'
const json_path_1 = require("./json-path");
/**
 * Extract a field from the State Machine data that gets passed around between states
 */
class Data {
    /**
     * Instead of using a literal string, get the value from a JSON path
     */
    static stringAt(path) {
        validateDataPath(path);
        return new json_path_1.JsonPathToken(path).toString();
    }
    /**
     * Instead of using a literal string list, get the value from a JSON path
     */
    static listAt(path) {
        validateDataPath(path);
        return core_1.Token.asList(new json_path_1.JsonPathToken(path));
    }
    /**
     * Instead of using a literal number, get the value from a JSON path
     */
    static numberAt(path) {
        validateDataPath(path);
        return core_1.Token.asNumber(new json_path_1.JsonPathToken(path));
    }
    /**
     * Use the entire data structure
     *
     * Will be an object at invocation time, but is represented in the CDK
     * application as a string.
     */
    static get entirePayload() {
        return new json_path_1.JsonPathToken('$').toString();
    }
    /**
     * Determines if the indicated string is an encoded JSON path
     *
     * @param value string to be evaluated
     */
    static isJsonPathString(value) {
        return !!json_path_1.jsonPathString(value);
    }
    constructor() {
    }
}
exports.Data = Data;
/**
 * Extract a field from the State Machine Context data
 *
 * @see https://docs.aws.amazon.com/step-functions/latest/dg/connect-to-resource.html#wait-token-contextobject
 */
class Context {
    /**
     * Instead of using a literal string, get the value from a JSON path
     */
    static stringAt(path) {
        validateContextPath(path);
        return new json_path_1.JsonPathToken(path).toString();
    }
    /**
     * Instead of using a literal number, get the value from a JSON path
     */
    static numberAt(path) {
        validateContextPath(path);
        return core_1.Token.asNumber(new json_path_1.JsonPathToken(path));
    }
    /**
     * Return the Task Token field
     *
     * External actions will need this token to report step completion
     * back to StepFunctions using the `SendTaskSuccess` or `SendTaskFailure`
     * calls.
     */
    static get taskToken() {
        return new json_path_1.JsonPathToken('$$.Task.Token').toString();
    }
    /**
     * Use the entire context data structure
     *
     * Will be an object at invocation time, but is represented in the CDK
     * application as a string.
     */
    static get entireContext() {
        return new json_path_1.JsonPathToken('$$').toString();
    }
    constructor() {
    }
}
exports.Context = Context;
/**
 * Helper functions to work with structures containing fields
 */
class FieldUtils {
    /**
     * Render a JSON structure containing fields to the right StepFunctions structure
     */
    static renderObject(obj) {
        return json_path_1.renderObject(obj);
    }
    /**
     * Return all JSON paths used in the given structure
     */
    static findReferencedPaths(obj) {
        return Array.from(json_path_1.findReferencedPaths(obj)).sort();
    }
    /**
     * Returns whether the given task structure contains the TaskToken field anywhere
     *
     * The field is considered included if the field itself or one of its containing
     * fields occurs anywhere in the payload.
     */
    static containsTaskToken(obj) {
        const paths = json_path_1.findReferencedPaths(obj);
        return paths.has('$$.Task.Token') || paths.has('$$.Task') || paths.has('$$');
    }
    constructor() {
    }
}
exports.FieldUtils = FieldUtils;
function validateDataPath(path) {
    if (path !== '$' && !path.startsWith('$.')) {
        throw new Error("Data JSON path values must either be exactly equal to '$' or start with '$.'");
    }
}
function validateContextPath(path) {
    if (path !== '$$' && !path.startsWith('$$.')) {
        throw new Error("Context JSON path values must either be exactly equal to '$$' or start with '$$.'");
    }
}
//# sourceMappingURL=data:application/json;base64,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