"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.SageMakerCreateTransformJob = void 0;
const ec2 = require("../../../aws-ec2"); // Automatically re-written from '@aws-cdk/aws-ec2'
const iam = require("../../../aws-iam"); // Automatically re-written from '@aws-cdk/aws-iam'
const sfn = require("../../../aws-stepfunctions"); // Automatically re-written from '@aws-cdk/aws-stepfunctions'
const core_1 = require("../../../core"); // Automatically re-written from '@aws-cdk/core'
const task_utils_1 = require("../private/task-utils");
const base_types_1 = require("./base-types");
/**
 * Class representing the SageMaker Create Training Job task.
 *
 *  @experimental
 */
class SageMakerCreateTransformJob extends sfn.TaskStateBase {
    constructor(scope, id, props) {
        super(scope, id, props);
        this.props = props;
        this.integrationPattern = props.integrationPattern || sfn.IntegrationPattern.REQUEST_RESPONSE;
        task_utils_1.validatePatternSupported(this.integrationPattern, SageMakerCreateTransformJob.SUPPORTED_INTEGRATION_PATTERNS);
        // set the sagemaker role or create new one
        if (props.role) {
            this._role = props.role;
        }
        // set the S3 Data type of the input data config objects to be 'S3Prefix' if not defined
        this.transformInput = props.transformInput.transformDataSource.s3DataSource.s3DataType
            ? props.transformInput
            : Object.assign({}, props.transformInput, {
                transformDataSource: { s3DataSource: { ...props.transformInput.transformDataSource.s3DataSource, s3DataType: base_types_1.S3DataType.S3_PREFIX } },
            });
        // set the default value for the transform resources
        this.transformResources = props.transformResources || {
            instanceCount: 1,
            instanceType: ec2.InstanceType.of(ec2.InstanceClass.M4, ec2.InstanceSize.XLARGE),
        };
        this.taskPolicies = this.makePolicyStatements();
    }
    renderTask() {
        return {
            Resource: task_utils_1.integrationResourceArn('sagemaker', 'createTransformJob', this.integrationPattern),
            Parameters: sfn.FieldUtils.renderObject(this.renderParameters()),
        };
    }
    /**
     * The execution role for the Sagemaker training job.
     *
     * Only available after task has been added to a state machine.
     */
    get role() {
        if (this._role === undefined) {
            throw new Error('role not available yet--use the object in a Task first');
        }
        return this._role;
    }
    renderParameters() {
        return {
            ...(this.props.batchStrategy ? { BatchStrategy: this.props.batchStrategy } : {}),
            ...this.renderEnvironment(this.props.environment),
            ...(this.props.maxConcurrentTransforms ? { MaxConcurrentTransforms: this.props.maxConcurrentTransforms } : {}),
            ...(this.props.maxPayload ? { MaxPayloadInMB: this.props.maxPayload.toMebibytes() } : {}),
            ModelName: this.props.modelName,
            ...this.renderTags(this.props.tags),
            ...this.renderTransformInput(this.transformInput),
            TransformJobName: this.props.transformJobName,
            ...this.renderTransformOutput(this.props.transformOutput),
            ...this.renderTransformResources(this.transformResources),
        };
    }
    renderTransformInput(input) {
        return {
            TransformInput: {
                ...(input.compressionType ? { CompressionType: input.compressionType } : {}),
                ...(input.contentType ? { ContentType: input.contentType } : {}),
                DataSource: {
                    S3DataSource: {
                        S3Uri: input.transformDataSource.s3DataSource.s3Uri,
                        S3DataType: input.transformDataSource.s3DataSource.s3DataType,
                    },
                },
                ...(input.splitType ? { SplitType: input.splitType } : {}),
            },
        };
    }
    renderTransformOutput(output) {
        return {
            TransformOutput: {
                S3OutputPath: output.s3OutputPath,
                ...(output.encryptionKey ? { KmsKeyId: output.encryptionKey.keyArn } : {}),
                ...(output.accept ? { Accept: output.accept } : {}),
                ...(output.assembleWith ? { AssembleWith: output.assembleWith } : {}),
            },
        };
    }
    renderTransformResources(resources) {
        return {
            TransformResources: {
                InstanceCount: resources.instanceCount,
                InstanceType: 'ml.' + resources.instanceType,
                ...(resources.volumeEncryptionKey ? { VolumeKmsKeyId: resources.volumeEncryptionKey.keyArn } : {}),
            },
        };
    }
    renderEnvironment(environment) {
        return environment ? { Environment: environment } : {};
    }
    renderTags(tags) {
        return tags ? { Tags: Object.keys(tags).map((key) => ({ Key: key, Value: tags[key] })) } : {};
    }
    makePolicyStatements() {
        const stack = core_1.Stack.of(this);
        // create new role if doesn't exist
        if (this._role === undefined) {
            this._role = new iam.Role(this, 'SagemakerTransformRole', {
                assumedBy: new iam.ServicePrincipal('sagemaker.amazonaws.com'),
                managedPolicies: [iam.ManagedPolicy.fromAwsManagedPolicyName('AmazonSageMakerFullAccess')],
            });
        }
        // https://docs.aws.amazon.com/step-functions/latest/dg/sagemaker-iam.html
        const policyStatements = [
            new iam.PolicyStatement({
                actions: ['sagemaker:CreateTransformJob', 'sagemaker:DescribeTransformJob', 'sagemaker:StopTransformJob'],
                resources: [
                    stack.formatArn({
                        service: 'sagemaker',
                        resource: 'transform-job',
                        resourceName: '*',
                    }),
                ],
            }),
            new iam.PolicyStatement({
                actions: ['sagemaker:ListTags'],
                resources: ['*'],
            }),
            new iam.PolicyStatement({
                actions: ['iam:PassRole'],
                resources: [this.role.roleArn],
                conditions: {
                    StringEquals: { 'iam:PassedToService': 'sagemaker.amazonaws.com' },
                },
            }),
        ];
        if (this.integrationPattern === sfn.IntegrationPattern.RUN_JOB) {
            policyStatements.push(new iam.PolicyStatement({
                actions: ['events:PutTargets', 'events:PutRule', 'events:DescribeRule'],
                resources: [
                    stack.formatArn({
                        service: 'events',
                        resource: 'rule',
                        resourceName: 'StepFunctionsGetEventsForSageMakerTransformJobsRule',
                    }),
                ],
            }));
        }
        return policyStatements;
    }
}
exports.SageMakerCreateTransformJob = SageMakerCreateTransformJob;
SageMakerCreateTransformJob.SUPPORTED_INTEGRATION_PATTERNS = [
    sfn.IntegrationPattern.REQUEST_RESPONSE,
    sfn.IntegrationPattern.RUN_JOB,
];
//# sourceMappingURL=data:application/json;base64,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