"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.SageMakerCreateTrainingJob = void 0;
const ec2 = require("../../../aws-ec2"); // Automatically re-written from '@aws-cdk/aws-ec2'
const iam = require("../../../aws-iam"); // Automatically re-written from '@aws-cdk/aws-iam'
const sfn = require("../../../aws-stepfunctions"); // Automatically re-written from '@aws-cdk/aws-stepfunctions'
const core_1 = require("../../../core"); // Automatically re-written from '@aws-cdk/core'
const task_utils_1 = require("../private/task-utils");
const base_types_1 = require("./base-types");
/**
 * Class representing the SageMaker Create Training Job task.
 *
 * @experimental
 */
class SageMakerCreateTrainingJob extends sfn.TaskStateBase {
    constructor(scope, id, props) {
        super(scope, id, props);
        this.props = props;
        /**
         * Allows specify security group connections for instances of this fleet.
         */
        this.connections = new ec2.Connections();
        this.securityGroups = [];
        this.integrationPattern = props.integrationPattern || sfn.IntegrationPattern.REQUEST_RESPONSE;
        task_utils_1.validatePatternSupported(this.integrationPattern, SageMakerCreateTrainingJob.SUPPORTED_INTEGRATION_PATTERNS);
        // set the default resource config if not defined.
        this.resourceConfig = props.resourceConfig || {
            instanceCount: 1,
            instanceType: ec2.InstanceType.of(ec2.InstanceClass.M4, ec2.InstanceSize.XLARGE),
            volumeSize: core_1.Size.gibibytes(10),
        };
        // set the stopping condition if not defined
        this.stoppingCondition = props.stoppingCondition || {
            maxRuntime: core_1.Duration.hours(1),
        };
        // check that either algorithm name or image is defined
        if (!props.algorithmSpecification.algorithmName && !props.algorithmSpecification.trainingImage) {
            throw new Error('Must define either an algorithm name or training image URI in the algorithm specification');
        }
        // set the input mode to 'File' if not defined
        this.algorithmSpecification = props.algorithmSpecification.trainingInputMode
            ? props.algorithmSpecification
            : { ...props.algorithmSpecification, trainingInputMode: base_types_1.InputMode.FILE };
        // set the S3 Data type of the input data config objects to be 'S3Prefix' if not defined
        this.inputDataConfig = props.inputDataConfig.map((config) => {
            if (!config.dataSource.s3DataSource.s3DataType) {
                return {
                    ...config,
                    dataSource: { s3DataSource: { ...config.dataSource.s3DataSource, s3DataType: base_types_1.S3DataType.S3_PREFIX } },
                };
            }
            else {
                return config;
            }
        });
        // add the security groups to the connections object
        if (props.vpcConfig) {
            this.vpc = props.vpcConfig.vpc;
            this.subnets = props.vpcConfig.subnets ? this.vpc.selectSubnets(props.vpcConfig.subnets).subnetIds : this.vpc.selectSubnets().subnetIds;
        }
        this.taskPolicies = this.makePolicyStatements();
    }
    /**
     * The execution role for the Sagemaker training job.
     *
     * Only available after task has been added to a state machine.
     */
    get role() {
        if (this._role === undefined) {
            throw new Error('role not available yet--use the object in a Task first');
        }
        return this._role;
    }
    get grantPrincipal() {
        if (this._grantPrincipal === undefined) {
            throw new Error('Principal not available yet--use the object in a Task first');
        }
        return this._grantPrincipal;
    }
    /**
     * Add the security group to all instances via the launch configuration
     * security groups array.
     *
     * @param securityGroup: The security group to add
     */
    addSecurityGroup(securityGroup) {
        this.securityGroups.push(securityGroup);
    }
    renderTask() {
        return {
            Resource: task_utils_1.integrationResourceArn('sagemaker', 'createTrainingJob', this.integrationPattern),
            Parameters: sfn.FieldUtils.renderObject(this.renderParameters()),
        };
    }
    renderParameters() {
        return {
            TrainingJobName: this.props.trainingJobName,
            RoleArn: this._role.roleArn,
            ...this.renderAlgorithmSpecification(this.algorithmSpecification),
            ...this.renderInputDataConfig(this.inputDataConfig),
            ...this.renderOutputDataConfig(this.props.outputDataConfig),
            ...this.renderResourceConfig(this.resourceConfig),
            ...this.renderStoppingCondition(this.stoppingCondition),
            ...this.renderHyperparameters(this.props.hyperparameters),
            ...this.renderTags(this.props.tags),
            ...this.renderVpcConfig(this.props.vpcConfig),
        };
    }
    renderAlgorithmSpecification(spec) {
        return {
            AlgorithmSpecification: {
                TrainingInputMode: spec.trainingInputMode,
                ...(spec.trainingImage ? { TrainingImage: spec.trainingImage.bind(this).imageUri } : {}),
                ...(spec.algorithmName ? { AlgorithmName: spec.algorithmName } : {}),
                ...(spec.metricDefinitions
                    ? { MetricDefinitions: spec.metricDefinitions.map((metric) => ({ Name: metric.name, Regex: metric.regex })) }
                    : {}),
            },
        };
    }
    renderInputDataConfig(config) {
        return {
            InputDataConfig: config.map((channel) => ({
                ChannelName: channel.channelName,
                DataSource: {
                    S3DataSource: {
                        S3Uri: channel.dataSource.s3DataSource.s3Location.bind(this, { forReading: true }).uri,
                        S3DataType: channel.dataSource.s3DataSource.s3DataType,
                        ...(channel.dataSource.s3DataSource.s3DataDistributionType
                            ? { S3DataDistributionType: channel.dataSource.s3DataSource.s3DataDistributionType }
                            : {}),
                        ...(channel.dataSource.s3DataSource.attributeNames ? { AtttributeNames: channel.dataSource.s3DataSource.attributeNames } : {}),
                    },
                },
                ...(channel.compressionType ? { CompressionType: channel.compressionType } : {}),
                ...(channel.contentType ? { ContentType: channel.contentType } : {}),
                ...(channel.inputMode ? { InputMode: channel.inputMode } : {}),
                ...(channel.recordWrapperType ? { RecordWrapperType: channel.recordWrapperType } : {}),
            })),
        };
    }
    renderOutputDataConfig(config) {
        return {
            OutputDataConfig: {
                S3OutputPath: config.s3OutputLocation.bind(this, { forWriting: true }).uri,
                ...(config.encryptionKey ? { KmsKeyId: config.encryptionKey.keyArn } : {}),
            },
        };
    }
    renderResourceConfig(config) {
        return {
            ResourceConfig: {
                InstanceCount: config.instanceCount,
                InstanceType: 'ml.' + config.instanceType,
                VolumeSizeInGB: config.volumeSize.toGibibytes(),
                ...(config.volumeEncryptionKey ? { VolumeKmsKeyId: config.volumeEncryptionKey.keyArn } : {}),
            },
        };
    }
    renderStoppingCondition(config) {
        return {
            StoppingCondition: {
                MaxRuntimeInSeconds: config.maxRuntime && config.maxRuntime.toSeconds(),
            },
        };
    }
    renderHyperparameters(params) {
        return params ? { HyperParameters: params } : {};
    }
    renderTags(tags) {
        return tags ? { Tags: Object.keys(tags).map((key) => ({ Key: key, Value: tags[key] })) } : {};
    }
    renderVpcConfig(config) {
        return config
            ? {
                VpcConfig: {
                    SecurityGroupIds: core_1.Lazy.listValue({ produce: () => this.securityGroups.map((sg) => sg.securityGroupId) }),
                    Subnets: this.subnets,
                },
            }
            : {};
    }
    makePolicyStatements() {
        // set the sagemaker role or create new one
        this._grantPrincipal = this._role =
            this.props.role ||
                new iam.Role(this, 'SagemakerRole', {
                    assumedBy: new iam.ServicePrincipal('sagemaker.amazonaws.com'),
                    inlinePolicies: {
                        CreateTrainingJob: new iam.PolicyDocument({
                            statements: [
                                new iam.PolicyStatement({
                                    actions: [
                                        'cloudwatch:PutMetricData',
                                        'logs:CreateLogStream',
                                        'logs:PutLogEvents',
                                        'logs:CreateLogGroup',
                                        'logs:DescribeLogStreams',
                                        'ecr:GetAuthorizationToken',
                                        ...(this.props.vpcConfig
                                            ? [
                                                'ec2:CreateNetworkInterface',
                                                'ec2:CreateNetworkInterfacePermission',
                                                'ec2:DeleteNetworkInterface',
                                                'ec2:DeleteNetworkInterfacePermission',
                                                'ec2:DescribeNetworkInterfaces',
                                                'ec2:DescribeVpcs',
                                                'ec2:DescribeDhcpOptions',
                                                'ec2:DescribeSubnets',
                                                'ec2:DescribeSecurityGroups',
                                            ]
                                            : []),
                                    ],
                                    resources: ['*'],
                                }),
                            ],
                        }),
                    },
                });
        if (this.props.outputDataConfig.encryptionKey) {
            this.props.outputDataConfig.encryptionKey.grantEncrypt(this._role);
        }
        if (this.props.resourceConfig && this.props.resourceConfig.volumeEncryptionKey) {
            this.props.resourceConfig.volumeEncryptionKey.grant(this._role, 'kms:CreateGrant');
        }
        // create a security group if not defined
        if (this.vpc && this.securityGroup === undefined) {
            this.securityGroup = new ec2.SecurityGroup(this, 'TrainJobSecurityGroup', {
                vpc: this.vpc,
            });
            this.connections.addSecurityGroup(this.securityGroup);
            this.securityGroups.push(this.securityGroup);
        }
        const stack = core_1.Stack.of(this);
        // https://docs.aws.amazon.com/step-functions/latest/dg/sagemaker-iam.html
        const policyStatements = [
            new iam.PolicyStatement({
                actions: ['sagemaker:CreateTrainingJob', 'sagemaker:DescribeTrainingJob', 'sagemaker:StopTrainingJob'],
                resources: [
                    stack.formatArn({
                        service: 'sagemaker',
                        resource: 'training-job',
                        // If the job name comes from input, we cannot target the policy to a particular ARN prefix reliably...
                        resourceName: sfn.Data.isJsonPathString(this.props.trainingJobName) ? '*' : `${this.props.trainingJobName}*`,
                    }),
                ],
            }),
            new iam.PolicyStatement({
                actions: ['sagemaker:ListTags'],
                resources: ['*'],
            }),
            new iam.PolicyStatement({
                actions: ['iam:PassRole'],
                resources: [this._role.roleArn],
                conditions: {
                    StringEquals: { 'iam:PassedToService': 'sagemaker.amazonaws.com' },
                },
            }),
        ];
        if (this.integrationPattern === sfn.IntegrationPattern.RUN_JOB) {
            policyStatements.push(new iam.PolicyStatement({
                actions: ['events:PutTargets', 'events:PutRule', 'events:DescribeRule'],
                resources: [
                    stack.formatArn({
                        service: 'events',
                        resource: 'rule',
                        resourceName: 'StepFunctionsGetEventsForSageMakerTrainingJobsRule',
                    }),
                ],
            }));
        }
        return policyStatements;
    }
}
exports.SageMakerCreateTrainingJob = SageMakerCreateTrainingJob;
SageMakerCreateTrainingJob.SUPPORTED_INTEGRATION_PATTERNS = [
    sfn.IntegrationPattern.REQUEST_RESPONSE,
    sfn.IntegrationPattern.RUN_JOB,
];
//# sourceMappingURL=data:application/json;base64,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