"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.AssembleWith = exports.SplitType = exports.BatchStrategy = exports.CompressionType = exports.InputMode = exports.RecordWrapperType = exports.S3DataDistributionType = exports.S3DataType = exports.DockerImage = exports.S3Location = void 0;
const aws_ecr_assets_1 = require("../../../aws-ecr-assets"); // Automatically re-written from '@aws-cdk/aws-ecr-assets'
const iam = require("../../../aws-iam"); // Automatically re-written from '@aws-cdk/aws-iam'
const sfn = require("../../../aws-stepfunctions"); // Automatically re-written from '@aws-cdk/aws-stepfunctions'
/**
 * Constructs `IS3Location` objects.
 *
 * @experimental
 */
class S3Location {
    /**
     * An `IS3Location` built with a determined bucket and key prefix.
     *
     * @param bucket    is the bucket where the objects are to be stored.
     * @param keyPrefix is the key prefix used by the location.
     */
    static fromBucket(bucket, keyPrefix) {
        return new StandardS3Location({ bucket, keyPrefix, uri: bucket.urlForObject(keyPrefix) });
    }
    /**
     * An `IS3Location` determined fully by a JSON Path from the task input.
     *
     * Due to the dynamic nature of those locations, the IAM grants that will be set by `grantRead` and `grantWrite`
     * apply to the `*` resource.
     *
     * @param expression the JSON expression resolving to an S3 location URI.
     */
    static fromJsonExpression(expression) {
        return new StandardS3Location({ uri: sfn.Data.stringAt(expression) });
    }
}
exports.S3Location = S3Location;
/**
 * Creates `IDockerImage` instances.
 *
 * @experimental
 */
class DockerImage {
    /**
     * Reference a Docker image stored in an ECR repository.
     *
     * @param repository the ECR repository where the image is hosted.
     * @param tag an optional `tag`
     */
    static fromEcrRepository(repository, tag = 'latest') {
        return new StandardDockerImage({ repository, imageUri: repository.repositoryUriForTag(tag) });
    }
    /**
     * Reference a Docker image which URI is obtained from the task's input.
     *
     * @param expression           the JSON path expression with the task input.
     * @param allowAnyEcrImagePull whether ECR access should be permitted (set to `false` if the image will never be in ECR).
     */
    static fromJsonExpression(expression, allowAnyEcrImagePull = true) {
        return new StandardDockerImage({ imageUri: expression, allowAnyEcrImagePull });
    }
    /**
     * Reference a Docker image by it's URI.
     *
     * When referencing ECR images, prefer using `inEcr`.
     *
     * @param imageUri the URI to the docker image.
     */
    static fromRegistry(imageUri) {
        return new StandardDockerImage({ imageUri });
    }
    /**
     * Reference a Docker image that is provided as an Asset in the current app.
     *
     * @param scope the scope in which to create the Asset.
     * @param id    the ID for the asset in the construct tree.
     * @param props the configuration props of the asset.
     */
    static fromAsset(scope, id, props) {
        const asset = new aws_ecr_assets_1.DockerImageAsset(scope, id, props);
        return new StandardDockerImage({ repository: asset.repository, imageUri: asset.imageUri });
    }
}
exports.DockerImage = DockerImage;
/**
 * S3 Data Type.
 *
 * @experimental
 */
var S3DataType;
(function (S3DataType) {
    /**
     * Manifest File Data Type
     */
    S3DataType["MANIFEST_FILE"] = "ManifestFile";
    /**
     * S3 Prefix Data Type
     */
    S3DataType["S3_PREFIX"] = "S3Prefix";
    /**
     * Augmented Manifest File Data Type
     */
    S3DataType["AUGMENTED_MANIFEST_FILE"] = "AugmentedManifestFile";
})(S3DataType = exports.S3DataType || (exports.S3DataType = {}));
/**
 * S3 Data Distribution Type.
 *
 * @experimental
 */
var S3DataDistributionType;
(function (S3DataDistributionType) {
    /**
     * Fully replicated S3 Data Distribution Type
     */
    S3DataDistributionType["FULLY_REPLICATED"] = "FullyReplicated";
    /**
     * Sharded By S3 Key Data Distribution Type
     */
    S3DataDistributionType["SHARDED_BY_S3_KEY"] = "ShardedByS3Key";
})(S3DataDistributionType = exports.S3DataDistributionType || (exports.S3DataDistributionType = {}));
/**
 * Define the format of the input data.
 *
 * @experimental
 */
var RecordWrapperType;
(function (RecordWrapperType) {
    /**
     * None record wrapper type
     */
    RecordWrapperType["NONE"] = "None";
    /**
     * RecordIO record wrapper type
     */
    RecordWrapperType["RECORD_IO"] = "RecordIO";
})(RecordWrapperType = exports.RecordWrapperType || (exports.RecordWrapperType = {}));
/**
 *  Input mode that the algorithm supports.
 *
 * @experimental
 */
var InputMode;
(function (InputMode) {
    /**
     * Pipe mode
     */
    InputMode["PIPE"] = "Pipe";
    /**
     * File mode.
     */
    InputMode["FILE"] = "File";
})(InputMode = exports.InputMode || (exports.InputMode = {}));
/**
 * Compression type of the data.
 *
 * @experimental
 */
var CompressionType;
(function (CompressionType) {
    /**
     * None compression type
     */
    CompressionType["NONE"] = "None";
    /**
     * Gzip compression type
     */
    CompressionType["GZIP"] = "Gzip";
})(CompressionType = exports.CompressionType || (exports.CompressionType = {}));
/**
 * Specifies the number of records to include in a mini-batch for an HTTP inference request.
 *
 * @experimental
 */
var BatchStrategy;
(function (BatchStrategy) {
    /**
     * Fits multiple records in a mini-batch.
     */
    BatchStrategy["MULTI_RECORD"] = "MultiRecord";
    /**
     * Use a single record when making an invocation request.
     */
    BatchStrategy["SINGLE_RECORD"] = "SingleRecord";
})(BatchStrategy = exports.BatchStrategy || (exports.BatchStrategy = {}));
/**
 * Method to use to split the transform job's data files into smaller batches.
 *
 * @experimental
 */
var SplitType;
(function (SplitType) {
    /**
     * Input data files are not split,
     */
    SplitType["NONE"] = "None";
    /**
     * Split records on a newline character boundary.
     */
    SplitType["LINE"] = "Line";
    /**
     * Split using MXNet RecordIO format.
     */
    SplitType["RECORD_IO"] = "RecordIO";
    /**
     * Split using TensorFlow TFRecord format.
     */
    SplitType["TF_RECORD"] = "TFRecord";
})(SplitType = exports.SplitType || (exports.SplitType = {}));
/**
 * How to assemble the results of the transform job as a single S3 object.
 *
 * @experimental
 */
var AssembleWith;
(function (AssembleWith) {
    /**
     * Concatenate the results in binary format.
     */
    AssembleWith["NONE"] = "None";
    /**
     * Add a newline character at the end of every transformed record.
     */
    AssembleWith["LINE"] = "Line";
})(AssembleWith = exports.AssembleWith || (exports.AssembleWith = {}));
class StandardDockerImage extends DockerImage {
    constructor(opts) {
        super();
        this.allowAnyEcrImagePull = !!opts.allowAnyEcrImagePull;
        this.imageUri = opts.imageUri;
        this.repository = opts.repository;
    }
    bind(task) {
        if (this.repository) {
            this.repository.grantPull(task);
        }
        if (this.allowAnyEcrImagePull) {
            task.grantPrincipal.addToPolicy(new iam.PolicyStatement({
                actions: [
                    'ecr:BatchCheckLayerAvailability',
                    'ecr:GetDownloadUrlForLayer',
                    'ecr:BatchGetImage',
                ],
                resources: ['*'],
            }));
        }
        return {
            imageUri: this.imageUri,
        };
    }
}
class StandardS3Location extends S3Location {
    constructor(opts) {
        super();
        this.bucket = opts.bucket;
        this.keyGlob = `${opts.keyPrefix || ''}*`;
        this.uri = opts.uri;
    }
    bind(task, opts) {
        if (this.bucket) {
            if (opts.forReading) {
                this.bucket.grantRead(task, this.keyGlob);
            }
            if (opts.forWriting) {
                this.bucket.grantWrite(task, this.keyGlob);
            }
        }
        else {
            const actions = new Array();
            if (opts.forReading) {
                actions.push('s3:GetObject', 's3:ListBucket');
            }
            if (opts.forWriting) {
                actions.push('s3:PutObject');
            }
            task.grantPrincipal.addToPolicy(new iam.PolicyStatement({ actions, resources: ['*'] }));
        }
        return { uri: this.uri };
    }
}
//# sourceMappingURL=data:application/json;base64,eyJ2ZXJzaW9uIjozLCJmaWxlIjoiYmFzZS10eXBlcy5qcyIsInNvdXJjZVJvb3QiOiIiLCJzb3VyY2VzIjpbImJhc2UtdHlwZXMudHMiXSwibmFtZXMiOltdLCJtYXBwaW5ncyI6Ijs7O0FBRUEsNERBQWtGLENBQUMsMERBQTBEO0FBQzdJLHdDQUF3QyxDQUFDLG1EQUFtRDtBQUc1RixrREFBa0QsQ0FBQyw2REFBNkQ7QUFvUGhIOzs7O0dBSUc7QUFDSCxNQUFzQixVQUFVO0lBQzVCOzs7OztPQUtHO0lBQ0ksTUFBTSxDQUFDLFVBQVUsQ0FBQyxNQUFrQixFQUFFLFNBQWlCO1FBQzFELE9BQU8sSUFBSSxrQkFBa0IsQ0FBQyxFQUFFLE1BQU0sRUFBRSxTQUFTLEVBQUUsR0FBRyxFQUFFLE1BQU0sQ0FBQyxZQUFZLENBQUMsU0FBUyxDQUFDLEVBQUUsQ0FBQyxDQUFDO0lBQzlGLENBQUM7SUFDRDs7Ozs7OztPQU9HO0lBQ0ksTUFBTSxDQUFDLGtCQUFrQixDQUFDLFVBQWtCO1FBQy9DLE9BQU8sSUFBSSxrQkFBa0IsQ0FBQyxFQUFFLEdBQUcsRUFBRSxHQUFHLENBQUMsSUFBSSxDQUFDLFFBQVEsQ0FBQyxVQUFVLENBQUMsRUFBRSxDQUFDLENBQUM7SUFDMUUsQ0FBQztDQUtKO0FBekJELGdDQXlCQztBQStCRDs7OztHQUlHO0FBQ0gsTUFBc0IsV0FBVztJQUM3Qjs7Ozs7T0FLRztJQUNJLE1BQU0sQ0FBQyxpQkFBaUIsQ0FBQyxVQUEyQixFQUFFLE1BQWMsUUFBUTtRQUMvRSxPQUFPLElBQUksbUJBQW1CLENBQUMsRUFBRSxVQUFVLEVBQUUsUUFBUSxFQUFFLFVBQVUsQ0FBQyxtQkFBbUIsQ0FBQyxHQUFHLENBQUMsRUFBRSxDQUFDLENBQUM7SUFDbEcsQ0FBQztJQUNEOzs7OztPQUtHO0lBQ0ksTUFBTSxDQUFDLGtCQUFrQixDQUFDLFVBQWtCLEVBQUUsb0JBQW9CLEdBQUcsSUFBSTtRQUM1RSxPQUFPLElBQUksbUJBQW1CLENBQUMsRUFBRSxRQUFRLEVBQUUsVUFBVSxFQUFFLG9CQUFvQixFQUFFLENBQUMsQ0FBQztJQUNuRixDQUFDO0lBQ0Q7Ozs7OztPQU1HO0lBQ0ksTUFBTSxDQUFDLFlBQVksQ0FBQyxRQUFnQjtRQUN2QyxPQUFPLElBQUksbUJBQW1CLENBQUMsRUFBRSxRQUFRLEVBQUUsQ0FBQyxDQUFDO0lBQ2pELENBQUM7SUFDRDs7Ozs7O09BTUc7SUFDSSxNQUFNLENBQUMsU0FBUyxDQUFDLEtBQWdCLEVBQUUsRUFBVSxFQUFFLEtBQTRCO1FBQzlFLE1BQU0sS0FBSyxHQUFHLElBQUksaUNBQWdCLENBQUMsS0FBSyxFQUFFLEVBQUUsRUFBRSxLQUFLLENBQUMsQ0FBQztRQUNyRCxPQUFPLElBQUksbUJBQW1CLENBQUMsRUFBRSxVQUFVLEVBQUUsS0FBSyxDQUFDLFVBQVUsRUFBRSxRQUFRLEVBQUUsS0FBSyxDQUFDLFFBQVEsRUFBRSxDQUFDLENBQUM7SUFDL0YsQ0FBQztDQUtKO0FBNUNELGtDQTRDQztBQUNEOzs7O0dBSUc7QUFDSCxJQUFZLFVBYVg7QUFiRCxXQUFZLFVBQVU7SUFDbEI7O09BRUc7SUFDSCw0Q0FBOEIsQ0FBQTtJQUM5Qjs7T0FFRztJQUNILG9DQUFzQixDQUFBO0lBQ3RCOztPQUVHO0lBQ0gsK0RBQWlELENBQUE7QUFDckQsQ0FBQyxFQWJXLFVBQVUsR0FBVixrQkFBVSxLQUFWLGtCQUFVLFFBYXJCO0FBQ0Q7Ozs7R0FJRztBQUNILElBQVksc0JBU1g7QUFURCxXQUFZLHNCQUFzQjtJQUM5Qjs7T0FFRztJQUNILDhEQUFvQyxDQUFBO0lBQ3BDOztPQUVHO0lBQ0gsOERBQW9DLENBQUE7QUFDeEMsQ0FBQyxFQVRXLHNCQUFzQixHQUF0Qiw4QkFBc0IsS0FBdEIsOEJBQXNCLFFBU2pDO0FBQ0Q7Ozs7R0FJRztBQUNILElBQVksaUJBU1g7QUFURCxXQUFZLGlCQUFpQjtJQUN6Qjs7T0FFRztJQUNILGtDQUFhLENBQUE7SUFDYjs7T0FFRztJQUNILDJDQUFzQixDQUFBO0FBQzFCLENBQUMsRUFUVyxpQkFBaUIsR0FBakIseUJBQWlCLEtBQWpCLHlCQUFpQixRQVM1QjtBQUNEOzs7O0dBSUc7QUFDSCxJQUFZLFNBU1g7QUFURCxXQUFZLFNBQVM7SUFDakI7O09BRUc7SUFDSCwwQkFBYSxDQUFBO0lBQ2I7O09BRUc7SUFDSCwwQkFBYSxDQUFBO0FBQ2pCLENBQUMsRUFUVyxTQUFTLEdBQVQsaUJBQVMsS0FBVCxpQkFBUyxRQVNwQjtBQUNEOzs7O0dBSUc7QUFDSCxJQUFZLGVBU1g7QUFURCxXQUFZLGVBQWU7SUFDdkI7O09BRUc7SUFDSCxnQ0FBYSxDQUFBO0lBQ2I7O09BRUc7SUFDSCxnQ0FBYSxDQUFBO0FBQ2pCLENBQUMsRUFUVyxlQUFlLEdBQWYsdUJBQWUsS0FBZix1QkFBZSxRQVMxQjtBQStHRDs7OztHQUlHO0FBQ0gsSUFBWSxhQVNYO0FBVEQsV0FBWSxhQUFhO0lBQ3JCOztPQUVHO0lBQ0gsNkNBQTRCLENBQUE7SUFDNUI7O09BRUc7SUFDSCwrQ0FBOEIsQ0FBQTtBQUNsQyxDQUFDLEVBVFcsYUFBYSxHQUFiLHFCQUFhLEtBQWIscUJBQWEsUUFTeEI7QUFDRDs7OztHQUlHO0FBQ0gsSUFBWSxTQWlCWDtBQWpCRCxXQUFZLFNBQVM7SUFDakI7O09BRUc7SUFDSCwwQkFBYSxDQUFBO0lBQ2I7O09BRUc7SUFDSCwwQkFBYSxDQUFBO0lBQ2I7O09BRUc7SUFDSCxtQ0FBc0IsQ0FBQTtJQUN0Qjs7T0FFRztJQUNILG1DQUFzQixDQUFBO0FBQzFCLENBQUMsRUFqQlcsU0FBUyxHQUFULGlCQUFTLEtBQVQsaUJBQVMsUUFpQnBCO0FBQ0Q7Ozs7R0FJRztBQUNILElBQVksWUFTWDtBQVRELFdBQVksWUFBWTtJQUNwQjs7T0FFRztJQUNILDZCQUFhLENBQUE7SUFDYjs7T0FFRztJQUNILDZCQUFhLENBQUE7QUFDakIsQ0FBQyxFQVRXLFlBQVksR0FBWixvQkFBWSxLQUFaLG9CQUFZLFFBU3ZCO0FBQ0QsTUFBTSxtQkFBb0IsU0FBUSxXQUFXO0lBSXpDLFlBQVksSUFJWDtRQUNHLEtBQUssRUFBRSxDQUFDO1FBQ1IsSUFBSSxDQUFDLG9CQUFvQixHQUFHLENBQUMsQ0FBQyxJQUFJLENBQUMsb0JBQW9CLENBQUM7UUFDeEQsSUFBSSxDQUFDLFFBQVEsR0FBRyxJQUFJLENBQUMsUUFBUSxDQUFDO1FBQzlCLElBQUksQ0FBQyxVQUFVLEdBQUcsSUFBSSxDQUFDLFVBQVUsQ0FBQztJQUN0QyxDQUFDO0lBQ00sSUFBSSxDQUFDLElBQW9CO1FBQzVCLElBQUksSUFBSSxDQUFDLFVBQVUsRUFBRTtZQUNqQixJQUFJLENBQUMsVUFBVSxDQUFDLFNBQVMsQ0FBQyxJQUFJLENBQUMsQ0FBQztTQUNuQztRQUNELElBQUksSUFBSSxDQUFDLG9CQUFvQixFQUFFO1lBQzNCLElBQUksQ0FBQyxjQUFjLENBQUMsV0FBVyxDQUFDLElBQUksR0FBRyxDQUFDLGVBQWUsQ0FBQztnQkFDcEQsT0FBTyxFQUFFO29CQUNMLGlDQUFpQztvQkFDakMsNEJBQTRCO29CQUM1QixtQkFBbUI7aUJBQ3RCO2dCQUNELFNBQVMsRUFBRSxDQUFDLEdBQUcsQ0FBQzthQUNuQixDQUFDLENBQUMsQ0FBQztTQUNQO1FBQ0QsT0FBTztZQUNILFFBQVEsRUFBRSxJQUFJLENBQUMsUUFBUTtTQUMxQixDQUFDO0lBQ04sQ0FBQztDQUNKO0FBQ0QsTUFBTSxrQkFBbUIsU0FBUSxVQUFVO0lBSXZDLFlBQVksSUFJWDtRQUNHLEtBQUssRUFBRSxDQUFDO1FBQ1IsSUFBSSxDQUFDLE1BQU0sR0FBRyxJQUFJLENBQUMsTUFBTSxDQUFDO1FBQzFCLElBQUksQ0FBQyxPQUFPLEdBQUcsR0FBRyxJQUFJLENBQUMsU0FBUyxJQUFJLEVBQUUsR0FBRyxDQUFDO1FBQzFDLElBQUksQ0FBQyxHQUFHLEdBQUcsSUFBSSxDQUFDLEdBQUcsQ0FBQztJQUN4QixDQUFDO0lBQ00sSUFBSSxDQUFDLElBQW9CLEVBQUUsSUFBMkI7UUFDekQsSUFBSSxJQUFJLENBQUMsTUFBTSxFQUFFO1lBQ2IsSUFBSSxJQUFJLENBQUMsVUFBVSxFQUFFO2dCQUNqQixJQUFJLENBQUMsTUFBTSxDQUFDLFNBQVMsQ0FBQyxJQUFJLEVBQUUsSUFBSSxDQUFDLE9BQU8sQ0FBQyxDQUFDO2FBQzdDO1lBQ0QsSUFBSSxJQUFJLENBQUMsVUFBVSxFQUFFO2dCQUNqQixJQUFJLENBQUMsTUFBTSxDQUFDLFVBQVUsQ0FBQyxJQUFJLEVBQUUsSUFBSSxDQUFDLE9BQU8sQ0FBQyxDQUFDO2FBQzlDO1NBQ0o7YUFDSTtZQUNELE1BQU0sT0FBTyxHQUFHLElBQUksS0FBSyxFQUFVLENBQUM7WUFDcEMsSUFBSSxJQUFJLENBQUMsVUFBVSxFQUFFO2dCQUNqQixPQUFPLENBQUMsSUFBSSxDQUFDLGNBQWMsRUFBRSxlQUFlLENBQUMsQ0FBQzthQUNqRDtZQUNELElBQUksSUFBSSxDQUFDLFVBQVUsRUFBRTtnQkFDakIsT0FBTyxDQUFDLElBQUksQ0FBQyxjQUFjLENBQUMsQ0FBQzthQUNoQztZQUNELElBQUksQ0FBQyxjQUFjLENBQUMsV0FBVyxDQUFDLElBQUksR0FBRyxDQUFDLGVBQWUsQ0FBQyxFQUFFLE9BQU8sRUFBRSxTQUFTLEVBQUUsQ0FBQyxHQUFHLENBQUMsRUFBRSxDQUFDLENBQUMsQ0FBQztTQUMzRjtRQUNELE9BQU8sRUFBRSxHQUFHLEVBQUUsSUFBSSxDQUFDLEdBQUcsRUFBRSxDQUFDO0lBQzdCLENBQUM7Q0FDSiIsInNvdXJjZXNDb250ZW50IjpbImltcG9ydCAqIGFzIGVjMiBmcm9tIFwiLi4vLi4vLi4vYXdzLWVjMlwiOyAvLyBBdXRvbWF0aWNhbGx5IHJlLXdyaXR0ZW4gZnJvbSAnQGF3cy1jZGsvYXdzLWVjMidcbmltcG9ydCAqIGFzIGVjciBmcm9tIFwiLi4vLi4vLi4vYXdzLWVjclwiOyAvLyBBdXRvbWF0aWNhbGx5IHJlLXdyaXR0ZW4gZnJvbSAnQGF3cy1jZGsvYXdzLWVjcidcbmltcG9ydCB7IERvY2tlckltYWdlQXNzZXQsIERvY2tlckltYWdlQXNzZXRQcm9wcyB9IGZyb20gXCIuLi8uLi8uLi9hd3MtZWNyLWFzc2V0c1wiOyAvLyBBdXRvbWF0aWNhbGx5IHJlLXdyaXR0ZW4gZnJvbSAnQGF3cy1jZGsvYXdzLWVjci1hc3NldHMnXG5pbXBvcnQgKiBhcyBpYW0gZnJvbSBcIi4uLy4uLy4uL2F3cy1pYW1cIjsgLy8gQXV0b21hdGljYWxseSByZS13cml0dGVuIGZyb20gJ0Bhd3MtY2RrL2F3cy1pYW0nXG5pbXBvcnQgKiBhcyBrbXMgZnJvbSBcIi4uLy4uLy4uL2F3cy1rbXNcIjsgLy8gQXV0b21hdGljYWxseSByZS13cml0dGVuIGZyb20gJ0Bhd3MtY2RrL2F3cy1rbXMnXG5pbXBvcnQgKiBhcyBzMyBmcm9tIFwiLi4vLi4vLi4vYXdzLXMzXCI7IC8vIEF1dG9tYXRpY2FsbHkgcmUtd3JpdHRlbiBmcm9tICdAYXdzLWNkay9hd3MtczMnXG5pbXBvcnQgKiBhcyBzZm4gZnJvbSBcIi4uLy4uLy4uL2F3cy1zdGVwZnVuY3Rpb25zXCI7IC8vIEF1dG9tYXRpY2FsbHkgcmUtd3JpdHRlbiBmcm9tICdAYXdzLWNkay9hd3Mtc3RlcGZ1bmN0aW9ucydcbmltcG9ydCB7IENvbnN0cnVjdCwgRHVyYXRpb24sIFNpemUgfSBmcm9tIFwiLi4vLi4vLi4vY29yZVwiOyAvLyBBdXRvbWF0aWNhbGx5IHJlLXdyaXR0ZW4gZnJvbSAnQGF3cy1jZGsvY29yZSdcbi8qKlxuICogVGFzayB0byB0cmFpbiBhIG1hY2hpbmUgbGVhcm5pbmcgbW9kZWwgdXNpbmcgQW1hem9uIFNhZ2VNYWtlclxuICogQGV4cGVyaW1lbnRhbFxuICovXG5leHBvcnQgaW50ZXJmYWNlIElTYWdlTWFrZXJUYXNrIGV4dGVuZHMgaWFtLklHcmFudGFibGUge1xufVxuLyoqXG4gKiBTcGVjaWZ5IHRoZSB0cmFpbmluZyBhbGdvcml0aG0gYW5kIGFsZ29yaXRobS1zcGVjaWZpYyBtZXRhZGF0YVxuICogQGV4cGVyaW1lbnRhbFxuICovXG5leHBvcnQgaW50ZXJmYWNlIEFsZ29yaXRobVNwZWNpZmljYXRpb24ge1xuICAgIC8qKlxuICAgICAqIE5hbWUgb2YgdGhlIGFsZ29yaXRobSByZXNvdXJjZSB0byB1c2UgZm9yIHRoZSB0cmFpbmluZyBqb2IuXG4gICAgICogVGhpcyBtdXN0IGJlIGFuIGFsZ29yaXRobSByZXNvdXJjZSB0aGF0IHlvdSBjcmVhdGVkIG9yIHN1YnNjcmliZSB0byBvbiBBV1MgTWFya2V0cGxhY2UuXG4gICAgICogSWYgeW91IHNwZWNpZnkgYSB2YWx1ZSBmb3IgdGhpcyBwYXJhbWV0ZXIsIHlvdSBjYW4ndCBzcGVjaWZ5IGEgdmFsdWUgZm9yIFRyYWluaW5nSW1hZ2UuXG4gICAgICpcbiAgICAgKiBAZGVmYXVsdCAtIE5vIGFsZ29yaXRobSBpcyBzcGVjaWZpZWRcbiAgICAgKi9cbiAgICByZWFkb25seSBhbGdvcml0aG1OYW1lPzogc3RyaW5nO1xuICAgIC8qKlxuICAgICAqIExpc3Qgb2YgbWV0cmljIGRlZmluaXRpb24gb2JqZWN0cy4gRWFjaCBvYmplY3Qgc3BlY2lmaWVzIHRoZSBtZXRyaWMgbmFtZSBhbmQgcmVndWxhciBleHByZXNzaW9ucyB1c2VkIHRvIHBhcnNlIGFsZ29yaXRobSBsb2dzLlxuICAgICAqXG4gICAgICogQGRlZmF1bHQgLSBObyBtZXRyaWNzXG4gICAgICovXG4gICAgcmVhZG9ubHkgbWV0cmljRGVmaW5pdGlvbnM/OiBNZXRyaWNEZWZpbml0aW9uW107XG4gICAgLyoqXG4gICAgICogUmVnaXN0cnkgcGF0aCBvZiB0aGUgRG9ja2VyIGltYWdlIHRoYXQgY29udGFpbnMgdGhlIHRyYWluaW5nIGFsZ29yaXRobS5cbiAgICAgKlxuICAgICAqIEBkZWZhdWx0IC0gTm8gRG9ja2VyIGltYWdlIGlzIHNwZWNpZmllZFxuICAgICAqL1xuICAgIHJlYWRvbmx5IHRyYWluaW5nSW1hZ2U/OiBEb2NrZXJJbWFnZTtcbiAgICAvKipcbiAgICAgKiBJbnB1dCBtb2RlIHRoYXQgdGhlIGFsZ29yaXRobSBzdXBwb3J0cy5cbiAgICAgKlxuICAgICAqIEBkZWZhdWx0ICdGaWxlJyBtb2RlXG4gICAgICovXG4gICAgcmVhZG9ubHkgdHJhaW5pbmdJbnB1dE1vZGU/OiBJbnB1dE1vZGU7XG59XG4vKipcbiAqICBEZXNjcmliZXMgdGhlIHRyYWluaW5nLCB2YWxpZGF0aW9uIG9yIHRlc3QgZGF0YXNldCBhbmQgdGhlIEFtYXpvbiBTMyBsb2NhdGlvbiB3aGVyZSBpdCBpcyBzdG9yZWQuXG4gKlxuICogQGV4cGVyaW1lbnRhbFxuICovXG5leHBvcnQgaW50ZXJmYWNlIENoYW5uZWwge1xuICAgIC8qKlxuICAgICAqIE5hbWUgb2YgdGhlIGNoYW5uZWxcbiAgICAgKi9cbiAgICByZWFkb25seSBjaGFubmVsTmFtZTogc3RyaW5nO1xuICAgIC8qKlxuICAgICAqIENvbXByZXNzaW9uIHR5cGUgaWYgdHJhaW5pbmcgZGF0YSBpcyBjb21wcmVzc2VkXG4gICAgICpcbiAgICAgKiBAZGVmYXVsdCAtIE5vbmVcbiAgICAgKi9cbiAgICByZWFkb25seSBjb21wcmVzc2lvblR5cGU/OiBDb21wcmVzc2lvblR5cGU7XG4gICAgLyoqXG4gICAgICogVGhlIE1JTUUgdHlwZSBvZiB0aGUgZGF0YS5cbiAgICAgKlxuICAgICAqIEBkZWZhdWx0IC0gTm9uZVxuICAgICAqL1xuICAgIHJlYWRvbmx5IGNvbnRlbnRUeXBlPzogc3RyaW5nO1xuICAgIC8qKlxuICAgICAqIExvY2F0aW9uIG9mIHRoZSBjaGFubmVsIGRhdGEuXG4gICAgICovXG4gICAgcmVhZG9ubHkgZGF0YVNvdXJjZTogRGF0YVNvdXJjZTtcbiAgICAvKipcbiAgICAgKiBJbnB1dCBtb2RlIHRvIHVzZSBmb3IgdGhlIGRhdGEgY2hhbm5lbCBpbiBhIHRyYWluaW5nIGpvYi5cbiAgICAgKlxuICAgICAqIEBkZWZhdWx0IC0gTm9uZVxuICAgICAqL1xuICAgIHJlYWRvbmx5IGlucHV0TW9kZT86IElucHV0TW9kZTtcbiAgICAvKipcbiAgICAgKiBTcGVjaWZ5IFJlY29yZElPIGFzIHRoZSB2YWx1ZSB3aGVuIGlucHV0IGRhdGEgaXMgaW4gcmF3IGZvcm1hdCBidXQgdGhlIHRyYWluaW5nIGFsZ29yaXRobSByZXF1aXJlcyB0aGUgUmVjb3JkSU8gZm9ybWF0LlxuICAgICAqIEluIHRoaXMgY2FzZSwgQW1hem9uIFNhZ2VNYWtlciB3cmFwcyBlYWNoIGluZGl2aWR1YWwgUzMgb2JqZWN0IGluIGEgUmVjb3JkSU8gcmVjb3JkLlxuICAgICAqIElmIHRoZSBpbnB1dCBkYXRhIGlzIGFscmVhZHkgaW4gUmVjb3JkSU8gZm9ybWF0LCB5b3UgZG9uJ3QgbmVlZCB0byBzZXQgdGhpcyBhdHRyaWJ1dGUuXG4gICAgICpcbiAgICAgKiBAZGVmYXVsdCAtIE5vbmVcbiAgICAgKi9cbiAgICByZWFkb25seSByZWNvcmRXcmFwcGVyVHlwZT86IFJlY29yZFdyYXBwZXJUeXBlO1xuICAgIC8qKlxuICAgICAqIFNodWZmbGUgY29uZmlnIG9wdGlvbiBmb3IgaW5wdXQgZGF0YSBpbiBhIGNoYW5uZWwuXG4gICAgICpcbiAgICAgKiBAZGVmYXVsdCAtIE5vbmVcbiAgICAgKi9cbiAgICByZWFkb25seSBzaHVmZmxlQ29uZmlnPzogU2h1ZmZsZUNvbmZpZztcbn1cbi8qKlxuICogQ29uZmlndXJhdGlvbiBmb3IgYSBzaHVmZmxlIG9wdGlvbiBmb3IgaW5wdXQgZGF0YSBpbiBhIGNoYW5uZWwuXG4gKlxuICogQGV4cGVyaW1lbnRhbFxuICovXG5leHBvcnQgaW50ZXJmYWNlIFNodWZmbGVDb25maWcge1xuICAgIC8qKlxuICAgICAqIERldGVybWluZXMgdGhlIHNodWZmbGluZyBvcmRlci5cbiAgICAgKi9cbiAgICByZWFkb25seSBzZWVkOiBudW1iZXI7XG59XG4vKipcbiAqIExvY2F0aW9uIG9mIHRoZSBjaGFubmVsIGRhdGEuXG4gKlxuICogQGV4cGVyaW1lbnRhbFxuICovXG5leHBvcnQgaW50ZXJmYWNlIERhdGFTb3VyY2Uge1xuICAgIC8qKlxuICAgICAqIFMzIGxvY2F0aW9uIG9mIHRoZSBkYXRhIHNvdXJjZSB0aGF0IGlzIGFzc29jaWF0ZWQgd2l0aCBhIGNoYW5uZWwuXG4gICAgICovXG4gICAgcmVhZG9ubHkgczNEYXRhU291cmNlOiBTM0RhdGFTb3VyY2U7XG59XG4vKipcbiAqIFMzIGxvY2F0aW9uIG9mIHRoZSBjaGFubmVsIGRhdGEuXG4gKlxuICogQHNlZSBodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vc2FnZW1ha2VyL2xhdGVzdC9BUElSZWZlcmVuY2UvQVBJX1MzRGF0YVNvdXJjZS5odG1sXG4gKlxuICogQGV4cGVyaW1lbnRhbFxuICovXG5leHBvcnQgaW50ZXJmYWNlIFMzRGF0YVNvdXJjZSB7XG4gICAgLyoqXG4gICAgICogTGlzdCBvZiBvbmUgb3IgbW9yZSBhdHRyaWJ1dGUgbmFtZXMgdG8gdXNlIHRoYXQgYXJlIGZvdW5kIGluIGEgc3BlY2lmaWVkIGF1Z21lbnRlZCBtYW5pZmVzdCBmaWxlLlxuICAgICAqXG4gICAgICogQGRlZmF1bHQgLSBObyBhdHRyaWJ1dGUgbmFtZXNcbiAgICAgKi9cbiAgICByZWFkb25seSBhdHRyaWJ1dGVOYW1lcz86IHN0cmluZ1tdO1xuICAgIC8qKlxuICAgICAqIFMzIERhdGEgRGlzdHJpYnV0aW9uIFR5cGVcbiAgICAgKlxuICAgICAqIEBkZWZhdWx0IC0gTm9uZVxuICAgICAqL1xuICAgIHJlYWRvbmx5IHMzRGF0YURpc3RyaWJ1dGlvblR5cGU/OiBTM0RhdGFEaXN0cmlidXRpb25UeXBlO1xuICAgIC8qKlxuICAgICAqIFMzIERhdGEgVHlwZVxuICAgICAqXG4gICAgICogQGRlZmF1bHQgUzNfUFJFRklYXG4gICAgICovXG4gICAgcmVhZG9ubHkgczNEYXRhVHlwZT86IFMzRGF0YVR5cGU7XG4gICAgLyoqXG4gICAgICogUzMgVXJpXG4gICAgICovXG4gICAgcmVhZG9ubHkgczNMb2NhdGlvbjogUzNMb2NhdGlvbjtcbn1cbi8qKlxuICogQ29uZmlndXJlcyB0aGUgUzMgYnVja2V0IHdoZXJlIFNhZ2VNYWtlciB3aWxsIHNhdmUgdGhlIHJlc3VsdCBvZiBtb2RlbCB0cmFpbmluZ1xuICogQGV4cGVyaW1lbnRhbFxuICovXG5leHBvcnQgaW50ZXJmYWNlIE91dHB1dERhdGFDb25maWcge1xuICAgIC8qKlxuICAgICAqIE9wdGlvbmFsIEtNUyBlbmNyeXB0aW9uIGtleSB0aGF0IEFtYXpvbiBTYWdlTWFrZXIgdXNlcyB0byBlbmNyeXB0IHRoZSBtb2RlbCBhcnRpZmFjdHMgYXQgcmVzdCB1c2luZyBBbWF6b24gUzMgc2VydmVyLXNpZGUgZW5jcnlwdGlvbi5cbiAgICAgKlxuICAgICAqIEBkZWZhdWx0IC0gQW1hem9uIFNhZ2VNYWtlciB1c2VzIHRoZSBkZWZhdWx0IEtNUyBrZXkgZm9yIEFtYXpvbiBTMyBmb3IgeW91ciByb2xlJ3MgYWNjb3VudFxuICAgICAqL1xuICAgIHJlYWRvbmx5IGVuY3J5cHRpb25LZXk/OiBrbXMuSUtleTtcbiAgICAvKipcbiAgICAgKiBJZGVudGlmaWVzIHRoZSBTMyBwYXRoIHdoZXJlIHlvdSB3YW50IEFtYXpvbiBTYWdlTWFrZXIgdG8gc3RvcmUgdGhlIG1vZGVsIGFydGlmYWN0cy5cbiAgICAgKi9cbiAgICByZWFkb25seSBzM091dHB1dExvY2F0aW9uOiBTM0xvY2F0aW9uO1xufVxuLyoqXG4gKiBTcGVjaWZpZXMgYSBsaW1pdCB0byBob3cgbG9uZyBhIG1vZGVsIHRyYWluaW5nIGpvYiBjYW4gcnVuLlxuICogV2hlbiB0aGUgam9iIHJlYWNoZXMgdGhlIHRpbWUgbGltaXQsIEFtYXpvbiBTYWdlTWFrZXIgZW5kcyB0aGUgdHJhaW5pbmcgam9iLlxuICpcbiAqIEBleHBlcmltZW50YWxcbiAqL1xuZXhwb3J0IGludGVyZmFjZSBTdG9wcGluZ0NvbmRpdGlvbiB7XG4gICAgLyoqXG4gICAgICogVGhlIG1heGltdW0gbGVuZ3RoIG9mIHRpbWUsIGluIHNlY29uZHMsIHRoYXQgdGhlIHRyYWluaW5nIG9yIGNvbXBpbGF0aW9uIGpvYiBjYW4gcnVuLlxuICAgICAqXG4gICAgICogQGRlZmF1bHQgLSAxIGhvdXJcbiAgICAgKi9cbiAgICByZWFkb25seSBtYXhSdW50aW1lPzogRHVyYXRpb247XG59XG4vKipcbiAqIFNwZWNpZmllcyB0aGUgcmVzb3VyY2VzLCBNTCBjb21wdXRlIGluc3RhbmNlcywgYW5kIE1MIHN0b3JhZ2Ugdm9sdW1lcyB0byBkZXBsb3kgZm9yIG1vZGVsIHRyYWluaW5nLlxuICpcbiAqIEBleHBlcmltZW50YWxcbiAqL1xuZXhwb3J0IGludGVyZmFjZSBSZXNvdXJjZUNvbmZpZyB7XG4gICAgLyoqXG4gICAgICogVGhlIG51bWJlciBvZiBNTCBjb21wdXRlIGluc3RhbmNlcyB0byB1c2UuXG4gICAgICpcbiAgICAgKiBAZGVmYXVsdCAxIGluc3RhbmNlLlxuICAgICAqL1xuICAgIHJlYWRvbmx5IGluc3RhbmNlQ291bnQ6IG51bWJlcjtcbiAgICAvKipcbiAgICAgKiBNTCBjb21wdXRlIGluc3RhbmNlIHR5cGUuXG4gICAgICpcbiAgICAgKiBAZGVmYXVsdCBpcyB0aGUgJ200LnhsYXJnZScgaW5zdGFuY2UgdHlwZS5cbiAgICAgKi9cbiAgICByZWFkb25seSBpbnN0YW5jZVR5cGU6IGVjMi5JbnN0YW5jZVR5cGU7XG4gICAgLyoqXG4gICAgICogS01TIGtleSB0aGF0IEFtYXpvbiBTYWdlTWFrZXIgdXNlcyB0byBlbmNyeXB0IGRhdGEgb24gdGhlIHN0b3JhZ2Ugdm9sdW1lIGF0dGFjaGVkIHRvIHRoZSBNTCBjb21wdXRlIGluc3RhbmNlKHMpIHRoYXQgcnVuIHRoZSB0cmFpbmluZyBqb2IuXG4gICAgICpcbiAgICAgKiBAZGVmYXVsdCAtIEFtYXpvbiBTYWdlTWFrZXIgdXNlcyB0aGUgZGVmYXVsdCBLTVMga2V5IGZvciBBbWF6b24gUzMgZm9yIHlvdXIgcm9sZSdzIGFjY291bnRcbiAgICAgKi9cbiAgICByZWFkb25seSB2b2x1bWVFbmNyeXB0aW9uS2V5Pzoga21zLklLZXk7XG4gICAgLyoqXG4gICAgICogU2l6ZSBvZiB0aGUgTUwgc3RvcmFnZSB2b2x1bWUgdGhhdCB5b3Ugd2FudCB0byBwcm92aXNpb24uXG4gICAgICpcbiAgICAgKiBAZGVmYXVsdCAxMCBHQiBFQlMgdm9sdW1lLlxuICAgICAqL1xuICAgIHJlYWRvbmx5IHZvbHVtZVNpemU6IFNpemU7XG59XG4vKipcbiAqIFNwZWNpZmllcyB0aGUgVlBDIHRoYXQgeW91IHdhbnQgeW91ciBBbWF6b24gU2FnZU1ha2VyIHRyYWluaW5nIGpvYiB0byBjb25uZWN0IHRvLlxuICpcbiAqIEBleHBlcmltZW50YWxcbiAqL1xuZXhwb3J0IGludGVyZmFjZSBWcGNDb25maWcge1xuICAgIC8qKlxuICAgICAqIFZQQ1xuICAgICAqL1xuICAgIHJlYWRvbmx5IHZwYzogZWMyLklWcGM7XG4gICAgLyoqXG4gICAgICogVlBDIHN1Ym5ldHMuXG4gICAgICpcbiAgICAgKiBAZGVmYXVsdCAtIFByaXZhdGUgU3VibmV0cyBhcmUgc2VsZWN0ZWRcbiAgICAgKi9cbiAgICByZWFkb25seSBzdWJuZXRzPzogZWMyLlN1Ym5ldFNlbGVjdGlvbjtcbn1cbi8qKlxuICogU3BlY2lmaWVzIHRoZSBtZXRyaWMgbmFtZSBhbmQgcmVndWxhciBleHByZXNzaW9ucyB1c2VkIHRvIHBhcnNlIGFsZ29yaXRobSBsb2dzLlxuICpcbiAqIEBleHBlcmltZW50YWxcbiAqL1xuZXhwb3J0IGludGVyZmFjZSBNZXRyaWNEZWZpbml0aW9uIHtcbiAgICAvKipcbiAgICAgKiBOYW1lIG9mIHRoZSBtZXRyaWMuXG4gICAgICovXG4gICAgcmVhZG9ubHkgbmFtZTogc3RyaW5nO1xuICAgIC8qKlxuICAgICAqIFJlZ3VsYXIgZXhwcmVzc2lvbiB0aGF0IHNlYXJjaGVzIHRoZSBvdXRwdXQgb2YgYSB0cmFpbmluZyBqb2IgYW5kIGdldHMgdGhlIHZhbHVlIG9mIHRoZSBtZXRyaWMuXG4gICAgICovXG4gICAgcmVhZG9ubHkgcmVnZXg6IHN0cmluZztcbn1cbi8qKlxuICogU3RvcmVzIGluZm9ybWF0aW9uIGFib3V0IHRoZSBsb2NhdGlvbiBvZiBhbiBvYmplY3QgaW4gQW1hem9uIFMzXG4gKlxuICogQGV4cGVyaW1lbnRhbFxuICovXG5leHBvcnQgaW50ZXJmYWNlIFMzTG9jYXRpb25Db25maWcge1xuICAgIC8qKlxuICAgICAqIFVuaXF1ZWx5IGlkZW50aWZpZXMgdGhlIHJlc291cmNlIGluIEFtYXpvbiBTM1xuICAgICAqL1xuICAgIHJlYWRvbmx5IHVyaTogc3RyaW5nO1xufVxuLyoqXG4gKiBDb25zdHJ1Y3RzIGBJUzNMb2NhdGlvbmAgb2JqZWN0cy5cbiAqXG4gKiBAZXhwZXJpbWVudGFsXG4gKi9cbmV4cG9ydCBhYnN0cmFjdCBjbGFzcyBTM0xvY2F0aW9uIHtcbiAgICAvKipcbiAgICAgKiBBbiBgSVMzTG9jYXRpb25gIGJ1aWx0IHdpdGggYSBkZXRlcm1pbmVkIGJ1Y2tldCBhbmQga2V5IHByZWZpeC5cbiAgICAgKlxuICAgICAqIEBwYXJhbSBidWNrZXQgICAgaXMgdGhlIGJ1Y2tldCB3aGVyZSB0aGUgb2JqZWN0cyBhcmUgdG8gYmUgc3RvcmVkLlxuICAgICAqIEBwYXJhbSBrZXlQcmVmaXggaXMgdGhlIGtleSBwcmVmaXggdXNlZCBieSB0aGUgbG9jYXRpb24uXG4gICAgICovXG4gICAgcHVibGljIHN0YXRpYyBmcm9tQnVja2V0KGJ1Y2tldDogczMuSUJ1Y2tldCwga2V5UHJlZml4OiBzdHJpbmcpOiBTM0xvY2F0aW9uIHtcbiAgICAgICAgcmV0dXJuIG5ldyBTdGFuZGFyZFMzTG9jYXRpb24oeyBidWNrZXQsIGtleVByZWZpeCwgdXJpOiBidWNrZXQudXJsRm9yT2JqZWN0KGtleVByZWZpeCkgfSk7XG4gICAgfVxuICAgIC8qKlxuICAgICAqIEFuIGBJUzNMb2NhdGlvbmAgZGV0ZXJtaW5lZCBmdWxseSBieSBhIEpTT04gUGF0aCBmcm9tIHRoZSB0YXNrIGlucHV0LlxuICAgICAqXG4gICAgICogRHVlIHRvIHRoZSBkeW5hbWljIG5hdHVyZSBvZiB0aG9zZSBsb2NhdGlvbnMsIHRoZSBJQU0gZ3JhbnRzIHRoYXQgd2lsbCBiZSBzZXQgYnkgYGdyYW50UmVhZGAgYW5kIGBncmFudFdyaXRlYFxuICAgICAqIGFwcGx5IHRvIHRoZSBgKmAgcmVzb3VyY2UuXG4gICAgICpcbiAgICAgKiBAcGFyYW0gZXhwcmVzc2lvbiB0aGUgSlNPTiBleHByZXNzaW9uIHJlc29sdmluZyB0byBhbiBTMyBsb2NhdGlvbiBVUkkuXG4gICAgICovXG4gICAgcHVibGljIHN0YXRpYyBmcm9tSnNvbkV4cHJlc3Npb24oZXhwcmVzc2lvbjogc3RyaW5nKTogUzNMb2NhdGlvbiB7XG4gICAgICAgIHJldHVybiBuZXcgU3RhbmRhcmRTM0xvY2F0aW9uKHsgdXJpOiBzZm4uRGF0YS5zdHJpbmdBdChleHByZXNzaW9uKSB9KTtcbiAgICB9XG4gICAgLyoqXG4gICAgICogQ2FsbGVkIHdoZW4gdGhlIFMzTG9jYXRpb24gaXMgYm91bmQgdG8gYSBTdGVwRnVuY3Rpb25zIHRhc2suXG4gICAgICovXG4gICAgcHVibGljIGFic3RyYWN0IGJpbmQodGFzazogSVNhZ2VNYWtlclRhc2ssIG9wdHM6IFMzTG9jYXRpb25CaW5kT3B0aW9ucyk6IFMzTG9jYXRpb25Db25maWc7XG59XG4vKipcbiAqIE9wdGlvbnMgZm9yIGJpbmRpbmcgYW4gUzMgTG9jYXRpb24uXG4gKlxuICogQGV4cGVyaW1lbnRhbFxuICovXG5leHBvcnQgaW50ZXJmYWNlIFMzTG9jYXRpb25CaW5kT3B0aW9ucyB7XG4gICAgLyoqXG4gICAgICogQWxsb3cgcmVhZGluZyBmcm9tIHRoZSBTMyBMb2NhdGlvbi5cbiAgICAgKlxuICAgICAqIEBkZWZhdWx0IGZhbHNlXG4gICAgICovXG4gICAgcmVhZG9ubHkgZm9yUmVhZGluZz86IGJvb2xlYW47XG4gICAgLyoqXG4gICAgICogQWxsb3cgd3JpdGluZyB0byB0aGUgUzMgTG9jYXRpb24uXG4gICAgICpcbiAgICAgKiBAZGVmYXVsdCBmYWxzZVxuICAgICAqL1xuICAgIHJlYWRvbmx5IGZvcldyaXRpbmc/OiBib29sZWFuO1xufVxuLyoqXG4gKiBDb25maWd1cmF0aW9uIGZvciBhIHVzaW5nIERvY2tlciBpbWFnZS5cbiAqXG4gKiBAZXhwZXJpbWVudGFsXG4gKi9cbmV4cG9ydCBpbnRlcmZhY2UgRG9ja2VySW1hZ2VDb25maWcge1xuICAgIC8qKlxuICAgICAqIFRoZSBmdWxseSBxdWFsaWZpZWQgVVJJIG9mIHRoZSBEb2NrZXIgaW1hZ2UuXG4gICAgICovXG4gICAgcmVhZG9ubHkgaW1hZ2VVcmk6IHN0cmluZztcbn1cbi8qKlxuICogQ3JlYXRlcyBgSURvY2tlckltYWdlYCBpbnN0YW5jZXMuXG4gKlxuICogQGV4cGVyaW1lbnRhbFxuICovXG5leHBvcnQgYWJzdHJhY3QgY2xhc3MgRG9ja2VySW1hZ2Uge1xuICAgIC8qKlxuICAgICAqIFJlZmVyZW5jZSBhIERvY2tlciBpbWFnZSBzdG9yZWQgaW4gYW4gRUNSIHJlcG9zaXRvcnkuXG4gICAgICpcbiAgICAgKiBAcGFyYW0gcmVwb3NpdG9yeSB0aGUgRUNSIHJlcG9zaXRvcnkgd2hlcmUgdGhlIGltYWdlIGlzIGhvc3RlZC5cbiAgICAgKiBAcGFyYW0gdGFnIGFuIG9wdGlvbmFsIGB0YWdgXG4gICAgICovXG4gICAgcHVibGljIHN0YXRpYyBmcm9tRWNyUmVwb3NpdG9yeShyZXBvc2l0b3J5OiBlY3IuSVJlcG9zaXRvcnksIHRhZzogc3RyaW5nID0gJ2xhdGVzdCcpOiBEb2NrZXJJbWFnZSB7XG4gICAgICAgIHJldHVybiBuZXcgU3RhbmRhcmREb2NrZXJJbWFnZSh7IHJlcG9zaXRvcnksIGltYWdlVXJpOiByZXBvc2l0b3J5LnJlcG9zaXRvcnlVcmlGb3JUYWcodGFnKSB9KTtcbiAgICB9XG4gICAgLyoqXG4gICAgICogUmVmZXJlbmNlIGEgRG9ja2VyIGltYWdlIHdoaWNoIFVSSSBpcyBvYnRhaW5lZCBmcm9tIHRoZSB0YXNrJ3MgaW5wdXQuXG4gICAgICpcbiAgICAgKiBAcGFyYW0gZXhwcmVzc2lvbiAgICAgICAgICAgdGhlIEpTT04gcGF0aCBleHByZXNzaW9uIHdpdGggdGhlIHRhc2sgaW5wdXQuXG4gICAgICogQHBhcmFtIGFsbG93QW55RWNySW1hZ2VQdWxsIHdoZXRoZXIgRUNSIGFjY2VzcyBzaG91bGQgYmUgcGVybWl0dGVkIChzZXQgdG8gYGZhbHNlYCBpZiB0aGUgaW1hZ2Ugd2lsbCBuZXZlciBiZSBpbiBFQ1IpLlxuICAgICAqL1xuICAgIHB1YmxpYyBzdGF0aWMgZnJvbUpzb25FeHByZXNzaW9uKGV4cHJlc3Npb246IHN0cmluZywgYWxsb3dBbnlFY3JJbWFnZVB1bGwgPSB0cnVlKTogRG9ja2VySW1hZ2Uge1xuICAgICAgICByZXR1cm4gbmV3IFN0YW5kYXJkRG9ja2VySW1hZ2UoeyBpbWFnZVVyaTogZXhwcmVzc2lvbiwgYWxsb3dBbnlFY3JJbWFnZVB1bGwgfSk7XG4gICAgfVxuICAgIC8qKlxuICAgICAqIFJlZmVyZW5jZSBhIERvY2tlciBpbWFnZSBieSBpdCdzIFVSSS5cbiAgICAgKlxuICAgICAqIFdoZW4gcmVmZXJlbmNpbmcgRUNSIGltYWdlcywgcHJlZmVyIHVzaW5nIGBpbkVjcmAuXG4gICAgICpcbiAgICAgKiBAcGFyYW0gaW1hZ2VVcmkgdGhlIFVSSSB0byB0aGUgZG9ja2VyIGltYWdlLlxuICAgICAqL1xuICAgIHB1YmxpYyBzdGF0aWMgZnJvbVJlZ2lzdHJ5KGltYWdlVXJpOiBzdHJpbmcpOiBEb2NrZXJJbWFnZSB7XG4gICAgICAgIHJldHVybiBuZXcgU3RhbmRhcmREb2NrZXJJbWFnZSh7IGltYWdlVXJpIH0pO1xuICAgIH1cbiAgICAvKipcbiAgICAgKiBSZWZlcmVuY2UgYSBEb2NrZXIgaW1hZ2UgdGhhdCBpcyBwcm92aWRlZCBhcyBhbiBBc3NldCBpbiB0aGUgY3VycmVudCBhcHAuXG4gICAgICpcbiAgICAgKiBAcGFyYW0gc2NvcGUgdGhlIHNjb3BlIGluIHdoaWNoIHRvIGNyZWF0ZSB0aGUgQXNzZXQuXG4gICAgICogQHBhcmFtIGlkICAgIHRoZSBJRCBmb3IgdGhlIGFzc2V0IGluIHRoZSBjb25zdHJ1Y3QgdHJlZS5cbiAgICAgKiBAcGFyYW0gcHJvcHMgdGhlIGNvbmZpZ3VyYXRpb24gcHJvcHMgb2YgdGhlIGFzc2V0LlxuICAgICAqL1xuICAgIHB1YmxpYyBzdGF0aWMgZnJvbUFzc2V0KHNjb3BlOiBDb25zdHJ1Y3QsIGlkOiBzdHJpbmcsIHByb3BzOiBEb2NrZXJJbWFnZUFzc2V0UHJvcHMpOiBEb2NrZXJJbWFnZSB7XG4gICAgICAgIGNvbnN0IGFzc2V0ID0gbmV3IERvY2tlckltYWdlQXNzZXQoc2NvcGUsIGlkLCBwcm9wcyk7XG4gICAgICAgIHJldHVybiBuZXcgU3RhbmRhcmREb2NrZXJJbWFnZSh7IHJlcG9zaXRvcnk6IGFzc2V0LnJlcG9zaXRvcnksIGltYWdlVXJpOiBhc3NldC5pbWFnZVVyaSB9KTtcbiAgICB9XG4gICAgLyoqXG4gICAgICogQ2FsbGVkIHdoZW4gdGhlIGltYWdlIGlzIHVzZWQgYnkgYSBTYWdlTWFrZXIgdGFzay5cbiAgICAgKi9cbiAgICBwdWJsaWMgYWJzdHJhY3QgYmluZCh0YXNrOiBJU2FnZU1ha2VyVGFzayk6IERvY2tlckltYWdlQ29uZmlnO1xufVxuLyoqXG4gKiBTMyBEYXRhIFR5cGUuXG4gKlxuICogQGV4cGVyaW1lbnRhbFxuICovXG5leHBvcnQgZW51bSBTM0RhdGFUeXBlIHtcbiAgICAvKipcbiAgICAgKiBNYW5pZmVzdCBGaWxlIERhdGEgVHlwZVxuICAgICAqL1xuICAgIE1BTklGRVNUX0ZJTEUgPSAnTWFuaWZlc3RGaWxlJyxcbiAgICAvKipcbiAgICAgKiBTMyBQcmVmaXggRGF0YSBUeXBlXG4gICAgICovXG4gICAgUzNfUFJFRklYID0gJ1MzUHJlZml4JyxcbiAgICAvKipcbiAgICAgKiBBdWdtZW50ZWQgTWFuaWZlc3QgRmlsZSBEYXRhIFR5cGVcbiAgICAgKi9cbiAgICBBVUdNRU5URURfTUFOSUZFU1RfRklMRSA9ICdBdWdtZW50ZWRNYW5pZmVzdEZpbGUnXG59XG4vKipcbiAqIFMzIERhdGEgRGlzdHJpYnV0aW9uIFR5cGUuXG4gKlxuICogQGV4cGVyaW1lbnRhbFxuICovXG5leHBvcnQgZW51bSBTM0RhdGFEaXN0cmlidXRpb25UeXBlIHtcbiAgICAvKipcbiAgICAgKiBGdWxseSByZXBsaWNhdGVkIFMzIERhdGEgRGlzdHJpYnV0aW9uIFR5cGVcbiAgICAgKi9cbiAgICBGVUxMWV9SRVBMSUNBVEVEID0gJ0Z1bGx5UmVwbGljYXRlZCcsXG4gICAgLyoqXG4gICAgICogU2hhcmRlZCBCeSBTMyBLZXkgRGF0YSBEaXN0cmlidXRpb24gVHlwZVxuICAgICAqL1xuICAgIFNIQVJERURfQllfUzNfS0VZID0gJ1NoYXJkZWRCeVMzS2V5J1xufVxuLyoqXG4gKiBEZWZpbmUgdGhlIGZvcm1hdCBvZiB0aGUgaW5wdXQgZGF0YS5cbiAqXG4gKiBAZXhwZXJpbWVudGFsXG4gKi9cbmV4cG9ydCBlbnVtIFJlY29yZFdyYXBwZXJUeXBlIHtcbiAgICAvKipcbiAgICAgKiBOb25lIHJlY29yZCB3cmFwcGVyIHR5cGVcbiAgICAgKi9cbiAgICBOT05FID0gJ05vbmUnLFxuICAgIC8qKlxuICAgICAqIFJlY29yZElPIHJlY29yZCB3cmFwcGVyIHR5cGVcbiAgICAgKi9cbiAgICBSRUNPUkRfSU8gPSAnUmVjb3JkSU8nXG59XG4vKipcbiAqICBJbnB1dCBtb2RlIHRoYXQgdGhlIGFsZ29yaXRobSBzdXBwb3J0cy5cbiAqXG4gKiBAZXhwZXJpbWVudGFsXG4gKi9cbmV4cG9ydCBlbnVtIElucHV0TW9kZSB7XG4gICAgLyoqXG4gICAgICogUGlwZSBtb2RlXG4gICAgICovXG4gICAgUElQRSA9ICdQaXBlJyxcbiAgICAvKipcbiAgICAgKiBGaWxlIG1vZGUuXG4gICAgICovXG4gICAgRklMRSA9ICdGaWxlJ1xufVxuLyoqXG4gKiBDb21wcmVzc2lvbiB0eXBlIG9mIHRoZSBkYXRhLlxuICpcbiAqIEBleHBlcmltZW50YWxcbiAqL1xuZXhwb3J0IGVudW0gQ29tcHJlc3Npb25UeXBlIHtcbiAgICAvKipcbiAgICAgKiBOb25lIGNvbXByZXNzaW9uIHR5cGVcbiAgICAgKi9cbiAgICBOT05FID0gJ05vbmUnLFxuICAgIC8qKlxuICAgICAqIEd6aXAgY29tcHJlc3Npb24gdHlwZVxuICAgICAqL1xuICAgIEdaSVAgPSAnR3ppcCdcbn1cbi8vXG4vLyBDcmVhdGUgVHJhbnNmb3JtIEpvYiB0eXBlc1xuLy9cbi8qKlxuICogIERhdGFzZXQgdG8gYmUgdHJhbnNmb3JtZWQgYW5kIHRoZSBBbWF6b24gUzMgbG9jYXRpb24gd2hlcmUgaXQgaXMgc3RvcmVkLlxuICpcbiAqICBAZXhwZXJpbWVudGFsXG4gKi9cbmV4cG9ydCBpbnRlcmZhY2UgVHJhbnNmb3JtSW5wdXQge1xuICAgIC8qKlxuICAgICAqIFRoZSBjb21wcmVzc2lvbiB0eXBlIG9mIHRoZSB0cmFuc2Zvcm0gZGF0YS5cbiAgICAgKlxuICAgICAqIEBkZWZhdWx0IE5PTkVcbiAgICAgKi9cbiAgICByZWFkb25seSBjb21wcmVzc2lvblR5cGU/OiBDb21wcmVzc2lvblR5cGU7XG4gICAgLyoqXG4gICAgICogTXVsdGlwdXJwb3NlIGludGVybmV0IG1haWwgZXh0ZW5zaW9uIChNSU1FKSB0eXBlIG9mIHRoZSBkYXRhLlxuICAgICAqXG4gICAgICogQGRlZmF1bHQgLSBOb25lXG4gICAgICovXG4gICAgcmVhZG9ubHkgY29udGVudFR5cGU/OiBzdHJpbmc7XG4gICAgLyoqXG4gICAgICogUzMgbG9jYXRpb24gb2YgdGhlIGNoYW5uZWwgZGF0YVxuICAgICAqL1xuICAgIHJlYWRvbmx5IHRyYW5zZm9ybURhdGFTb3VyY2U6IFRyYW5zZm9ybURhdGFTb3VyY2U7XG4gICAgLyoqXG4gICAgICogTWV0aG9kIHRvIHVzZSB0byBzcGxpdCB0aGUgdHJhbnNmb3JtIGpvYidzIGRhdGEgZmlsZXMgaW50byBzbWFsbGVyIGJhdGNoZXMuXG4gICAgICpcbiAgICAgKiBAZGVmYXVsdCBOT05FXG4gICAgICovXG4gICAgcmVhZG9ubHkgc3BsaXRUeXBlPzogU3BsaXRUeXBlO1xufVxuLyoqXG4gKiBTMyBsb2NhdGlvbiBvZiB0aGUgaW5wdXQgZGF0YSB0aGF0IHRoZSBtb2RlbCBjYW4gY29uc3VtZS5cbiAqXG4gKiAgQGV4cGVyaW1lbnRhbFxuICovXG5leHBvcnQgaW50ZXJmYWNlIFRyYW5zZm9ybURhdGFTb3VyY2Uge1xuICAgIC8qKlxuICAgICAqIFMzIGxvY2F0aW9uIG9mIHRoZSBpbnB1dCBkYXRhXG4gICAgICovXG4gICAgcmVhZG9ubHkgczNEYXRhU291cmNlOiBUcmFuc2Zvcm1TM0RhdGFTb3VyY2U7XG59XG4vKipcbiAqIExvY2F0aW9uIG9mIHRoZSBjaGFubmVsIGRhdGEuXG4gKlxuICogIEBleHBlcmltZW50YWxcbiAqL1xuZXhwb3J0IGludGVyZmFjZSBUcmFuc2Zvcm1TM0RhdGFTb3VyY2Uge1xuICAgIC8qKlxuICAgICAqIFMzIERhdGEgVHlwZVxuICAgICAqXG4gICAgICogQGRlZmF1bHQgJ1MzUHJlZml4J1xuICAgICAqL1xuICAgIHJlYWRvbmx5IHMzRGF0YVR5cGU/OiBTM0RhdGFUeXBlO1xuICAgIC8qKlxuICAgICAqIElkZW50aWZpZXMgZWl0aGVyIGEga2V5IG5hbWUgcHJlZml4IG9yIGEgbWFuaWZlc3QuXG4gICAgICovXG4gICAgcmVhZG9ubHkgczNVcmk6IHN0cmluZztcbn1cbi8qKlxuICogUzMgbG9jYXRpb24gd2hlcmUgeW91IHdhbnQgQW1hem9uIFNhZ2VNYWtlciB0byBzYXZlIHRoZSByZXN1bHRzIGZyb20gdGhlIHRyYW5zZm9ybSBqb2IuXG4gKlxuICogIEBleHBlcmltZW50YWxcbiAqL1xuZXhwb3J0IGludGVyZmFjZSBUcmFuc2Zvcm1PdXRwdXQge1xuICAgIC8qKlxuICAgICAqIE1JTUUgdHlwZSB1c2VkIHRvIHNwZWNpZnkgdGhlIG91dHB1dCBkYXRhLlxuICAgICAqXG4gICAgICogQGRlZmF1bHQgLSBOb25lXG4gICAgICovXG4gICAgcmVhZG9ubHkgYWNjZXB0Pzogc3RyaW5nO1xuICAgIC8qKlxuICAgICAqIERlZmluZXMgaG93IHRvIGFzc2VtYmxlIHRoZSByZXN1bHRzIG9mIHRoZSB0cmFuc2Zvcm0gam9iIGFzIGEgc2luZ2xlIFMzIG9iamVjdC5cbiAgICAgKlxuICAgICAqIEBkZWZhdWx0IC0gTm9uZVxuICAgICAqL1xuICAgIHJlYWRvbmx5IGFzc2VtYmxlV2l0aD86IEFzc2VtYmxlV2l0aDtcbiAgICAvKipcbiAgICAgKiBBV1MgS01TIGtleSB0aGF0IEFtYXpvbiBTYWdlTWFrZXIgdXNlcyB0byBlbmNyeXB0IHRoZSBtb2RlbCBhcnRpZmFjdHMgYXQgcmVzdCB1c2luZyBBbWF6b24gUzMgc2VydmVyLXNpZGUgZW5jcnlwdGlvbi5cbiAgICAgKlxuICAgICAqIEBkZWZhdWx0IC0gZGVmYXVsdCBLTVMga2V5IGZvciBBbWF6b24gUzMgZm9yIHlvdXIgcm9sZSdzIGFjY291bnQuXG4gICAgICovXG4gICAgcmVhZG9ubHkgZW5jcnlwdGlvbktleT86IGttcy5JS2V5O1xuICAgIC8qKlxuICAgICAqIFMzIHBhdGggd2hlcmUgeW91IHdhbnQgQW1hem9uIFNhZ2VNYWtlciB0byBzdG9yZSB0aGUgcmVzdWx0cyBvZiB0aGUgdHJhbnNmb3JtIGpvYi5cbiAgICAgKi9cbiAgICByZWFkb25seSBzM091dHB1dFBhdGg6IHN0cmluZztcbn1cbi8qKlxuICogTUwgY29tcHV0ZSBpbnN0YW5jZXMgZm9yIHRoZSB0cmFuc2Zvcm0gam9iLlxuICpcbiAqICBAZXhwZXJpbWVudGFsXG4gKi9cbmV4cG9ydCBpbnRlcmZhY2UgVHJhbnNmb3JtUmVzb3VyY2VzIHtcbiAgICAvKipcbiAgICAgKiBOdW1iZXIgb2YgTUwgY29tcHV0ZSBpbnN0YW5jZXMgdG8gdXNlIGluIHRoZSB0cmFuc2Zvcm0gam9iXG4gICAgICovXG4gICAgcmVhZG9ubHkgaW5zdGFuY2VDb3VudDogbnVtYmVyO1xuICAgIC8qKlxuICAgICAqIE1MIGNvbXB1dGUgaW5zdGFuY2UgdHlwZSBmb3IgdGhlIHRyYW5zZm9ybSBqb2IuXG4gICAgICovXG4gICAgcmVhZG9ubHkgaW5zdGFuY2VUeXBlOiBlYzIuSW5zdGFuY2VUeXBlO1xuICAgIC8qKlxuICAgICAqIEFXUyBLTVMga2V5IHRoYXQgQW1hem9uIFNhZ2VNYWtlciB1c2VzIHRvIGVuY3J5cHQgZGF0YSBvbiB0aGUgc3RvcmFnZSB2b2x1bWUgYXR0YWNoZWQgdG8gdGhlIE1MIGNvbXB1dGUgaW5zdGFuY2UocykuXG4gICAgICpcbiAgICAgKiBAZGVmYXVsdCAtIE5vbmVcbiAgICAgKi9cbiAgICByZWFkb25seSB2b2x1bWVFbmNyeXB0aW9uS2V5Pzoga21zLklLZXk7XG59XG4vKipcbiAqIFNwZWNpZmllcyB0aGUgbnVtYmVyIG9mIHJlY29yZHMgdG8gaW5jbHVkZSBpbiBhIG1pbmktYmF0Y2ggZm9yIGFuIEhUVFAgaW5mZXJlbmNlIHJlcXVlc3QuXG4gKlxuICogQGV4cGVyaW1lbnRhbFxuICovXG5leHBvcnQgZW51bSBCYXRjaFN0cmF0ZWd5IHtcbiAgICAvKipcbiAgICAgKiBGaXRzIG11bHRpcGxlIHJlY29yZHMgaW4gYSBtaW5pLWJhdGNoLlxuICAgICAqL1xuICAgIE1VTFRJX1JFQ09SRCA9ICdNdWx0aVJlY29yZCcsXG4gICAgLyoqXG4gICAgICogVXNlIGEgc2luZ2xlIHJlY29yZCB3aGVuIG1ha2luZyBhbiBpbnZvY2F0aW9uIHJlcXVlc3QuXG4gICAgICovXG4gICAgU0lOR0xFX1JFQ09SRCA9ICdTaW5nbGVSZWNvcmQnXG59XG4vKipcbiAqIE1ldGhvZCB0byB1c2UgdG8gc3BsaXQgdGhlIHRyYW5zZm9ybSBqb2IncyBkYXRhIGZpbGVzIGludG8gc21hbGxlciBiYXRjaGVzLlxuICpcbiAqIEBleHBlcmltZW50YWxcbiAqL1xuZXhwb3J0IGVudW0gU3BsaXRUeXBlIHtcbiAgICAvKipcbiAgICAgKiBJbnB1dCBkYXRhIGZpbGVzIGFyZSBub3Qgc3BsaXQsXG4gICAgICovXG4gICAgTk9ORSA9ICdOb25lJyxcbiAgICAvKipcbiAgICAgKiBTcGxpdCByZWNvcmRzIG9uIGEgbmV3bGluZSBjaGFyYWN0ZXIgYm91bmRhcnkuXG4gICAgICovXG4gICAgTElORSA9ICdMaW5lJyxcbiAgICAvKipcbiAgICAgKiBTcGxpdCB1c2luZyBNWE5ldCBSZWNvcmRJTyBmb3JtYXQuXG4gICAgICovXG4gICAgUkVDT1JEX0lPID0gJ1JlY29yZElPJyxcbiAgICAvKipcbiAgICAgKiBTcGxpdCB1c2luZyBUZW5zb3JGbG93IFRGUmVjb3JkIGZvcm1hdC5cbiAgICAgKi9cbiAgICBURl9SRUNPUkQgPSAnVEZSZWNvcmQnXG59XG4vKipcbiAqIEhvdyB0byBhc3NlbWJsZSB0aGUgcmVzdWx0cyBvZiB0aGUgdHJhbnNmb3JtIGpvYiBhcyBhIHNpbmdsZSBTMyBvYmplY3QuXG4gKlxuICogQGV4cGVyaW1lbnRhbFxuICovXG5leHBvcnQgZW51bSBBc3NlbWJsZVdpdGgge1xuICAgIC8qKlxuICAgICAqIENvbmNhdGVuYXRlIHRoZSByZXN1bHRzIGluIGJpbmFyeSBmb3JtYXQuXG4gICAgICovXG4gICAgTk9ORSA9ICdOb25lJyxcbiAgICAvKipcbiAgICAgKiBBZGQgYSBuZXdsaW5lIGNoYXJhY3RlciBhdCB0aGUgZW5kIG9mIGV2ZXJ5IHRyYW5zZm9ybWVkIHJlY29yZC5cbiAgICAgKi9cbiAgICBMSU5FID0gJ0xpbmUnXG59XG5jbGFzcyBTdGFuZGFyZERvY2tlckltYWdlIGV4dGVuZHMgRG9ja2VySW1hZ2Uge1xuICAgIHByaXZhdGUgcmVhZG9ubHkgYWxsb3dBbnlFY3JJbWFnZVB1bGw6IGJvb2xlYW47XG4gICAgcHJpdmF0ZSByZWFkb25seSBpbWFnZVVyaTogc3RyaW5nO1xuICAgIHByaXZhdGUgcmVhZG9ubHkgcmVwb3NpdG9yeT86IGVjci5JUmVwb3NpdG9yeTtcbiAgICBjb25zdHJ1Y3RvcihvcHRzOiB7XG4gICAgICAgIGFsbG93QW55RWNySW1hZ2VQdWxsPzogYm9vbGVhbjtcbiAgICAgICAgaW1hZ2VVcmk6IHN0cmluZztcbiAgICAgICAgcmVwb3NpdG9yeT86IGVjci5JUmVwb3NpdG9yeTtcbiAgICB9KSB7XG4gICAgICAgIHN1cGVyKCk7XG4gICAgICAgIHRoaXMuYWxsb3dBbnlFY3JJbWFnZVB1bGwgPSAhIW9wdHMuYWxsb3dBbnlFY3JJbWFnZVB1bGw7XG4gICAgICAgIHRoaXMuaW1hZ2VVcmkgPSBvcHRzLmltYWdlVXJpO1xuICAgICAgICB0aGlzLnJlcG9zaXRvcnkgPSBvcHRzLnJlcG9zaXRvcnk7XG4gICAgfVxuICAgIHB1YmxpYyBiaW5kKHRhc2s6IElTYWdlTWFrZXJUYXNrKTogRG9ja2VySW1hZ2VDb25maWcge1xuICAgICAgICBpZiAodGhpcy5yZXBvc2l0b3J5KSB7XG4gICAgICAgICAgICB0aGlzLnJlcG9zaXRvcnkuZ3JhbnRQdWxsKHRhc2spO1xuICAgICAgICB9XG4gICAgICAgIGlmICh0aGlzLmFsbG93QW55RWNySW1hZ2VQdWxsKSB7XG4gICAgICAgICAgICB0YXNrLmdyYW50UHJpbmNpcGFsLmFkZFRvUG9saWN5KG5ldyBpYW0uUG9saWN5U3RhdGVtZW50KHtcbiAgICAgICAgICAgICAgICBhY3Rpb25zOiBbXG4gICAgICAgICAgICAgICAgICAgICdlY3I6QmF0Y2hDaGVja0xheWVyQXZhaWxhYmlsaXR5JyxcbiAgICAgICAgICAgICAgICAgICAgJ2VjcjpHZXREb3dubG9hZFVybEZvckxheWVyJyxcbiAgICAgICAgICAgICAgICAgICAgJ2VjcjpCYXRjaEdldEltYWdlJyxcbiAgICAgICAgICAgICAgICBdLFxuICAgICAgICAgICAgICAgIHJlc291cmNlczogWycqJ10sXG4gICAgICAgICAgICB9KSk7XG4gICAgICAgIH1cbiAgICAgICAgcmV0dXJuIHtcbiAgICAgICAgICAgIGltYWdlVXJpOiB0aGlzLmltYWdlVXJpLFxuICAgICAgICB9O1xuICAgIH1cbn1cbmNsYXNzIFN0YW5kYXJkUzNMb2NhdGlvbiBleHRlbmRzIFMzTG9jYXRpb24ge1xuICAgIHByaXZhdGUgcmVhZG9ubHkgYnVja2V0PzogczMuSUJ1Y2tldDtcbiAgICBwcml2YXRlIHJlYWRvbmx5IGtleUdsb2I6IHN0cmluZztcbiAgICBwcml2YXRlIHJlYWRvbmx5IHVyaTogc3RyaW5nO1xuICAgIGNvbnN0cnVjdG9yKG9wdHM6IHtcbiAgICAgICAgYnVja2V0PzogczMuSUJ1Y2tldDtcbiAgICAgICAga2V5UHJlZml4Pzogc3RyaW5nO1xuICAgICAgICB1cmk6IHN0cmluZztcbiAgICB9KSB7XG4gICAgICAgIHN1cGVyKCk7XG4gICAgICAgIHRoaXMuYnVja2V0ID0gb3B0cy5idWNrZXQ7XG4gICAgICAgIHRoaXMua2V5R2xvYiA9IGAke29wdHMua2V5UHJlZml4IHx8ICcnfSpgO1xuICAgICAgICB0aGlzLnVyaSA9IG9wdHMudXJpO1xuICAgIH1cbiAgICBwdWJsaWMgYmluZCh0YXNrOiBJU2FnZU1ha2VyVGFzaywgb3B0czogUzNMb2NhdGlvbkJpbmRPcHRpb25zKTogUzNMb2NhdGlvbkNvbmZpZyB7XG4gICAgICAgIGlmICh0aGlzLmJ1Y2tldCkge1xuICAgICAgICAgICAgaWYgKG9wdHMuZm9yUmVhZGluZykge1xuICAgICAgICAgICAgICAgIHRoaXMuYnVja2V0LmdyYW50UmVhZCh0YXNrLCB0aGlzLmtleUdsb2IpO1xuICAgICAgICAgICAgfVxuICAgICAgICAgICAgaWYgKG9wdHMuZm9yV3JpdGluZykge1xuICAgICAgICAgICAgICAgIHRoaXMuYnVja2V0LmdyYW50V3JpdGUodGFzaywgdGhpcy5rZXlHbG9iKTtcbiAgICAgICAgICAgIH1cbiAgICAgICAgfVxuICAgICAgICBlbHNlIHtcbiAgICAgICAgICAgIGNvbnN0IGFjdGlvbnMgPSBuZXcgQXJyYXk8c3RyaW5nPigpO1xuICAgICAgICAgICAgaWYgKG9wdHMuZm9yUmVhZGluZykge1xuICAgICAgICAgICAgICAgIGFjdGlvbnMucHVzaCgnczM6R2V0T2JqZWN0JywgJ3MzOkxpc3RCdWNrZXQnKTtcbiAgICAgICAgICAgIH1cbiAgICAgICAgICAgIGlmIChvcHRzLmZvcldyaXRpbmcpIHtcbiAgICAgICAgICAgICAgICBhY3Rpb25zLnB1c2goJ3MzOlB1dE9iamVjdCcpO1xuICAgICAgICAgICAgfVxuICAgICAgICAgICAgdGFzay5ncmFudFByaW5jaXBhbC5hZGRUb1BvbGljeShuZXcgaWFtLlBvbGljeVN0YXRlbWVudCh7IGFjdGlvbnMsIHJlc291cmNlczogWycqJ10gfSkpO1xuICAgICAgICB9XG4gICAgICAgIHJldHVybiB7IHVyaTogdGhpcy51cmkgfTtcbiAgICB9XG59XG4iXX0=