"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.BatchSubmitJob = void 0;
const iam = require("../../../aws-iam"); // Automatically re-written from '@aws-cdk/aws-iam'
const sfn = require("../../../aws-stepfunctions"); // Automatically re-written from '@aws-cdk/aws-stepfunctions'
const core_1 = require("../../../core"); // Automatically re-written from '@aws-cdk/core'
const task_utils_1 = require("../private/task-utils");
/**
 * Task to submits an AWS Batch job from a job definition.
 *
 * @see https://docs.aws.amazon.com/step-functions/latest/dg/connect-batch.html
 */
class BatchSubmitJob extends sfn.TaskStateBase {
    constructor(scope, id, props) {
        var _a, _b;
        super(scope, id, props);
        this.props = props;
        this.integrationPattern = (_a = props.integrationPattern) !== null && _a !== void 0 ? _a : sfn.IntegrationPattern.RUN_JOB;
        task_utils_1.validatePatternSupported(this.integrationPattern, BatchSubmitJob.SUPPORTED_INTEGRATION_PATTERNS);
        // validate arraySize limits
        core_1.withResolved(props.arraySize, (arraySize) => {
            if (arraySize !== undefined && (arraySize < 2 || arraySize > 10000)) {
                throw new Error(`arraySize must be between 2 and 10,000. Received ${arraySize}.`);
            }
        });
        // validate dependency size
        if (props.dependsOn && props.dependsOn.length > 20) {
            throw new Error(`dependencies must be 20 or less. Received ${props.dependsOn.length}.`);
        }
        // validate attempts
        core_1.withResolved(props.attempts, (attempts) => {
            if (attempts !== undefined && (attempts < 1 || attempts > 10)) {
                throw new Error(`attempts must be between 1 and 10. Received ${attempts}.`);
            }
        });
        // validate timeout
        // tslint:disable-next-line:no-unused-expression
        props.timeout !== undefined && core_1.withResolved(props.timeout.toSeconds(), (timeout) => {
            if (timeout < 60) {
                throw new Error(`attempt duration must be greater than 60 seconds. Received ${timeout} seconds.`);
            }
        });
        // This is required since environment variables must not start with AWS_BATCH;
        // this naming convention is reserved for variables that are set by the AWS Batch service.
        if ((_b = props.containerOverrides) === null || _b === void 0 ? void 0 : _b.environment) {
            Object.keys(props.containerOverrides.environment).forEach(key => {
                if (key.match(/^AWS_BATCH/)) {
                    throw new Error(`Invalid environment variable name: ${key}. Environment variable names starting with 'AWS_BATCH' are reserved.`);
                }
            });
        }
        this.taskPolicies = this.configurePolicyStatements();
    }
    renderTask() {
        var _a;
        return {
            Resource: task_utils_1.integrationResourceArn('batch', 'submitJob', this.integrationPattern),
            Parameters: sfn.FieldUtils.renderObject({
                JobDefinition: this.props.jobDefinition.jobDefinitionArn,
                JobName: this.props.jobName,
                JobQueue: this.props.jobQueue.jobQueueArn,
                Parameters: (_a = this.props.payload) === null || _a === void 0 ? void 0 : _a.value,
                ArrayProperties: this.props.arraySize !== undefined
                    ? { Size: this.props.arraySize }
                    : undefined,
                ContainerOverrides: this.props.containerOverrides
                    ? this.configureContainerOverrides(this.props.containerOverrides)
                    : undefined,
                DependsOn: this.props.dependsOn
                    ? this.props.dependsOn.map(jobDependency => ({
                        JobId: jobDependency.jobId,
                        Type: jobDependency.type,
                    }))
                    : undefined,
                RetryStrategy: this.props.attempts !== undefined
                    ? { Attempts: this.props.attempts }
                    : undefined,
                Timeout: this.props.timeout
                    ? { AttemptDurationSeconds: this.props.timeout.toSeconds() }
                    : undefined,
            }),
            TimeoutSeconds: undefined,
        };
    }
    configurePolicyStatements() {
        return [
            // Resource level access control for job-definition requires revision which batch does not support yet
            // Using the alternative permissions as mentioned here:
            // https://docs.aws.amazon.com/batch/latest/userguide/batch-supported-iam-actions-resources.html
            new iam.PolicyStatement({
                resources: [
                    core_1.Stack.of(this).formatArn({
                        service: 'batch',
                        resource: 'job-definition',
                        resourceName: '*',
                    }),
                    this.props.jobQueue.jobQueueArn,
                ],
                actions: ['batch:SubmitJob'],
            }),
            new iam.PolicyStatement({
                resources: [
                    core_1.Stack.of(this).formatArn({
                        service: 'events',
                        resource: 'rule/StepFunctionsGetEventsForBatchJobsRule',
                    }),
                ],
                actions: ['events:PutTargets', 'events:PutRule', 'events:DescribeRule'],
            }),
        ];
    }
    configureContainerOverrides(containerOverrides) {
        var _a, _b;
        let environment;
        if (containerOverrides.environment) {
            environment = Object.entries(containerOverrides.environment).map(([key, value]) => ({
                Name: key,
                Value: value,
            }));
        }
        let resources;
        if (containerOverrides.gpuCount) {
            resources = [
                {
                    Type: 'GPU',
                    Value: `${containerOverrides.gpuCount}`,
                },
            ];
        }
        return {
            Command: containerOverrides.command,
            Environment: environment,
            InstanceType: (_a = containerOverrides.instanceType) === null || _a === void 0 ? void 0 : _a.toString(),
            Memory: (_b = containerOverrides.memory) === null || _b === void 0 ? void 0 : _b.toMebibytes(),
            ResourceRequirements: resources,
            Vcpus: containerOverrides.vcpus,
        };
    }
}
exports.BatchSubmitJob = BatchSubmitJob;
BatchSubmitJob.SUPPORTED_INTEGRATION_PATTERNS = [
    sfn.IntegrationPattern.REQUEST_RESPONSE,
    sfn.IntegrationPattern.RUN_JOB,
];
//# sourceMappingURL=data:application/json;base64,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