"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.Expires = exports.StorageClass = exports.ServerSideEncryption = exports.CacheControl = exports.BucketDeployment = void 0;
const iam = require("../../aws-iam"); // Automatically re-written from '@aws-cdk/aws-iam'
const lambda = require("../../aws-lambda"); // Automatically re-written from '@aws-cdk/aws-lambda'
const cdk = require("../../core"); // Automatically re-written from '@aws-cdk/core'
const crypto = require("crypto");
const fs = require("fs");
const path = require("path");
const now = Date.now();
const handlerCodeBundle = path.join(__dirname, '..', 'lambda', 'bundle.zip');
const handlerSourceDirectory = path.join(__dirname, '..', 'lambda', 'src');
class BucketDeployment extends cdk.Construct {
    constructor(scope, id, props) {
        super(scope, id);
        if (props.distributionPaths && !props.distribution) {
            throw new Error('Distribution must be specified if distribution paths are specified');
        }
        const assetHash = calcSourceHash(handlerSourceDirectory);
        const handler = new lambda.SingletonFunction(this, 'CustomResourceHandler', {
            uuid: this.renderSingletonUuid(props.memoryLimit),
            code: lambda.Code.fromAsset(handlerCodeBundle, { assetHash }),
            runtime: lambda.Runtime.PYTHON_3_6,
            handler: 'index.handler',
            lambdaPurpose: 'Custom::CDKBucketDeployment',
            timeout: cdk.Duration.minutes(15),
            role: props.role,
            memorySize: props.memoryLimit,
        });
        const handlerRole = handler.role;
        if (!handlerRole) {
            throw new Error('lambda.SingletonFunction should have created a Role');
        }
        const sources = props.sources.map((source) => source.bind(this, { handlerRole }));
        props.destinationBucket.grantReadWrite(handler);
        if (props.distribution) {
            handler.addToRolePolicy(new iam.PolicyStatement({
                effect: iam.Effect.ALLOW,
                actions: ['cloudfront:GetInvalidation', 'cloudfront:CreateInvalidation'],
                resources: ['*'],
            }));
        }
        new cdk.CustomResource(this, 'CustomResource', {
            serviceToken: handler.functionArn,
            resourceType: 'Custom::CDKBucketDeployment',
            properties: {
                SourceBucketNames: sources.map(source => source.bucket.bucketName),
                SourceObjectKeys: sources.map(source => source.zipObjectKey),
                DestinationBucketName: props.destinationBucket.bucketName,
                DestinationBucketKeyPrefix: props.destinationKeyPrefix,
                RetainOnDelete: props.retainOnDelete,
                UserMetadata: props.metadata ? mapUserMetadata(props.metadata) : undefined,
                SystemMetadata: mapSystemMetadata(props),
                DistributionId: props.distribution ? props.distribution.distributionId : undefined,
                DistributionPaths: props.distributionPaths,
            },
        });
    }
    renderSingletonUuid(memoryLimit) {
        let uuid = '8693BB64-9689-44B6-9AAF-B0CC9EB8756C';
        // if user specify a custom memory limit, define another singleton handler
        // with this configuration. otherwise, it won't be possible to use multiple
        // configurations since we have a singleton.
        if (memoryLimit) {
            if (cdk.Token.isUnresolved(memoryLimit)) {
                throw new Error('Can\'t use tokens when specifying "memoryLimit" since we use it to identify the singleton custom resource handler');
            }
            uuid += `-${memoryLimit.toString()}MiB`;
        }
        return uuid;
    }
}
exports.BucketDeployment = BucketDeployment;
/**
 * We need a custom source hash calculation since the bundle.zip file
 * contains python dependencies installed during build and results in a
 * non-deterministic behavior.
 *
 * So we just take the `src/` directory of our custom resoruce code.
 */
function calcSourceHash(srcDir) {
    const sha = crypto.createHash('sha256');
    for (const file of fs.readdirSync(srcDir)) {
        const data = fs.readFileSync(path.join(srcDir, file));
        sha.update(`<file name=${file}>`);
        sha.update(data);
        sha.update('</file>');
    }
    return sha.digest('hex');
}
/**
 * Metadata
 */
function mapUserMetadata(metadata) {
    const mapKey = (key) => key.toLowerCase().startsWith('x-amzn-meta-')
        ? key.toLowerCase()
        : `x-amzn-meta-${key.toLowerCase()}`;
    return Object.keys(metadata).reduce((o, key) => ({ ...o, [mapKey(key)]: metadata[key] }), {});
}
function mapSystemMetadata(metadata) {
    const res = {};
    if (metadata.cacheControl) {
        res['cache-control'] = metadata.cacheControl.map(c => c.value).join(', ');
    }
    if (metadata.expires) {
        res.expires = metadata.expires.value;
    }
    if (metadata.contentDisposition) {
        res['content-disposition'] = metadata.contentDisposition;
    }
    if (metadata.contentEncoding) {
        res['content-encoding'] = metadata.contentEncoding;
    }
    if (metadata.contentLanguage) {
        res['content-language'] = metadata.contentLanguage;
    }
    if (metadata.contentType) {
        res['content-type'] = metadata.contentType;
    }
    if (metadata.serverSideEncryption) {
        res.sse = metadata.serverSideEncryption;
    }
    if (metadata.storageClass) {
        res['storage-class'] = metadata.storageClass;
    }
    if (metadata.websiteRedirectLocation) {
        res['website-redirect'] = metadata.websiteRedirectLocation;
    }
    if (metadata.serverSideEncryptionAwsKmsKeyId) {
        res['sse-kms-key-id'] = metadata.serverSideEncryptionAwsKmsKeyId;
    }
    if (metadata.serverSideEncryptionCustomerAlgorithm) {
        res['sse-c-copy-source'] = metadata.serverSideEncryptionCustomerAlgorithm;
    }
    return Object.keys(res).length === 0 ? undefined : res;
}
/**
 * Used for HTTP cache-control header, which influences downstream caches.
 * @see https://docs.aws.amazon.com/AmazonS3/latest/dev/UsingMetadata.html#SysMetadata
 */
class CacheControl {
    constructor(value) {
        this.value = value;
    }
    static mustRevalidate() { return new CacheControl('must-revalidate'); }
    static noCache() { return new CacheControl('no-cache'); }
    static noTransform() { return new CacheControl('no-transform'); }
    static setPublic() { return new CacheControl('public'); }
    static setPrivate() { return new CacheControl('private'); }
    static proxyRevalidate() { return new CacheControl('proxy-revalidate'); }
    static maxAge(t) { return new CacheControl(`max-age=${t.toSeconds()}`); }
    static sMaxAge(t) { return new CacheControl(`s-maxage=${t.toSeconds()}`); }
    static fromString(s) { return new CacheControl(s); }
}
exports.CacheControl = CacheControl;
/**
 * Indicates whether server-side encryption is enabled for the object, and whether that encryption is
 * from the AWS Key Management Service (AWS KMS) or from Amazon S3 managed encryption (SSE-S3).
 * @see https://docs.aws.amazon.com/AmazonS3/latest/dev/UsingMetadata.html#SysMetadata
 */
var ServerSideEncryption;
(function (ServerSideEncryption) {
    ServerSideEncryption["AES_256"] = "AES256";
    ServerSideEncryption["AWS_KMS"] = "aws:kms";
})(ServerSideEncryption = exports.ServerSideEncryption || (exports.ServerSideEncryption = {}));
/**
 * Storage class used for storing the object.
 * @see https://docs.aws.amazon.com/AmazonS3/latest/dev/UsingMetadata.html#SysMetadata
 */
var StorageClass;
(function (StorageClass) {
    StorageClass["STANDARD"] = "STANDARD";
    StorageClass["REDUCED_REDUNDANCY"] = "REDUCED_REDUNDANCY";
    StorageClass["STANDARD_IA"] = "STANDARD_IA";
    StorageClass["ONEZONE_IA"] = "ONEZONE_IA";
    StorageClass["INTELLIGENT_TIERING"] = "INTELLIGENT_TIERING";
    StorageClass["GLACIER"] = "GLACIER";
    StorageClass["DEEP_ARCHIVE"] = "DEEP_ARCHIVE";
})(StorageClass = exports.StorageClass || (exports.StorageClass = {}));
/**
 * Used for HTTP expires header, which influences downstream caches. Does NOT influence deletion of the object.
 * @see https://docs.aws.amazon.com/AmazonS3/latest/dev/UsingMetadata.html#SysMetadata
 */
class Expires {
    constructor(value) {
        this.value = value;
    }
    /**
     * Expire at the specified date
     * @param d date to expire at
     */
    static atDate(d) { return new Expires(d.toUTCString()); }
    /**
     * Expire at the specified timestamp
     * @param t timestamp in unix milliseconds
     */
    static atTimestamp(t) { return Expires.atDate(new Date(t)); }
    /**
     * Expire once the specified duration has passed since deployment time
     * @param t the duration to wait before expiring
     */
    static after(t) { return Expires.atDate(new Date(now + t.toMilliseconds())); }
    static fromString(s) { return new Expires(s); }
}
exports.Expires = Expires;
//# sourceMappingURL=data:application/json;base64,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