"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.findGitPath = exports.findPkgPath = exports.nodeMajorVersion = exports.parseStackTrace = void 0;
const fs = require("fs");
const path = require("path");
// From https://github.com/errwischt/stacktrace-parser/blob/master/src/stack-trace-parser.js
const STACK_RE = /^\s*at (?:((?:\[object object\])?[^\\/]+(?: \[as \S+\])?) )?\(?(.*?):(\d+)(?::(\d+))?\)?\s*$/i;
/**
 * Parses the stack trace of an error
 */
function parseStackTrace(error) {
    const err = error || new Error();
    if (!err.stack) {
        return [];
    }
    const lines = err.stack.split('\n');
    const stackTrace = [];
    for (const line of lines) {
        const results = STACK_RE.exec(line);
        if (results) {
            stackTrace.push({
                file: results[2],
                methodName: results[1],
                lineNumber: parseInt(results[3], 10),
                column: parseInt(results[4], 10),
            });
        }
    }
    return stackTrace;
}
exports.parseStackTrace = parseStackTrace;
/**
 * Returns the major version of node installation
 */
function nodeMajorVersion() {
    return parseInt(process.versions.node.split('.')[0], 10);
}
exports.nodeMajorVersion = nodeMajorVersion;
/**
 * Finds the closest path containg a path
 */
function findClosestPathContaining(p) {
    for (const nodeModulesPath of module.paths) {
        if (fs.existsSync(path.join(path.dirname(nodeModulesPath), p))) {
            return path.dirname(nodeModulesPath);
        }
    }
    return undefined;
}
/**
 * Finds closest package.json path
 */
function findPkgPath() {
    return findClosestPathContaining('package.json');
}
exports.findPkgPath = findPkgPath;
/**
 * Finds closest .git/
 */
function findGitPath() {
    return findClosestPathContaining(`.git${path.sep}`);
}
exports.findGitPath = findGitPath;
//# sourceMappingURL=data:application/json;base64,eyJ2ZXJzaW9uIjozLCJmaWxlIjoidXRpbC5qcyIsInNvdXJjZVJvb3QiOiIiLCJzb3VyY2VzIjpbInV0aWwudHMiXSwibmFtZXMiOltdLCJtYXBwaW5ncyI6Ijs7O0FBQUEseUJBQXlCO0FBQ3pCLDZCQUE2QjtBQUM3Qiw0RkFBNEY7QUFDNUYsTUFBTSxRQUFRLEdBQUcsK0ZBQStGLENBQUM7QUFVakg7O0dBRUc7QUFDSCxTQUFnQixlQUFlLENBQUMsS0FBYTtJQUN6QyxNQUFNLEdBQUcsR0FBRyxLQUFLLElBQUksSUFBSSxLQUFLLEVBQUUsQ0FBQztJQUNqQyxJQUFJLENBQUMsR0FBRyxDQUFDLEtBQUssRUFBRTtRQUNaLE9BQU8sRUFBRSxDQUFDO0tBQ2I7SUFDRCxNQUFNLEtBQUssR0FBRyxHQUFHLENBQUMsS0FBSyxDQUFDLEtBQUssQ0FBQyxJQUFJLENBQUMsQ0FBQztJQUNwQyxNQUFNLFVBQVUsR0FBaUIsRUFBRSxDQUFDO0lBQ3BDLEtBQUssTUFBTSxJQUFJLElBQUksS0FBSyxFQUFFO1FBQ3RCLE1BQU0sT0FBTyxHQUFHLFFBQVEsQ0FBQyxJQUFJLENBQUMsSUFBSSxDQUFDLENBQUM7UUFDcEMsSUFBSSxPQUFPLEVBQUU7WUFDVCxVQUFVLENBQUMsSUFBSSxDQUFDO2dCQUNaLElBQUksRUFBRSxPQUFPLENBQUMsQ0FBQyxDQUFDO2dCQUNoQixVQUFVLEVBQUUsT0FBTyxDQUFDLENBQUMsQ0FBQztnQkFDdEIsVUFBVSxFQUFFLFFBQVEsQ0FBQyxPQUFPLENBQUMsQ0FBQyxDQUFDLEVBQUUsRUFBRSxDQUFDO2dCQUNwQyxNQUFNLEVBQUUsUUFBUSxDQUFDLE9BQU8sQ0FBQyxDQUFDLENBQUMsRUFBRSxFQUFFLENBQUM7YUFDbkMsQ0FBQyxDQUFDO1NBQ047S0FDSjtJQUNELE9BQU8sVUFBVSxDQUFDO0FBQ3RCLENBQUM7QUFuQkQsMENBbUJDO0FBQ0Q7O0dBRUc7QUFDSCxTQUFnQixnQkFBZ0I7SUFDNUIsT0FBTyxRQUFRLENBQUMsT0FBTyxDQUFDLFFBQVEsQ0FBQyxJQUFJLENBQUMsS0FBSyxDQUFDLEdBQUcsQ0FBQyxDQUFDLENBQUMsQ0FBQyxFQUFFLEVBQUUsQ0FBQyxDQUFDO0FBQzdELENBQUM7QUFGRCw0Q0FFQztBQUNEOztHQUVHO0FBQ0gsU0FBUyx5QkFBeUIsQ0FBQyxDQUFTO0lBQ3hDLEtBQUssTUFBTSxlQUFlLElBQUksTUFBTSxDQUFDLEtBQUssRUFBRTtRQUN4QyxJQUFJLEVBQUUsQ0FBQyxVQUFVLENBQUMsSUFBSSxDQUFDLElBQUksQ0FBQyxJQUFJLENBQUMsT0FBTyxDQUFDLGVBQWUsQ0FBQyxFQUFFLENBQUMsQ0FBQyxDQUFDLEVBQUU7WUFDNUQsT0FBTyxJQUFJLENBQUMsT0FBTyxDQUFDLGVBQWUsQ0FBQyxDQUFDO1NBQ3hDO0tBQ0o7SUFDRCxPQUFPLFNBQVMsQ0FBQztBQUNyQixDQUFDO0FBQ0Q7O0dBRUc7QUFDSCxTQUFnQixXQUFXO0lBQ3ZCLE9BQU8seUJBQXlCLENBQUMsY0FBYyxDQUFDLENBQUM7QUFDckQsQ0FBQztBQUZELGtDQUVDO0FBQ0Q7O0dBRUc7QUFDSCxTQUFnQixXQUFXO0lBQ3ZCLE9BQU8seUJBQXlCLENBQUMsT0FBTyxJQUFJLENBQUMsR0FBRyxFQUFFLENBQUMsQ0FBQztBQUN4RCxDQUFDO0FBRkQsa0NBRUMiLCJzb3VyY2VzQ29udGVudCI6WyJpbXBvcnQgKiBhcyBmcyBmcm9tICdmcyc7XG5pbXBvcnQgKiBhcyBwYXRoIGZyb20gJ3BhdGgnO1xuLy8gRnJvbSBodHRwczovL2dpdGh1Yi5jb20vZXJyd2lzY2h0L3N0YWNrdHJhY2UtcGFyc2VyL2Jsb2IvbWFzdGVyL3NyYy9zdGFjay10cmFjZS1wYXJzZXIuanNcbmNvbnN0IFNUQUNLX1JFID0gL15cXHMqYXQgKD86KCg/OlxcW29iamVjdCBvYmplY3RcXF0pP1teXFxcXC9dKyg/OiBcXFthcyBcXFMrXFxdKT8pICk/XFwoPyguKj8pOihcXGQrKSg/OjooXFxkKykpP1xcKT9cXHMqJC9pO1xuLyoqXG4gKiBBIHBhcnNlZCBzdGFjayB0cmFjZSBsaW5lXG4gKi9cbmV4cG9ydCBpbnRlcmZhY2UgU3RhY2tUcmFjZSB7XG4gICAgcmVhZG9ubHkgZmlsZTogc3RyaW5nO1xuICAgIHJlYWRvbmx5IG1ldGhvZE5hbWU/OiBzdHJpbmc7XG4gICAgcmVhZG9ubHkgbGluZU51bWJlcjogbnVtYmVyO1xuICAgIHJlYWRvbmx5IGNvbHVtbjogbnVtYmVyO1xufVxuLyoqXG4gKiBQYXJzZXMgdGhlIHN0YWNrIHRyYWNlIG9mIGFuIGVycm9yXG4gKi9cbmV4cG9ydCBmdW5jdGlvbiBwYXJzZVN0YWNrVHJhY2UoZXJyb3I/OiBFcnJvcik6IFN0YWNrVHJhY2VbXSB7XG4gICAgY29uc3QgZXJyID0gZXJyb3IgfHwgbmV3IEVycm9yKCk7XG4gICAgaWYgKCFlcnIuc3RhY2spIHtcbiAgICAgICAgcmV0dXJuIFtdO1xuICAgIH1cbiAgICBjb25zdCBsaW5lcyA9IGVyci5zdGFjay5zcGxpdCgnXFxuJyk7XG4gICAgY29uc3Qgc3RhY2tUcmFjZTogU3RhY2tUcmFjZVtdID0gW107XG4gICAgZm9yIChjb25zdCBsaW5lIG9mIGxpbmVzKSB7XG4gICAgICAgIGNvbnN0IHJlc3VsdHMgPSBTVEFDS19SRS5leGVjKGxpbmUpO1xuICAgICAgICBpZiAocmVzdWx0cykge1xuICAgICAgICAgICAgc3RhY2tUcmFjZS5wdXNoKHtcbiAgICAgICAgICAgICAgICBmaWxlOiByZXN1bHRzWzJdLFxuICAgICAgICAgICAgICAgIG1ldGhvZE5hbWU6IHJlc3VsdHNbMV0sXG4gICAgICAgICAgICAgICAgbGluZU51bWJlcjogcGFyc2VJbnQocmVzdWx0c1szXSwgMTApLFxuICAgICAgICAgICAgICAgIGNvbHVtbjogcGFyc2VJbnQocmVzdWx0c1s0XSwgMTApLFxuICAgICAgICAgICAgfSk7XG4gICAgICAgIH1cbiAgICB9XG4gICAgcmV0dXJuIHN0YWNrVHJhY2U7XG59XG4vKipcbiAqIFJldHVybnMgdGhlIG1ham9yIHZlcnNpb24gb2Ygbm9kZSBpbnN0YWxsYXRpb25cbiAqL1xuZXhwb3J0IGZ1bmN0aW9uIG5vZGVNYWpvclZlcnNpb24oKTogbnVtYmVyIHtcbiAgICByZXR1cm4gcGFyc2VJbnQocHJvY2Vzcy52ZXJzaW9ucy5ub2RlLnNwbGl0KCcuJylbMF0sIDEwKTtcbn1cbi8qKlxuICogRmluZHMgdGhlIGNsb3Nlc3QgcGF0aCBjb250YWluZyBhIHBhdGhcbiAqL1xuZnVuY3Rpb24gZmluZENsb3Nlc3RQYXRoQ29udGFpbmluZyhwOiBzdHJpbmcpOiBzdHJpbmcgfCB1bmRlZmluZWQge1xuICAgIGZvciAoY29uc3Qgbm9kZU1vZHVsZXNQYXRoIG9mIG1vZHVsZS5wYXRocykge1xuICAgICAgICBpZiAoZnMuZXhpc3RzU3luYyhwYXRoLmpvaW4ocGF0aC5kaXJuYW1lKG5vZGVNb2R1bGVzUGF0aCksIHApKSkge1xuICAgICAgICAgICAgcmV0dXJuIHBhdGguZGlybmFtZShub2RlTW9kdWxlc1BhdGgpO1xuICAgICAgICB9XG4gICAgfVxuICAgIHJldHVybiB1bmRlZmluZWQ7XG59XG4vKipcbiAqIEZpbmRzIGNsb3Nlc3QgcGFja2FnZS5qc29uIHBhdGhcbiAqL1xuZXhwb3J0IGZ1bmN0aW9uIGZpbmRQa2dQYXRoKCk6IHN0cmluZyB8IHVuZGVmaW5lZCB7XG4gICAgcmV0dXJuIGZpbmRDbG9zZXN0UGF0aENvbnRhaW5pbmcoJ3BhY2thZ2UuanNvbicpO1xufVxuLyoqXG4gKiBGaW5kcyBjbG9zZXN0IC5naXQvXG4gKi9cbmV4cG9ydCBmdW5jdGlvbiBmaW5kR2l0UGF0aCgpOiBzdHJpbmcgfCB1bmRlZmluZWQge1xuICAgIHJldHVybiBmaW5kQ2xvc2VzdFBhdGhDb250YWluaW5nKGAuZ2l0JHtwYXRoLnNlcH1gKTtcbn1cbiJdfQ==