"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.Bundling = void 0;
const lambda = require("../../aws-lambda"); // Automatically re-written from '@aws-cdk/aws-lambda'
const cdk = require("../../core"); // Automatically re-written from '@aws-cdk/core'
const fs = require("fs");
const path = require("path");
const util_1 = require("./util");
/**
 * Parcel code
 */
class Bundling {
    static parcel(options) {
        var _a;
        // Original package.json path and content
        let pkgPath = util_1.findPkgPath();
        if (!pkgPath) {
            throw new Error('Cannot find a `package.json` in this project.');
        }
        pkgPath = path.join(pkgPath, 'package.json');
        const originalPkg = fs.readFileSync(pkgPath);
        const originalPkgJson = JSON.parse(originalPkg.toString());
        // Update engines.node in package.json to set the right Babel target
        setEngines(options.nodeVersion, pkgPath, originalPkgJson);
        // Entry file path relative to container path
        const containerEntryPath = path.join(cdk.AssetStaging.BUNDLING_INPUT_DIR, path.relative(options.projectRoot, path.resolve(options.entry)));
        try {
            const command = [
                'parcel', 'build', containerEntryPath.replace(/\\/g, '/'),
                '--out-dir', cdk.AssetStaging.BUNDLING_OUTPUT_DIR,
                '--out-file', 'index.js',
                '--global', options.global,
                '--target', 'node',
                '--bundle-node-modules',
                '--log-level', '2',
                !options.minify && '--no-minify',
                !options.sourceMaps && '--no-source-maps',
                ...(options.cacheDir ? ['--cache-dir', '/parcel-cache'] : []),
            ].filter(Boolean);
            return lambda.Code.fromAsset(options.projectRoot, {
                assetHashType: cdk.AssetHashType.BUNDLE,
                bundling: {
                    image: cdk.BundlingDockerImage.fromAsset(path.join(__dirname, '../parcel-bundler'), {
                        buildArgs: {
                            NODE_TAG: (_a = options.nodeDockerTag) !== null && _a !== void 0 ? _a : `${process.versions.node}-alpine`,
                        },
                    }),
                    environment: options.environment,
                    volumes: options.cacheDir
                        ? [{ containerPath: '/parcel-cache', hostPath: options.cacheDir }]
                        : [],
                    workingDirectory: path.dirname(containerEntryPath).replace(/\\/g, '/'),
                    command,
                },
            });
        }
        finally {
            restorePkg(pkgPath, originalPkg);
        }
    }
}
exports.Bundling = Bundling;
function setEngines(nodeVersion, pkgPath, originalPkgJson) {
    // Update engines.node (Babel target)
    const updateData = {
        engines: {
            node: `>= ${nodeVersion}`,
        },
    };
    // Write new package.json
    if (Object.keys(updateData).length !== 0) {
        fs.writeFileSync(pkgPath, JSON.stringify({
            ...originalPkgJson,
            ...updateData,
        }, null, 2));
    }
}
function restorePkg(pkgPath, originalPkg) {
    fs.writeFileSync(pkgPath, originalPkg);
}
//# sourceMappingURL=data:application/json;base64,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