"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.KinesisEventSource = void 0;
const stream_1 = require("./stream");
/**
 * Use an Amazon Kinesis stream as an event source for AWS Lambda.
 */
class KinesisEventSource extends stream_1.StreamEventSource {
    constructor(stream, props) {
        super(props);
        this.stream = stream;
        this._eventSourceMappingId = undefined;
        if (this.props.batchSize !== undefined && (this.props.batchSize < 1 || this.props.batchSize > 10000)) {
            throw new Error(`Maximum batch size must be between 1 and 10000 inclusive (given ${this.props.batchSize})`);
        }
    }
    bind(target) {
        const eventSourceMapping = target.addEventSourceMapping(`KinesisEventSource:${this.stream.node.uniqueId}`, this.enrichMappingOptions({ eventSourceArn: this.stream.streamArn }));
        this._eventSourceMappingId = eventSourceMapping.eventSourceMappingId;
        this.stream.grantRead(target);
        // The `grantRead` API provides all the permissions recommended by the Kinesis team for reading a stream.
        // `DescribeStream` permissions are not required to read a stream as it's covered by the `DescribeStreamSummary`
        // and `SubscribeToShard` APIs.
        // The Lambda::EventSourceMapping resource validates against the `DescribeStream` permission. So we add it explicitly.
        // FIXME This permission can be removed when the event source mapping resource drops it from validation.
        this.stream.grant(target, 'kinesis:DescribeStream');
    }
    /**
     * The identifier for this EventSourceMapping
     */
    get eventSourceMappingId() {
        if (!this._eventSourceMappingId) {
            throw new Error('KinesisEventSource is not yet bound to an event source mapping');
        }
        return this._eventSourceMappingId;
    }
}
exports.KinesisEventSource = KinesisEventSource;
//# sourceMappingURL=data:application/json;base64,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