"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.ListenerCondition = void 0;
/**
 * ListenerCondition providers definition.
 */
class ListenerCondition {
    /**
     * Create a host-header listener rule condition
     *
     * @param values Hosts for host headers
     */
    static hostHeaders(values) {
        return new HostHeaderListenerCondition(values);
    }
    /**
     * Create a http-header listener rule condition
     *
     * @param name HTTP header name
     * @param values HTTP header values
     */
    static httpHeader(name, values) {
        return new HttpHeaderListenerCondition(name, values);
    }
    /**
     * Create a http-request-method listener rule condition
     *
     * @param values HTTP request methods
     */
    static httpRequestMethods(values) {
        return new HttpRequestMethodListenerCondition(values);
    }
    /**
     * Create a path-pattern listener rule condition
     *
     * @param values Path patterns
     */
    static pathPatterns(values) {
        return new PathPatternListenerCondition(values);
    }
    /**
     * Create a query-string listener rule condition
     *
     * @param values Query string key/value pairs
     */
    static queryStrings(values) {
        return new QueryStringListenerCondition(values);
    }
    /**
     * Create a source-ip listener rule condition
     *
     * @param values Source ips
     */
    static sourceIps(values) {
        return new SourceIpListenerCondition(values);
    }
}
exports.ListenerCondition = ListenerCondition;
/**
 * Host header config of the listener rule condition
 */
class HostHeaderListenerCondition extends ListenerCondition {
    constructor(values) {
        super();
        this.values = values;
    }
    renderRawCondition() {
        return {
            field: 'host-header',
            hostHeaderConfig: {
                values: this.values,
            },
        };
    }
}
/**
 * HTTP header config of the listener rule condition
 */
class HttpHeaderListenerCondition extends ListenerCondition {
    constructor(name, values) {
        super();
        this.name = name;
        this.values = values;
    }
    renderRawCondition() {
        return {
            field: 'http-header',
            httpHeaderConfig: {
                httpHeaderName: this.name,
                values: this.values,
            },
        };
    }
}
/**
 * HTTP reqeust method config of the listener rule condition
 */
class HttpRequestMethodListenerCondition extends ListenerCondition {
    constructor(values) {
        super();
        this.values = values;
    }
    renderRawCondition() {
        return {
            field: 'http-request-method',
            httpRequestMethodConfig: {
                values: this.values,
            },
        };
    }
}
/**
 * Path pattern config of the listener rule condition
 */
class PathPatternListenerCondition extends ListenerCondition {
    constructor(values) {
        super();
        this.values = values;
    }
    renderRawCondition() {
        return {
            field: 'path-pattern',
            pathPatternConfig: {
                values: this.values,
            },
        };
    }
}
/**
 * Query string config of the listener rule condition
 */
class QueryStringListenerCondition extends ListenerCondition {
    constructor(values) {
        super();
        this.values = values;
    }
    renderRawCondition() {
        return {
            field: 'query-string',
            queryStringConfig: {
                values: this.values,
            },
        };
    }
}
/**
 * Source ip config of the listener rule condition
 */
class SourceIpListenerCondition extends ListenerCondition {
    constructor(values) {
        super();
        this.values = values;
    }
    renderRawCondition() {
        return {
            field: 'source-ip',
            sourceIpConfig: {
                values: this.values,
            },
        };
    }
}
//# sourceMappingURL=data:application/json;base64,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