"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.HttpCodeTarget = exports.HttpCodeElb = exports.ApplicationLoadBalancer = void 0;
const cloudwatch = require("../../../aws-cloudwatch"); // Automatically re-written from '@aws-cdk/aws-cloudwatch'
const ec2 = require("../../../aws-ec2"); // Automatically re-written from '@aws-cdk/aws-ec2'
const core_1 = require("../../../core"); // Automatically re-written from '@aws-cdk/core'
const base_load_balancer_1 = require("../shared/base-load-balancer");
const application_listener_1 = require("./application-listener");
/**
 * Define an Application Load Balancer
 *
 * @resource AWS::ElasticLoadBalancingV2::LoadBalancer
 */
class ApplicationLoadBalancer extends base_load_balancer_1.BaseLoadBalancer {
    constructor(scope, id, props) {
        super(scope, id, props, {
            type: 'application',
            securityGroups: core_1.Lazy.listValue({ produce: () => [this.securityGroup.securityGroupId] }),
            ipAddressType: props.ipAddressType,
        });
        this.securityGroup = props.securityGroup || new ec2.SecurityGroup(this, 'SecurityGroup', {
            vpc: props.vpc,
            description: `Automatically created Security Group for ELB ${this.node.uniqueId}`,
            allowAllOutbound: false,
        });
        this.connections = new ec2.Connections({ securityGroups: [this.securityGroup] });
        if (props.http2Enabled === false) {
            this.setAttribute('routing.http2.enabled', 'false');
        }
        if (props.idleTimeout !== undefined) {
            this.setAttribute('idle_timeout.timeout_seconds', props.idleTimeout.toSeconds().toString());
        }
    }
    /**
     * Import an existing Application Load Balancer
     */
    static fromApplicationLoadBalancerAttributes(scope, id, attrs) {
        return new ImportedApplicationLoadBalancer(scope, id, attrs);
    }
    /**
     * Add a new listener to this load balancer
     */
    addListener(id, props) {
        return new application_listener_1.ApplicationListener(this, id, {
            loadBalancer: this,
            ...props,
        });
    }
    /**
     * Return the given named metric for this Application Load Balancer
     *
     * @default Average over 5 minutes
     */
    metric(metricName, props) {
        return new cloudwatch.Metric({
            namespace: 'AWS/ApplicationELB',
            metricName,
            dimensions: { LoadBalancer: this.loadBalancerFullName },
            ...props,
        });
    }
    /**
     * The total number of concurrent TCP connections active from clients to the
     * load balancer and from the load balancer to targets.
     *
     * @default Sum over 5 minutes
     */
    metricActiveConnectionCount(props) {
        return this.metric('ActiveConnectionCount', {
            statistic: 'sum',
            ...props,
        });
    }
    /**
     * The number of TLS connections initiated by the client that did not
     * establish a session with the load balancer. Possible causes include a
     * mismatch of ciphers or protocols.
     *
     * @default Sum over 5 minutes
     */
    metricClientTlsNegotiationErrorCount(props) {
        return this.metric('ClientTLSNegotiationErrorCount', {
            statistic: 'sum',
            ...props,
        });
    }
    /**
     * The number of load balancer capacity units (LCU) used by your load balancer.
     *
     * @default Sum over 5 minutes
     */
    metricConsumedLCUs(props) {
        return this.metric('ConsumedLCUs', {
            statistic: 'sum',
            ...props,
        });
    }
    /**
     * The number of fixed-response actions that were successful.
     *
     * @default Sum over 5 minutes
     */
    metricHttpFixedResponseCount(props) {
        return this.metric('HTTP_Fixed_Response_Count', {
            statistic: 'Sum',
            ...props,
        });
    }
    /**
     * The number of redirect actions that were successful.
     *
     * @default Sum over 5 minutes
     */
    metricHttpRedirectCount(props) {
        return this.metric('HTTP_Redirect_Count', {
            statistic: 'Sum',
            ...props,
        });
    }
    /**
     * The number of redirect actions that couldn't be completed because the URL
     * in the response location header is larger than 8K.
     *
     * @default Sum over 5 minutes
     */
    metricHttpRedirectUrlLimitExceededCount(props) {
        return this.metric('HTTP_Redirect_Url_Limit_Exceeded_Count', {
            statistic: 'Sum',
            ...props,
        });
    }
    /**
     * The number of HTTP 3xx/4xx/5xx codes that originate from the load balancer.
     *
     * This does not include any response codes generated by the targets.
     *
     * @default Sum over 5 minutes
     */
    metricHttpCodeElb(code, props) {
        return this.metric(code, {
            statistic: 'Sum',
            ...props,
        });
    }
    /**
     * The number of HTTP 2xx/3xx/4xx/5xx response codes generated by all targets
     * in the load balancer.
     *
     * This does not include any response codes generated by the load balancer.
     *
     * @default Sum over 5 minutes
     */
    metricHttpCodeTarget(code, props) {
        return this.metric(code, {
            statistic: 'Sum',
            ...props,
        });
    }
    /**
     * The total number of bytes processed by the load balancer over IPv6.
     *
     * @default Sum over 5 minutes
     */
    metricIpv6ProcessedBytes(props) {
        return this.metric('IPv6ProcessedBytes', {
            statistic: 'Sum',
            ...props,
        });
    }
    /**
     * The number of IPv6 requests received by the load balancer.
     *
     * @default Sum over 5 minutes
     */
    metricIpv6RequestCount(props) {
        return this.metric('IPv6RequestCount', {
            statistic: 'Sum',
            ...props,
        });
    }
    /**
     * The total number of new TCP connections established from clients to the
     * load balancer and from the load balancer to targets.
     *
     * @default Sum over 5 minutes
     */
    metricNewConnectionCount(props) {
        return this.metric('NewConnectionCount', {
            statistic: 'Sum',
            ...props,
        });
    }
    /**
     * The total number of bytes processed by the load balancer over IPv4 and IPv6.
     *
     * @default Sum over 5 minutes
     */
    metricProcessedBytes(props) {
        return this.metric('ProcessedBytes', {
            statistic: 'Sum',
            ...props,
        });
    }
    /**
     * The number of connections that were rejected because the load balancer had
     * reached its maximum number of connections.
     *
     * @default Sum over 5 minutes
     */
    metricRejectedConnectionCount(props) {
        return this.metric('RejectedConnectionCount', {
            statistic: 'Sum',
            ...props,
        });
    }
    /**
     * The number of requests processed over IPv4 and IPv6.
     *
     * This count includes only the requests with a response generated by a target of the load balancer.
     *
     * @default Sum over 5 minutes
     */
    metricRequestCount(props) {
        return this.metric('RequestCount', {
            statistic: 'Sum',
            ...props,
        });
    }
    /**
     * The number of rules processed by the load balancer given a request rate averaged over an hour.
     *
     * @default Sum over 5 minutes
     */
    metricRuleEvaluations(props) {
        return this.metric('RuleEvaluations', {
            statistic: 'Sum',
            ...props,
        });
    }
    /**
     * The number of connections that were not successfully established between the load balancer and target.
     *
     * @default Sum over 5 minutes
     */
    metricTargetConnectionErrorCount(props) {
        return this.metric('TargetConnectionErrorCount', {
            statistic: 'Sum',
            ...props,
        });
    }
    /**
     * The time elapsed, in seconds, after the request leaves the load balancer until a response from the target is received.
     *
     * @default Average over 5 minutes
     */
    metricTargetResponseTime(props) {
        return this.metric('TargetResponseTime', {
            statistic: 'Average',
            ...props,
        });
    }
    /**
     * The number of TLS connections initiated by the load balancer that did not establish a session with the target.
     *
     * Possible causes include a mismatch of ciphers or protocols.
     *
     * @default Sum over 5 minutes
     */
    metricTargetTLSNegotiationErrorCount(props) {
        return this.metric('TargetTLSNegotiationErrorCount', {
            statistic: 'Sum',
            ...props,
        });
    }
    /**
     * The number of user authentications that could not be completed
     *
     * Because an authenticate action was misconfigured, the load balancer
     * couldn't establish a connection with the IdP, or the load balancer
     * couldn't complete the authentication flow due to an internal error.
     *
     * @default Sum over 5 minutes
     */
    metricElbAuthError(props) {
        return this.metric('ELBAuthError', {
            statistic: 'Sum',
            ...props,
        });
    }
    /**
     * The number of user authentications that could not be completed because the
     * IdP denied access to the user or an authorization code was used more than
     * once.
     *
     * @default Sum over 5 minutes
     */
    metricElbAuthFailure(props) {
        return this.metric('ELBAuthFailure', {
            statistic: 'Sum',
            ...props,
        });
    }
    /**
     * The time elapsed, in milliseconds, to query the IdP for the ID token and user info.
     *
     * If one or more of these operations fail, this is the time to failure.
     *
     * @default Average over 5 minutes
     */
    metricElbAuthLatency(props) {
        return this.metric('ELBAuthLatency', {
            statistic: 'Average',
            ...props,
        });
    }
    /**
     * The number of authenticate actions that were successful.
     *
     * This metric is incremented at the end of the authentication workflow,
     * after the load balancer has retrieved the user claims from the IdP.
     *
     * @default Sum over 5 minutes
     */
    metricElbAuthSuccess(props) {
        return this.metric('ELBAuthSuccess', {
            statistic: 'Sum',
            ...props,
        });
    }
}
exports.ApplicationLoadBalancer = ApplicationLoadBalancer;
/**
 * Count of HTTP status originating from the load balancer
 *
 * This count does not include any response codes generated by the targets.
 */
var HttpCodeElb;
(function (HttpCodeElb) {
    /**
     * The number of HTTP 3XX redirection codes that originate from the load balancer.
     */
    HttpCodeElb["ELB_3XX_COUNT"] = "HTTPCode_ELB_3XX_Count";
    /**
     * The number of HTTP 4XX client error codes that originate from the load balancer.
     *
     * Client errors are generated when requests are malformed or incomplete.
     * These requests have not been received by the target. This count does not
     * include any response codes generated by the targets.
     */
    HttpCodeElb["ELB_4XX_COUNT"] = "HTTPCode_ELB_4XX_Count";
    /**
     * The number of HTTP 5XX server error codes that originate from the load balancer.
     */
    HttpCodeElb["ELB_5XX_COUNT"] = "HTTPCode_ELB_5XX_Count";
})(HttpCodeElb = exports.HttpCodeElb || (exports.HttpCodeElb = {}));
/**
 * Count of HTTP status originating from the targets
 */
var HttpCodeTarget;
(function (HttpCodeTarget) {
    /**
     * The number of 2xx response codes from targets
     */
    HttpCodeTarget["TARGET_2XX_COUNT"] = "HTTPCode_Target_2XX_Count";
    /**
     * The number of 3xx response codes from targets
     */
    HttpCodeTarget["TARGET_3XX_COUNT"] = "HTTPCode_Target_3XX_Count";
    /**
     * The number of 4xx response codes from targets
     */
    HttpCodeTarget["TARGET_4XX_COUNT"] = "HTTPCode_Target_4XX_Count";
    /**
     * The number of 5xx response codes from targets
     */
    HttpCodeTarget["TARGET_5XX_COUNT"] = "HTTPCode_Target_5XX_Count";
})(HttpCodeTarget = exports.HttpCodeTarget || (exports.HttpCodeTarget = {}));
/**
 * An ApplicationLoadBalancer that has been defined elsewhere
 */
class ImportedApplicationLoadBalancer extends core_1.Resource {
    constructor(scope, id, props) {
        super(scope, id);
        this.props = props;
        this.vpc = props.vpc;
        this.loadBalancerArn = props.loadBalancerArn;
        this.connections = new ec2.Connections({
            securityGroups: [ec2.SecurityGroup.fromSecurityGroupId(this, 'SecurityGroup', props.securityGroupId, {
                    allowAllOutbound: props.securityGroupAllowsAllOutbound,
                })],
        });
    }
    addListener(id, props) {
        return new application_listener_1.ApplicationListener(this, id, {
            loadBalancer: this,
            ...props,
        });
    }
    get loadBalancerCanonicalHostedZoneId() {
        if (this.props.loadBalancerCanonicalHostedZoneId) {
            return this.props.loadBalancerCanonicalHostedZoneId;
        }
        // tslint:disable-next-line:max-line-length
        throw new Error(`'loadBalancerCanonicalHostedZoneId' was not provided when constructing Application Load Balancer ${this.node.path} from attributes`);
    }
    get loadBalancerDnsName() {
        if (this.props.loadBalancerDnsName) {
            return this.props.loadBalancerDnsName;
        }
        // tslint:disable-next-line:max-line-length
        throw new Error(`'loadBalancerDnsName' was not provided when constructing Application Load Balancer ${this.node.path} from attributes`);
    }
}
//# sourceMappingURL=data:application/json;base64,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