"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.NetworkMultipleTargetGroupsFargateService = void 0;
const aws_ecs_1 = require("../../../aws-ecs"); // Automatically re-written from '@aws-cdk/aws-ecs'
const network_multiple_target_groups_service_base_1 = require("../base/network-multiple-target-groups-service-base");
/**
 * A Fargate service running on an ECS cluster fronted by a network load balancer.
 */
class NetworkMultipleTargetGroupsFargateService extends network_multiple_target_groups_service_base_1.NetworkMultipleTargetGroupsServiceBase {
    /**
     * Constructs a new instance of the NetworkMultipleTargetGroupsFargateService class.
     */
    constructor(scope, id, props = {}) {
        super(scope, id, props);
        this.assignPublicIp = props.assignPublicIp !== undefined ? props.assignPublicIp : false;
        if (props.taskDefinition && props.taskImageOptions) {
            throw new Error('You must specify only one of TaskDefinition or TaskImageOptions.');
        }
        else if (props.taskDefinition) {
            this.taskDefinition = props.taskDefinition;
        }
        else if (props.taskImageOptions) {
            const taskImageOptions = props.taskImageOptions;
            this.taskDefinition = new aws_ecs_1.FargateTaskDefinition(this, 'TaskDef', {
                memoryLimitMiB: props.memoryLimitMiB,
                cpu: props.cpu,
                executionRole: taskImageOptions.executionRole,
                taskRole: taskImageOptions.taskRole,
                family: taskImageOptions.family,
            });
            const containerName = taskImageOptions.containerName !== undefined ? taskImageOptions.containerName : 'web';
            const container = this.taskDefinition.addContainer(containerName, {
                image: taskImageOptions.image,
                logging: this.logDriver,
                environment: taskImageOptions.environment,
                secrets: taskImageOptions.secrets,
            });
            if (taskImageOptions.containerPorts) {
                for (const containerPort of taskImageOptions.containerPorts) {
                    container.addPortMappings({
                        containerPort,
                    });
                }
            }
        }
        else {
            throw new Error('You must specify one of: taskDefinition or image');
        }
        if (!this.taskDefinition.defaultContainer) {
            throw new Error('At least one essential container must be specified');
        }
        if (this.taskDefinition.defaultContainer.portMappings.length === 0) {
            this.taskDefinition.defaultContainer.addPortMappings({
                containerPort: 80,
            });
        }
        this.service = this.createFargateService(props);
        if (props.targetGroups) {
            this.addPortMappingForTargets(this.taskDefinition.defaultContainer, props.targetGroups);
            this.targetGroup = this.registerECSTargets(this.service, this.taskDefinition.defaultContainer, props.targetGroups);
        }
        else {
            this.targetGroup = this.listener.addTargets('ECS', {
                targets: [this.service],
                port: this.taskDefinition.defaultContainer.portMappings[0].containerPort,
            });
        }
    }
    createFargateService(props) {
        return new aws_ecs_1.FargateService(this, 'Service', {
            cluster: this.cluster,
            desiredCount: this.desiredCount,
            taskDefinition: this.taskDefinition,
            assignPublicIp: this.assignPublicIp,
            serviceName: props.serviceName,
            healthCheckGracePeriod: props.healthCheckGracePeriod,
            propagateTags: props.propagateTags,
            enableECSManagedTags: props.enableECSManagedTags,
            cloudMapOptions: props.cloudMapOptions,
            platformVersion: props.platformVersion,
        });
    }
}
exports.NetworkMultipleTargetGroupsFargateService = NetworkMultipleTargetGroupsFargateService;
//# sourceMappingURL=data:application/json;base64,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