"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.FlowLog = exports.FlowLogDestination = exports.FlowLogResourceType = exports.FlowLogDestinationType = exports.FlowLogTrafficType = void 0;
const iam = require("../../aws-iam"); // Automatically re-written from '@aws-cdk/aws-iam'
const logs = require("../../aws-logs"); // Automatically re-written from '@aws-cdk/aws-logs'
const s3 = require("../../aws-s3"); // Automatically re-written from '@aws-cdk/aws-s3'
const core_1 = require("../../core"); // Automatically re-written from '@aws-cdk/core'
const ec2_generated_1 = require("./ec2.generated");
/**
 * The type of VPC traffic to log
 *
 * @experimental
 */
var FlowLogTrafficType;
(function (FlowLogTrafficType) {
    /**
     * Only log accepts
     */
    FlowLogTrafficType["ACCEPT"] = "ACCEPT";
    /**
     * Log all requests
     */
    FlowLogTrafficType["ALL"] = "ALL";
    /**
     * Only log rejects
     */
    FlowLogTrafficType["REJECT"] = "REJECT";
})(FlowLogTrafficType = exports.FlowLogTrafficType || (exports.FlowLogTrafficType = {}));
/**
 * The available destination types for Flow Logs
 * @experimental
 */
var FlowLogDestinationType;
(function (FlowLogDestinationType) {
    /**
     * Send flow logs to CloudWatch Logs Group
     */
    FlowLogDestinationType["CLOUD_WATCH_LOGS"] = "cloud-watch-logs";
    /**
     * Send flow logs to S3 Bucket
     */
    FlowLogDestinationType["S3"] = "s3";
})(FlowLogDestinationType = exports.FlowLogDestinationType || (exports.FlowLogDestinationType = {}));
/**
 * The type of resource to create the flow log for
 *
 * @experimental
 */
class FlowLogResourceType {
    /**
     * The subnet to attach the Flow Log to
     */
    static fromSubnet(subnet) {
        return {
            resourceType: 'Subnet',
            resourceId: subnet.subnetId,
        };
    }
    /**
     * The VPC to attach the Flow Log to
     */
    static fromVpc(vpc) {
        return {
            resourceType: 'VPC',
            resourceId: vpc.vpcId,
        };
    }
    /**
     * The Network Interface to attach the Flow Log to
     */
    static fromNetworkInterfaceId(id) {
        return {
            resourceType: 'NetworkInterface',
            resourceId: id,
        };
    }
}
exports.FlowLogResourceType = FlowLogResourceType;
/**
 * The destination type for the flow log
 *
 * @experimental
 */
class FlowLogDestination {
    /**
     * Use CloudWatch logs as the destination
     */
    static toCloudWatchLogs(logGroup, iamRole) {
        return new CloudWatchLogsDestination({
            logDestinationType: FlowLogDestinationType.CLOUD_WATCH_LOGS,
            logGroup,
            iamRole,
        });
    }
    /**
     * Use S3 as the destination
     */
    static toS3(bucket) {
        return new S3Destination({
            logDestinationType: FlowLogDestinationType.S3,
            s3Bucket: bucket,
        });
    }
}
exports.FlowLogDestination = FlowLogDestination;
/**
 * @experimental
 */
class S3Destination extends FlowLogDestination {
    constructor(props) {
        super();
        this.props = props;
    }
    bind(scope, _flowLog) {
        let s3Bucket;
        if (this.props.s3Bucket === undefined) {
            s3Bucket = new s3.Bucket(scope, 'Bucket', {
                encryption: s3.BucketEncryption.UNENCRYPTED,
                removalPolicy: core_1.RemovalPolicy.RETAIN,
            });
        }
        else {
            s3Bucket = this.props.s3Bucket;
        }
        return {
            logDestinationType: FlowLogDestinationType.S3,
            s3Bucket,
        };
    }
}
/**
 * @experimental
 */
class CloudWatchLogsDestination extends FlowLogDestination {
    constructor(props) {
        super();
        this.props = props;
    }
    bind(scope, _flowLog) {
        let iamRole;
        let logGroup;
        if (this.props.iamRole === undefined) {
            iamRole = new iam.Role(scope, 'IAMRole', {
                roleName: core_1.PhysicalName.GENERATE_IF_NEEDED,
                assumedBy: new iam.ServicePrincipal('vpc-flow-logs.amazonaws.com'),
            });
        }
        else {
            iamRole = this.props.iamRole;
        }
        if (this.props.logGroup === undefined) {
            logGroup = new logs.LogGroup(scope, 'LogGroup');
        }
        else {
            logGroup = this.props.logGroup;
        }
        iamRole.addToPolicy(new iam.PolicyStatement({
            actions: [
                'logs:CreateLogStream',
                'logs:PutLogEvents',
                'logs:DescribeLogStreams',
            ],
            effect: iam.Effect.ALLOW,
            resources: [logGroup.logGroupArn],
        }));
        iamRole.addToPolicy(new iam.PolicyStatement({
            actions: ['iam:PassRole'],
            effect: iam.Effect.ALLOW,
            resources: [iamRole.roleArn],
        }));
        return {
            logDestinationType: FlowLogDestinationType.CLOUD_WATCH_LOGS,
            logGroup,
            iamRole,
        };
    }
}
/**
 * The base class for a Flow Log
 *
 * @experimental
 */
class FlowLogBase extends core_1.Resource {
}
/**
 * A VPC flow log.
 * @resource AWS::EC2::FlowLog
 *
 * @experimental
 */
class FlowLog extends FlowLogBase {
    constructor(scope, id, props) {
        super(scope, id, {
            physicalName: props.flowLogName,
        });
        const destination = props.destination || FlowLogDestination.toCloudWatchLogs();
        const destinationConfig = destination.bind(this, this);
        this.logGroup = destinationConfig.logGroup;
        this.bucket = destinationConfig.s3Bucket;
        this.iamRole = destinationConfig.iamRole;
        const flowLog = new ec2_generated_1.CfnFlowLog(this, 'FlowLog', {
            deliverLogsPermissionArn: this.iamRole ? this.iamRole.roleArn : undefined,
            logDestinationType: destinationConfig.logDestinationType,
            logGroupName: this.logGroup ? this.logGroup.logGroupName : undefined,
            resourceId: props.resourceType.resourceId,
            resourceType: props.resourceType.resourceType,
            trafficType: props.trafficType
                ? props.trafficType
                : FlowLogTrafficType.ALL,
            logDestination: this.bucket ? this.bucket.bucketArn : undefined,
        });
        this.flowLogId = flowLog.ref;
    }
    /**
     * Import a Flow Log by it's Id
     */
    static fromFlowLogId(scope, id, flowLogId) {
        class Import extends FlowLogBase {
            constructor() {
                super(...arguments);
                this.flowLogId = flowLogId;
            }
        }
        return new Import(scope, id);
    }
}
exports.FlowLog = FlowLog;
//# sourceMappingURL=data:application/json;base64,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