"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.DatabaseInstance = void 0;
const ec2 = require("../../aws-ec2"); // Automatically re-written from '@aws-cdk/aws-ec2'
const cdk = require("../../core"); // Automatically re-written from '@aws-cdk/core'
const docdb_generated_1 = require("./docdb.generated");
const endpoint_1 = require("./endpoint");
/**
 * A new or imported database instance.
 */
class DatabaseInstanceBase extends cdk.Resource {
    /**
     * Import an existing database instance.
     */
    static fromDatabaseInstanceAttributes(scope, id, attrs) {
        class Import extends DatabaseInstanceBase {
            constructor() {
                super(...arguments);
                this.defaultPort = ec2.Port.tcp(attrs.port);
                this.instanceIdentifier = attrs.instanceIdentifier;
                this.dbInstanceEndpointAddress = attrs.instanceEndpointAddress;
                this.dbInstanceEndpointPort = attrs.port.toString();
                this.instanceEndpoint = new endpoint_1.Endpoint(attrs.instanceEndpointAddress, attrs.port);
            }
        }
        return new Import(scope, id);
    }
    /**
     * The instance arn.
     */
    get instanceArn() {
        return cdk.Stack.of(this).formatArn({
            service: 'docdb',
            resource: 'db',
            sep: ':',
            resourceName: this.instanceIdentifier,
        });
    }
}
/**
 * A database instance
 *
 * @resource AWS::DocDB::DBInstance
 */
class DatabaseInstance extends DatabaseInstanceBase {
    constructor(scope, id, props) {
        super(scope, id);
        const instance = new docdb_generated_1.CfnDBInstance(this, 'Resource', {
            dbClusterIdentifier: props.cluster.clusterIdentifier,
            dbInstanceClass: `db.${props.instanceClass}`,
            autoMinorVersionUpgrade: props.autoMinorVersionUpgrade,
            availabilityZone: props.availabilityZone,
            dbInstanceIdentifier: props.dbInstanceName,
            preferredMaintenanceWindow: props.preferredMaintenanceWindow,
        });
        this.cluster = props.cluster;
        this.instanceIdentifier = instance.ref;
        this.dbInstanceEndpointAddress = instance.attrEndpoint;
        this.dbInstanceEndpointPort = instance.attrPort;
        // create a number token that represents the port of the instance
        const portAttribute = cdk.Token.asNumber(instance.attrPort);
        this.instanceEndpoint = new endpoint_1.Endpoint(instance.attrEndpoint, portAttribute);
        instance.applyRemovalPolicy(props.removalPolicy, {
            applyToUpdateReplacePolicy: true,
        });
    }
}
exports.DatabaseInstance = DatabaseInstance;
//# sourceMappingURL=data:application/json;base64,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