"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.CustomRule = exports.ManagedRule = exports.MaximumExecutionFrequency = void 0;
const events = require("../../aws-events"); // Automatically re-written from '@aws-cdk/aws-events'
const iam = require("../../aws-iam"); // Automatically re-written from '@aws-cdk/aws-iam'
const core_1 = require("../../core"); // Automatically re-written from '@aws-cdk/core'
const config_generated_1 = require("./config.generated");
/**
 * A new or imported rule.
 */
class RuleBase extends core_1.Resource {
    /**
     * Defines a CloudWatch event rule which triggers for rule events. Use
     * `rule.addEventPattern(pattern)` to specify a filter.
     */
    onEvent(id, options = {}) {
        const rule = new events.Rule(this, id, options);
        rule.addEventPattern({
            source: ['aws.config'],
            detail: {
                configRuleName: [this.configRuleName],
            },
        });
        rule.addTarget(options.target);
        return rule;
    }
    /**
     * Defines a CloudWatch event rule which triggers for rule compliance events.
     */
    onComplianceChange(id, options = {}) {
        const rule = this.onEvent(id, options);
        rule.addEventPattern({
            detailType: ['Config Rules Compliance Change'],
        });
        return rule;
    }
    /**
     * Defines a CloudWatch event rule which triggers for rule re-evaluation status events.
     */
    onReEvaluationStatus(id, options = {}) {
        const rule = this.onEvent(id, options);
        rule.addEventPattern({
            detailType: ['Config Rules Re-evaluation Status'],
        });
        return rule;
    }
}
/**
 * A new managed or custom rule.
 */
class RuleNew extends RuleBase {
    /**
     * Imports an existing rule.
     *
     * @param configRuleName the name of the rule
     */
    static fromConfigRuleName(scope, id, configRuleName) {
        class Import extends RuleBase {
            constructor() {
                super(...arguments);
                this.configRuleName = configRuleName;
            }
        }
        return new Import(scope, id);
    }
    /**
     * Restrict scope of changes to a specific resource.
     *
     * @see https://docs.aws.amazon.com/config/latest/developerguide/resource-config-reference.html#supported-resources
     *
     * @param type the resource type
     * @param identifier the resource identifier
     */
    scopeToResource(type, identifier) {
        this.scopeTo({
            complianceResourceId: identifier,
            complianceResourceTypes: [type],
        });
    }
    /**
     * Restrict scope of changes to specific resource types.
     *
     * @see https://docs.aws.amazon.com/config/latest/developerguide/resource-config-reference.html#supported-resources
     *
     * @param types resource types
     */
    scopeToResources(...types) {
        this.scopeTo({
            complianceResourceTypes: types,
        });
    }
    /**
     * Restrict scope of changes to a specific tag.
     *
     * @param key the tag key
     * @param value the tag value
     */
    scopeToTag(key, value) {
        this.scopeTo({
            tagKey: key,
            tagValue: value,
        });
    }
    scopeTo(scope) {
        if (!this.isManaged && !this.isCustomWithChanges) {
            throw new Error('Cannot scope rule when `configurationChanges` is set to false.');
        }
        this.scope = scope;
    }
}
/**
 * The maximum frequency at which the AWS Config rule runs evaluations.
 */
var MaximumExecutionFrequency;
(function (MaximumExecutionFrequency) {
    /**
     * 1 hour.
     */
    MaximumExecutionFrequency["ONE_HOUR"] = "One_Hour";
    /**
     * 3 hours.
     */
    MaximumExecutionFrequency["THREE_HOURS"] = "Three_Hours";
    /**
     * 6 hours.
     */
    MaximumExecutionFrequency["SIX_HOURS"] = "Six_Hours";
    /**
     * 12 hours.
     */
    MaximumExecutionFrequency["TWELVE_HOURS"] = "Twelve_Hours";
    /**
     * 24 hours.
     */
    MaximumExecutionFrequency["TWENTY_FOUR_HOURS"] = "TwentyFour_Hours";
})(MaximumExecutionFrequency = exports.MaximumExecutionFrequency || (exports.MaximumExecutionFrequency = {}));
/**
 * A new managed rule.
 *
 * @resource AWS::Config::ConfigRule
 */
class ManagedRule extends RuleNew {
    constructor(scope, id, props) {
        super(scope, id, {
            physicalName: props.configRuleName,
        });
        const rule = new config_generated_1.CfnConfigRule(this, 'Resource', {
            configRuleName: this.physicalName,
            description: props.description,
            inputParameters: props.inputParameters,
            maximumExecutionFrequency: props.maximumExecutionFrequency,
            scope: core_1.Lazy.anyValue({ produce: () => this.scope }),
            source: {
                owner: 'AWS',
                sourceIdentifier: props.identifier,
            },
        });
        this.configRuleName = rule.ref;
        this.configRuleArn = rule.attrArn;
        this.configRuleId = rule.attrConfigRuleId;
        this.configRuleComplianceType = rule.attrComplianceType;
        this.isManaged = true;
    }
}
exports.ManagedRule = ManagedRule;
/**
 * A new custom rule.
 *
 * @resource AWS::Config::ConfigRule
 */
class CustomRule extends RuleNew {
    constructor(scope, id, props) {
        super(scope, id, {
            physicalName: props.configRuleName,
        });
        if (!props.configurationChanges && !props.periodic) {
            throw new Error('At least one of `configurationChanges` or `periodic` must be set to true.');
        }
        const sourceDetails = [];
        if (props.configurationChanges) {
            sourceDetails.push({
                eventSource: 'aws.config',
                messageType: 'ConfigurationItemChangeNotification',
            });
            sourceDetails.push({
                eventSource: 'aws.config',
                messageType: 'OversizedConfigurationItemChangeNotification',
            });
        }
        if (props.periodic) {
            sourceDetails.push({
                eventSource: 'aws.config',
                maximumExecutionFrequency: props.maximumExecutionFrequency,
                messageType: 'ScheduledNotification',
            });
        }
        props.lambdaFunction.addPermission('Permission', {
            principal: new iam.ServicePrincipal('config.amazonaws.com'),
        });
        if (props.lambdaFunction.role) {
            props.lambdaFunction.role.addManagedPolicy(iam.ManagedPolicy.fromAwsManagedPolicyName('service-role/AWSConfigRulesExecutionRole'));
        }
        // The lambda permission must be created before the rule
        this.node.addDependency(props.lambdaFunction);
        const rule = new config_generated_1.CfnConfigRule(this, 'Resource', {
            configRuleName: this.physicalName,
            description: props.description,
            inputParameters: props.inputParameters,
            maximumExecutionFrequency: props.maximumExecutionFrequency,
            scope: core_1.Lazy.anyValue({ produce: () => this.scope }),
            source: {
                owner: 'CUSTOM_LAMBDA',
                sourceDetails,
                sourceIdentifier: props.lambdaFunction.functionArn,
            },
        });
        this.configRuleName = rule.ref;
        this.configRuleArn = rule.attrArn;
        this.configRuleId = rule.attrConfigRuleId;
        this.configRuleComplianceType = rule.attrComplianceType;
        if (props.configurationChanges) {
            this.isCustomWithChanges = true;
        }
    }
}
exports.CustomRule = CustomRule;
//# sourceMappingURL=data:application/json;base64,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