"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.GitHubSourceAction = exports.GitHubTrigger = void 0;
const codepipeline = require("../../../aws-codepipeline"); // Automatically re-written from '@aws-cdk/aws-codepipeline'
const action_1 = require("../action");
const common_1 = require("../common");
/**
 * If and how the GitHub source action should be triggered
 */
var GitHubTrigger;
(function (GitHubTrigger) {
    GitHubTrigger["NONE"] = "None";
    GitHubTrigger["POLL"] = "Poll";
    GitHubTrigger["WEBHOOK"] = "WebHook";
})(GitHubTrigger = exports.GitHubTrigger || (exports.GitHubTrigger = {}));
/**
 * Source that is provided by a GitHub repository.
 */
class GitHubSourceAction extends action_1.Action {
    constructor(props) {
        super({
            ...props,
            category: codepipeline.ActionCategory.SOURCE,
            owner: 'ThirdParty',
            provider: 'GitHub',
            artifactBounds: common_1.sourceArtifactBounds(),
            outputs: [props.output],
        });
        this.props = props;
    }
    /** The variables emitted by this action. */
    get variables() {
        return {
            repositoryName: this.variableExpression('RepositoryName'),
            branchName: this.variableExpression('BranchName'),
            authorDate: this.variableExpression('AuthorDate'),
            committerDate: this.variableExpression('CommitterDate'),
            commitId: this.variableExpression('CommitId'),
            commitMessage: this.variableExpression('CommitMessage'),
            commitUrl: this.variableExpression('CommitUrl'),
        };
    }
    bound(scope, stage, _options) {
        if (!this.props.trigger || this.props.trigger === GitHubTrigger.WEBHOOK) {
            new codepipeline.CfnWebhook(scope, 'WebhookResource', {
                authentication: 'GITHUB_HMAC',
                authenticationConfiguration: {
                    secretToken: this.props.oauthToken.toString(),
                },
                filters: [
                    {
                        jsonPath: '$.ref',
                        matchEquals: 'refs/heads/{Branch}',
                    },
                ],
                targetAction: this.actionProperties.actionName,
                targetPipeline: stage.pipeline.pipelineName,
                targetPipelineVersion: 1,
                registerWithThirdParty: true,
            });
        }
        return {
            configuration: {
                Owner: this.props.owner,
                Repo: this.props.repo,
                Branch: this.props.branch || 'master',
                OAuthToken: this.props.oauthToken.toString(),
                PollForSourceChanges: this.props.trigger === GitHubTrigger.POLL,
            },
        };
    }
}
exports.GitHubSourceAction = GitHubSourceAction;
//# sourceMappingURL=data:application/json;base64,eyJ2ZXJzaW9uIjozLCJmaWxlIjoic291cmNlLWFjdGlvbi5qcyIsInNvdXJjZVJvb3QiOiIiLCJzb3VyY2VzIjpbInNvdXJjZS1hY3Rpb24udHMiXSwibmFtZXMiOltdLCJtYXBwaW5ncyI6Ijs7O0FBQUEsMERBQTBELENBQUMsNERBQTREO0FBRXZILHNDQUFtQztBQUNuQyxzQ0FBaUQ7QUFDakQ7O0dBRUc7QUFDSCxJQUFZLGFBSVg7QUFKRCxXQUFZLGFBQWE7SUFDckIsOEJBQWEsQ0FBQTtJQUNiLDhCQUFhLENBQUE7SUFDYixvQ0FBbUIsQ0FBQTtBQUN2QixDQUFDLEVBSlcsYUFBYSxHQUFiLHFCQUFhLEtBQWIscUJBQWEsUUFJeEI7QUE4REQ7O0dBRUc7QUFDSCxNQUFhLGtCQUFtQixTQUFRLGVBQU07SUFFMUMsWUFBWSxLQUE4QjtRQUN0QyxLQUFLLENBQUM7WUFDRixHQUFHLEtBQUs7WUFDUixRQUFRLEVBQUUsWUFBWSxDQUFDLGNBQWMsQ0FBQyxNQUFNO1lBQzVDLEtBQUssRUFBRSxZQUFZO1lBQ25CLFFBQVEsRUFBRSxRQUFRO1lBQ2xCLGNBQWMsRUFBRSw2QkFBb0IsRUFBRTtZQUN0QyxPQUFPLEVBQUUsQ0FBQyxLQUFLLENBQUMsTUFBTSxDQUFDO1NBQzFCLENBQUMsQ0FBQztRQUNILElBQUksQ0FBQyxLQUFLLEdBQUcsS0FBSyxDQUFDO0lBQ3ZCLENBQUM7SUFDRCw0Q0FBNEM7SUFDNUMsSUFBVyxTQUFTO1FBQ2hCLE9BQU87WUFDSCxjQUFjLEVBQUUsSUFBSSxDQUFDLGtCQUFrQixDQUFDLGdCQUFnQixDQUFDO1lBQ3pELFVBQVUsRUFBRSxJQUFJLENBQUMsa0JBQWtCLENBQUMsWUFBWSxDQUFDO1lBQ2pELFVBQVUsRUFBRSxJQUFJLENBQUMsa0JBQWtCLENBQUMsWUFBWSxDQUFDO1lBQ2pELGFBQWEsRUFBRSxJQUFJLENBQUMsa0JBQWtCLENBQUMsZUFBZSxDQUFDO1lBQ3ZELFFBQVEsRUFBRSxJQUFJLENBQUMsa0JBQWtCLENBQUMsVUFBVSxDQUFDO1lBQzdDLGFBQWEsRUFBRSxJQUFJLENBQUMsa0JBQWtCLENBQUMsZUFBZSxDQUFDO1lBQ3ZELFNBQVMsRUFBRSxJQUFJLENBQUMsa0JBQWtCLENBQUMsV0FBVyxDQUFDO1NBQ2xELENBQUM7SUFDTixDQUFDO0lBQ1MsS0FBSyxDQUFDLEtBQWdCLEVBQUUsS0FBMEIsRUFBRSxRQUF3QztRQUNsRyxJQUFJLENBQUMsSUFBSSxDQUFDLEtBQUssQ0FBQyxPQUFPLElBQUksSUFBSSxDQUFDLEtBQUssQ0FBQyxPQUFPLEtBQUssYUFBYSxDQUFDLE9BQU8sRUFBRTtZQUNyRSxJQUFJLFlBQVksQ0FBQyxVQUFVLENBQUMsS0FBSyxFQUFFLGlCQUFpQixFQUFFO2dCQUNsRCxjQUFjLEVBQUUsYUFBYTtnQkFDN0IsMkJBQTJCLEVBQUU7b0JBQ3pCLFdBQVcsRUFBRSxJQUFJLENBQUMsS0FBSyxDQUFDLFVBQVUsQ0FBQyxRQUFRLEVBQUU7aUJBQ2hEO2dCQUNELE9BQU8sRUFBRTtvQkFDTDt3QkFDSSxRQUFRLEVBQUUsT0FBTzt3QkFDakIsV0FBVyxFQUFFLHFCQUFxQjtxQkFDckM7aUJBQ0o7Z0JBQ0QsWUFBWSxFQUFFLElBQUksQ0FBQyxnQkFBZ0IsQ0FBQyxVQUFVO2dCQUM5QyxjQUFjLEVBQUUsS0FBSyxDQUFDLFFBQVEsQ0FBQyxZQUFZO2dCQUMzQyxxQkFBcUIsRUFBRSxDQUFDO2dCQUN4QixzQkFBc0IsRUFBRSxJQUFJO2FBQy9CLENBQUMsQ0FBQztTQUNOO1FBQ0QsT0FBTztZQUNILGFBQWEsRUFBRTtnQkFDWCxLQUFLLEVBQUUsSUFBSSxDQUFDLEtBQUssQ0FBQyxLQUFLO2dCQUN2QixJQUFJLEVBQUUsSUFBSSxDQUFDLEtBQUssQ0FBQyxJQUFJO2dCQUNyQixNQUFNLEVBQUUsSUFBSSxDQUFDLEtBQUssQ0FBQyxNQUFNLElBQUksUUFBUTtnQkFDckMsVUFBVSxFQUFFLElBQUksQ0FBQyxLQUFLLENBQUMsVUFBVSxDQUFDLFFBQVEsRUFBRTtnQkFDNUMsb0JBQW9CLEVBQUUsSUFBSSxDQUFDLEtBQUssQ0FBQyxPQUFPLEtBQUssYUFBYSxDQUFDLElBQUk7YUFDbEU7U0FDSixDQUFDO0lBQ04sQ0FBQztDQUNKO0FBdERELGdEQXNEQyIsInNvdXJjZXNDb250ZW50IjpbImltcG9ydCAqIGFzIGNvZGVwaXBlbGluZSBmcm9tIFwiLi4vLi4vLi4vYXdzLWNvZGVwaXBlbGluZVwiOyAvLyBBdXRvbWF0aWNhbGx5IHJlLXdyaXR0ZW4gZnJvbSAnQGF3cy1jZGsvYXdzLWNvZGVwaXBlbGluZSdcbmltcG9ydCB7IENvbnN0cnVjdCwgU2VjcmV0VmFsdWUgfSBmcm9tIFwiLi4vLi4vLi4vY29yZVwiOyAvLyBBdXRvbWF0aWNhbGx5IHJlLXdyaXR0ZW4gZnJvbSAnQGF3cy1jZGsvY29yZSdcbmltcG9ydCB7IEFjdGlvbiB9IGZyb20gJy4uL2FjdGlvbic7XG5pbXBvcnQgeyBzb3VyY2VBcnRpZmFjdEJvdW5kcyB9IGZyb20gJy4uL2NvbW1vbic7XG4vKipcbiAqIElmIGFuZCBob3cgdGhlIEdpdEh1YiBzb3VyY2UgYWN0aW9uIHNob3VsZCBiZSB0cmlnZ2VyZWRcbiAqL1xuZXhwb3J0IGVudW0gR2l0SHViVHJpZ2dlciB7XG4gICAgTk9ORSA9ICdOb25lJyxcbiAgICBQT0xMID0gJ1BvbGwnLFxuICAgIFdFQkhPT0sgPSAnV2ViSG9vaydcbn1cbi8qKlxuICogVGhlIENvZGVQaXBlbGluZSB2YXJpYWJsZXMgZW1pdHRlZCBieSBHaXRIdWIgc291cmNlIEFjdGlvbi5cbiAqL1xuZXhwb3J0IGludGVyZmFjZSBHaXRIdWJTb3VyY2VWYXJpYWJsZXMge1xuICAgIC8qKiBUaGUgbmFtZSBvZiB0aGUgcmVwb3NpdG9yeSB0aGlzIGFjdGlvbiBwb2ludHMgdG8uICovXG4gICAgcmVhZG9ubHkgcmVwb3NpdG9yeU5hbWU6IHN0cmluZztcbiAgICAvKiogVGhlIG5hbWUgb2YgdGhlIGJyYW5jaCB0aGlzIGFjdGlvbiB0cmFja3MuICovXG4gICAgcmVhZG9ubHkgYnJhbmNoTmFtZTogc3RyaW5nO1xuICAgIC8qKiBUaGUgZGF0ZSB0aGUgY3VycmVudGx5IGxhc3QgY29tbWl0IG9uIHRoZSB0cmFja2VkIGJyYW5jaCB3YXMgYXV0aG9yZWQsIGluIElTTy04NjAxIGZvcm1hdC4gKi9cbiAgICByZWFkb25seSBhdXRob3JEYXRlOiBzdHJpbmc7XG4gICAgLyoqIFRoZSBkYXRlIHRoZSBjdXJyZW50bHkgbGFzdCBjb21taXQgb24gdGhlIHRyYWNrZWQgYnJhbmNoIHdhcyBjb21taXR0ZWQsIGluIElTTy04NjAxIGZvcm1hdC4gKi9cbiAgICByZWFkb25seSBjb21taXR0ZXJEYXRlOiBzdHJpbmc7XG4gICAgLyoqIFRoZSBTSEExIGhhc2ggb2YgdGhlIGN1cnJlbnRseSBsYXN0IGNvbW1pdCBvbiB0aGUgdHJhY2tlZCBicmFuY2guICovXG4gICAgcmVhZG9ubHkgY29tbWl0SWQ6IHN0cmluZztcbiAgICAvKiogVGhlIG1lc3NhZ2Ugb2YgdGhlIGN1cnJlbnRseSBsYXN0IGNvbW1pdCBvbiB0aGUgdHJhY2tlZCBicmFuY2guICovXG4gICAgcmVhZG9ubHkgY29tbWl0TWVzc2FnZTogc3RyaW5nO1xuICAgIC8qKiBUaGUgR2l0SHViIEFQSSBVUkwgb2YgdGhlIGN1cnJlbnRseSBsYXN0IGNvbW1pdCBvbiB0aGUgdHJhY2tlZCBicmFuY2guICovXG4gICAgcmVhZG9ubHkgY29tbWl0VXJsOiBzdHJpbmc7XG59XG4vKipcbiAqIENvbnN0cnVjdGlvbiBwcm9wZXJ0aWVzIG9mIHRoZSB7QGxpbmsgR2l0SHViU291cmNlQWN0aW9uIEdpdEh1YiBzb3VyY2UgYWN0aW9ufS5cbiAqL1xuZXhwb3J0IGludGVyZmFjZSBHaXRIdWJTb3VyY2VBY3Rpb25Qcm9wcyBleHRlbmRzIGNvZGVwaXBlbGluZS5Db21tb25BY3Rpb25Qcm9wcyB7XG4gICAgLyoqXG4gICAgICpcbiAgICAgKi9cbiAgICByZWFkb25seSBvdXRwdXQ6IGNvZGVwaXBlbGluZS5BcnRpZmFjdDtcbiAgICAvKipcbiAgICAgKiBUaGUgR2l0SHViIGFjY291bnQvdXNlciB0aGF0IG93bnMgdGhlIHJlcG8uXG4gICAgICovXG4gICAgcmVhZG9ubHkgb3duZXI6IHN0cmluZztcbiAgICAvKipcbiAgICAgKiBUaGUgbmFtZSBvZiB0aGUgcmVwbywgd2l0aG91dCB0aGUgdXNlcm5hbWUuXG4gICAgICovXG4gICAgcmVhZG9ubHkgcmVwbzogc3RyaW5nO1xuICAgIC8qKlxuICAgICAqIFRoZSBicmFuY2ggdG8gdXNlLlxuICAgICAqXG4gICAgICogQGRlZmF1bHQgXCJtYXN0ZXJcIlxuICAgICAqL1xuICAgIHJlYWRvbmx5IGJyYW5jaD86IHN0cmluZztcbiAgICAvKipcbiAgICAgKiBBIEdpdEh1YiBPQXV0aCB0b2tlbiB0byB1c2UgZm9yIGF1dGhlbnRpY2F0aW9uLlxuICAgICAqXG4gICAgICogSXQgaXMgcmVjb21tZW5kZWQgdG8gdXNlIGEgU2VjcmV0cyBNYW5hZ2VyIGBTZWNyZXRgIHRvIG9idGFpbiB0aGUgdG9rZW46XG4gICAgICpcbiAgICAgKiAgIGNvbnN0IG9hdXRoID0gY2RrLlNlY3JldFZhbHVlLnNlY3JldHNNYW5hZ2VyKCdteS1naXRodWItdG9rZW4nKTtcbiAgICAgKiAgIG5ldyBHaXRIdWJTb3VyY2UodGhpcywgJ0dpdEh1YkFjdGlvbicsIHsgb2F1dGhUb2tlbjogb2F1dGgsIC4uLiB9KTtcbiAgICAgKi9cbiAgICByZWFkb25seSBvYXV0aFRva2VuOiBTZWNyZXRWYWx1ZTtcbiAgICAvKipcbiAgICAgKiBIb3cgQVdTIENvZGVQaXBlbGluZSBzaG91bGQgYmUgdHJpZ2dlcmVkXG4gICAgICpcbiAgICAgKiBXaXRoIHRoZSBkZWZhdWx0IHZhbHVlIFwiV0VCSE9PS1wiLCBhIHdlYmhvb2sgaXMgY3JlYXRlZCBpbiBHaXRIdWIgdGhhdCB0cmlnZ2VycyB0aGUgYWN0aW9uXG4gICAgICogV2l0aCBcIlBPTExcIiwgQ29kZVBpcGVsaW5lIHBlcmlvZGljYWxseSBjaGVja3MgdGhlIHNvdXJjZSBmb3IgY2hhbmdlc1xuICAgICAqIFdpdGggXCJOb25lXCIsIHRoZSBhY3Rpb24gaXMgbm90IHRyaWdnZXJlZCB0aHJvdWdoIGNoYW5nZXMgaW4gdGhlIHNvdXJjZVxuICAgICAqXG4gICAgICogQGRlZmF1bHQgR2l0SHViVHJpZ2dlci5XRUJIT09LXG4gICAgICovXG4gICAgcmVhZG9ubHkgdHJpZ2dlcj86IEdpdEh1YlRyaWdnZXI7XG59XG4vKipcbiAqIFNvdXJjZSB0aGF0IGlzIHByb3ZpZGVkIGJ5IGEgR2l0SHViIHJlcG9zaXRvcnkuXG4gKi9cbmV4cG9ydCBjbGFzcyBHaXRIdWJTb3VyY2VBY3Rpb24gZXh0ZW5kcyBBY3Rpb24ge1xuICAgIHByaXZhdGUgcmVhZG9ubHkgcHJvcHM6IEdpdEh1YlNvdXJjZUFjdGlvblByb3BzO1xuICAgIGNvbnN0cnVjdG9yKHByb3BzOiBHaXRIdWJTb3VyY2VBY3Rpb25Qcm9wcykge1xuICAgICAgICBzdXBlcih7XG4gICAgICAgICAgICAuLi5wcm9wcyxcbiAgICAgICAgICAgIGNhdGVnb3J5OiBjb2RlcGlwZWxpbmUuQWN0aW9uQ2F0ZWdvcnkuU09VUkNFLFxuICAgICAgICAgICAgb3duZXI6ICdUaGlyZFBhcnR5JyxcbiAgICAgICAgICAgIHByb3ZpZGVyOiAnR2l0SHViJyxcbiAgICAgICAgICAgIGFydGlmYWN0Qm91bmRzOiBzb3VyY2VBcnRpZmFjdEJvdW5kcygpLFxuICAgICAgICAgICAgb3V0cHV0czogW3Byb3BzLm91dHB1dF0sXG4gICAgICAgIH0pO1xuICAgICAgICB0aGlzLnByb3BzID0gcHJvcHM7XG4gICAgfVxuICAgIC8qKiBUaGUgdmFyaWFibGVzIGVtaXR0ZWQgYnkgdGhpcyBhY3Rpb24uICovXG4gICAgcHVibGljIGdldCB2YXJpYWJsZXMoKTogR2l0SHViU291cmNlVmFyaWFibGVzIHtcbiAgICAgICAgcmV0dXJuIHtcbiAgICAgICAgICAgIHJlcG9zaXRvcnlOYW1lOiB0aGlzLnZhcmlhYmxlRXhwcmVzc2lvbignUmVwb3NpdG9yeU5hbWUnKSxcbiAgICAgICAgICAgIGJyYW5jaE5hbWU6IHRoaXMudmFyaWFibGVFeHByZXNzaW9uKCdCcmFuY2hOYW1lJyksXG4gICAgICAgICAgICBhdXRob3JEYXRlOiB0aGlzLnZhcmlhYmxlRXhwcmVzc2lvbignQXV0aG9yRGF0ZScpLFxuICAgICAgICAgICAgY29tbWl0dGVyRGF0ZTogdGhpcy52YXJpYWJsZUV4cHJlc3Npb24oJ0NvbW1pdHRlckRhdGUnKSxcbiAgICAgICAgICAgIGNvbW1pdElkOiB0aGlzLnZhcmlhYmxlRXhwcmVzc2lvbignQ29tbWl0SWQnKSxcbiAgICAgICAgICAgIGNvbW1pdE1lc3NhZ2U6IHRoaXMudmFyaWFibGVFeHByZXNzaW9uKCdDb21taXRNZXNzYWdlJyksXG4gICAgICAgICAgICBjb21taXRVcmw6IHRoaXMudmFyaWFibGVFeHByZXNzaW9uKCdDb21taXRVcmwnKSxcbiAgICAgICAgfTtcbiAgICB9XG4gICAgcHJvdGVjdGVkIGJvdW5kKHNjb3BlOiBDb25zdHJ1Y3QsIHN0YWdlOiBjb2RlcGlwZWxpbmUuSVN0YWdlLCBfb3B0aW9uczogY29kZXBpcGVsaW5lLkFjdGlvbkJpbmRPcHRpb25zKTogY29kZXBpcGVsaW5lLkFjdGlvbkNvbmZpZyB7XG4gICAgICAgIGlmICghdGhpcy5wcm9wcy50cmlnZ2VyIHx8IHRoaXMucHJvcHMudHJpZ2dlciA9PT0gR2l0SHViVHJpZ2dlci5XRUJIT09LKSB7XG4gICAgICAgICAgICBuZXcgY29kZXBpcGVsaW5lLkNmbldlYmhvb2soc2NvcGUsICdXZWJob29rUmVzb3VyY2UnLCB7XG4gICAgICAgICAgICAgICAgYXV0aGVudGljYXRpb246ICdHSVRIVUJfSE1BQycsXG4gICAgICAgICAgICAgICAgYXV0aGVudGljYXRpb25Db25maWd1cmF0aW9uOiB7XG4gICAgICAgICAgICAgICAgICAgIHNlY3JldFRva2VuOiB0aGlzLnByb3BzLm9hdXRoVG9rZW4udG9TdHJpbmcoKSxcbiAgICAgICAgICAgICAgICB9LFxuICAgICAgICAgICAgICAgIGZpbHRlcnM6IFtcbiAgICAgICAgICAgICAgICAgICAge1xuICAgICAgICAgICAgICAgICAgICAgICAganNvblBhdGg6ICckLnJlZicsXG4gICAgICAgICAgICAgICAgICAgICAgICBtYXRjaEVxdWFsczogJ3JlZnMvaGVhZHMve0JyYW5jaH0nLFxuICAgICAgICAgICAgICAgICAgICB9LFxuICAgICAgICAgICAgICAgIF0sXG4gICAgICAgICAgICAgICAgdGFyZ2V0QWN0aW9uOiB0aGlzLmFjdGlvblByb3BlcnRpZXMuYWN0aW9uTmFtZSxcbiAgICAgICAgICAgICAgICB0YXJnZXRQaXBlbGluZTogc3RhZ2UucGlwZWxpbmUucGlwZWxpbmVOYW1lLFxuICAgICAgICAgICAgICAgIHRhcmdldFBpcGVsaW5lVmVyc2lvbjogMSxcbiAgICAgICAgICAgICAgICByZWdpc3RlcldpdGhUaGlyZFBhcnR5OiB0cnVlLFxuICAgICAgICAgICAgfSk7XG4gICAgICAgIH1cbiAgICAgICAgcmV0dXJuIHtcbiAgICAgICAgICAgIGNvbmZpZ3VyYXRpb246IHtcbiAgICAgICAgICAgICAgICBPd25lcjogdGhpcy5wcm9wcy5vd25lcixcbiAgICAgICAgICAgICAgICBSZXBvOiB0aGlzLnByb3BzLnJlcG8sXG4gICAgICAgICAgICAgICAgQnJhbmNoOiB0aGlzLnByb3BzLmJyYW5jaCB8fCAnbWFzdGVyJyxcbiAgICAgICAgICAgICAgICBPQXV0aFRva2VuOiB0aGlzLnByb3BzLm9hdXRoVG9rZW4udG9TdHJpbmcoKSxcbiAgICAgICAgICAgICAgICBQb2xsRm9yU291cmNlQ2hhbmdlczogdGhpcy5wcm9wcy50cmlnZ2VyID09PSBHaXRIdWJUcmlnZ2VyLlBPTEwsXG4gICAgICAgICAgICB9LFxuICAgICAgICB9O1xuICAgIH1cbn1cbiJdfQ==