"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.CloudFormationDeleteStackAction = exports.CloudFormationCreateUpdateStackAction = exports.CloudFormationCreateReplaceChangeSetAction = exports.CloudFormationExecuteChangeSetAction = void 0;
const cloudformation = require("../../../aws-cloudformation"); // Automatically re-written from '@aws-cdk/aws-cloudformation'
const codepipeline = require("../../../aws-codepipeline"); // Automatically re-written from '@aws-cdk/aws-codepipeline'
const iam = require("../../../aws-iam"); // Automatically re-written from '@aws-cdk/aws-iam'
const cdk = require("../../../core"); // Automatically re-written from '@aws-cdk/core'
const action_1 = require("../action");
/**
 * Base class for Actions that execute CloudFormation
 */
class CloudFormationAction extends action_1.Action {
    constructor(props, inputs) {
        super({
            ...props,
            provider: 'CloudFormation',
            category: codepipeline.ActionCategory.DEPLOY,
            artifactBounds: {
                minInputs: 0,
                maxInputs: 10,
                minOutputs: 0,
                maxOutputs: 1,
            },
            inputs,
            outputs: props.outputFileName
                ? [props.output || new codepipeline.Artifact(`${props.actionName}_${props.stackName}_Artifact`)]
                : undefined,
        });
        this.props = props;
    }
    bound(_scope, _stage, options) {
        const singletonPolicy = SingletonPolicy.forRole(options.role);
        if ((this.actionProperties.outputs || []).length > 0) {
            options.bucket.grantReadWrite(singletonPolicy);
        }
        else if ((this.actionProperties.inputs || []).length > 0) {
            options.bucket.grantRead(singletonPolicy);
        }
        return {
            configuration: {
                StackName: this.props.stackName,
                OutputFileName: this.props.outputFileName,
            },
        };
    }
}
/**
 * CodePipeline action to execute a prepared change set.
 */
class CloudFormationExecuteChangeSetAction extends CloudFormationAction {
    constructor(props) {
        super(props, undefined);
        this.props2 = props;
    }
    bound(scope, stage, options) {
        SingletonPolicy.forRole(options.role).grantExecuteChangeSet(this.props2);
        const actionConfig = super.bound(scope, stage, options);
        return {
            ...actionConfig,
            configuration: {
                ...actionConfig.configuration,
                ActionMode: 'CHANGE_SET_EXECUTE',
                ChangeSetName: this.props2.changeSetName,
            },
        };
    }
}
exports.CloudFormationExecuteChangeSetAction = CloudFormationExecuteChangeSetAction;
// tslint:enable:max-line-length
/**
 * Base class for all CloudFormation actions that execute or stage deployments.
 */
class CloudFormationDeployAction extends CloudFormationAction {
    constructor(props, inputs) {
        super(props, (props.extraInputs || []).concat(inputs || []));
        this.props2 = props;
    }
    /**
     * Add statement to the service role assumed by CloudFormation while executing this action.
     */
    addToDeploymentRolePolicy(statement) {
        return this.getDeploymentRole('method addToRolePolicy()').addToPolicy(statement);
    }
    get deploymentRole() {
        return this.getDeploymentRole('property role()');
    }
    bound(scope, stage, options) {
        if (this.props2.deploymentRole) {
            this._deploymentRole = this.props2.deploymentRole;
        }
        else {
            const roleStack = cdk.Stack.of(options.role);
            const pipelineStack = cdk.Stack.of(scope);
            if (roleStack.account !== pipelineStack.account) {
                // pass role is not allowed for cross-account access - so,
                // create the deployment Role in the other account!
                this._deploymentRole = new iam.Role(roleStack, `${stage.pipeline.node.uniqueId}-${stage.stageName}-${this.actionProperties.actionName}-DeploymentRole`, {
                    assumedBy: new iam.ServicePrincipal('cloudformation.amazonaws.com'),
                    roleName: cdk.PhysicalName.GENERATE_IF_NEEDED,
                });
            }
            else {
                this._deploymentRole = new iam.Role(scope, 'Role', {
                    assumedBy: new iam.ServicePrincipal('cloudformation.amazonaws.com'),
                });
            }
            // the deployment role might need read access to the pipeline's bucket
            // (for example, if it's deploying a Lambda function),
            // and even if it has admin permissions, it won't be enough,
            // as it needs to be added to the key's resource policy
            // (and the bucket's, if the access is cross-account)
            options.bucket.grantRead(this._deploymentRole);
            if (this.props2.adminPermissions) {
                this._deploymentRole.addToPolicy(new iam.PolicyStatement({
                    actions: ['*'],
                    resources: ['*'],
                }));
            }
        }
        SingletonPolicy.forRole(options.role).grantPassRole(this._deploymentRole);
        const capabilities = this.props2.adminPermissions && this.props2.capabilities === undefined
            ? [cloudformation.CloudFormationCapabilities.NAMED_IAM]
            : this.props2.capabilities;
        const actionConfig = super.bound(scope, stage, options);
        return {
            ...actionConfig,
            configuration: {
                ...actionConfig.configuration,
                // None evaluates to empty string which is falsey and results in undefined
                Capabilities: parseCapabilities(capabilities),
                RoleArn: this.deploymentRole.roleArn,
                ParameterOverrides: cdk.Stack.of(scope).toJsonString(this.props2.parameterOverrides),
                TemplateConfiguration: this.props2.templateConfiguration
                    ? this.props2.templateConfiguration.location
                    : undefined,
                StackName: this.props2.stackName,
            },
        };
    }
    getDeploymentRole(member) {
        if (this._deploymentRole) {
            return this._deploymentRole;
        }
        else {
            throw new Error(`Cannot use the ${member} before the Action has been added to a Pipeline`);
        }
    }
}
/**
 * CodePipeline action to prepare a change set.
 *
 * Creates the change set if it doesn't exist based on the stack name and template that you submit.
 * If the change set exists, AWS CloudFormation deletes it, and then creates a new one.
 */
class CloudFormationCreateReplaceChangeSetAction extends CloudFormationDeployAction {
    constructor(props) {
        super(props, props.templateConfiguration
            ? [props.templatePath.artifact, props.templateConfiguration.artifact]
            : [props.templatePath.artifact]);
        this.props3 = props;
    }
    bound(scope, stage, options) {
        // the super call order is to preserve the existing order of statements in policies
        const actionConfig = super.bound(scope, stage, options);
        SingletonPolicy.forRole(options.role).grantCreateReplaceChangeSet(this.props3);
        return {
            ...actionConfig,
            configuration: {
                ...actionConfig.configuration,
                ActionMode: 'CHANGE_SET_REPLACE',
                ChangeSetName: this.props3.changeSetName,
                TemplatePath: this.props3.templatePath.location,
            },
        };
    }
}
exports.CloudFormationCreateReplaceChangeSetAction = CloudFormationCreateReplaceChangeSetAction;
/**
 * CodePipeline action to deploy a stack.
 *
 * Creates the stack if the specified stack doesn't exist. If the stack exists,
 * AWS CloudFormation updates the stack. Use this action to update existing
 * stacks.
 *
 * AWS CodePipeline won't replace the stack, and will fail deployment if the
 * stack is in a failed state. Use `ReplaceOnFailure` for an action that
 * will delete and recreate the stack to try and recover from failed states.
 *
 * Use this action to automatically replace failed stacks without recovering or
 * troubleshooting them. You would typically choose this mode for testing.
 */
class CloudFormationCreateUpdateStackAction extends CloudFormationDeployAction {
    constructor(props) {
        super(props, props.templateConfiguration
            ? [props.templatePath.artifact, props.templateConfiguration.artifact]
            : [props.templatePath.artifact]);
        this.props3 = props;
    }
    bound(scope, stage, options) {
        // the super call order is to preserve the existing order of statements in policies
        const actionConfig = super.bound(scope, stage, options);
        SingletonPolicy.forRole(options.role).grantCreateUpdateStack(this.props3);
        return {
            ...actionConfig,
            configuration: {
                ...actionConfig.configuration,
                ActionMode: this.props3.replaceOnFailure ? 'REPLACE_ON_FAILURE' : 'CREATE_UPDATE',
                TemplatePath: this.props3.templatePath.location,
            },
        };
    }
}
exports.CloudFormationCreateUpdateStackAction = CloudFormationCreateUpdateStackAction;
/**
 * CodePipeline action to delete a stack.
 *
 * Deletes a stack. If you specify a stack that doesn't exist, the action completes successfully
 * without deleting a stack.
 */
class CloudFormationDeleteStackAction extends CloudFormationDeployAction {
    constructor(props) {
        super(props, undefined);
        this.props3 = props;
    }
    bound(scope, stage, options) {
        // the super call order is to preserve the existing order of statements in policies
        const actionConfig = super.bound(scope, stage, options);
        SingletonPolicy.forRole(options.role).grantDeleteStack(this.props3);
        return {
            ...actionConfig,
            configuration: {
                ...actionConfig.configuration,
                ActionMode: 'DELETE_ONLY',
            },
        };
    }
}
exports.CloudFormationDeleteStackAction = CloudFormationDeleteStackAction;
/**
 * Manages a bunch of singleton-y statements on the policy of an IAM Role.
 * Dedicated methods can be used to add specific permissions to the role policy
 * using as few statements as possible (adding resources to existing compatible
 * statements instead of adding new statements whenever possible).
 *
 * Statements created outside of this class are not considered when adding new
 * permissions.
 */
class SingletonPolicy extends cdk.Construct {
    constructor(role) {
        super(role, SingletonPolicy.UUID);
        this.role = role;
        this.statements = {};
        this.grantPrincipal = role;
    }
    /**
     * Obtain a SingletonPolicy for a given role.
     * @param role the Role this policy is bound to.
     * @returns the SingletonPolicy for this role.
     */
    static forRole(role) {
        const found = role.node.tryFindChild(SingletonPolicy.UUID);
        return found || new SingletonPolicy(role);
    }
    grantExecuteChangeSet(props) {
        this.statementFor({
            actions: [
                'cloudformation:DescribeStacks',
                'cloudformation:DescribeChangeSet',
                'cloudformation:ExecuteChangeSet',
            ],
            conditions: { StringEqualsIfExists: { 'cloudformation:ChangeSetName': props.changeSetName } },
        }).addResources(this.stackArnFromProps(props));
    }
    grantCreateReplaceChangeSet(props) {
        this.statementFor({
            actions: [
                'cloudformation:CreateChangeSet',
                'cloudformation:DeleteChangeSet',
                'cloudformation:DescribeChangeSet',
                'cloudformation:DescribeStacks',
            ],
            conditions: { StringEqualsIfExists: { 'cloudformation:ChangeSetName': props.changeSetName } },
        }).addResources(this.stackArnFromProps(props));
    }
    grantCreateUpdateStack(props) {
        const actions = [
            'cloudformation:DescribeStack*',
            'cloudformation:CreateStack',
            'cloudformation:UpdateStack',
            'cloudformation:GetTemplate*',
            'cloudformation:ValidateTemplate',
            'cloudformation:GetStackPolicy',
            'cloudformation:SetStackPolicy',
        ];
        if (props.replaceOnFailure) {
            actions.push('cloudformation:DeleteStack');
        }
        this.statementFor({ actions }).addResources(this.stackArnFromProps(props));
    }
    grantDeleteStack(props) {
        this.statementFor({
            actions: [
                'cloudformation:DescribeStack*',
                'cloudformation:DeleteStack',
            ],
        }).addResources(this.stackArnFromProps(props));
    }
    grantPassRole(role) {
        this.statementFor({ actions: ['iam:PassRole'] }).addResources(role.roleArn);
    }
    statementFor(template) {
        const key = keyFor(template);
        if (!(key in this.statements)) {
            this.statements[key] = new iam.PolicyStatement({ actions: template.actions });
            if (template.conditions) {
                this.statements[key].addConditions(template.conditions);
            }
            this.role.addToPolicy(this.statements[key]);
        }
        return this.statements[key];
        function keyFor(props) {
            const actions = `${props.actions.sort().join('\x1F')}`;
            const conditions = formatConditions(props.conditions);
            return `${actions}\x1D${conditions}`;
            function formatConditions(cond) {
                if (cond == null) {
                    return '';
                }
                let result = '';
                for (const op of Object.keys(cond).sort()) {
                    result += `${op}\x1E`;
                    const condition = cond[op];
                    for (const attribute of Object.keys(condition).sort()) {
                        const value = condition[attribute];
                        result += `${value}\x1F`;
                    }
                }
                return result;
            }
        }
    }
    stackArnFromProps(props) {
        return cdk.Stack.of(this).formatArn({
            region: props.region,
            service: 'cloudformation',
            resource: 'stack',
            resourceName: `${props.stackName}/*`,
        });
    }
}
SingletonPolicy.UUID = '8389e75f-0810-4838-bf64-d6f85a95cf83';
function parseCapabilities(capabilities) {
    if (capabilities === undefined) {
        return undefined;
    }
    else if (capabilities.length === 1) {
        const capability = capabilities.toString();
        return (capability === '') ? undefined : capability;
    }
    else if (capabilities.length > 1) {
        return capabilities.join(',');
    }
    return undefined;
}
//# sourceMappingURL=data:application/json;base64,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