"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.Resolver = exports.MappingTemplate = exports.Values = exports.AttributeValuesStep = exports.AttributeValues = exports.PartitionKey = exports.PrimaryKey = exports.SortKeyStep = exports.PartitionKeyStep = exports.Assign = exports.KeyCondition = exports.LambdaDataSource = exports.HttpDataSource = exports.DynamoDbDataSource = exports.NoneDataSource = exports.BackedDataSource = exports.BaseDataSource = exports.GraphQLApi = exports.FieldLogLevel = exports.UserPoolDefaultAction = exports.AuthorizationType = void 0;
const aws_iam_1 = require("../../aws-iam"); // Automatically re-written from '@aws-cdk/aws-iam'
const core_1 = require("../../core"); // Automatically re-written from '@aws-cdk/core'
const fs_1 = require("fs");
const appsync_generated_1 = require("./appsync.generated");
/**
 * enum with all possible values for AppSync authorization type
 */
var AuthorizationType;
(function (AuthorizationType) {
    /**
     * API Key authorization type
     */
    AuthorizationType["API_KEY"] = "API_KEY";
    /**
     * AWS IAM authorization type. Can be used with Cognito Identity Pool federated credentials
     */
    AuthorizationType["IAM"] = "AWS_IAM";
    /**
     * Cognito User Pool authorization type
     */
    AuthorizationType["USER_POOL"] = "AMAZON_COGNITO_USER_POOLS";
    /**
     * OpenID Connect authorization type
     */
    AuthorizationType["OIDC"] = "OPENID_CONNECT";
})(AuthorizationType = exports.AuthorizationType || (exports.AuthorizationType = {}));
/**
 * enum with all possible values for Cognito user-pool default actions
 */
var UserPoolDefaultAction;
(function (UserPoolDefaultAction) {
    /**
     * ALLOW access to API
     */
    UserPoolDefaultAction["ALLOW"] = "ALLOW";
    /**
     * DENY access to API
     */
    UserPoolDefaultAction["DENY"] = "DENY";
})(UserPoolDefaultAction = exports.UserPoolDefaultAction || (exports.UserPoolDefaultAction = {}));
/**
 * log-level for fields in AppSync
 */
var FieldLogLevel;
(function (FieldLogLevel) {
    /**
     * No logging
     */
    FieldLogLevel["NONE"] = "NONE";
    /**
     * Error logging
     */
    FieldLogLevel["ERROR"] = "ERROR";
    /**
     * All logging
     */
    FieldLogLevel["ALL"] = "ALL";
})(FieldLogLevel = exports.FieldLogLevel || (exports.FieldLogLevel = {}));
/**
 * An AppSync GraphQL API
 */
class GraphQLApi extends core_1.Construct {
    constructor(scope, id, props) {
        var _a, _b, _c, _d, _e, _f, _g, _h, _j, _k;
        super(scope, id);
        this.validateAuthorizationProps(props);
        const defaultAuthorizationType = ((_b = (_a = props.authorizationConfig) === null || _a === void 0 ? void 0 : _a.defaultAuthorization) === null || _b === void 0 ? void 0 : _b.authorizationType) ||
            AuthorizationType.API_KEY;
        let apiLogsRole;
        if (props.logConfig) {
            apiLogsRole = new aws_iam_1.Role(this, 'ApiLogsRole', {
                assumedBy: new aws_iam_1.ServicePrincipal('appsync'),
            });
            apiLogsRole.addManagedPolicy(aws_iam_1.ManagedPolicy.fromAwsManagedPolicyName('service-role/AWSAppSyncPushToCloudWatchLogs'));
        }
        this.api = new appsync_generated_1.CfnGraphQLApi(this, 'Resource', {
            name: props.name,
            authenticationType: defaultAuthorizationType,
            ...(props.logConfig && {
                logConfig: {
                    cloudWatchLogsRoleArn: apiLogsRole ? apiLogsRole.roleArn : undefined,
                    excludeVerboseContent: props.logConfig.excludeVerboseContent,
                    fieldLogLevel: props.logConfig.fieldLogLevel
                        ? props.logConfig.fieldLogLevel.toString()
                        : undefined,
                },
            }),
            openIdConnectConfig: ((_d = (_c = props.authorizationConfig) === null || _c === void 0 ? void 0 : _c.defaultAuthorization) === null || _d === void 0 ? void 0 : _d.authorizationType) ===
                AuthorizationType.OIDC
                ? this.formatOpenIdConnectConfig(props.authorizationConfig.defaultAuthorization
                    .openIdConnectConfig)
                : undefined,
            userPoolConfig: ((_f = (_e = props.authorizationConfig) === null || _e === void 0 ? void 0 : _e.defaultAuthorization) === null || _f === void 0 ? void 0 : _f.authorizationType) ===
                AuthorizationType.USER_POOL
                ? this.formatUserPoolConfig(props.authorizationConfig.defaultAuthorization.userPoolConfig)
                : undefined,
            additionalAuthenticationProviders: this.formatAdditionalAuthenticationProviders(props),
        });
        this.apiId = this.api.attrApiId;
        this.arn = this.api.attrArn;
        this.graphQlUrl = this.api.attrGraphQlUrl;
        this.name = this.api.name;
        if (defaultAuthorizationType === AuthorizationType.API_KEY ||
            ((_h = (_g = props.authorizationConfig) === null || _g === void 0 ? void 0 : _g.additionalAuthorizationModes) === null || _h === void 0 ? void 0 : _h.findIndex((authMode) => authMode.authorizationType === AuthorizationType.API_KEY)) !== -1) {
            const apiKeyConfig = ((_k = (_j = props.authorizationConfig) === null || _j === void 0 ? void 0 : _j.defaultAuthorization) === null || _k === void 0 ? void 0 : _k.apiKeyConfig) || {
                name: 'DefaultAPIKey',
                description: 'Default API Key created by CDK',
            };
            this.createAPIKey(apiKeyConfig);
        }
        let definition;
        if (props.schemaDefinition) {
            definition = props.schemaDefinition;
        }
        else if (props.schemaDefinitionFile) {
            definition = fs_1.readFileSync(props.schemaDefinitionFile).toString('UTF-8');
        }
        else {
            throw new Error('Missing Schema definition. Provide schemaDefinition or schemaDefinitionFile');
        }
        this.schema = new appsync_generated_1.CfnGraphQLSchema(this, 'Schema', {
            apiId: this.apiId,
            definition,
        });
    }
    /**
     * the configured API key, if present
     */
    get apiKey() {
        return this._apiKey;
    }
    /**
     * add a new dummy data source to this API
     * @param name The name of the data source
     * @param description The description of the data source
     */
    addNoneDataSource(name, description) {
        return new NoneDataSource(this, `${name}DS`, {
            api: this,
            description,
            name,
        });
    }
    /**
     * add a new DynamoDB data source to this API
     * @param name The name of the data source
     * @param description The description of the data source
     * @param table The DynamoDB table backing this data source [disable-awslint:ref-via-interface]
     */
    addDynamoDbDataSource(name, description, table) {
        return new DynamoDbDataSource(this, `${name}DS`, {
            api: this,
            description,
            name,
            table,
        });
    }
    /**
     * add a new http data source to this API
     * @param name The name of the data source
     * @param description The description of the data source
     * @param endpoint The http endpoint
     */
    addHttpDataSource(name, description, endpoint) {
        return new HttpDataSource(this, `${name}DS`, {
            api: this,
            description,
            endpoint,
            name,
        });
    }
    /**
     * add a new Lambda data source to this API
     * @param name The name of the data source
     * @param description The description of the data source
     * @param lambdaFunction The Lambda function to call to interact with this data source
     */
    addLambdaDataSource(name, description, lambdaFunction) {
        return new LambdaDataSource(this, `${name}DS`, {
            api: this,
            description,
            name,
            lambdaFunction,
        });
    }
    validateAuthorizationProps(props) {
        var _a, _b, _c, _d, _e, _f, _g;
        const defaultAuthorizationType = ((_b = (_a = props.authorizationConfig) === null || _a === void 0 ? void 0 : _a.defaultAuthorization) === null || _b === void 0 ? void 0 : _b.authorizationType) ||
            AuthorizationType.API_KEY;
        if (defaultAuthorizationType === AuthorizationType.OIDC &&
            !((_d = (_c = props.authorizationConfig) === null || _c === void 0 ? void 0 : _c.defaultAuthorization) === null || _d === void 0 ? void 0 : _d.openIdConnectConfig)) {
            throw new Error('Missing default OIDC Configuration');
        }
        if (defaultAuthorizationType === AuthorizationType.USER_POOL &&
            !((_f = (_e = props.authorizationConfig) === null || _e === void 0 ? void 0 : _e.defaultAuthorization) === null || _f === void 0 ? void 0 : _f.userPoolConfig)) {
            throw new Error('Missing default User Pool Configuration');
        }
        if ((_g = props.authorizationConfig) === null || _g === void 0 ? void 0 : _g.additionalAuthorizationModes) {
            props.authorizationConfig.additionalAuthorizationModes.forEach((authorizationMode) => {
                if (authorizationMode.authorizationType === AuthorizationType.API_KEY &&
                    defaultAuthorizationType === AuthorizationType.API_KEY) {
                    throw new Error("You can't duplicate API_KEY in additional authorization config. See https://docs.aws.amazon.com/appsync/latest/devguide/security.html");
                }
                if (authorizationMode.authorizationType === AuthorizationType.IAM &&
                    defaultAuthorizationType === AuthorizationType.IAM) {
                    throw new Error("You can't duplicate IAM in additional authorization config. See https://docs.aws.amazon.com/appsync/latest/devguide/security.html");
                }
                if (authorizationMode.authorizationType === AuthorizationType.OIDC &&
                    !authorizationMode.openIdConnectConfig) {
                    throw new Error('Missing OIDC Configuration inside an additional authorization mode');
                }
                if (authorizationMode.authorizationType ===
                    AuthorizationType.USER_POOL &&
                    !authorizationMode.userPoolConfig) {
                    throw new Error('Missing User Pool Configuration inside an additional authorization mode');
                }
            });
        }
    }
    formatOpenIdConnectConfig(config) {
        return {
            authTtl: config.tokenExpiryFromAuth,
            clientId: config.clientId,
            iatTtl: config.tokenExpiryFromIssue,
            issuer: config.oidcProvider,
        };
    }
    formatUserPoolConfig(config) {
        return {
            userPoolId: config.userPool.userPoolId,
            awsRegion: config.userPool.stack.region,
            appIdClientRegex: config.appIdClientRegex,
            defaultAction: config.defaultAction || 'ALLOW',
        };
    }
    createAPIKey(config) {
        let expires;
        if (config.expires) {
            expires = new Date(config.expires).valueOf();
            const days = (d) => Date.now() + core_1.Duration.days(d).toMilliseconds();
            if (expires < days(1) || expires > days(365)) {
                throw Error('API key expiration must be between 1 and 365 days.');
            }
            expires = Math.round(expires / 1000);
        }
        const key = new appsync_generated_1.CfnApiKey(this, `${config.name || 'DefaultAPIKey'}ApiKey`, {
            expires,
            description: config.description || 'Default API Key created by CDK',
            apiId: this.apiId,
        });
        this._apiKey = key.attrApiKey;
    }
    formatAdditionalAuthorizationModes(authModes) {
        return authModes.reduce((acc, authMode) => [
            ...acc,
            {
                authenticationType: authMode.authorizationType,
                userPoolConfig: authMode.authorizationType === AuthorizationType.USER_POOL
                    ? this.formatUserPoolConfig(authMode.userPoolConfig)
                    : undefined,
                openIdConnectConfig: authMode.authorizationType === AuthorizationType.OIDC
                    ? this.formatOpenIdConnectConfig(authMode.openIdConnectConfig)
                    : undefined,
            },
        ], []);
    }
    formatAdditionalAuthenticationProviders(props) {
        var _a;
        const authModes = (_a = props.authorizationConfig) === null || _a === void 0 ? void 0 : _a.additionalAuthorizationModes;
        return authModes ? this.formatAdditionalAuthorizationModes(authModes) : undefined;
    }
}
exports.GraphQLApi = GraphQLApi;
/**
 * Abstract AppSync datasource implementation. Do not use directly but use subclasses for concrete datasources
 */
class BaseDataSource extends core_1.Construct {
    constructor(scope, id, props, extended) {
        var _a;
        super(scope, id);
        if (extended.type !== 'NONE') {
            this.serviceRole = props.serviceRole || new aws_iam_1.Role(this, 'ServiceRole', { assumedBy: new aws_iam_1.ServicePrincipal('appsync') });
        }
        this.ds = new appsync_generated_1.CfnDataSource(this, 'Resource', {
            apiId: props.api.apiId,
            name: props.name,
            description: props.description,
            serviceRoleArn: (_a = this.serviceRole) === null || _a === void 0 ? void 0 : _a.roleArn,
            ...extended,
        });
        this.name = props.name;
        this.api = props.api;
    }
    /**
     * creates a new resolver for this datasource and API using the given properties
     */
    createResolver(props) {
        return new Resolver(this, `${props.typeName}${props.fieldName}Resolver`, {
            api: this.api,
            dataSource: this,
            ...props,
        });
    }
}
exports.BaseDataSource = BaseDataSource;
/**
 * Abstract AppSync datasource implementation. Do not use directly but use subclasses for resource backed datasources
 */
class BackedDataSource extends BaseDataSource {
    constructor(scope, id, props, extended) {
        super(scope, id, props, extended);
        this.grantPrincipal = this.serviceRole;
    }
}
exports.BackedDataSource = BackedDataSource;
/**
 * An AppSync dummy datasource
 */
class NoneDataSource extends BaseDataSource {
    constructor(scope, id, props) {
        super(scope, id, props, {
            type: 'NONE',
        });
    }
}
exports.NoneDataSource = NoneDataSource;
/**
 * An AppSync datasource backed by a DynamoDB table
 */
class DynamoDbDataSource extends BackedDataSource {
    constructor(scope, id, props) {
        super(scope, id, props, {
            type: 'AMAZON_DYNAMODB',
            dynamoDbConfig: {
                tableName: props.table.tableName,
                awsRegion: props.table.stack.region,
                useCallerCredentials: props.useCallerCredentials,
            },
        });
        if (props.readOnlyAccess) {
            props.table.grantReadData(this);
        }
        else {
            props.table.grantReadWriteData(this);
        }
    }
}
exports.DynamoDbDataSource = DynamoDbDataSource;
/**
 * An AppSync datasource backed by a http endpoint
 */
class HttpDataSource extends BaseDataSource {
    constructor(scope, id, props) {
        super(scope, id, props, {
            httpConfig: {
                endpoint: props.endpoint,
            },
            type: 'HTTP',
        });
    }
}
exports.HttpDataSource = HttpDataSource;
/**
 * An AppSync datasource backed by a Lambda function
 */
class LambdaDataSource extends BackedDataSource {
    constructor(scope, id, props) {
        super(scope, id, props, {
            type: 'AWS_LAMBDA',
            lambdaConfig: {
                lambdaFunctionArn: props.lambdaFunction.functionArn,
            },
        });
        props.lambdaFunction.grantInvoke(this);
    }
}
exports.LambdaDataSource = LambdaDataSource;
function concatAndDedup(left, right) {
    return left.concat(right).filter((elem, index, self) => {
        return index === self.indexOf(elem);
    });
}
/**
 * Utility class to represent DynamoDB key conditions.
 */
class BaseKeyCondition {
    and(cond) {
        return new (class extends BaseKeyCondition {
            constructor(left, right) {
                super();
                this.left = left;
                this.right = right;
            }
            renderCondition() {
                return `${this.left.renderCondition()} AND ${this.right.renderCondition()}`;
            }
            keyNames() {
                return concatAndDedup(this.left.keyNames(), this.right.keyNames());
            }
            args() {
                return concatAndDedup(this.left.args(), this.right.args());
            }
        })(this, cond);
    }
    renderExpressionNames() {
        return this.keyNames()
            .map((keyName) => {
            return `"#${keyName}" : "${keyName}"`;
        })
            .join(', ');
    }
    renderExpressionValues() {
        return this.args()
            .map((arg) => {
            return `":${arg}" : $util.dynamodb.toDynamoDBJson($ctx.args.${arg})`;
        })
            .join(', ');
    }
}
/**
 * Utility class to represent DynamoDB "begins_with" key conditions.
 */
class BeginsWith extends BaseKeyCondition {
    constructor(keyName, arg) {
        super();
        this.keyName = keyName;
        this.arg = arg;
    }
    renderCondition() {
        return `begins_with(#${this.keyName}, :${this.arg})`;
    }
    keyNames() {
        return [this.keyName];
    }
    args() {
        return [this.arg];
    }
}
/**
 * Utility class to represent DynamoDB binary key conditions.
 */
class BinaryCondition extends BaseKeyCondition {
    constructor(keyName, op, arg) {
        super();
        this.keyName = keyName;
        this.op = op;
        this.arg = arg;
    }
    renderCondition() {
        return `#${this.keyName} ${this.op} :${this.arg}`;
    }
    keyNames() {
        return [this.keyName];
    }
    args() {
        return [this.arg];
    }
}
/**
 * Utility class to represent DynamoDB "between" key conditions.
 */
class Between extends BaseKeyCondition {
    constructor(keyName, arg1, arg2) {
        super();
        this.keyName = keyName;
        this.arg1 = arg1;
        this.arg2 = arg2;
    }
    renderCondition() {
        return `#${this.keyName} BETWEEN :${this.arg1} AND :${this.arg2}`;
    }
    keyNames() {
        return [this.keyName];
    }
    args() {
        return [this.arg1, this.arg2];
    }
}
/**
 * Factory class for DynamoDB key conditions.
 */
class KeyCondition {
    constructor(cond) {
        this.cond = cond;
    }
    /**
     * Condition k = arg, true if the key attribute k is equal to the Query argument
     */
    static eq(keyName, arg) {
        return new KeyCondition(new BinaryCondition(keyName, '=', arg));
    }
    /**
     * Condition k < arg, true if the key attribute k is less than the Query argument
     */
    static lt(keyName, arg) {
        return new KeyCondition(new BinaryCondition(keyName, '<', arg));
    }
    /**
     * Condition k <= arg, true if the key attribute k is less than or equal to the Query argument
     */
    static le(keyName, arg) {
        return new KeyCondition(new BinaryCondition(keyName, '<=', arg));
    }
    /**
     * Condition k > arg, true if the key attribute k is greater than the the Query argument
     */
    static gt(keyName, arg) {
        return new KeyCondition(new BinaryCondition(keyName, '>', arg));
    }
    /**
     * Condition k >= arg, true if the key attribute k is greater or equal to the Query argument
     */
    static ge(keyName, arg) {
        return new KeyCondition(new BinaryCondition(keyName, '>=', arg));
    }
    /**
     * Condition (k, arg). True if the key attribute k begins with the Query argument.
     */
    static beginsWith(keyName, arg) {
        return new KeyCondition(new BeginsWith(keyName, arg));
    }
    /**
     * Condition k BETWEEN arg1 AND arg2, true if k >= arg1 and k <= arg2.
     */
    static between(keyName, arg1, arg2) {
        return new KeyCondition(new Between(keyName, arg1, arg2));
    }
    /**
     * Conjunction between two conditions.
     */
    and(keyCond) {
        return new KeyCondition(this.cond.and(keyCond.cond));
    }
    /**
     * Renders the key condition to a VTL string.
     */
    renderTemplate() {
        return `"query" : {
            "expression" : "${this.cond.renderCondition()}",
            "expressionNames" : {
                ${this.cond.renderExpressionNames()}
            },
            "expressionValues" : {
                ${this.cond.renderExpressionValues()}
            }
        }`;
    }
}
exports.KeyCondition = KeyCondition;
/**
 * Utility class representing the assigment of a value to an attribute.
 */
class Assign {
    constructor(attr, arg) {
        this.attr = attr;
        this.arg = arg;
    }
    /**
     * Renders the assignment as a VTL string.
     */
    renderAsAssignment() {
        return `"${this.attr}" : $util.dynamodb.toDynamoDBJson(${this.arg})`;
    }
    /**
     * Renders the assignment as a map element.
     */
    putInMap(map) {
        return `$util.qr($${map}.put("${this.attr}", ${this.arg}))`;
    }
}
exports.Assign = Assign;
/**
 * Utility class to allow assigning a value or an auto-generated id
 * to a partition key.
 */
class PartitionKeyStep {
    constructor(key) {
        this.key = key;
    }
    /**
     * Assign an auto-generated value to the partition key.
     */
    is(val) {
        return new PartitionKey(new Assign(this.key, `$ctx.args.${val}`));
    }
    /**
     * Assign an auto-generated value to the partition key.
     */
    auto() {
        return new PartitionKey(new Assign(this.key, '$util.autoId()'));
    }
}
exports.PartitionKeyStep = PartitionKeyStep;
/**
 * Utility class to allow assigning a value or an auto-generated id
 * to a sort key.
 */
class SortKeyStep {
    constructor(pkey, skey) {
        this.pkey = pkey;
        this.skey = skey;
    }
    /**
     * Assign an auto-generated value to the sort key.
     */
    is(val) {
        return new PrimaryKey(this.pkey, new Assign(this.skey, `$ctx.args.${val}`));
    }
    /**
     * Assign an auto-generated value to the sort key.
     */
    auto() {
        return new PrimaryKey(this.pkey, new Assign(this.skey, '$util.autoId()'));
    }
}
exports.SortKeyStep = SortKeyStep;
/**
 * Specifies the assignment to the primary key. It either
 * contains the full primary key or only the partition key.
 */
class PrimaryKey {
    constructor(pkey, skey) {
        this.pkey = pkey;
        this.skey = skey;
    }
    /**
     * Allows assigning a value to the partition key.
     */
    static partition(key) {
        return new PartitionKeyStep(key);
    }
    /**
     * Renders the key assignment to a VTL string.
     */
    renderTemplate() {
        const assignments = [this.pkey.renderAsAssignment()];
        if (this.skey) {
            assignments.push(this.skey.renderAsAssignment());
        }
        return `"key" : {
      ${assignments.join(',')}
    }`;
    }
}
exports.PrimaryKey = PrimaryKey;
/**
 * Specifies the assignment to the partition key. It can be
 * enhanced with the assignment of the sort key.
 */
class PartitionKey extends PrimaryKey {
    constructor(pkey) {
        super(pkey);
    }
    /**
     * Allows assigning a value to the sort key.
     */
    sort(key) {
        return new SortKeyStep(this.pkey, key);
    }
}
exports.PartitionKey = PartitionKey;
/**
 * Specifies the attribute value assignments.
 */
class AttributeValues {
    constructor(container, assignments = []) {
        this.container = container;
        this.assignments = assignments;
    }
    /**
     * Allows assigning a value to the specified attribute.
     */
    attribute(attr) {
        return new AttributeValuesStep(attr, this.container, this.assignments);
    }
    /**
     * Renders the variables required for `renderTemplate`.
     */
    renderVariables() {
        return `#set($input = ${this.container})
      ${this.assignments.map(a => a.putInMap('input')).join('\n')}`;
    }
    /**
     * Renders the attribute value assingments to a VTL string.
     */
    renderTemplate() {
        return '"attributeValues": $util.dynamodb.toMapValuesJson($input)';
    }
}
exports.AttributeValues = AttributeValues;
/**
 * Utility class to allow assigning a value to an attribute.
 */
class AttributeValuesStep {
    constructor(attr, container, assignments) {
        this.attr = attr;
        this.container = container;
        this.assignments = assignments;
    }
    /**
     * Assign the value to the current attribute.
     */
    is(val) {
        this.assignments.push(new Assign(this.attr, val));
        return new AttributeValues(this.container, this.assignments);
    }
}
exports.AttributeValuesStep = AttributeValuesStep;
/**
 * Factory class for attribute value assignments.
 */
class Values {
    /**
     * Treats the specified object as a map of assignments, where the property
     * names represent attribute names. It’s opinionated about how it represents
     * some of the nested objects: e.g., it will use lists (“L”) rather than sets
     * (“SS”, “NS”, “BS”). By default it projects the argument container ("$ctx.args").
     */
    static projecting(arg) {
        return new AttributeValues('$ctx.args' + (arg ? `.${arg}` : ''));
    }
    /**
     * Allows assigning a value to the specified attribute.
     */
    static attribute(attr) {
        return new AttributeValues('{}').attribute(attr);
    }
}
exports.Values = Values;
/**
 * MappingTemplates for AppSync resolvers
 */
class MappingTemplate {
    /**
     * Create a mapping template from the given string
     */
    static fromString(template) {
        return new StringMappingTemplate(template);
    }
    /**
     * Create a mapping template from the given file
     */
    static fromFile(fileName) {
        return new StringMappingTemplate(fs_1.readFileSync(fileName).toString('UTF-8'));
    }
    /**
     * Mapping template for a result list from DynamoDB
     */
    static dynamoDbResultList() {
        return this.fromString('$util.toJson($ctx.result.items)');
    }
    /**
     * Mapping template for a single result item from DynamoDB
     */
    static dynamoDbResultItem() {
        return this.fromString('$util.toJson($ctx.result)');
    }
    /**
     * Mapping template to scan a DynamoDB table to fetch all entries
     */
    static dynamoDbScanTable() {
        return this.fromString('{"version" : "2017-02-28", "operation" : "Scan"}');
    }
    /**
     * Mapping template to query a set of items from a DynamoDB table
     *
     * @param cond the key condition for the query
     */
    static dynamoDbQuery(cond) {
        return this.fromString(`{"version" : "2017-02-28", "operation" : "Query", ${cond.renderTemplate()}}`);
    }
    /**
     * Mapping template to get a single item from a DynamoDB table
     *
     * @param keyName the name of the hash key field
     * @param idArg the name of the Query argument
     */
    static dynamoDbGetItem(keyName, idArg) {
        return this.fromString(`{"version": "2017-02-28", "operation": "GetItem", "key": {"${keyName}": $util.dynamodb.toDynamoDBJson($ctx.args.${idArg})}}`);
    }
    /**
     * Mapping template to delete a single item from a DynamoDB table
     *
     * @param keyName the name of the hash key field
     * @param idArg the name of the Mutation argument
     */
    static dynamoDbDeleteItem(keyName, idArg) {
        return this.fromString(`{"version": "2017-02-28", "operation": "DeleteItem", "key": {"${keyName}": $util.dynamodb.toDynamoDBJson($ctx.args.${idArg})}}`);
    }
    /**
     * Mapping template to save a single item to a DynamoDB table
     *
     * @param key the assigment of Mutation values to the primary key
     * @param values the assignment of Mutation values to the table attributes
     */
    static dynamoDbPutItem(key, values) {
        return this.fromString(`
      ${values.renderVariables()}
      {
        "version": "2017-02-28",
        "operation": "PutItem",
        ${key.renderTemplate()},
        ${values.renderTemplate()}
      }`);
    }
    /**
     * Mapping template to invoke a Lambda function
     *
     * @param payload the VTL template snippet of the payload to send to the lambda.
     * If no payload is provided all available context fields are sent to the Lambda function
     */
    static lambdaRequest(payload = '$util.toJson($ctx)') {
        return this.fromString(`{"version": "2017-02-28", "operation": "Invoke", "payload": ${payload}}`);
    }
    /**
     * Mapping template to return the Lambda result to the caller
     */
    static lambdaResult() {
        return this.fromString('$util.toJson($ctx.result)');
    }
}
exports.MappingTemplate = MappingTemplate;
class StringMappingTemplate extends MappingTemplate {
    constructor(template) {
        super();
        this.template = template;
    }
    renderTemplate() {
        return this.template;
    }
}
/**
 * An AppSync resolver
 */
class Resolver extends core_1.Construct {
    constructor(scope, id, props) {
        super(scope, id);
        this.resolver = new appsync_generated_1.CfnResolver(this, 'Resource', {
            apiId: props.api.apiId,
            typeName: props.typeName,
            fieldName: props.fieldName,
            dataSourceName: props.dataSource ? props.dataSource.name : undefined,
            kind: props.pipelineConfig ? 'PIPELINE' : 'UNIT',
            requestMappingTemplate: props.requestMappingTemplate ? props.requestMappingTemplate.renderTemplate() : undefined,
            responseMappingTemplate: props.responseMappingTemplate ? props.responseMappingTemplate.renderTemplate() : undefined,
        });
        this.resolver.addDependsOn(props.api.schema);
        if (props.dataSource) {
            this.resolver.addDependsOn(props.dataSource.ds);
        }
        this.arn = this.resolver.attrResolverArn;
    }
}
exports.Resolver = Resolver;
//# sourceMappingURL=data:application/json;base64,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