"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.UsagePlan = exports.Period = void 0;
const core_1 = require("../../core"); // Automatically re-written from '@aws-cdk/core'
const apigateway_generated_1 = require("./apigateway.generated");
const util_1 = require("./util");
/**
 * Time period for which quota settings apply.
 */
var Period;
(function (Period) {
    Period["DAY"] = "DAY";
    Period["WEEK"] = "WEEK";
    Period["MONTH"] = "MONTH";
})(Period = exports.Period || (exports.Period = {}));
class UsagePlan extends core_1.Resource {
    constructor(scope, id, props = {}) {
        super(scope, id);
        this.apiStages = new Array();
        let resource;
        resource = new apigateway_generated_1.CfnUsagePlan(this, 'Resource', {
            apiStages: core_1.Lazy.anyValue({ produce: () => this.renderApiStages(this.apiStages) }),
            description: props.description,
            quota: this.renderQuota(props),
            throttle: this.renderThrottle(props.throttle),
            usagePlanName: props.name,
        });
        this.apiStages.push(...(props.apiStages || []));
        this.usagePlanId = resource.ref;
        // Add ApiKey when
        if (props.apiKey) {
            this.addApiKey(props.apiKey);
        }
    }
    /**
     * Adds an ApiKey.
     *
     * @param apiKey
     */
    addApiKey(apiKey) {
        const prefix = 'UsagePlanKeyResource';
        // Postfixing apikey id only from the 2nd child, to keep physicalIds of UsagePlanKey for existing CDK apps unmodifed.
        const id = this.node.tryFindChild(prefix) ? `${prefix}:${apiKey.node.uniqueId}` : prefix;
        new apigateway_generated_1.CfnUsagePlanKey(this, id, {
            keyId: apiKey.keyId,
            keyType: "API_KEY" /* API_KEY */,
            usagePlanId: this.usagePlanId,
        });
    }
    /**
     * Adds an apiStage.
     * @param apiStage
     */
    addApiStage(apiStage) {
        this.apiStages.push(apiStage);
    }
    /**
     *
     * @param props
     */
    renderApiStages(apiStages) {
        if (apiStages && apiStages.length > 0) {
            const stages = [];
            apiStages.forEach((apiStage) => {
                stages.push(this.createStage(apiStage));
            });
            return stages;
        }
        return undefined;
    }
    createStage(apiStage) {
        const stage = apiStage.stage ? apiStage.stage.stageName.toString() : undefined;
        const apiId = apiStage.stage ? apiStage.stage.restApi.restApiId : undefined;
        const throttle = this.renderThrottlePerMethod(apiStage.throttle);
        return {
            apiId,
            stage,
            throttle,
        };
    }
    renderQuota(props) {
        if (props.quota === undefined) {
            return undefined;
        }
        else {
            const limit = props.quota ? props.quota.limit : undefined;
            util_1.validateInteger(limit, 'Throttle quota limit');
            const ret = {
                limit: limit ? limit : undefined,
                offset: props.quota ? props.quota.offset : undefined,
                period: props.quota ? props.quota.period : undefined,
            };
            return ret;
        }
    }
    renderThrottle(props) {
        let ret;
        if (props !== undefined) {
            const burstLimit = props.burstLimit;
            util_1.validateInteger(burstLimit, 'Throttle burst limit');
            const rateLimit = props.rateLimit;
            util_1.validateInteger(rateLimit, 'Throttle rate limit');
            ret = {
                burstLimit: burstLimit ? burstLimit : undefined,
                rateLimit: rateLimit ? rateLimit : undefined,
            };
        }
        return ret;
    }
    renderThrottlePerMethod(throttlePerMethod) {
        const ret = {};
        if (throttlePerMethod && throttlePerMethod.length > 0) {
            throttlePerMethod.forEach((value) => {
                const method = value.method;
                // this methodId is resource path and method for example /GET or /pets/GET
                const methodId = `${method.resource.path}/${method.httpMethod}`;
                ret[methodId] = this.renderThrottle(value.throttle);
            });
        }
        return ret;
    }
}
exports.UsagePlan = UsagePlan;
//# sourceMappingURL=data:application/json;base64,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