"""requirements.py


This file is used by the SQLAlchemy 0.8 testing suite to mark various
optional behaviors as non-supported.

"""
from sqlalchemy.testing.requirements import SuiteRequirements

from sqlalchemy.testing import exclusions

"""
Copyright 2019 Amrit Baveja

Licensed under the Apache License, Version 2.0 (the "License");
you may not use this file except in compliance with the License.
You may obtain a copy of the License at

    http://www.apache.org/licenses/LICENSE-2.0

Unless required by applicable law or agreed to in writing, software
distributed under the License is distributed on an "AS IS" BASIS,
WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
See the License for the specific language governing permissions and
limitations under the License.
"""


# exclusions.closed() not supported on DB
# exclusions.open() special supported (atypical) in DB

class Requirements(SuiteRequirements):
    @property
    def text_type(self):
        return exclusions.open()

    @property
    def parens_in_union_contained_select_w_limit_offset(self):
        return exclusions.closed()  # this is not supported in dialect

    @property
    def parens_in_union_contained_select_wo_limit_offset(self):
        return exclusions.closed()

    @property
    def schemas(self):
        return exclusions.open()  # schemas are supported in SM

    @property
    def view_reflection(self):
        return exclusions.open()  # view reflection is supported

    @property
    def view_column_reflection(self):
        return exclusions.open()  # view col reflection is supported

    @property
    def unique_constraint_reflection(self):
        return exclusions.closed()

    @property
    def index_reflection(self):
        return exclusions.closed()

    @property
    def autocommit(self):
        return exclusions.open()

    @property
    def ctes(self):
        return exclusions.closed()
        # Common table expressions are NOT supported in this dialect

    @property
    def temp_table_reflection(self):
        return exclusions.closed()

    # although we do support temporary tables,
    # the SQLAlchemy test cases for temporary tables
    # use autogenerated (identity) columns, which fail
    # as our database DOESN'T support 'em w/ temp tables

    @property
    def independent_connections(self):
        return exclusions.closed()

    @property
    def on_update_cascade(self):
        """"target database must support ON UPDATE..CASCADE behavior in
        foreign keys."""

        return exclusions.closed()

    @property
    def cross_schema_fk_reflection(self):
        """"target database must support ON UPDATE..CASCADE behavior in
        foreign keys."""

        return exclusions.closed()

    @property
    def order_by_col_from_union(self):
        return exclusions.closed()

    @property
    def datetime_microseconds(self):
        """target dialect supports representation of Python
        datetime.datetime() with microsecond objects."""

        return exclusions.closed()

    @property
    def time_microseconds(self):
        """target dialect supports representation of Python
        datetime.time() with microsecond objects."""

        return exclusions.closed()

    @property
    def unbounded_varchar(self):
        """Target database must support VARCHAR with no length"""

        return exclusions.closed()

    @property
    def offset(self):
        return exclusions.open()

    @property
    def window_functions(self):
        """Target database must support window functions."""
        return exclusions.open()

    @property
    def precision_numerics_enotation_small(self):
        """target backend supports Decimal() objects using E notation
        to represent very small values."""
        return exclusions.open()

    @property
    def precision_numerics_enotation_large(self):
        """target backend supports Decimal() objects using E notation
        to represent very large values."""
        return exclusions.open()

    @property
    def precision_numerics_many_significant_digits(self):
        """target backend supports values with many digits on both sides,
        such as 319438950232418390.273596, 87673.594069654243

        """
        return exclusions.fails_if(lambda: True,
                                   "Throws error SQL0604N, regarding Decimal(38, 12)"
                                   )

    @property
    def precision_numerics_retains_significant_digits(self):
        """A precision numeric type will return empty significant digits,
        i.e. a value such as 10.000 will come back in Decimal form with
        the .000 maintained."""

        return exclusions.open()
