"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.ProfilingGroup = exports.ComputePlatform = void 0;
const jsiiDeprecationWarnings = require("../.warnings.jsii.js");
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const aws_iam_1 = require("@aws-cdk/aws-iam");
const core_1 = require("@aws-cdk/core");
const codeguruprofiler_generated_1 = require("./codeguruprofiler.generated");
/**
 * The compute platform of the profiling group.
 */
var ComputePlatform;
(function (ComputePlatform) {
    /**
     * Use AWS_LAMBDA if your application runs on AWS Lambda.
     */
    ComputePlatform["AWS_LAMBDA"] = "AWSLambda";
    /**
     * Use Default if your application runs on a compute platform that is not AWS Lambda,
     * such an Amazon EC2 instance, an on-premises server, or a different platform.
     */
    ComputePlatform["DEFAULT"] = "Default";
})(ComputePlatform = exports.ComputePlatform || (exports.ComputePlatform = {}));
class ProfilingGroupBase extends core_1.Resource {
    /**
     * Grant access to publish profiling information to the Profiling Group to the given identity.
     *
     * This will grant the following permissions:
     *
     *  - codeguru-profiler:ConfigureAgent
     *  - codeguru-profiler:PostAgentProfile
     *
     * @param grantee Principal to grant publish rights to
     */
    grantPublish(grantee) {
        // https://docs.aws.amazon.com/codeguru/latest/profiler-ug/security-iam.html#security-iam-access-control
        return aws_iam_1.Grant.addToPrincipal({
            grantee,
            actions: ['codeguru-profiler:ConfigureAgent', 'codeguru-profiler:PostAgentProfile'],
            resourceArns: [this.profilingGroupArn],
        });
    }
    /**
     * Grant access to read profiling information from the Profiling Group to the given identity.
     *
     * This will grant the following permissions:
     *
     *  - codeguru-profiler:GetProfile
     *  - codeguru-profiler:DescribeProfilingGroup
     *
     * @param grantee Principal to grant read rights to
     */
    grantRead(grantee) {
        // https://docs.aws.amazon.com/codeguru/latest/profiler-ug/security-iam.html#security-iam-access-control
        return aws_iam_1.Grant.addToPrincipal({
            grantee,
            actions: ['codeguru-profiler:GetProfile', 'codeguru-profiler:DescribeProfilingGroup'],
            resourceArns: [this.profilingGroupArn],
        });
    }
}
/**
 * A new Profiling Group.
 */
class ProfilingGroup extends ProfilingGroupBase {
    constructor(scope, id, props = {}) {
        super(scope, id, {
            physicalName: props.profilingGroupName ?? core_1.Lazy.string({ produce: () => this.generateUniqueId() }),
        });
        try {
            jsiiDeprecationWarnings._aws_cdk_aws_codeguruprofiler_ProfilingGroupProps(props);
        }
        catch (error) {
            if (process.env.JSII_DEBUG !== "1" && error.name === "DeprecationError") {
                Error.captureStackTrace(error, ProfilingGroup);
            }
            throw error;
        }
        const profilingGroup = new codeguruprofiler_generated_1.CfnProfilingGroup(this, 'ProfilingGroup', {
            profilingGroupName: this.physicalName,
            computePlatform: props.computePlatform,
        });
        this.profilingGroupName = this.getResourceNameAttribute(profilingGroup.ref);
        this.profilingGroupArn = this.getResourceArnAttribute(profilingGroup.attrArn, {
            service: 'codeguru-profiler',
            resource: 'profilingGroup',
            resourceName: this.physicalName,
        });
    }
    /**
     * Import an existing Profiling Group provided a Profiling Group Name.
     *
     * @param scope The parent creating construct
     * @param id The construct's name
     * @param profilingGroupName Profiling Group Name
     */
    static fromProfilingGroupName(scope, id, profilingGroupName) {
        const stack = core_1.Stack.of(scope);
        return this.fromProfilingGroupArn(scope, id, stack.formatArn({
            service: 'codeguru-profiler',
            resource: 'profilingGroup',
            resourceName: profilingGroupName,
        }));
    }
    /**
     * Import an existing Profiling Group provided an ARN.
     *
     * @param scope The parent creating construct
     * @param id The construct's name
     * @param profilingGroupArn Profiling Group ARN
     */
    static fromProfilingGroupArn(scope, id, profilingGroupArn) {
        class Import extends ProfilingGroupBase {
            constructor() {
                super(...arguments);
                this.profilingGroupName = core_1.Stack.of(scope).splitArn(profilingGroupArn, core_1.ArnFormat.SLASH_RESOURCE_NAME).resource;
                this.profilingGroupArn = profilingGroupArn;
            }
        }
        return new Import(scope, id);
    }
    generateUniqueId() {
        const name = core_1.Names.uniqueId(this);
        if (name.length > 240) {
            return name.substring(0, 120) + name.substring(name.length - 120);
        }
        return name;
    }
}
exports.ProfilingGroup = ProfilingGroup;
_a = JSII_RTTI_SYMBOL_1;
ProfilingGroup[_a] = { fqn: "@aws-cdk/aws-codeguruprofiler.ProfilingGroup", version: "1.197.0" };
//# sourceMappingURL=data:application/json;base64,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