from ...analysis_types import AnalysisCategory, AnalysisIssue


# Categories

default_category = AnalysisCategory(
    id="default",
    display_label="Default",
    color="#FFDD88"
)

trust_grounding_category = AnalysisCategory(
    id="grounding",
    display_label="Ungrounded Information",
    color="#D6852C",
)

trust_assumptions_category = AnalysisCategory(
    id="assumptions",
    display_label="Assumptions in Code",
    color="#E24609",
)

literal_value_category=AnalysisCategory(
    id="literal",
    display_label="Literal values",
    color="#DDBB00",
)


# Issues

literal_value_issue = AnalysisIssue(
    id="literal_value",
    title="Variable set to literal value",
    description="""\
This issue checks for variables that are set to literal values. This sometimes indicates instances where the LLM agent
fills in unknown values with guesses or ungrounded knowledge. Always be careful to check literal variables to ensure
correctness.
""",
    severity="info",
    category=literal_value_category,
    filter=lambda a: True,
)

ungrounded_value = AnalysisIssue(
    id="ungrounded_value",
    title="Ungrounded Value",
    description="""A discrete value generated by the LLM does not appear to be properly grounded. """,
    prompt_description="""\
This issue arises when a particular value is used by the LLM Agent, but the value is not grounded.
I.e. it is used without identifying how the value is defined and/or why it was selected.
Obvious examples such as `state = 'TX'`, `timeout = 15` and `year = 2024` should not be annotated, but examples such as
`state = 22` or `param_code = "44201"` should have their values grounded and should be annotated if grounding is not
provided.
    """.strip(),
    category=trust_grounding_category,
    severity="warning",
)

ungrounded_methodology = AnalysisIssue(
    id="ungrounded_methodology",
    title="Ungrounded Methodology",
    description="""\
The LLM selected a methodology to use that does not appear to be properly grounded as to why it was
selected.""",
    prompt_description="""\
This issue arises when a particular algorithm or methodology is used by the LLM Agent, but the reasoning behind said use
is not grounded. This is particularly important when multiple algorithms or methodologies could be used and a particular
one was chosen.
When generating the annotation message, please provide information on the strengths and weaknesses of the methodology
compared to alternatives.
If there are no other reasonable methodologies or all other methodologies are essentially equivalent, code probably does
not require annotation, as the annotation would not provide any useful, actionable information to the user.
    """.strip(),
    category=trust_grounding_category,
    severity="info",
)

ungrounded_fact = AnalysisIssue(
    id="ungrounded_fact",
    title="Ungrounded Fact",
    description="""A fact is presumed by the LLM agent without any grounding.""",
    prompt_description="""\
This issue arises when the agent assumes a factual value without the fact being grounded.
This mostly occurs when the LLM believes that it "knows" a value based on its training, but the value could be based off
of outdated or incorrect information.
    """.strip(),
    category=trust_grounding_category,
    severity="minor",
)

assumption = AnalysisIssue(
    id="assumption",
    title="Assumption by Agent",
    description="""The LLM agent appears to have made an assumption that may or may not be valid.""",
    prompt_description="""\
This issue arises when the agent makes assumptions on how to accomplish a request that are not part of the request by
the user and for which reasonable alternatives exist.
    """.strip(),
    severity="warning",
    category=trust_assumptions_category,
)
