import codecs
import dill
from typing import TypeAlias, Optional, Literal, TypeVar, Generic, cast
from tree_sitter import Language, Parser, Tree, Point

from pydantic import BaseModel, Field

from archytas.react import ReActAgent, tool, LoopController
from archytas.tool_utils import LoopControllerRef
from .analyzer import AnalysisCategory
from .analysis_types import AnalysisAnnotation, AnalysisIssue


class AnalysisResult(BaseModel):
    """
    Representation of a lint-style analysis that will be generated by an LLM agent and passed to the code_analysis tool.
    """
    issue_type: str = Field(description=(
        "Unique ID indicating the issue type represented. This value should always match one of the issue types "
        "provided in the prompt instructions."
    ))
    analysis_rule_id: str = Field(description="Unique ID indicating which of the provided rules this analysis falls under.")
    extra_info: Optional[str] = Field(
        description=(
            "Optional extra information that may be useful to the user to understand the analysis, particularly in cases "
            "where the issue or rule is vague or potentially confusing."
        ),
        default=None,
    )
    cell_id: Optional[str] = Field(
        description=(
            "The ID of the cell the code is generated from, if available from the context."
        ),
        default=None,
    )
    code_start_line: int = Field(description="Starting line number for which the analysis applies.")
    code_end_line: int = Field(description=(
        "Ending line number for which the analysis applies. If the analysis is for a single line, this value should be "
        "the same as the value for `code_start_line`."
    ))
    code_start_line_pos: Optional[int] = Field(
        default=None,
        description=(
            "Optional position in start line where analysis starts. If not provided, defaults to start of starting line."
    ))
    code_end_line_pos: Optional[int] = Field(
        default=None,
        description=(
            "Optional position in end line where analysis ends. If not provided, defaults to end of ending line."
    ))


class AnalysisAgent(ReActAgent):
    """
    You are a code evaluation agent, whose job it is to evaluate code that is generated by a LLM and to provide helpful
    information so that they can more easily identify potential issues with the code and to understand the code and what
    it is doing more efficiently.

    You should always run the tool code_analysis, providing it with the results of your analysis of the code provided in
    the context.
    This tool will format the analysis and present it to the user.
    """.strip()

    def __init__(
        self, *, model = None, api_key = None, tools = None, max_errors = 3, max_react_steps = None,
        thought_handler = ..., messages = None, custom_prelude = None,
        code: str = None,
        issues: list[AnalysisIssue] = None,
        beaker_kernel = None,
        **kwargs
    ):
        super().__init__(
            model=model,
            api_key=api_key,
            tools=tools,
            allow_ask_user=False,
            max_errors=max_errors,
            max_react_steps=max_react_steps,
            messages=messages,
            custom_prelude=custom_prelude,
            spinner=None,
            rich_print=False,
            thought_handler=self.thought_callback,
            **kwargs
        )
        self.beaker_kernel = beaker_kernel
        if code is not None:
            self.add_context(f"""\
Code to run rules against:
```
{code}
```
""")
        issues = issues if isinstance(issues, list) else []
        self.issues = [issue for issue in issues if hasattr(issue, 'prompt_description')]

    def thought_callback(self, thought, tool_name, tool_input):
        if self.beaker_kernel:
            self.beaker_kernel.log(
                event_type="code_analysis_thought",
                content={
                    "thought": thought,
                    "tool_name": tool_name,
                    "tool_input": tool_input,
                }
            )

    @property
    def instruction_prompt(self):
        content: list[str] = [
            """\
 Below are the types of annotations/issues that you should use when analyzing code and generating analyses.
 All generated analyses should match exactly one of these rules.
 """
        ]
        for issue in self.issues:
            content.append(
                f"""\
## {issue.title} ({issue.id})
    id: {issue.id}
    severity: {issue.severity}
    description: \\
```
{issue.prompt_description}
```"""
            )
        return "\n".join(content)

    @tool
    def code_analysis(self, analysis_list: list[AnalysisResult], loop: LoopControllerRef) -> list[AnalysisResult]:
        """
        Given the code and rules above, please inspect the code for matching issues. For each issue identified, please
        generate an Analysis object

        Include all instances in the code that a rule applies to. It is fine to include the same line(s) more than once
        if more than one rule applies to that code segment.

        If successful, this tool will immediately stop the react loop and return the analyses to the invoking process.

        Arguments:
            analysis_list (list[AnalysisObject]): A list of analysis objects indicating the type and location of the issue.
        Returns:
            list[AnalysisObject]: List of analyses provided to the tool directly to the calling function.
        """

        loop.set_state(loop.STOP_SUCCESS)
        output =  dill.dumps(analysis_list)
        # Return must be a string, and dill.dumps returns bytes so decode appropriately
        return codecs.encode(output, 'base64').decode()
