from abc import ABCMeta
from typing import Sequence

from dungeonsheets.content import Content
from dungeonsheets.content_registry import default_content_registry


default_content_registry.add_module(__name__)


class SubtableFactory(ABCMeta):
    """Meta class to append subtables to the docstring of a RandomTable..
    
    For classes using this metaclass, the *subtables* attribute, if
    present, should be a list of subtables that are to be
    included. For each entry on that list, it will first be resolved
    into a RandomTable class, if appropriate, then its docstring will
    be added to the docstring of the calling class.
    
    """
    def __init__(self, name, bases, attrs):
        # Resolve subtables to RandomTable classes
        for idx, subtable in enumerate(self.subtables):
            TheTable = self._resolve_mechanic(subtable, SuperClass=RandomTable)
            self.subtables[idx] = TheTable
        # Append docstrings for subtables
        docstring = self.__doc__ if self.__doc__ is not None else ""
        for table in self.subtables:
            docstring += f"\n\n**{table.name}**\n\n{table.__doc__}\n"
        self.__doc__ = docstring


class RandomTable(Content, metaclass=SubtableFactory):
    """A generic table for rolling treasure, monsters, etc.

    Additional tables can be included by using the *subtables*
    attribute. A use case for this is to create a table for rolling
    random treasure, which may include subtables for gems, art, magic
    items, etc. By including these as subtables, each subtable could
    be included by itself if the verbosity of the full *Treasure*
    table is not needed.

    Attributes
    ==========
    subtables
      A sequence of other random tables that will be included in this
      table.
    
    """
    name = "Generic Random Table"
    subtables: Sequence = []


class ConjureAnimals(RandomTable):
    """
    +-----+-----------------------------------------------+
    | 1d4 | Number of Beasts                              |
    +=====+===============================================+
    | 1   | One beast of challenge rating 2               |
    +-----+-----------------------------------------------+
    | 2   | Two beasts of challenge rating 1              |
    +-----+-----------------------------------------------+
    | 3   | Four beasts of challenge rating 1/2           |
    +-----+-----------------------------------------------+
    | 4   | Eight beasts of challenge rating 1/4 or lower |
    +-----+-----------------------------------------------+
    
    +-------+---------------------------+
    | 1d20  | CR2 Beasts                |
    +=======+===========================+
    | 1-2   | Allosaurus                |
    +-------+---------------------------+
    | 3-4   | Giant Boar                |
    +-------+---------------------------+
    | 5-6   | Giant Constrictor Snake   |
    +-------+---------------------------+
    | 7-8   | Giant Elk                 |
    +-------+---------------------------+
    | 9-10  | Hunter Shark              |
    +-------+---------------------------+
    | 11    | Plesiosaurus              |
    +-------+---------------------------+
    | 12-13 | Polar Bear                |
    +-------+---------------------------+
    | 14-15 | Rhinoceros                |
    +-------+---------------------------+
    | 16-17 | Saber-toothed Tiger       |
    +-------+---------------------------+
    | 18-19 | Swarm of Poisonous Snakes |
    +-------+---------------------------+
    | 20    | Roll on CR 1 Beast Table  |
    +-------+---------------------------+
    
    +------+----------------------------+
    | 1d12 | Challenge Rating 1 Beasts  |
    +======+============================+
    | 1    | Brown Bear                 |
    +------+----------------------------+
    | 2    | Dire Wolf                  |
    +------+----------------------------+
    | 3    | Fire Snake                 |
    +------+----------------------------+
    | 4    | Giant Eagle                |
    +------+----------------------------+
    | 5    | Giant Hyena                |
    +------+----------------------------+
    | 6    | Giant Octopus              |
    +------+----------------------------+
    | 7    | Giant Spider               |
    +------+----------------------------+
    | 8    | Giant Toad                 |
    +------+----------------------------+
    | 9    | Giant Vulture              |
    +------+----------------------------+
    | 10   | Lion                       |
    +------+----------------------------+
    | 11   | Tiger                      |
    +------+----------------------------+
    | 12   | Roll on CR 1/2 Beast Table |
    +------+----------------------------+
    
    +-------+---------------------------------+
    | 1d20  | Challenge Rating 1/2 Beasts     |
    +=======+=================================+
    | 1-2   | Ape                             |
    +-------+---------------------------------+
    | 3-4   | Black Bear                      |
    +-------+---------------------------------+
    | 5-6   | Crocodile                       |
    +-------+---------------------------------+
    | 7-8   | Giant Goat                      |
    +-------+---------------------------------+
    | 9-10  | Giant Sea Horse                 |
    +-------+---------------------------------+
    | 11-12 | Giant Wasp                      |
    +-------+---------------------------------+
    | 13-14 | Reef Shark                      |
    +-------+---------------------------------+
    | 15-16 | Swarm of Insects (below)        |
    +-------+---------------------------------+
    | 17-18 | Warhorse                        |
    +-------+---------------------------------+
    | 19    | Worg                            |
    +-------+---------------------------------+
    | 20    | Roll on Lesser Beast Menu Table |
    +-------+---------------------------------+
    
    +-----+------------------+
    | 1d6 | Swarm of Insects |
    +=====+==================+
    | 1   | Ant              |
    +-----+------------------+
    | 2   | Beatles          |
    +-----+------------------+
    | 3   | Centipedes       |
    +-----+------------------+
    | 4   | Locusts          |
    +-----+------------------+
    | 5   | Spiders          |
    +-----+------------------+
    | 6   | Wasps            |
    +-----+------------------+
    
    +-----+------------------------------+
    | 1d6 | CR 1/4 and Lesser Beast Menu |
    +=====+==============================+
    | 1-2 | Menu A                       |
    +-----+------------------------------+
    | 3-4 | Menu B                       |
    +-----+------------------------------+
    | 5-6 | Menu C                       |
    +-----+------------------------------+
    
    +------+---------------------+
    | 1d20 | Lesser Beast Menu A |
    +======+=====================+
    | 1    | Axe Beak            |
    +------+---------------------+
    | 2    | Baboon              |
    +------+---------------------+
    | 3    | Badger              |
    +------+---------------------+
    | 4    | Bat                 |
    +------+---------------------+
    | 5    | Blood Hawk          |
    +------+---------------------+
    | 6    | Boar                |
    +------+---------------------+
    | 7    | Camel               |
    +------+---------------------+
    | 8    | Cat                 |
    +------+---------------------+
    | 9    | Chicken¹            |
    +------+---------------------+
    | 10   | Constrictor Snake   |
    +------+---------------------+
    | 11   | Crab                |
    +------+---------------------+
    | 12   | Deer                |
    +------+---------------------+
    | 13   | Draft Horse         |
    +------+---------------------+
    | 14   | Eagle               |
    +------+---------------------+
    | 15   | Elk                 |
    +------+---------------------+
    | 16   | Flying Snake        |
    +------+---------------------+
    | 17   | Frog                |
    +------+---------------------+
    | 18   | Giant Badger        |
    +------+---------------------+
    | 19   | Giant Bat           |
    +------+---------------------+
    | 20   | Giant Centipede     |
    +------+---------------------+
    
    ¹Chicken
      Raven stats with Advantage on checks to wake up targets instead
      of mimicry
    
    +------+--------------------------+
    | 1d20 | Lesser Beast Menu B      |
    +======+==========================+
    | 1    | Giant Crab               |
    +------+--------------------------+
    | 2    | Giant Fire Beetle        |
    +------+--------------------------+
    | 3    | Giant Frog               |
    +------+--------------------------+
    | 4    | Giant Lizard             |
    +------+--------------------------+
    | 5    | Giant Owl                |
    +------+--------------------------+
    | 6    | Giant Poisonous Snake    |
    +------+--------------------------+
    | 7    | Giant Rat                |
    +------+--------------------------+
    | 8    | Giant Weasel             |
    +------+--------------------------+
    | 9    | Giant Wolf Spider        |
    +------+--------------------------+
    | 10   | Goat                     |
    +------+--------------------------+
    | 11   | Hawk                     |
    +------+--------------------------+
    | 12   | Hyena                    |
    +------+--------------------------+
    | 13   | Jackal                   |
    +------+--------------------------+
    | 14   | Lemur²                   |
    +------+--------------------------+
    | 15   | Lizard                   |
    +------+--------------------------+
    | 16   | Mastiff                  |
    +------+--------------------------+
    | 17   | Mule                     |
    +------+--------------------------+
    | 18   | Newt³                    |
    +------+--------------------------+
    | 19   | Octopus                  |
    +------+--------------------------+
    | 20   | Octopus, Cascadian Tree⁴ |
    +------+--------------------------+
    
    ²Lemur
      Weasel stats with a common Climb speed instead of a bite attack
    ³Newt
      Lizard stats with Amphibious instead of a bite attack
    ⁴Octopus, Cascadian Tree:
      Octopus stats with Amphibious and a 10 ft land speed instead of
      camouflage
    
    +------+---------------------+
    | 1d20 | Lesser Beast Menu C |
    +======+=====================+
    | 1    | Owl                 |
    +------+---------------------+
    | 2    | Panther             |
    +------+---------------------+
    | 3    | Poisonous Snake     |
    +------+---------------------+
    | 4    | Pony                |
    +------+---------------------+
    | 5    | Pteranodon          |
    +------+---------------------+
    | 6    | Quipper             |
    +------+---------------------+
    | 7    | Rat                 |
    +------+---------------------+
    | 8    | Raven               |
    +------+---------------------+
    | 9    | Riding Horse        |
    +------+---------------------+
    | 10   | Scorpion            |
    +------+---------------------+
    | 11   | Sea Horse           |
    +------+---------------------+
    | 12   | Shocker Lizard⁵     |
    +------+---------------------+
    | 13   | Spider              |
    +------+---------------------+
    | 14   | Swarm of Bats       |
    +------+---------------------+
    | 15   | Swarm of Rats       |
    +------+---------------------+
    | 16   | Swarm of Ravens     |
    +------+---------------------+
    | 17   | Turtle⁶             |
    +------+---------------------+
    | 18   | Vulture             |
    +------+---------------------+
    | 19   | Weasel              |
    +------+---------------------+
    | 20   | Wolf                |
    +------+---------------------+
    
    ⁵Shocker Lizard
      Lizard stats with Static Electricity ranged attack of 1d6
      Electricity damage Close/Medium.
    ⁶Turtle
      Lizard stats with 14 natural armor and no climb speed.
    
    """
    # https://the-azure-triskele.obsidianportal.com/wikis/conjure-animals-table
    name = "Conjure Animals"


class IndividualTreasure0To4(RandomTable):
    """
    +-------+----------+----------+----------+----------+---------+
    | d100  | CP       | SP       | EP       | GP       | PP      |
    +=======+==========+==========+==========+==========+=========+
    | 01‒30 | 5d6 (17) | –        | –        | –        | –       |
    +-------+----------+----------+----------+----------+---------+
    | 31‒60 | –        | 4d6 (14) | –        | –        | –       |
    +-------+----------+----------+----------+----------+---------+
    | 61‒70 | –        | –        | 3d6 (10) | –        | –       |
    +-------+----------+----------+----------+----------+---------+
    | 71‒95 | –        | –        | –        | 3d6 (10) | –       |
    +-------+----------+----------+----------+----------+---------+
    | 96‒00 | –        | –        | –        | –        | 1d6 (3) |
    +-------+----------+----------+----------+----------+---------+
    
    """
    name = "Individual Treasure: Challenge 0–4"


class IndividualTreasure5To10(RandomTable):
    """
    +-------+--------------------+----------------+----------------+----------------+----------+
    | d100  | CP                 | SP             | EP             | GP             | PP       |
    +=======+====================+================+================+================+==========+
    | 01‒30 | 4d6 × 100 (14,000) | –              | 1d6 × 100 (35) | –              | –        |
    +-------+--------------------+----------------+----------------+----------------+----------+
    | 31‒60 | –                  | 6d6 × 10 (210) | –              | 2d6 × 10 (70)  | –        |
    +-------+--------------------+----------------+----------------+----------------+----------+
    | 61‒70 | –                  | –              | 3d6 × 10 (105) | 2d6 × 10 (70)  | –        |
    +-------+--------------------+----------------+----------------+----------------+----------+
    | 71‒95 | –                  | –              | –              | 4d6 × 10 (140) | –        |
    +-------+--------------------+----------------+----------------+----------------+----------+
    | 96‒00 | –                  | –              | –              | 2d6 × 10 (70)  | 3d6 (10) |
    +-------+--------------------+----------------+----------------+----------------+----------+
    
    """
    name = "Individual Treasure: Challenge 5‒10"


class IndividualTreasure11To16(RandomTable):
    """
    +-------+----+-------------------+-----------------+-----------------+---------------+
    | d100  | CP | SP                | EP              | GP              | PP            |
    +=======+====+===================+=================+=================+===============+
    | 01‒20 | –  | 4d6 × 100 (1,400) | –               | 1d6 × 100 (350) | –             |
    +-------+----+-------------------+-----------------+-----------------+---------------+
    | 21‒35 | –  | –                 | 1d6 × 100 (350) | 1d6 × 100 (350) | –             |
    +-------+----+-------------------+-----------------+-----------------+---------------+
    | 36‒75 | –  | –                 | –               | 2d6 × 100 (700) | 1d6 × 10 (35) |
    +-------+----+-------------------+-----------------+----------------+----------------+
    | 75‒00 | –  | –                 | –               | 2d6 × 100 (700) | 2d6 × 10 (70) |
    +-------+----+-------------------+-----------------+----------------+----------------+
    
    """
    name = "Individual Treasure: Challenge 11‒16"


class IndividualTreasure17Plus(RandomTable):
    """
    +-------+----+----+---------------------+---------------------+-----------------+
    | d100  | CP | SP | EP                  | GP                  | PP              |
    +=======+====+====+=====================+=====================+=================+
    | 01‒15 | –  | –  | 2d6 × 1,000 (7,000) | 8d6 × 100 (2,800)   | –               |
    +-------+----+----+---------------------+---------------------+-----------------+
    | 16‒55 | –  | –  | –                   | 1d6 × 1,000 (3,500) | 1d6 × 100 (350) |
    +-------+----+----+---------------------+---------------------+-----------------+
    | 56‒00 | –  | –  | –                   | 1d6 × 1,000 (3,500) | 2d6 × 100 (700) |
    +-------+----+----+---------------------+---------------------+-----------------+
    
    """
    name = "Individual Treasure: Challenge 17+"


class IndividualTreasure(RandomTable):
    """"""
    name = "Individual Treasure"
    subtables = [IndividualTreasure0To4, IndividualTreasure5To10,
                 IndividualTreasure11To16, IndividualTreasure17Plus]


class HoardTreasure0To4(RandomTable):
    """
    +-------+-------------------+-------------------+----+---------------+----+
    |       | CP                | SP                | EP | GP            | PP |
    +=======+===================+===================+====+===============+====+
    | Coins | 6d6 × 100 (2,100) | 3d6 × 100 (1,050) | –  | 2d6 × 10 (70) | –  |
    +-------+-------------------+-------------------+----+---------------+----+
    
    +-------+---------------------------+---------------------------------------+
    | d100  | Gems or Art Objects       | Magic Items                           |
    +=======+===========================+=======================================+
    | 01‒06 | –                         | –                                     |
    +-------+---------------------------+---------------------------------------+
    | 07‒16 | 2d6 (7) 10 gp gems        | –                                     |
    +-------+---------------------------+---------------------------------------+
    | 17‒26 | 2d4 (5) 25 gp art objects | –                                     |
    +-------+---------------------------+---------------------------------------+
    | 27‒36 | 2d6 (7) 50 gp gems        | –                                     |
    +-------+---------------------------+---------------------------------------+
    | 37‒44 | 2d6 (7) 10 gp gems        | Roll 1d6 times on Magic Item Table A. |
    +-------+---------------------------+---------------------------------------+
    | 45‒52 | 2d4 (5) 25 gp art objects | Roll 1d6 times on Magic Item Table A. |
    +-------+---------------------------+---------------------------------------+
    | 53‒60 | 2d6 (7) 50 gp gems        | Roll 1d6 times on Magic Item Table A. |
    +-------+---------------------------+---------------------------------------+
    | 61‒65 | 2d6 (7) 10 gp gems        | Roll 1d4 times on Magic Item Table B. |
    +-------+---------------------------+---------------------------------------+
    | 66‒70 | 2d4 (5) 25 gp art objects | Roll 1d4 times on Magic Item Table B. |
    +-------+---------------------------+---------------------------------------+
    | 71‒75 | 2d6 (6) 50 gp gems        | Roll 1d4 times on Magic Item Table B. |
    +-------+---------------------------+---------------------------------------+
    | 76‒78 | 2d6 (7) 10 gp gems        | Roll 1d4 times on Magic Item Table C. |
    +-------+---------------------------+---------------------------------------+
    | 79‒80 | 2d4 (5) 25 gp art objects | Roll 1d4 times on Magic Item Table C. |
    +-------+---------------------------+---------------------------------------+
    | 81‒85 | 2d6 (7) 50 gp gems        | Roll 1d4 times on Magic Item Table C. |
    +-------+---------------------------+---------------------------------------+
    | 86‒92 | 2d4 (5) 25 gp art objects | Roll 1d4 times on Magic Item Table F. |
    +-------+---------------------------+---------------------------------------+
    | 93‒97 | 2d6 (7) 50 gp gems        | Roll 1d4 times on Magic Item Table F. |
    +-------+---------------------------+---------------------------------------+
    | 98‒99 | 2d4 (5) 25 gp art objects | Roll once on Magic Item Table G.      |
    +-------+---------------------------+---------------------------------------+
    | 00    | 2d6 (7) 50 gp gems        | Roll once on Magic Item Table G.      |
    +-------+---------------------------+---------------------------------------+
    
    """
    name = "Hoard Treasure: Challenge 0‒4"


class HoardTreasure5To10(RandomTable):
    """
    +-------+-----------------+---------------------+----+-------------------+---------------+
    |       | CP              | SP                  | EP | GP                | PP            |
    +=======+=================+=====================+====+===================+===============+
    | Coins | 2d6 × 100 (700) | 2d6 × 1,000 (7,000) | –  | 6d6 × 100 (2,100) | 3d6 × 10 (105 |
    +-------+-----------------+---------------------+----+-------------------+---------------+
    
    +-------+----------------------------+---------------------------------------+
    | d100  | Gems or Art 0bjects        | Magic Items                           |
    +=======+============================+=======================================+
    | 01‒04 | –                          | –                                     |
    +-------+----------------------------+---------------------------------------+
    | 05‒10 | 2d4 (5) 25 gp art objects  | –                                     |
    +-------+----------------------------+---------------------------------------+
    | 11‒16 | 3d6 (10) 50 gp gems        | –                                     |
    +-------+----------------------------+---------------------------------------+
    | 17‒22 | 3d6 (10) 100 gp gems       | –                                     |
    +-------+----------------------------+---------------------------------------+
    | 23‒28 | 2d4 (5) 250 gp art objects | –                                     |
    +-------+----------------------------+---------------------------------------+
    | 29‒32 | 2d4 (5) 25 gp art objects  | Roll 1d6 times on Magic Item Table A. |
    +-------+----------------------------+---------------------------------------+
    | 33‒36 | 3d6 (10) 50 gp gems        | Roll 1d6 times on Magic Item Table A. |
    +-------+----------------------------+---------------------------------------+
    | 37‒40 | 3d6 (10) 100 gp gems       | Roll 1d6 times on Magic Item Table A. |
    +-------+----------------------------+---------------------------------------+
    | 41‒44 | 2d4 (5) 250 gp art objects | Roll 1d6 times on Magic Item Table A. |
    +-------+----------------------------+---------------------------------------+
    | 45‒49 | 2d4 (5) 25 gp art objects  | Roll 1d4 times on Magic Item Table B. |
    +-------+----------------------------+---------------------------------------+
    | 50‒54 | 3d6 (10) 50 gp gems        | Roll 1d4 times on Magic Item Table B. |
    +-------+----------------------------+---------------------------------------+
    | 55‒59 | 3d6 (10) 100 gp gems       | Roll 1d4 times on Magic Item Table B. |
    +-------+----------------------------+---------------------------------------+
    | 60‒63 | 2d4 (5) 250 gp art objects | Roll 1d4 times on Magic Item Table B. |
    +-------+----------------------------+---------------------------------------+
    | 64‒66 | 2d4 (5) 25 gp art objects  | Roll 1d4 times on Magic Item Table C. |
    +-------+----------------------------+---------------------------------------+
    | 67‒69 | 3d6 (10) 50 gp gems        | Roll 1d4 times on Magic Item Table C. |
    +-------+----------------------------+---------------------------------------+
    | 70‒72 | 3d6 (10) 100 gp gems       | Roll 1d4 times on Magic Item Table C. |
    +-------+----------------------------+---------------------------------------+
    | 73‒74 | 2d4 (5) 250 gp art objects | Roll 1d4 times on Magic Item Table C. |
    +-------+----------------------------+---------------------------------------+
    | 75‒76 | 2d4 (5) 25 gp art objects  | Roll once on Magic Item Table D.      |
    +-------+----------------------------+---------------------------------------+
    | 77‒78 | 3d6 (10) 50 gp gems        | Roll once on Magic Item Table D.      |
    +-------+----------------------------+---------------------------------------+
    | 79    | 3d6 (10) 100 gp gems       | Roll once on Magic I tem Table D.     |
    +-------+----------------------------+---------------------------------------+
    | 80    | 2d4 (5) 250 gp art objects | Roll once on Magic Item Table D.      |
    +-------+----------------------------+---------------------------------------+
    | 81‒84 | 2d4 (5) 25 gp art objects  | Roll 1d4 times on Magic Item Table F. |
    +-------+----------------------------+---------------------------------------+
    | 85‒88 | 3d6 (10) 50 gp gems        | Roll 1d4 times on Magic Item Table F. |
    +-------+----------------------------+---------------------------------------+
    | 89‒91 | 3d6 (10) 100 gp gems       | Roll 1d4 times on Magic Item Table F. |
    +-------+----------------------------+---------------------------------------+
    | 92‒94 | 2d4 (5) 250 gp art objects | Roll 1d4 times on Magic Item Table F. |
    +-------+----------------------------+---------------------------------------+
    | 95‒96 | 3d6 (10) 100 gp gems       | Roll 1d4 times on Magic Item Table G. |
    +-------+----------------------------+---------------------------------------+
    | 97‒98 | 2d4 (5) 250 gp art objects | Roll 1d4 times on Magic Item Table G. |
    +-------+----------------------------+---------------------------------------+
    | 99    | 3d6 (10) 100 gp gems       | Roll once on Magic Item Table H.      |
    +-------+----------------------------+---------------------------------------+
    | 00    | 2d4 (5) 250 gp art objects | Roll once on Magic Item Table H.      |
    +-------+----------------------------+---------------------------------------+
    
    """
    name = "Treasure Hoard: Challenge 5‒10"


class HoardTreasure11To16(RandomTable):
    """
    +-------+----+----+----+-------------------+------------------+
    |       | CP | SP | EP | GP                | PP               |
    +=======+====+====+====+===================+==================+
    | Coins | –  | –  | –  | 4d6 × 1000 (1400) | 5d6 × 100 (1750) |
    +-------+----+----+----+-------------------+------------------+
    
    +-------+----------------------------+---------------------------------------------------------------------------+
    | d100  | Gems or Art Objects        | Magic Items                                                               |
    +=======+============================+===========================================================================+
    | 01‒03 | –                          | –                                                                         |
    +-------+----------------------------+---------------------------------------------------------------------------+
    | 04–06 | 2d4 (5) 250 gp art objects | –                                                                         |
    +-------+----------------------------+---------------------------------------------------------------------------+
    | 07–10 | 2d4 (5) 750 gp art objects | –                                                                         |
    +-------+----------------------------+---------------------------------------------------------------------------+
    | 11–12 | 3d6 (10) 500 gp gems       | –                                                                         |
    +-------+----------------------------+---------------------------------------------------------------------------+
    | 13–15 | 3d6 (10) 1,000 gp gems     | –                                                                         |
    +-------+----------------------------+---------------------------------------------------------------------------+
    | 16–19 | 2d4 (5) 250 gp art objects | Roll 1d4 times on Magic Item Table A and 1d6 times on Magic Item Table B. |
    +-------+----------------------------+---------------------------------------------------------------------------+
    | 20–23 | 2d4 (5) 750 gp art objects | Roll 1d4 times on Magic Item Table A and 1d6 times on Magic Item Table B. |
    +-------+----------------------------+---------------------------------------------------------------------------+
    | 24–26 | 3d6 (10) 500 gp gems       | Roll 1d4 times on Magic Item Table A and 1d6 times on Magic Item Table B. |
    +-------+----------------------------+---------------------------------------------------------------------------+
    | 27–29 | 3d6 (10) 1,000 gp gems     | Roll 1d4 times on Magic Item Table A and 1d6 times on Magic Item Table B. |
    +-------+----------------------------+---------------------------------------------------------------------------+
    | 30–35 | 2d4 (5) 250 gp art objects | Roll 1d6 times on Magic Item Table C.                                     |
    +-------+----------------------------+---------------------------------------------------------------------------+
    | 36–40 | 2d4 (5) 750 gp art objects | Roll 1d6 times on Magic Item Table C.                                     |
    +-------+----------------------------+---------------------------------------------------------------------------+
    | 41–45 | 3d6 (10) 500 gp gems       | Roll 1d6 times on Magic Item Table C.                                     |
    +-------+----------------------------+---------------------------------------------------------------------------+
    | 46–50 | 3d6 (10) 1,000 gp gems     | Roll 1d6 times on Magic Item Table C.                                     |
    +-------+----------------------------+---------------------------------------------------------------------------+
    | 51–54 | 2d4 (5) 250 gp art objects | Roll 1d4 times on Magic Item Table D                                      |
    +-------+----------------------------+---------------------------------------------------------------------------+
    | 55–58 | 2d4 (5) 750 gp art objects | Roll 1d4 times on Magic Item Table D.                                     |
    +-------+----------------------------+---------------------------------------------------------------------------+
    | 59–62 | 3d6 (10) 500 gp gems       | Roll 1d4 times on Magic Item Table D.                                     |
    +-------+----------------------------+---------------------------------------------------------------------------+
    | 63–66 | 3d6 (10) 1,000 gp gems     | Roll 1d4 times on Magic Item Table D.                                     |
    +-------+----------------------------+---------------------------------------------------------------------------+
    | 67–68 | 2d4 (5) 250 gp art objects | Roll once on Magic Item Table E.                                          |
    +-------+----------------------------+---------------------------------------------------------------------------+
    | 69–70 | 2d4 (5) 750 gp art objects | Roll once on Magic Item Table E.                                          |
    +-------+----------------------------+---------------------------------------------------------------------------+
    | 71–72 | 3d6 (10) 500 gp gems       | Roll once on Magic Item Table E.                                          |
    +-------+----------------------------+---------------------------------------------------------------------------+
    | 73–74 | 3d6 (10) 1,000 gp gems     | Roll once on Magic Item Table E.                                          |
    +-------+----------------------------+---------------------------------------------------------------------------+
    | 75–76 | 2d4 (5) 250 gp art objects | Roll once on Magic Item Table F and 1d4 times on Magic Item Table G.      |
    +-------+----------------------------+---------------------------------------------------------------------------+
    | 77–78 | 2d4 (5) 750 gp art objects | Roll once on Magic Item Table F and 1d4 times on Magic Item Table G.      |
    +-------+----------------------------+---------------------------------------------------------------------------+
    | 79–80 | 3d6 (10) 500 gp gems       | Roll once on Magic Item Table F and 1d4 times on Magic Item Table G.      |
    +-------+----------------------------+---------------------------------------------------------------------------+
    | 81–82 | 3d6 (1 0) 1,000 gp gems    | Roll once on Magic Item Table F and 1d4 times on Magic Item Table G.      |
    +-------+----------------------------+---------------------------------------------------------------------------+
    | 83–85 | 2d4 (5) 250 gp art objects | Roll1d4 times on Magic Item Table H.                                      |
    +-------+----------------------------+---------------------------------------------------------------------------+
    | 86–88 | 2d4 (5) 750 gp art objects | Roll 1d4 times on Magic Item Table H.                                     |
    +-------+----------------------------+---------------------------------------------------------------------------+
    | 89–90 | 3d6 (10) 500 gp gems       | Roll 1d4 times on Magic Item Table H.                                     |
    +-------+----------------------------+---------------------------------------------------------------------------+
    | 91–92 | 3d6 (10) 1,000 gp gems     | Roll 1d4 times on Magic Item Table H.                                     |
    +-------+----------------------------+---------------------------------------------------------------------------+
    | 93–94 | 2d4 (5) 250 gp art objects | Roll once on Magic Item Table I.                                          |
    +-------+----------------------------+---------------------------------------------------------------------------+
    | 95–96 | 3d6 (10) 500 gp gems       | Roll once on Magic Item Table I.                                          |
    +-------+----------------------------+---------------------------------------------------------------------------+
    | 97–98 | 3d6 (10) 1,000 gp gems     | Roll once on Magic Item Table I.                                          |
    +-------+----------------------------+---------------------------------------------------------------------------+
    | 99–00 | 3d6 (10) 1,000 gp gems     | Roll once on Magic Item Table I.                                          |
    +-------+----------------------------+---------------------------------------------------------------------------+
    
    """
    name = "Treasure Hoard: Challenge 11‒16"


class HoardTreasure17Plus(RandomTable):
    """
    +-------+----+----+----+---------------------+--------------------+
    |       | CP | SP | EP | GP                  | PP                 |
    +=======+====+====+====+=====================+====================+
    | Coins | –  | –  | –  | 12d6 X 1000 (42000) | 8d6 x 1000 (28000) |
    +-------+----+----+----+---------------------+--------------------+

    +--------+-------------------------------+----------------------------------------------------------------------+
    | d100   | Gems or Art Objects           | Magic Items                                                          |
    +========+===============================+======================================================================+
    | 01–02  | –                             | –                                                                    |
    +--------+-------------------------------+----------------------------------------------------------------------+
    | 03–05  | 3d6 (10) 1, 000 gp gems       | Roll 1d8 times on Magic Item Table C.                                |
    +--------+-------------------------------+----------------------------------------------------------------------+
    | 06–08  | 1d10 (5) 2,500 gp art objects | Roll 1d8 times on Magic Item Table C.                                |
    +--------+-------------------------------+----------------------------------------------------------------------+
    | 09–11  | 1d4 (2) 7,500 gp art objects  | Roll 1d8 times on Magic Item Table C.                                |
    +--------+-------------------------------+----------------------------------------------------------------------+
    | 12–14  | 1d8 (4) 5,000 gp gems         | Roll 1d8 times on Magic Item Table C.                                |
    +--------+-------------------------------+----------------------------------------------------------------------+
    | 15–22  | 3d6 (10) 1,000 gp gems        | Roll 1d6 times on Magic Item Table D.                                |
    +--------+-------------------------------+----------------------------------------------------------------------+
    | 23–30  | 1dl0 (5) 2,500 gp art objects | Roll 1d6 times on Magic Item Table D.                                |
    +--------+-------------------------------+----------------------------------------------------------------------+
    | 31–38  | 1d4 (2) 7, 500 gp art objects | Roll 1d6 times on Magic Item Table D.                                |
    +--------+-------------------------------+----------------------------------------------------------------------+
    | 39–46  | 1d8 (4) 5,000 gp gems         | Roll 1d6 times on Magic Item Table D                                 |
    +--------+-------------------------------+----------------------------------------------------------------------+
    | 47–52  | 3d6 (10) 1,000 gp gems        | Roll 1d6 times on Magic Item Table E.                                |
    +--------+-------------------------------+----------------------------------------------------------------------+
    | 53–58  | 1d10 (5) 2,500 gp art objects | Roll1d6 times on Magic Item Table E.                                 |
    +--------+-------------------------------+----------------------------------------------------------------------+
    | 59–63  | 1d4 (2) 7,500 gp art objects  | Roll 1d6 times on Magic Item Table E.                                |
    +--------+-------------------------------+----------------------------------------------------------------------+
    | 64–68  | 1d8 (4) 5, 000 gp gems        | Roll 1d6 times on Magic Item Table E.                                |
    +--------+-------------------------------+----------------------------------------------------------------------+
    | 69     | 3d6 (1 0) 1,000 gp gems       | Roll 1d4 times on Magic Item Table G.                                |
    +--------+-------------------------------+----------------------------------------------------------------------+
    | 70     | 1d10 (5) 2,500 gp art objects | Roll 1d4 times on Magic Item Table G.                                |
    +--------+-------------------------------+----------------------------------------------------------------------+
    | 71     | 1d4 (2) 7,500 gp art objects  | Roll1d4 times on Magic Item Table G.                                 |
    +--------+-------------------------------+----------------------------------------------------------------------+
    | 72     | 1d8 (4) 5,000 gp gems         | Roll 1d4 times on Magic Item Table G.                                |
    +--------+-------------------------------+----------------------------------------------------------------------+
    | 73–74  | 3d6 (10) 1,000 gp gems        | Roll 1d4 times on Magic Item Table H.                                |
    +--------+-------------------------------+----------------------------------------------------------------------+
    | 75–76  | ld10 (5) 2,500 gp art objects | Roll 1d4 times on Magic Item Table H.                                |
    +--------+-------------------------------+----------------------------------------------------------------------+
    | 77–78  | 1d4 (2) 7,500 gp art objects  | Roll 1d4 times on Magic Item Table H.                                |
    +--------+-------------------------------+----------------------------------------------------------------------+
    | 79–80  | 1d8 (4) 5,000 gp gems         | Roll 1d4 times on Magic Item Table H.                                |
    +--------+-------------------------------+----------------------------------------------------------------------+
    | 81–85  | 3d6 (10) 1,000 gp gems        | Roll 1d4 times on Magic Item Table I.                                |
    +--------+-------------------------------+----------------------------------------------------------------------+
    | 86–90  | 1d10 (5) 2,500 gp art objects | Roll 1d4 times on Magic Item Table I.                                |
    +--------+-------------------------------+----------------------------------------------------------------------+
    | 91–95  | ld4 (2) 7,500 gp art objects  | Roll once on Magic Item Table F and 1d4 times on Magic Item Table G. |
    +--------+-------------------------------+----------------------------------------------------------------------+
    | 96–100 | 1d8 (4) 5,000 gp gems         | Roll 1d4 times on Magic Item Table I.                                |
    +--------+-------------------------------+----------------------------------------------------------------------+
    
    """
    name = "Treasure Hoard: Challenge 17+"


class MagicItemTableA(RandomTable):
    """
    +-------+---------------------------+
    | d100  | Magic Item                |
    +=======+===========================+
    | 01–50 | Potion of healing         |
    +-------+---------------------------+
    | 51–60 | Spell scroll (cantrip)    |
    +-------+---------------------------+
    | 61–70 | Potion of climbing        |
    +-------+---------------------------+
    | 71–90 | Spell scroll (1st level)  |
    +-------+---------------------------+
    | 91–94 | Spell scroll (2nd level)  |
    +-------+---------------------------+
    | 95–98 | Potion of greater healing |
    +-------+---------------------------+
    | 99    | Bag of holding            |
    +-------+---------------------------+
    | 00    | Driftglobe                |
    +-------+---------------------------+
    
    """
    name = "Magic Item Table A"


class MagicItemTableB(RandomTable):
    """
    +-------+---------------------------------+
    | d100  | Magic Item                      |
    +=======+=================================+
    | 01–15 | Potion of greater healing       |
    +-------+---------------------------------+
    | 16–22 | Potion of fire breath           |
    +-------+---------------------------------+
    | 23–29 | Potion of resistance            |
    +-------+---------------------------------+
    | 30–34 | Ammunition, +1                  |
    +-------+---------------------------------+
    | 35–39 | Potion of animal friendship     |
    +-------+---------------------------------+
    | 40–44 | Potion of hill giant strength   |
    +-------+---------------------------------+
    | 45–49 | Potion of growth                |
    +-------+---------------------------------+
    | 50–54 | Potion of water breathing       |
    +-------+---------------------------------+
    | 55–59 | Spell scroll (2nd level)        |
    +-------+---------------------------------+
    | 60–64 | Spell scroll (3rd level)        |
    +-------+---------------------------------+
    | 65–67 | Bag of holding                  |
    +-------+---------------------------------+
    | 68–70 | Keoghtom's ointment             |
    +-------+---------------------------------+
    | 71–73 | Oil of slipperiness             |
    +-------+---------------------------------+
    | 74–75 | Dust of disappearance           |
    +-------+---------------------------------+
    | 76–77 | Dust of dryness                 |
    +-------+---------------------------------+
    | 78–79 | Dust of sneezing and choking    |
    +-------+---------------------------------+
    | 80–81 | Elemental gem                   |
    +-------+---------------------------------+
    | 82–83 | Philter of love                 |
    +-------+---------------------------------+
    | 84    | Alchemy jug                     |
    +-------+---------------------------------+
    | 85    | Cap of water breathing          |
    +-------+---------------------------------+
    | 86    | Cloak of the manta ray          |
    +-------+---------------------------------+
    | 87    | Driftglobe                      |
    +-------+---------------------------------+
    | 88    | Goggles of night                |
    +-------+---------------------------------+
    | 89    | Helm of comprehending languages |
    +-------+---------------------------------+
    | 90    | Immovable rod                   |
    +-------+---------------------------------+
    | 91    | Lantern of revealing            |
    +-------+---------------------------------+
    | 92    | Mariner's armor                 |
    +-------+---------------------------------+
    | 93    | Mithral armor                   |
    +-------+---------------------------------+
    | 94    | Potion of poison                |
    +-------+---------------------------------+
    | 95    | Ring of swimming                |
    +-------+---------------------------------+
    | 96    | Robe of useful items            |
    +-------+---------------------------------+
    | 97    | Rope of climbing                |
    +-------+---------------------------------+
    | 98    | Saddle of the cavalier          |
    +-------+---------------------------------+
    | 99    | Wand of magic detection         |
    +-------+---------------------------------+
    | 100   | Wand of secrets                 |
    +-------+---------------------------------+
    
    """
    name = "Magic Item Table B"


class MagicItemTableC(RandomTable):
    """
    +-------+--------------------------------+
    | d100  | Magic Item                     |
    +=======+================================+
    | 01–15 | Potion of superior healing     |
    +-------+--------------------------------+
    | 16–22 | Spell scroll (4thlevel)        |
    +-------+--------------------------------+
    | 23–27 | Ammunition, +2                 |
    +-------+--------------------------------+
    | 28–32 | Potion of clairvoyance         |
    +-------+--------------------------------+
    | 33–37 | Potion of diminution           |
    +-------+--------------------------------+
    | 38–42 | Potion of gaseous form         |
    +-------+--------------------------------+
    | 43–47 | Potion of frost giant strength |
    +-------+--------------------------------+
    | 48–52 | Potion of stone giant strength |
    +-------+--------------------------------+
    | 53–57 | Potion of heroism              |
    +-------+--------------------------------+
    | 58–62 | Potion of invulnerability      |
    +-------+--------------------------------+
    | 63–67 | Potion of mind reading         |
    +-------+--------------------------------+
    | 68–72 | Spell scroll (5thlevel)        |
    +-------+--------------------------------+
    | 73–75 | Elixir of health               |
    +-------+--------------------------------+
    | 76–78 | Oil of etherealness            |
    +-------+--------------------------------+
    | 79–81 | Potion of fire giant strength  |
    +-------+--------------------------------+
    | 82–84 | Quaal's feather token          |
    +-------+--------------------------------+
    | 85–87 | Scroll of protection           |
    +-------+--------------------------------+
    | 88–89 | Bag of beans                   |
    +-------+--------------------------------+
    | 90-91 | Bead of force                  |
    +-------+--------------------------------+
    | 92    | Chime of opening               |
    +-------+--------------------------------+
    | 93    | Decanter of endless water      |
    +-------+--------------------------------+
    | 94    | Eyes of minute seeing          |
    +-------+--------------------------------+
    | 95    | Folding boat                   |
    +-------+--------------------------------+
    | 96    | Heward's handy haversack       |
    +-------+--------------------------------+
    | 97    | Horseshoes of speed            |
    +-------+--------------------------------+
    | 98    | Necklace of fireballs          |
    +-------+--------------------------------+
    | 99    | Periapt of health              |
    +-------+--------------------------------+
    | 100   | Sending Stones                 |
    +-------+--------------------------------+
    
    """
    name = "Magic Item Table C"


class MagicItemTableD(RandomTable):
    """
    +-------+--------------------------------+
    | d100  | Magic Item                     |
    +=======+================================+
    | 01–20 | Potion of supreme healing      |
    +-------+--------------------------------+
    | 21–30 | Potion of invisibility         |
    +-------+--------------------------------+
    | 31–40 | Potion of speed                |
    +-------+--------------------------------+
    | 41–50 | Spell scroll (6thlevel)        |
    +-------+--------------------------------+
    | 51–57 | Spell scroll (7thlevel)        |
    +-------+--------------------------------+
    | 58–62 | Ammunition, +3                 |
    +-------+--------------------------------+
    | 63–67 | Oil of sharpness               |
    +-------+--------------------------------+
    | 68–72 | Potion of flying               |
    +-------+--------------------------------+
    | 73–77 | Potion of cloud giant strength |
    +-------+--------------------------------+
    | 78–82 | Potion of longevity            |
    +-------+--------------------------------+
    | 83–87 | Potion of vitality             |
    +-------+--------------------------------+
    | 88–92 | Spell scroll (8thlevel)        |
    +-------+--------------------------------+
    | 93–95 | Horseshoes of a zephyr         |
    +-------+--------------------------------+
    | 96–98 | Nolzur's marvelous pigments    |
    +-------+--------------------------------+
    | 99    | Bag of devouring               |
    +-------+--------------------------------+
    | 100   | Portable hole                  |
    +-------+--------------------------------+
    
    """
    name = "Magic Item Table D"


class MagicItemTableE(RandomTable):
    """
    +--------+--------------------------------+
    | d100   | Magic Item                     |
    +========+================================+
    | 01–30  | Spell scroll (8thlevel)        |
    +--------+--------------------------------+
    | 31–55  | Potion of storm giant strength |
    +--------+--------------------------------+
    | 56–70  | Poti on of supreme healing     |
    +--------+--------------------------------+
    | 71–85  | Spell scroll (9st level)       |
    +--------+--------------------------------+
    | 86–93  | Universal solvent              |
    +--------+--------------------------------+
    | 94–98  | Arrow of slaying               |
    +--------+--------------------------------+
    | 99-100 | Sovereign glue                 |
    +--------+--------------------------------+
    
    """
    name = "Magic Item Table E"


class MagicItemTableF(RandomTable):
    """
    +-------+------------------------------------------------+
    |  d100 | Magic Item                                     |
    +=======+================================================+
    | 01–15 | Weapon, +1                                     |
    +-------+------------------------------------------------+
    | 16–18 | Shield,+ 1                                     |
    +-------+------------------------------------------------+
    | 19–21 | Sentinel shield                                |
    +-------+------------------------------------------------+
    | 22–23 | Amulet of proof against detection and location |
    +-------+------------------------------------------------+
    | 24–25 | Boots of elvenkind                             |
    +-------+------------------------------------------------+
    | 26–27 | Boots of striding and springing                |
    +-------+------------------------------------------------+
    | 27–29 | Bracers of archery                             |
    +-------+------------------------------------------------+
    | 30–31 | Brooch of shielding                            |
    +-------+------------------------------------------------+
    | 32–33 | Broom of flying                                |
    +-------+------------------------------------------------+
    | 34–35 | Cloak of elvenkind                             |
    +-------+------------------------------------------------+
    | 36–37 | Cloak of protection                            |
    +-------+------------------------------------------------+
    | 38–39 | Gauntlets of ogre power                        |
    +-------+------------------------------------------------+
    | 40–41 | Hat of disguise                                |
    +-------+------------------------------------------------+
    | 42–43 | Javelin of lightning                           |
    +-------+------------------------------------------------+
    | 44–45 | Pearl of power                                 |
    +-------+------------------------------------------------+
    | 46–47 | Rod of the pact keeper, + 1                    |
    +-------+------------------------------------------------+
    | 48–49 | Slippers of spider climbing                    |
    +-------+------------------------------------------------+
    | 50–51 | Staff of the adder                             |
    +-------+------------------------------------------------+
    | 52-53 | Staff of the python                            |
    +-------+------------------------------------------------+
    | 54-55 | Sword of vengeance                             |
    +-------+------------------------------------------------+
    | 56–57 | Trident of fish command                        |
    +-------+------------------------------------------------+
    | 58–59 | Wand of magic missiles                         |
    +-------+------------------------------------------------+
    | 60–61 | Wand of the war mage, + 1                      |
    +-------+------------------------------------------------+
    | 62–63 | Wand of web                                    |
    +-------+------------------------------------------------+
    | 64-65 | Weapon of warning                              |
    +-------+------------------------------------------------+
    |    66 | Adamantine armor (chain mail)                  |
    +-------+------------------------------------------------+
    |    67 | Adamantine armor (chain shirt)                 |
    +-------+------------------------------------------------+
    |    68 | Adamantine armor (scale mail)                  |
    +-------+------------------------------------------------+
    |    69 | Bag of tricks (gray)                           |
    +-------+------------------------------------------------+
    |    70 | Bag of tricks (rust)                           |
    +-------+------------------------------------------------+
    |    71 | Bag of tricks (tan)                            |
    +-------+------------------------------------------------+
    |    72 | Boots of the winterlands                       |
    +-------+------------------------------------------------+
    |    73 | Circlet of blasting                            |
    +-------+------------------------------------------------+
    |    74 | Deck of illusions                              |
    +-------+------------------------------------------------+
    |    75 | Eversmoking bottle                             |
    +-------+------------------------------------------------+
    |    76 | Eyes of charming                               |
    +-------+------------------------------------------------+
    |    77 | Eyes of the eagle                              |
    +-------+------------------------------------------------+
    |    78 | Figurine of wondrous power (silver raven)      |
    +-------+------------------------------------------------+
    |    79 | Gem of brightness                              |
    +-------+------------------------------------------------+
    |    80 | Gloves of missile snaring                      |
    +-------+------------------------------------------------+
    |    81 | Gloves of swimming and climbing                |
    +-------+------------------------------------------------+
    |    82 | Gloves of thievery                             |
    +-------+------------------------------------------------+
    |    83 | Headband of intellect                          |
    +-------+------------------------------------------------+
    |    84 | Helm of telepathy                              |
    +-------+------------------------------------------------+
    |    85 | Instrument of the bards (Doss lute)            |
    +-------+------------------------------------------------+
    |    86 | Instrument of the bards (Fochlucan bandore)    |
    +-------+------------------------------------------------+
    |    87 | Instrument of the bards (Mac-Fuimidh cittern)  |
    +-------+------------------------------------------------+
    |    88 | Medallion of thoughts                          |
    +-------+------------------------------------------------+
    |    89 | Necklace of adaptation                         |
    +-------+------------------------------------------------+
    |    90 | Periapt of wound closure                       |
    +-------+------------------------------------------------+
    |    91 | Pipes of haunting                              |
    +-------+------------------------------------------------+
    |    92 | Pipes of the sewers                            |
    +-------+------------------------------------------------+
    |    93 | Ring of jumping                                |
    +-------+------------------------------------------------+
    |    94 | Ring of mind shielding                         |
    +-------+------------------------------------------------+
    |    95 | Ring of warmth                                 |
    +-------+------------------------------------------------+
    |    96 | Ring of water walking                          |
    +-------+------------------------------------------------+
    |    97 | Quiver of Ehlonna                              |
    +-------+------------------------------------------------+
    |    98 | Stone of good luck                             |
    +-------+------------------------------------------------+
    |    99 | Wind fan                                       |
    +-------+------------------------------------------------+
    |   100 | Winged boots                                   |
    +-------+------------------------------------------------+
    
    """
    name = "Magic Item Table F"


class MagicItemTableG(RandomTable):
    """"
    +-------+-------------------------------------------+
    |  d100 | Magic Item                                |
    +=======+===========================================+
    | 01–11 | Weapon, +2                                |
    +-------+-------------------------------------------+
    | 12–14 | Figurine of wondrous power (roll d8)      |
    +-------+-------------------------------------------+
    |    -- | 1: Bronze griffon                         |
    +-------+-------------------------------------------+
    |    -- | 2: Ebony fly                              |
    +-------+-------------------------------------------+
    |    -- | 3: Golden lions                           |
    +-------+-------------------------------------------+
    |    -- | 4: Ivory goats                            |
    +-------+-------------------------------------------+
    |    -- | 5: Marble elephant                        |
    +-------+-------------------------------------------+
    |    -- | 6-7: Onyx dog                             |
    +-------+-------------------------------------------+
    |    -- | 8: Serpentine owl                         |
    +-------+-------------------------------------------+
    |    15 | Adamantine armor (breastplate)            |
    +-------+-------------------------------------------+
    |    16 | Adamantine armor (splint)                 |
    +-------+-------------------------------------------+
    |    17 | Amulet of health                          |
    +-------+-------------------------------------------+
    |    18 | Armor of vulnerability                    |
    +-------+-------------------------------------------+
    |    19 | Arrow-catching shield                     |
    +-------+-------------------------------------------+
    |    20 | Belt of dwarvenkind                       |
    +-------+-------------------------------------------+
    |    21 | Belt of hill giant strength               |
    +-------+-------------------------------------------+
    |    22 | Berserker axe                             |
    +-------+-------------------------------------------+
    |    23 | Boots of levitation                       |
    +-------+-------------------------------------------+
    |    24 | Boots of speed                            |
    +-------+-------------------------------------------+
    |    25 | Bowl of commanding water elementals       |
    +-------+-------------------------------------------+
    |    26 | Bracers of defense                        |
    +-------+-------------------------------------------+
    |    27 | Brazier of commanding fire elementals     |
    +-------+-------------------------------------------+
    |    28 | Cape of the mountebank                    |
    +-------+-------------------------------------------+
    |    29 | Censer of controlling air elementals      |
    +-------+-------------------------------------------+
    |    30 | Armor, +1 chain mail                      |
    +-------+-------------------------------------------+
    |    31 | Armor of resistance (chain mail)          |
    +-------+-------------------------------------------+
    |    32 | Armor of resistance (chain shirt)         |
    +-------+-------------------------------------------+
    |    33 | Armor,+ 1 chain shirt                     |
    +-------+-------------------------------------------+
    |    34 | Cloak of displacement                     |
    +-------+-------------------------------------------+
    |    35 | Cloak of the bat                          |
    +-------+-------------------------------------------+
    |    36 | Cube of force                             |
    +-------+-------------------------------------------+
    |    37 | Daern's instant fortress                  |
    +-------+-------------------------------------------+
    |    38 | Dagger of venom                           |
    +-------+-------------------------------------------+
    |    39 | Dimensional shackles                      |
    +-------+-------------------------------------------+
    |    40 | Dragon slayer                             |
    +-------+-------------------------------------------+
    |    41 | Elven chain                               |
    +-------+-------------------------------------------+
    |    42 | Flame tongue                              |
    +-------+-------------------------------------------+
    |    43 | Gem of seeing                             |
    +-------+-------------------------------------------+
    |    44 | Giant slayer                              |
    +-------+-------------------------------------------+
    |    45 | Clamoured studded leather                 |
    +-------+-------------------------------------------+
    |    46 | Helm of teleportation                     |
    +-------+-------------------------------------------+
    |    47 | Horn of blasting                          |
    +-------+-------------------------------------------+
    |    48 | Horn of Valhalla (silver or brass)        |
    +-------+-------------------------------------------+
    |    49 | Instrument of the bards (Canaithmandolin) |
    +-------+-------------------------------------------+
    |    50 | Instrument ofthe bards (Cii lyre)         |
    +-------+-------------------------------------------+
    |    51 | loun stone (awareness)                    |
    +-------+-------------------------------------------+
    |    52 | loun stone (protection)                   |
    +-------+-------------------------------------------+
    |    53 | loun stone (reserve)                      |
    +-------+-------------------------------------------+
    |    54 | loun stone (sustenance)                   |
    +-------+-------------------------------------------+
    |    55 | Iron bands of Bilarro                     |
    +-------+-------------------------------------------+
    |    56 | Armor, + 1 leather                        |
    +-------+-------------------------------------------+
    |    57 | Armor of resistance (leather)             |
    +-------+-------------------------------------------+
    |    58 | Mace of disruption                        |
    +-------+-------------------------------------------+
    |    59 | Mace of smiting                           |
    +-------+-------------------------------------------+
    |    60 | Mace of terror                            |
    +-------+-------------------------------------------+
    |    61 | Mantle of spell resistance                |
    +-------+-------------------------------------------+
    |    62 | Necklace of prayer beads                  |
    +-------+-------------------------------------------+
    |    63 | Periapt of proof against poison           |
    +-------+-------------------------------------------+
    |    64 | Ring of animal influence                  |
    +-------+-------------------------------------------+
    |    65 | Ring of evasion                           |
    +-------+-------------------------------------------+
    |    66 | Ring of feather falling                   |
    +-------+-------------------------------------------+
    |    67 | Ring of free action                       |
    +-------+-------------------------------------------+
    |    68 | Ring of protection                        |
    +-------+-------------------------------------------+
    |    69 | Ring of resistance                        |
    +-------+-------------------------------------------+
    |    70 | Ring of spell storing                     |
    +-------+-------------------------------------------+
    |    71 | Ring of the ram                           |
    +-------+-------------------------------------------+
    |    72 | Ring of X-ray vision                      |
    +-------+-------------------------------------------+
    |    73 | Robe of eyes                              |
    +-------+-------------------------------------------+
    |    74 | Rod of rulership                          |
    +-------+-------------------------------------------+
    |    75 | Rod of the pact keeper, +2                |
    +-------+-------------------------------------------+
    |    76 | Rope of entanglement                      |
    +-------+-------------------------------------------+
    |    77 | Armor, +1 scale mail                      |
    +-------+-------------------------------------------+
    |    78 | Armor of resistance (scale mail)          |
    +-------+-------------------------------------------+
    |    79 | Shield, +2                                |
    +-------+-------------------------------------------+
    |    80 | Shield of missile attraction              |
    +-------+-------------------------------------------+
    |    81 | Staff of charming                         |
    +-------+-------------------------------------------+
    |    82 | Staff of healing                          |
    +-------+-------------------------------------------+
    |    83 | Staff of swarming insects                 |
    +-------+-------------------------------------------+
    |    84 | Staff of the woodlands                    |
    +-------+-------------------------------------------+
    |    85 | Staff of withering                        |
    +-------+-------------------------------------------+
    |    86 | Stone of controlling earthelementals      |
    +-------+-------------------------------------------+
    |    87 | Sun blade                                 |
    +-------+-------------------------------------------+
    |    88 | Sword of life stealing                    |
    +-------+-------------------------------------------+
    |    89 | Sword of wounding                         |
    +-------+-------------------------------------------+
    |    90 | Tentacle rod                              |
    +-------+-------------------------------------------+
    |    91 | Vicious weapon                            |
    +-------+-------------------------------------------+
    |    92 | Wand of binding                           |
    +-------+-------------------------------------------+
    |    93 | Wand of enemy detection                   |
    +-------+-------------------------------------------+
    |    94 | Wand of fear                              |
    +-------+-------------------------------------------+
    |    95 | Wand of fireballs                         |
    +-------+-------------------------------------------+
    |    96 | Wand of lightning bolts                   |
    +-------+-------------------------------------------+
    |    97 | Wand of paralysis                         |
    +-------+-------------------------------------------+
    |    98 | Wand of the war mage, +2                  |
    +-------+-------------------------------------------+
    |    99 | Wand of wonder                            |
    +-------+-------------------------------------------+
    |   100 | Wings of flying                           |
    +-------+-------------------------------------------+
    
    """
    name = "Magic Item Table G"


class MagicItemTableH(RandomTable):
    """
    +-------+---------------------------------------------+
    |  d100 | Magic Item                                  |
    +=======+=============================================+
    | 01–10 | Weapon, +3                                  |
    +-------+---------------------------------------------+
    | 11–12 | Amulet of the planes                        |
    +-------+---------------------------------------------+
    | 13–14 | Carpet of flying                            |
    +-------+---------------------------------------------+
    | 15–16 | Crystal ball (very rare version)            |
    +-------+---------------------------------------------+
    | 17–18 | Ring of regeneration                        |
    +-------+---------------------------------------------+
    | 19–20 | Ring of shooting stars                      |
    +-------+---------------------------------------------+
    | 21–22 | Ring of telekinesis                         |
    +-------+---------------------------------------------+
    | 23–24 | Robe of scintillating colors                |
    +-------+---------------------------------------------+
    | 25–26 | Robe of stars                               |
    +-------+---------------------------------------------+
    | 27–28 | Rod of absorption                           |
    +-------+---------------------------------------------+
    | 29–30 | Rod of alertness                            |
    +-------+---------------------------------------------+
    | 31–32 | Rod of security                             |
    +-------+---------------------------------------------+
    | 33–34 | Rod of the pact keeper, +3                  |
    +-------+---------------------------------------------+
    | 35–36 | Scimitar of speed                           |
    +-------+---------------------------------------------+
    | 37–38 | Shield, +3                                  |
    +-------+---------------------------------------------+
    | 39–40 | Staff of fire                               |
    +-------+---------------------------------------------+
    | 41–42 | Staff of frost                              |
    +-------+---------------------------------------------+
    | 43–44 | Staff of power                              |
    +-------+---------------------------------------------+
    | 45-46 | Staff of striking                           |
    +-------+---------------------------------------------+
    | 47-48 | Staff of thunder and lightning              |
    +-------+---------------------------------------------+
    | 49–50 | Sword of sharpnes                           |
    +-------+---------------------------------------------+
    | 51–52 | Wand of polymorph                           |
    +-------+---------------------------------------------+
    | 53–54 | Wand of the war mage, + 3                   |
    +-------+---------------------------------------------+
    |    55 | Adamantine armor (half plate)               |
    +-------+---------------------------------------------+
    |    56 | Adamantine armor (plate)                    |
    +-------+---------------------------------------------+
    |    57 | Animated shield                             |
    +-------+---------------------------------------------+
    |    58 | Belt of fire giant strength                 |
    +-------+---------------------------------------------+
    |    59 | Belt of frost (or stone) giant strength     |
    +-------+---------------------------------------------+
    |    60 | Armor, + 1 breastplate                      |
    +-------+---------------------------------------------+
    |    61 | Armor of resistance (breastplate)           |
    +-------+---------------------------------------------+
    |    62 | Candle of invocation                        |
    +-------+---------------------------------------------+
    |    63 | Armor, +2 chain mail                        |
    +-------+---------------------------------------------+
    |    64 | Armor, +2 chain shirt                       |
    +-------+---------------------------------------------+
    |    65 | Cloak of arachnida                          |
    +-------+---------------------------------------------+
    |    66 | Dancing sword                               |
    +-------+---------------------------------------------+
    |    67 | Demon armor                                 |
    +-------+---------------------------------------------+
    |    68 | Dragon scale mail                           |
    +-------+---------------------------------------------+
    |    69 | Dwarven plate                               |
    +-------+---------------------------------------------+
    |    70 | Dwarven thrower                             |
    +-------+---------------------------------------------+
    |    71 | Efreeti bottle                              |
    +-------+---------------------------------------------+
    |    72 | Figurine of wondrous power (obsidian steed) |
    +-------+---------------------------------------------+
    |    73 | Frost brand                                 |
    +-------+---------------------------------------------+
    |    74 | Helm of brilliance                          |
    +-------+---------------------------------------------+
    |    75 | Horn ofValhalla (bronze)                    |
    +-------+---------------------------------------------+
    |    76 | Instrument of the bards (Anstruthharp)      |
    +-------+---------------------------------------------+
    |    77 | loun stone (absorption)                     |
    +-------+---------------------------------------------+
    |    78 | loun stone (agility)                        |
    +-------+---------------------------------------------+
    |    79 | loun stone (fortitude)                      |
    +-------+---------------------------------------------+
    |    80 | loun stone (insight)                        |
    +-------+---------------------------------------------+
    |    81 | loun stone (intellect)                      |
    +-------+---------------------------------------------+
    |    82 | loun stone (leadership)                     |
    +-------+---------------------------------------------+
    |    83 | loun stone (strength)                       |
    +-------+---------------------------------------------+
    |    84 | Armor, +2 leather                           |
    +-------+---------------------------------------------+
    |    85 | Manual of bodily health                     |
    +-------+---------------------------------------------+
    |    86 | Manual of gainful exercise                  |
    +-------+---------------------------------------------+
    |    87 | Manual of golems                            |
    +-------+---------------------------------------------+
    |    88 | Manual of quickness of action               |
    +-------+---------------------------------------------+
    |    89 | Mirror of life trapping                     |
    +-------+---------------------------------------------+
    |    90 | Nine lives stealer                          |
    +-------+---------------------------------------------+
    |    91 | Oathbow                                     |
    +-------+---------------------------------------------+
    |    92 | Armor, +2 scale mail                        |
    +-------+---------------------------------------------+
    |    93 | Spellguard shield                           |
    +-------+---------------------------------------------+
    |    94 | Armor, + 1 splint                           |
    +-------+---------------------------------------------+
    |    95 | Armor of resistance (splint)                |
    +-------+---------------------------------------------+
    |    96 | Armor, + 1 studded leather                  |
    +-------+---------------------------------------------+
    |    97 | Armor of resistance (studded leather)       |
    +-------+---------------------------------------------+
    |    98 | Tome of clear thought                       |
    +-------+---------------------------------------------+
    |    99 | Tome of leadership and influence            |
    +-------+---------------------------------------------+
    |   100 | Tome of understanding                       |
    +-------+---------------------------------------------+
    
    """
    name = "Magic Item Table H"


class MagicItemTableI(RandomTable):
    """
    +-------+---------------------------------------+
    |  d100 | Magic Item                            |
    +=======+=======================================+
    | 01–05 | Defender                              |
    +-------+---------------------------------------+
    | 06–10 | Hammer of thunderbolts                |
    +-------+---------------------------------------+
    | 11–15 | Luck Blade                            |
    +-------+---------------------------------------+
    | 16–20 | Sword of answering                    |
    +-------+---------------------------------------+
    | 21–23 | Holy avenger                          |
    +-------+---------------------------------------+
    | 24–26 | Ring of djinni summoning              |
    +-------+---------------------------------------+
    | 27–29 | Ring of invisibility                  |
    +-------+---------------------------------------+
    | 30–32 | Ring of spell turning                 |
    +-------+---------------------------------------+
    | 36–38 | Rod of lordly might                   |
    +-------+---------------------------------------+
    | 39–41 | Vorpal sword                          |
    +-------+---------------------------------------+
    | 42–43 | Belt of cloud giant strength          |
    +-------+---------------------------------------+
    | 44–45 | Armor, +2 breastplate                 |
    +-------+---------------------------------------+
    | 46–47 | Armor, +3 chain mail                  |
    +-------+---------------------------------------+
    | 48–49 | Armor, +3 chain shirt                 |
    +-------+---------------------------------------+
    | 50–51 | Cloak of invisibility                 |
    +-------+---------------------------------------+
    | 52–53 | Crystal ball (legendary version)      |
    +-------+---------------------------------------+
    | 54-55 | Armor, + 1 half plate                 |
    +-------+---------------------------------------+
    | 56-57 | Iron flask                            |
    +-------+---------------------------------------+
    | 58-59 | Armor, +3 leather                     |
    +-------+---------------------------------------+
    | 60-61 | Armor, +1 plate                       |
    +-------+---------------------------------------+
    | 62-63 | Robe of the archmagi                  |
    +-------+---------------------------------------+
    | 64-65 | Rod of resurrection                   |
    +-------+---------------------------------------+
    | 66-67 | Armor, +1 scale mail                  |
    +-------+---------------------------------------+
    | 68-69 | Scarab of protection                  |
    +-------+---------------------------------------+
    | 70-71 | Armor, +2 splint                      |
    +-------+---------------------------------------+
    | 72-73 | Armor, +2 studded leather             |
    +-------+---------------------------------------+
    | 74-75 | Well of many worlds                   |
    +-------+---------------------------------------+
    |    76 | Magic armor (roll dl2)                |
    +-------+---------------------------------------+
    |    -- | 1-2: Armor, +2 half plate             |
    +-------+---------------------------------------+
    |    -- | 3-4: Armor, +2 plate                  |
    +-------+---------------------------------------+
    |    -- | 5-6: Armor, +3 studded leather        |
    +-------+---------------------------------------+
    |    -- | 7-8: Armor, +3 breastplate            |
    +-------+---------------------------------------+
    |    -- | 9-10: Armor, +3 splint                |
    +-------+---------------------------------------+
    |    -- | 11: Armor, +3 half plate              |
    +-------+---------------------------------------+
    |    -- | 12: Armor, +3 plate                   |
    +-------+---------------------------------------+
    |    77 | Apparatus of Kwalish                  |
    +-------+---------------------------------------+
    |    78 | Armor of invulnerability              |
    +-------+---------------------------------------+
    |    79 | Belt of storm giant strength          |
    +-------+---------------------------------------+
    |    80 | Cubic gate                            |
    +-------+---------------------------------------+
    |    81 | Deck of many things                   |
    +-------+---------------------------------------+
    |    82 | Efreeti chain                         |
    +-------+---------------------------------------+
    |    83 | Armor of resistance (half plate)      |
    +-------+---------------------------------------+
    |    84 | Horn ofValhalla (iron)                |
    +-------+---------------------------------------+
    |    85 | Instrument of the bards (OIIamh harp) |
    +-------+---------------------------------------+
    |    86 | loun stone (greater absorption)       |
    +-------+---------------------------------------+
    |    87 | loun stone (mastery)                  |
    +-------+---------------------------------------+
    |    88 | loun stone (regeneration)             |
    +-------+---------------------------------------+
    |    89 | Plate armor of etherealness           |
    +-------+---------------------------------------+
    |    90 | Plate armor of resistance             |
    +-------+---------------------------------------+
    |    91 | Ring of air elemental command         |
    +-------+---------------------------------------+
    |    92 | Ring of earthelemental command        |
    +-------+---------------------------------------+
    |    93 | Ring of fire elemental command        |
    +-------+---------------------------------------+
    |    94 | Ring of three wishes                  |
    +-------+---------------------------------------+
    |    95 | Ring of water elemental command       |
    +-------+---------------------------------------+
    |    96 | Sphere of annihilation                |
    +-------+---------------------------------------+
    |    97 | Talisman of pure good                 |
    +-------+---------------------------------------+
    |    98 | Talisman of the sphere                |
    +-------+---------------------------------------+
    |    99 | Talisman of ultimate evil             |
    +-------+---------------------------------------+
    |   100 | Tome of the stilled tongue            |
    +-------+---------------------------------------+
    
    """
    name = "Magic Item Table I"


class Treasure(RandomTable):
    """"""
    name = "Treasure"
    subtables = [IndividualTreasure0To4, IndividualTreasure5To10,
                 IndividualTreasure11To16, IndividualTreasure17Plus,
                 HoardTreasure0To4, HoardTreasure5To10,
                 HoardTreasure11To16, HoardTreasure17Plus,
                 MagicItemTableA, MagicItemTableB, MagicItemTableC,
                 MagicItemTableD, MagicItemTableE, MagicItemTableF,
                 MagicItemTableG, MagicItemTableH, MagicItemTableI]


class Curses(RandomTable):
    """

    - 1: The character cannot turn right until the curse is lifted.
    - 2: Characters feet always sink at least 1 inch into any surface they walk on (the at least accounts for walking on water, as in if they try to walk on water they sink normally)
    - 3: All [food] type becomes tasteless (meat, vegetables, fruit)
    - 4: When the character fires a ranged weapon, the ammunition always breaks on impact (no effect on damage)
    - 5: Until the curse is lifted, when the character falls to 0hp, roll a D100. If you roll equal to or below the CR of the creature that cursed you, you instantly die.
    - 6: A player must close every door they walk through, even if there are people behind them.
    - 7: A player's weapon becomes lodged inside the body of their enemy after any stab attack, a strength check (DC 15) is needed to free the weapon.
    - 8: A player's weapons become twice as heavy, requiring two actions to strike once, until the curse is lifted.
    - 9: A player is stalked by an imp, who simply follows him, saying nothing, always staring. No one else can see the imp.
    - 10: The player's backpack is enchanted, to always give the player an item they needed in the past, but never what they will need in the future or present.
    - 11: The next item the cursed player grabs is bound to them forever, they can never get rid of it.
    - 12: Everytime the cursed character kills someone stealthily, the slain thing lets out an incredibly loud scream that can be heard from 500 ft away, even if it wouldn't be possible for the dead thing to scream.
    - 13: Character takes on the appearance and smell of being undead, but isn't.
    - 14: Characters must only answer questions with lies, unless they are asked about the reason for their behaviour (ex: "are you cursed?" "Are you lying on purpose?") In which case they must respond in the affirmative.
    - 15: Characters must agree to every suggestion or request made within 30 feet of them. Curse is broken after a week.
    - 16: The cursed character takes 1 damage whenever a creature within 30 feet of them takes any damage.
    - 17: Character cannot willingly kill/spare the life of any living creature (choose depending on character personality).
    - 18: Character becomes incapable of visually perceiving living creatures.
    - 19: Characters low-light vision and high-light vision switch (i.e. sunlight is effectively dark, but can see areas in shadow as if they were brightly lit).
    - 20: Roll a d100. After the amount of dies shown on dice, the character explodes for (as per a 5th level Fireball) the next time they take a long rest, then is immediately put under the effects of a Reincarnation spell. The cycle continues until a Wish spell dispels it.
    - 21: Character is struck with blindness, but can accurately identify objects by taste through the air up to 60 feet away.
    - 22: Butt switches place with face. Switches every time either orifice expels any substance.
    - 23: Your CHA stat becomes your CHAR stat, determining your effectiveness at cooking up a mean barbeque. Reflavour spells and skill checks accordingly.
    - 24: When the target of the curse next goes to sleep, they dream of a burning lake. The dreams progress, becoming nightmares over time. The target instinctively becomes aware of the direction of the lake, and must save vs Wis or spend that day trying to reach the lake. The target must save every day to prevent the condition progressing, taking a penalty to mental rolls for every stage it advances. To completely recover, the target must make 3 saves in a row, if they fail a save it regresses to its initial condition, and if they fail 3 times in a row the target becomes maddened until they reach the lake. Upon reaching the lake they will see it is not engulfed in flames, and will take d6 Psychic damage for the number of days they have been affected by the urges.
    - 25: The first ritual performed after being cursed succeeds instantly, but when they next sleep the target must save vs. Con. If they fail, their skin dries and their body catches alight, taking d6 damage per turn. The fire can be put out by magical or mundane conventions.
    - 26: The cursed begins aging at 5 years an hour. When they reach 100 years, they die, and an infant crawls from their body’s clothing. It continues to age at the same rate until it reaches 20. Same character, same memories.
    - 27: As the curse is activated, the target's hands detach from their wrists and scuttle away, and new hands grow in their place. For the rest of the day, every time they cast a spell, the same thing happens. The hands remain animate until destroyed, and will do their best to make terrible mischief.
    - 28: A thunderous voice narrates everything the cursed does, says, or thinks for the next d4 hours.
    - 29: For the next d4 days, every time the cursed attempts to speak, including to cast a spell, they must Save vs Int or instead deliver a lengthy and discursive monologue on: 1: bean cultivation; 2: the daily schedule of an emperor who died thousands of years ago; 3: the spiritual beliefs of spiders; 4: the life cycle of the cherub; 5: the various manias, phobias, or perversions of the nearest, most powerful monarch; 6: the correct method of preparing, storing, and administering a heretofore unknown and spectacularly deadly poison; 7: the best tourist destinations in the nearest village; 8: famous fish poets; 9: the dangers of breathing; 10: the magical properties of cheese; On a repeated roll, the target must continue their lecture from where they left off before.
    - 30: Until the curse is lifted the character constantly sniffs and has a runny nose. Disadvantage on stealth, persuasion and deception checks.
    - 31: Character can not control the volume at which he speaks. Player rolls a D6 every time their character speaks, even rolls are spoken in a whisper, odd rolls are shouting.
    - 32: Animals and children are always aware of your presence and are able to locate you without difficulty.
    - 33: "Curse of Popularity" - In populated area with non-hostile NPCs, everyone knows who you are and will not leave you alone. Roll a charisma check/save (DC varies). If failed, you are viewed in an unfavorable light. If passed, you are viewed in a favorable one.
    - 34: Everytime a player deals damage the same amount is reflected back to a random party member.
    - 35: All food and drink consumed immediately tastes of rotten flesh a successful fortitude save of DC 15 can overcome this taste.
    - 36: Whenever the PC comes into a hallway/corridor they are compelled to Sprint at full speed to the end. Will save to resist at DMs discretion.
    - 37: The PC must only speak in rhyme.
    - 38: The PC gains a new fear based on popular vote of the party until dispelled.
    - 39: All the player's equipment glows brightly for 24 hours. All of it.
    - 40: The player becomes magnetic.
    - 41: It is always raining in a 5ft cube around the player. The intensity randomly varies from a drizzle to a downpour and can exist even underwater or indoors.
    - 42: The character finds themselves unable to open any containers or doors which require a twisting motion.
    - 43: The character perceives traps everywhere where none exist.
    - 44: The cursed becomes lactose intolerant. Consuming any dairy leads to 1d4 hour(s) of insufferable gas & diarrhea.
    - 45: The player must compulsively juggle items any time the player has two or more of an object in easy reach. DC 10+the number of items being juggled Acrobatics check, or an item gets dropped, with appropriate consequences.
    - 46: Boots squeak loudly with each step.
    - 47: Effects of alcohol are heavily amplified, so that even drinking one drop of a fairly weak alcoholic beverage will make the PC drunk. Drinking a full glass of a strong alcoholic beverage could potentially cause death.
    - 48: All of the PC's armor and clothing teleported off their body and always floats just out of reach. Any attempt to put on other clothing or armor produces the same effect.
    - 49: Character's known languages are randomly determined after a long rest. Roll 1d8 per standard language known & 1d8 per exotic language known. You decide whether to exclude common from these rolls or not.
    - 50: After a long rest a random amount of GP the character is carrying is randomly changed to an amount of either Electrum, Silver or Copper pieces of the same worth, increasing number of coins. eg. (1d20 Amount, Roll 1d6 to determine type) Won't take long for pockets to become overflowing if character doesn't spend loose change.
    - 51: Any divination spells where the caster or target is within a certain range of a character are retargeted to that character.
    - 52: The cursed begins to weep tears of blood uncontrollably, reducing their hit point maximum by 1 for every hour the curse remains active. The cursed dies if this effect reduces its hit point maximum to 0.
    - 53: The cursed is compelled to repeat the last word of each sentence they say 3 times, each time speaking a little bit softer than the last. If the curse remains active for more than 24 hours, the cursed is compelled to dramatically flick their hands open and closed with each echo.
    - 54: Cursed characters are hated by all cats until cured. Every cat will hiss and attempt to swipe and bite the character. Irregardless of the cat is successful or not the cat will run away and hide. If the cat is successful in the attack any wounds caused will not heal (even with healing spells and potions) and will continuously weep foul smelling pus.
    - 55: The character cannot be convinced by any means that magic exists. They rationalize magical events away by using insane, impossible logic.
    - 56: The character believes themselves to have swapped bodies with the nearest person. Nothing has happened.
    - 57: One of the character's limbs no longer has any bones. It flops around uselessly until the bones have successfully regrown in 1d4 days.
    - 58: The character is unable to sleep when others are sleeping in a 60' radius.
    - 59: Once the character has fallen asleep they cannot be awoken by any non-magical means until 8 hours have passed.
    - 60: The character must consume 1d4+1 times the amount of food and drink a normal person does do sustain themselves. They experience terrible thirst and hunger pains. Treat as exhaustion if they do not actively maintain this regimen.
    - 61: The character cannot see anyone within 10 feet of them.
    - 62: The character finds a wooden spoon in their bag. Every time they retrieve an item they find another wooden spoon. Every time they investigate an area they find another wooden spoon. Every time the search a body they find another wooden spoon. If they intentionally attempt to locate, retrieve, or use a spoon the task is impossible.
    - 63: Incapable of ignoring direct orders given to their person.
    - 64: When splashed with cold water transforms the character into the opposite gender. Warm water temporarily reverses the transition.
    - 65: Must make one significant lie per day.
    - 66: Automatically fails all swimming checks; it's as if the character weights 10 times their normal weight while in water.
    - 67: Turned into a lycanthrope... with the form of a rabbit.
    - 68: All creatures of a specific species are invisible to the character.
    - 69: A perpetually magical darkness surrounds the character for 25 feet. It is transmittable by touch.
    - 70: When killed for the first time each day, the wounds heal and they instead stabilize. If they are not killed once a day, they are permanently slain.
    - 71: Makes an unarmed attack against themselves whenever they say 'what'.
    - 72: Characters ears and eyes switch place. PC cannot look straight ahead. This lasts until the next full day ends.
    - 73: PC summons a little foot tall naked man with a hat that goofs around and makes as much noise as possible. Everyone can hear him and see his impact in the area, but cannot see him. He has no name and will not speak back. He just speaks in sounds and screams.
    - 74: PC thinks their eyes have the same effect as a beholder and use them as much as possible in combat.
    - 75: PCs teeth are as weak as glass.
    - 76: PCs weapon changes to the next material they touch.
    - 77: PC is followed by all bugs within 20 yards. (Bonus points if there are ant hills around).
    - 78: PC grows a mouth in their chest. You can feed the mouth but you do not know what it will do. Overtime the mouth will grow if unfed.
    - 79: PCs money all goes down a material (gold turns to silver) until curse is lifted. Copper turns into wooden toy coins that children would play with.
    - 80: All plants the PC touches turn to dust for the next week. (Bonus points if a druid gets this)
    - 81: All potions being held by PC give a delusion effect (example: PC thinks they are invisible but are not.)
    - 82: All damage given to the PC for the next 12 minutes are irreversible.
    - 83: The next person the PC touches switches all items.
    - 84: PC's armor or clothing (whichever applicable) is made entirely out of shards of glass magically held together.
    - 85: Character must kill one humanoid per week or die themselves.
    - 86: Character must read one book per week or die, it must be a book they have not read before.
    - 87: Every dawn, gravity reverses for the character for one hour.
    - 88: Whenever the character physically harms another sentient being the character must apologize.
    - 89: Whenever the character is on a sea vehicle of any kind the character vomits anything he/she eats and cannot sleep.
    - 90: The PCs hand's tense up, and are stuck in a fist until the curse is removed.
    - 91: The character's left and right hands, and/or left and right feet switch sides until the curse is lifted.
    - 92: This curse hardens all food this character tries to eat like stone, unless they have the correct eating utensil to eat it. (ie They need a spoon to eat soup, a fork to eat pie, a knife to cut meat, etc).
    - 93: The player believes that their mentor/parent has just died.
    - 94: The player receives at least one false vision from their deity a day.
    - 95: The cursed player can no longer fail the expectations of those they come across. (For example, if an NPC learns the player's name and they believe their name to be elfish, they will become an elf. If they expect that he is a weakling due to rumors they have heard of him losing an important fight, the player looses some of his strength. If they thought he would be taller, the character would become taller, etc...)
    - 96: The player is cursed to look down at the ground; they can no longer make eye contact with others, unless they are able to look down on them...
    - 97: Until this player's curse is lifted, as long as they wear shoes/boots, they will feel as if their feet are walking on burning hot coals.
    - 98: This player has been cursed to be afraid of the sun.
    - 99: This curse makes the player compelled to hug all characters they come across, even if it would be inappropriate or awkward.
    - 100: This curse makes the character forcibly say gibberish every time they cast a spell.
    
    """
    name = "Curses"
