"""
A library which provides implementations of the kernel of the
Alchemical Integral Transform (AIT) in 1D, 2D, 3D.

Throughout this code, Hartree atomic units are used.

"""

# Regulator for numerically instable fractions like v_B/v_A
_reg = 1e-8

# factorial function
def _fc(n):
    if n == 0 or n == 1:
        return 1
    else:
        return n*_fc(n-1)

# binomial coefficient
def _bn(n,k):
    return _fc(n)/(_fc(k)*_fc(n-k))

# Built-in function for 3D potentials of molecules
def partial_v_mol_3D(mole, n_x, n_y, n_z, x, y, z, nuc_rad = 0):
    """
    A function for the external potential in 3D of a given molecule and its spatial derivatives.
    These derivatives are analytical up to and including third order :math:`n_x+n_y+n_z \leq 3`,
    and defined recursively via finite differences for higher orders.

    Parameters
    ----------
    mole : (..., 4) array
        A list of lists of the 4D coordinates (nuclear charge Z_i, coordinates x_i, y_i, z_i of all atoms,
        i.e. `mole = [[Z_1, x_1, y_1, z_1], [Z_2, x_2, y_2, z_2], ...]`
    n_x, n_y, n_z : int
        Order of the derivative
    x, y, z : float
        coordinates
    nuc_rad : float, optional
        An optional nuclear radius :math:`\eta` such that the Coulomb potential is rendered finite everywhere:
        .. math:: -Z_i [(x - x_i)^2 + (y - y_i)^2 + (z - z_i)^2]^{-1/2} \rightarrow -Z_i [(x - x_i)^2 + (y - y_i)^2 + (z - z_i)^2 + \eta^2]^{-1/2}

    Returns
    -------
    float
        the :math:`n_x+n_y+n_z`-th derivative of the external potential of `mole`
        with nuclear radius `nuc_rad` at `x,y,z`,
         i.e. :math:`\frac{\partial^{n_x + n_y + n_z} }{\partial x^{n_x} \partial y^{n_y} \partial z^{n_z} } v_{\text{mole}}(x,y,z)`


    """
    sum = 0
    # ----------------------------0-th derivatives------------------------------
    if n_x == 0 and n_y == 0 and n_z == 0:
        for i in range(len(mole)):
            sum += -mole[i][0]/((nuc_rad**2+(x-mole[i][1])**2+(y-mole[i][2])**2+(z-mole[i][3])**2)**(1/2) + _reg)
    # ----------------------------1-st derivatives------------------------------
    elif n_x == 1 and n_y == 0 and n_z == 0:
        for i in range(len(mole)):
            sum += mole[i][0]*(x-mole[i][1])/((nuc_rad**2+(x-mole[i][1])**2+(y-mole[i][2])**2+(z-mole[i][3])**2)**(3/2) + _reg)
    elif n_x == 0 and n_y == 1 and n_z == 0:
        for i in range(len(mole)):
            sum += mole[i][0]*(y-mole[i][2])/((nuc_rad**2+(x-mole[i][1])**2+(y-mole[i][2])**2+(z-mole[i][3])**2)**(3/2) + _reg)
    elif n_x == 0 and n_y == 0 and n_z == 1:
        for i in range(len(mole)):
            sum += mole[i][0]*(z-mole[i][3])/((nuc_rad**2+(x-mole[i][1])**2+(y-mole[i][2])**2+(z-mole[i][3])**2)**(3/2) + _reg)
    # ----------------------------2-nd derivatives------------------------------
    elif n_x == 2 and n_y == 0 and n_z == 0:
        for i in range(len(mole)):
            sum += mole[i][0]*(-2*(x-mole[i][1])**2+(y-mole[i][2])**2+(z-mole[i][3])**2)/((nuc_rad**2+(x-mole[i][1])**2+(y-mole[i][2])**2+(z-mole[i][3])**2)**(5/2) + _reg)
    elif n_x == 0 and n_y == 2 and n_z == 0:
        for i in range(len(mole)):
            sum += mole[i][0]*(-2*(y-mole[i][2])**2+(x-mole[i][1])**2+(z-mole[i][3])**2)/((nuc_rad**2+(x-mole[i][1])**2+(y-mole[i][2])**2+(z-mole[i][3])**2)**(5/2) + _reg)
    elif n_x == 0 and n_y == 0 and n_z == 2:
        for i in range(len(mole)):
            sum += mole[i][0]*(-2*(z-mole[i][3])**2+(x-mole[i][1])**2+(y-mole[i][2])**2)/((nuc_rad**2+(x-mole[i][1])**2+(y-mole[i][2])**2+(z-mole[i][3])**2)**(5/2) + _reg)
    elif n_x == 1 and n_y == 1 and n_z == 0:
        for i in range(len(mole)):
            sum += -3*mole[i][0]*(x-mole[i][1])*(y-mole[i][2])/((nuc_rad**2+(x-mole[i][1])**2+(y-mole[i][2])**2+(z-mole[i][3])**2)**(5/2) + _reg)
    elif n_x == 1 and n_y == 0 and n_z == 1:
        for i in range(len(mole)):
            sum += -3*mole[i][0]*(x-mole[i][1])*(z-mole[i][3])/((nuc_rad**2+(x-mole[i][1])**2+(y-mole[i][2])**2+(z-mole[i][3])**2)**(5/2) + _reg)
    elif n_x == 0 and n_y == 1 and n_z == 1:
        for i in range(len(mole)):
            sum += -3*mole[i][0]*(y-mole[i][2])*(z-mole[i][3])/((nuc_rad**2+(x-mole[i][1])**2+(y-mole[i][2])**2+(z-mole[i][3])**2)**(5/2) + _reg)
    # ----------------------------3-rd derivatives------------------------------
    elif n_x == 3 and n_y == 0 and n_z == 0:
        for i in range(len(mole)):
            sum += mole[i][0]*6*((x-mole[i][1])**3 - (x-mole[i][1])*(y-mole[i][2])**2 - (x-mole[i][1])*(z-mole[i][3])**2)/((nuc_rad**2+(x-mole[i][1])**2+(y-mole[i][2])**2+(z-mole[i][3])**2)**(7/2) + _reg)
    elif n_x == 0 and n_y == 3 and n_z == 0:
        for i in range(len(mole)):
            sum += mole[i][0]*6*((y-mole[i][2])**3 - (y-mole[i][2])*(x-mole[i][1])**2 - (y-mole[i][2])*(z-mole[i][3])**2)/((nuc_rad**2+(x-mole[i][1])**2+(y-mole[i][2])**2+(z-mole[i][3])**2)**(7/2) + _reg)
    elif n_x == 0 and n_y == 0 and n_z == 3:
        for i in range(len(mole)):
            sum += mole[i][0]*6*((z-mole[i][3])**3 - (z-mole[i][3])*(x-mole[i][1])**2 - (z-mole[i][3])*(y-mole[i][2])**2)/((nuc_rad**2+(x-mole[i][1])**2+(y-mole[i][2])**2+(z-mole[i][3])**2)**(7/2) + _reg)
    elif n_x == 2 and n_y == 1 and n_z == 0:
        for i in range(len(mole)):
            sum += -mole[i][0]*3*(y - mole[i][2])*( (-4)*(x-mole[i][1])**2+(y-mole[i][2])**2+(z-mole[i][3])**2 )/((nuc_rad**2+(x-mole[i][1])**2+(y-mole[i][2])**2+(z-mole[i][3])**2)**(7/2) + _reg)
    elif n_x == 2 and n_y == 0 and n_z == 1:
        for i in range(len(mole)):
            sum += -mole[i][0]*3*(z - mole[i][3])*( (-4)*(x-mole[i][1])**2+(y-mole[i][2])**2+(z-mole[i][3])**2 )/((nuc_rad**2+(x-mole[i][1])**2+(y-mole[i][2])**2+(z-mole[i][3])**2)**(7/2) + _reg)
    elif n_x == 1 and n_y == 2 and n_z == 0:
        for i in range(len(mole)):
            sum += -mole[i][0]*3*(x - mole[i][1])*( (x-mole[i][1])**2+(-4)*(y-mole[i][2])**2+(z-mole[i][3])**2 )/((nuc_rad**2+(x-mole[i][1])**2+(y-mole[i][2])**2+(z-mole[i][3])**2)**(7/2) + _reg)
    elif n_x == 0 and n_y == 2 and n_z == 1:
        for i in range(len(mole)):
            sum += -mole[i][0]*3*(z - mole[i][3])*( (x-mole[i][1])**2+(-4)*(y-mole[i][2])**2+(z-mole[i][3])**2 )/((nuc_rad**2+(x-mole[i][1])**2+(y-mole[i][2])**2+(z-mole[i][3])**2)**(7/2) + _reg)
    elif n_x == 1 and n_y == 0 and n_z == 2:
        for i in range(len(mole)):
            sum += -mole[i][0]*3*(x - mole[i][1])*( (x-mole[i][1])**2+(y-mole[i][2])**2+(-4)*(z-mole[i][3])**2 )/((nuc_rad**2+(x-mole[i][1])**2+(y-mole[i][2])**2+(z-mole[i][3])**2)**(7/2) + _reg)
    elif n_x == 0 and n_y == 1 and n_z == 2:
        for i in range(len(mole)):
            sum += -mole[i][0]*3*(y - mole[i][2])*( (x-mole[i][1])**2+(y-mole[i][2])**2+(-4)*(z-mole[i][3])**2 )/((nuc_rad**2+(x-mole[i][1])**2+(y-mole[i][2])**2+(z-mole[i][3])**2)**(7/2) + _reg)
    elif n_x == 1 and n_y == 1 and n_z == 1:
        for i in range(len(mole)):
            sum += mole[i][0]*15*( (x-mole[i][1])*(y-mole[i][2])*(z-mole[i][3]) )/((nuc_rad**2+(x-mole[i][1])**2+(y-mole[i][2])**2+(z-mole[i][3])**2)**(7/2) + _reg)
    # --------------------------higher derivatives------------------------------
    elif n_x + n_y + n_z > 3:
        h = 0.01
        if n_x > 3:
            sum += (partial_v_mol_3D(mole,n_x - 1, n_y, n_z, x+h, y, z) - partial_v_mol_3D(mole,n_x - 1, n_y, n_z, x-h, y, z))/(2*h)
        if n_y > 3:
            sum += (partial_v_mol_3D(mole,n_x, n_y - 1, n_z, x, y+h, z) - partial_v_mol_3D(mole,n_x, n_y - 1, n_z, x, y-h, z))/(2*h)
        if n_z > 3:
            sum += (partial_v_mol_3D(mole,n_x, n_y, n_z - 1, x, y, z+h) - partial_v_mol_3D(mole,n_x, n_y, n_z - 1, x, y, z-h))/(2*h)
    # --------------------------------------------------------------------------
    else:
        raise ValueError("The "+str(n_x + n_y + n_z)+"-th derivative is not supported!")
    return sum

def _Diophantine_1D(p):
    """
    Find all (1+p-1)-tuples of non-negative integers which fulfill:
    k_1 + ... + k_(p-1) = p-1
    k_1 + 2*k_2 + ... + (p-1)*k_(p-1) = mu_x

    Return an array of arrays: [[mu_x, k_1, ..., k_(p-1)], ..., [mu_x, k_1, ..., k_(p-1)]]
    """
    if p == 2:
        return [[1, 1]]
    elif p == 3:
        return [[1, 0, 1], [2, 2, 0]]
    elif p == 4:
        return [[1, 0, 0, 1], [2, 1, 1, 0], [3, 3, 0, 0]]
    elif p == 5:
        return [[1, 0, 0, 0, 1], [2, 0, 2, 0, 0], [2, 1, 0, 1, 0], [3, 2, 1, 0, 0], [4, 4, 0, 0, 0]]
    elif p == 6:
        return [[1, 0, 0, 0, 0, 1], [2, 0, 1, 1, 0, 0], [2, 1, 0, 0, 1, 0], [3, 1, 2, 0, 0, 0], [3, 2, 0, 1, 0, 0], [4, 3, 1, 0, 0, 0], [5, 5, 0, 0, 0, 0]]
    elif p == 7:
        return [[1, 0, 0, 0, 0, 0, 1], [2, 0, 0, 2, 0, 0, 0], [2, 0, 1, 0, 1, 0, 0], [2, 1, 0, 0, 0, 1, 0], [3, 0, 3, 0, 0, 0, 0], [3, 1, 1, 1, 0, 0, 0], [3, 2, 0, 0, 1, 0, 0], [4, 2, 2, 0, 0, 0, 0], [4, 3, 0, 1, 0, 0, 0], [5, 4, 1, 0, 0, 0, 0], [6, 6, 0, 0, 0, 0, 0]]
    elif p == 8:
        return [[1, 0, 0, 0, 0, 0, 0, 1], [2, 0, 0, 1, 1, 0, 0, 0], [2, 0, 1, 0, 0, 1, 0, 0], [2, 1, 0, 0, 0, 0, 1, 0], [3, 0, 2, 1, 0, 0, 0, 0], [3, 1, 0, 2, 0, 0, 0, 0], [3, 1, 1, 0, 1, 0, 0, 0], [3, 2, 0, 0, 0, 1, 0, 0], [4, 1, 3, 0, 0, 0, 0, 0], [4, 2, 1, 1, 0, 0, 0, 0], [4, 3, 0, 0, 1, 0, 0, 0], [5, 3, 2, 0, 0, 0, 0, 0], [5, 4, 0, 1, 0, 0, 0, 0], [6, 5, 1, 0, 0, 0, 0, 0], [7, 7, 0, 0, 0, 0, 0, 0]]
    elif p == 9:
        return [[1, 0, 0, 0, 0, 0, 0, 0, 1], [2, 0, 0, 0, 2, 0, 0, 0, 0], [2, 0, 0, 1, 0, 1, 0, 0, 0], [2, 0, 1, 0, 0, 0, 1, 0, 0], [2, 1, 0, 0, 0, 0, 0, 1, 0], [3, 0, 1, 2, 0, 0, 0, 0, 0], [3, 0, 2, 0, 1, 0, 0, 0, 0], [3, 1, 0, 1, 1, 0, 0, 0, 0], [3, 1, 1, 0, 0, 1, 0, 0, 0], [3, 2, 0, 0, 0, 0, 1, 0, 0], [4, 0, 4, 0, 0, 0, 0, 0, 0], [4, 1, 2, 1, 0, 0, 0, 0, 0], [4, 2, 0, 2, 0, 0, 0, 0, 0], [4, 2, 1, 0, 1, 0, 0, 0, 0], [4, 3, 0, 0, 0, 1, 0, 0, 0], [5, 2, 3, 0, 0, 0, 0, 0, 0], [5, 3, 1, 1, 0, 0, 0, 0, 0], [5, 4, 0, 0, 1, 0, 0, 0, 0], [6, 4, 2, 0, 0, 0, 0, 0, 0], [6, 5, 0, 1, 0, 0, 0, 0, 0], [7, 6, 1, 0, 0, 0, 0, 0, 0], [8, 8, 0, 0, 0, 0, 0, 0, 0]]
    else:
        raise ValueError("p = "+str(p)+" is not supported!")

def _Diophantine_2D(p):
    """
    Find all (2+p-1)-tuples of non-negative integers which fulfill:
    k_1 + ... + k_(p-1) = p-1
    k_1 + 2*k_2 + ... + (p-1)*k_(p-1) = mu_x + mu_y

    Return an array of arrays: [[mu_x, mu_y, k_1, ..., k_(p-1)], ..., [mu_x, mu_y, k_1, ..., k_(p-1)]]
    """
    if p == 2:
        return [[1, 0, 1], [0, 1, 1]]
    elif p == 3:
        return [[0, 1, 0, 1], [0, 2, 2, 0], [1, 0, 0, 1], [1, 1, 2, 0], [2, 0, 2, 0]]
    elif p == 4:
        return [[0, 1, 0, 0, 1], [0, 2, 1, 1, 0], [0, 3, 3, 0, 0], [1, 0, 0, 0, 1], [1, 1, 1, 1, 0], [1, 2, 3, 0, 0], [2, 0, 1, 1, 0], [2, 1, 3, 0, 0], [3, 0, 3, 0, 0]]
    elif p == 5:
        return [[0, 1, 0, 0, 0, 1], [0, 2, 0, 2, 0, 0], [0, 2, 1, 0, 1, 0], [0, 3, 2, 1, 0, 0], [0, 4, 4, 0, 0, 0], [1, 0, 0, 0, 0, 1], [1, 1, 0, 2, 0, 0], [1, 1, 1, 0, 1, 0], [1, 2, 2, 1, 0, 0], [1, 3, 4, 0, 0, 0], [2, 0, 0, 2, 0, 0], [2, 0, 1, 0, 1, 0], [2, 1, 2, 1, 0, 0], [2, 2, 4, 0, 0, 0], [3, 0, 2, 1, 0, 0], [3, 1, 4, 0, 0, 0], [4, 0, 4, 0, 0, 0]]
    elif p == 6:
        return [[0, 1, 0, 0, 0, 0, 1], [0, 2, 0, 1, 1, 0, 0], [0, 2, 1, 0, 0, 1, 0], [0, 3, 1, 2, 0, 0, 0], [0, 3, 2, 0, 1, 0, 0], [0, 4, 3, 1, 0, 0, 0], [0, 5, 5, 0, 0, 0, 0], [1, 0, 0, 0, 0, 0, 1], [1, 1, 0, 1, 1, 0, 0], [1, 1, 1, 0, 0, 1, 0], [1, 2, 1, 2, 0, 0, 0], [1, 2, 2, 0, 1, 0, 0], [1, 3, 3, 1, 0, 0, 0], [1, 4, 5, 0, 0, 0, 0], [2, 0, 0, 1, 1, 0, 0], [2, 0, 1, 0, 0, 1, 0], [2, 1, 1, 2, 0, 0, 0], [2, 1, 2, 0, 1, 0, 0], [2, 2, 3, 1, 0, 0, 0], [2, 3, 5, 0, 0, 0, 0], [3, 0, 1, 2, 0, 0, 0], [3, 0, 2, 0, 1, 0, 0], [3, 1, 3, 1, 0, 0, 0], [3, 2, 5, 0, 0, 0, 0], [4, 0, 3, 1, 0, 0, 0], [4, 1, 5, 0, 0, 0, 0], [5, 0, 5, 0, 0, 0, 0]]
    elif p == 7:
        return [[0, 1, 0, 0, 0, 0, 0, 1], [0, 2, 0, 0, 2, 0, 0, 0], [0, 2, 0, 1, 0, 1, 0, 0], [0, 2, 1, 0, 0, 0, 1, 0], [0, 3, 0, 3, 0, 0, 0, 0], [0, 3, 1, 1, 1, 0, 0, 0], [0, 3, 2, 0, 0, 1, 0, 0], [0, 4, 2, 2, 0, 0, 0, 0], [0, 4, 3, 0, 1, 0, 0, 0], [0, 5, 4, 1, 0, 0, 0, 0], [0, 6, 6, 0, 0, 0, 0, 0], [1, 0, 0, 0, 0, 0, 0, 1], [1, 1, 0, 0, 2, 0, 0, 0], [1, 1, 0, 1, 0, 1, 0, 0], [1, 1, 1, 0, 0, 0, 1, 0], [1, 2, 0, 3, 0, 0, 0, 0], [1, 2, 1, 1, 1, 0, 0, 0], [1, 2, 2, 0, 0, 1, 0, 0], [1, 3, 2, 2, 0, 0, 0, 0], [1, 3, 3, 0, 1, 0, 0, 0], [1, 4, 4, 1, 0, 0, 0, 0], [1, 5, 6, 0, 0, 0, 0, 0], [2, 0, 0, 0, 2, 0, 0, 0], [2, 0, 0, 1, 0, 1, 0, 0], [2, 0, 1, 0, 0, 0, 1, 0], [2, 1, 0, 3, 0, 0, 0, 0], [2, 1, 1, 1, 1, 0, 0, 0], [2, 1, 2, 0, 0, 1, 0, 0], [2, 2, 2, 2, 0, 0, 0, 0], [2, 2, 3, 0, 1, 0, 0, 0], [2, 3, 4, 1, 0, 0, 0, 0], [2, 4, 6, 0, 0, 0, 0, 0], [3, 0, 0, 3, 0, 0, 0, 0], [3, 0, 1, 1, 1, 0, 0, 0], [3, 0, 2, 0, 0, 1, 0, 0], [3, 1, 2, 2, 0, 0, 0, 0], [3, 1, 3, 0, 1, 0, 0, 0], [3, 2, 4, 1, 0, 0, 0, 0], [3, 3, 6, 0, 0, 0, 0, 0], [4, 0, 2, 2, 0, 0, 0, 0], [4, 0, 3, 0, 1, 0, 0, 0], [4, 1, 4, 1, 0, 0, 0, 0], [4, 2, 6, 0, 0, 0, 0, 0], [5, 0, 4, 1, 0, 0, 0, 0], [5, 1, 6, 0, 0, 0, 0, 0], [6, 0, 6, 0, 0, 0, 0, 0]]
    elif p == 8:
        return [[0, 1, 0, 0, 0, 0, 0, 0, 1], [0, 2, 0, 0, 1, 1, 0, 0, 0], [0, 2, 0, 1, 0, 0, 1, 0, 0], [0, 2, 1, 0, 0, 0, 0, 1, 0], [0, 3, 0, 2, 1, 0, 0, 0, 0], [0, 3, 1, 0, 2, 0, 0, 0, 0], [0, 3, 1, 1, 0, 1, 0, 0, 0], [0, 3, 2, 0, 0, 0, 1, 0, 0], [0, 4, 1, 3, 0, 0, 0, 0, 0], [0, 4, 2, 1, 1, 0, 0, 0, 0], [0, 4, 3, 0, 0, 1, 0, 0, 0], [0, 5, 3, 2, 0, 0, 0, 0, 0], [0, 5, 4, 0, 1, 0, 0, 0, 0], [0, 6, 5, 1, 0, 0, 0, 0, 0], [0, 7, 7, 0, 0, 0, 0, 0, 0], [1, 0, 0, 0, 0, 0, 0, 0, 1], [1, 1, 0, 0, 1, 1, 0, 0, 0], [1, 1, 0, 1, 0, 0, 1, 0, 0], [1, 1, 1, 0, 0, 0, 0, 1, 0], [1, 2, 0, 2, 1, 0, 0, 0, 0], [1, 2, 1, 0, 2, 0, 0, 0, 0], [1, 2, 1, 1, 0, 1, 0, 0, 0], [1, 2, 2, 0, 0, 0, 1, 0, 0], [1, 3, 1, 3, 0, 0, 0, 0, 0], [1, 3, 2, 1, 1, 0, 0, 0, 0], [1, 3, 3, 0, 0, 1, 0, 0, 0], [1, 4, 3, 2, 0, 0, 0, 0, 0], [1, 4, 4, 0, 1, 0, 0, 0, 0], [1, 5, 5, 1, 0, 0, 0, 0, 0], [1, 6, 7, 0, 0, 0, 0, 0, 0], [2, 0, 0, 0, 1, 1, 0, 0, 0], [2, 0, 0, 1, 0, 0, 1, 0, 0], [2, 0, 1, 0, 0, 0, 0, 1, 0], [2, 1, 0, 2, 1, 0, 0, 0, 0], [2, 1, 1, 0, 2, 0, 0, 0, 0], [2, 1, 1, 1, 0, 1, 0, 0, 0], [2, 1, 2, 0, 0, 0, 1, 0, 0], [2, 2, 1, 3, 0, 0, 0, 0, 0], [2, 2, 2, 1, 1, 0, 0, 0, 0], [2, 2, 3, 0, 0, 1, 0, 0, 0], [2, 3, 3, 2, 0, 0, 0, 0, 0], [2, 3, 4, 0, 1, 0, 0, 0, 0], [2, 4, 5, 1, 0, 0, 0, 0, 0], [2, 5, 7, 0, 0, 0, 0, 0, 0], [3, 0, 0, 2, 1, 0, 0, 0, 0], [3, 0, 1, 0, 2, 0, 0, 0, 0], [3, 0, 1, 1, 0, 1, 0, 0, 0], [3, 0, 2, 0, 0, 0, 1, 0, 0], [3, 1, 1, 3, 0, 0, 0, 0, 0], [3, 1, 2, 1, 1, 0, 0, 0, 0], [3, 1, 3, 0, 0, 1, 0, 0, 0], [3, 2, 3, 2, 0, 0, 0, 0, 0], [3, 2, 4, 0, 1, 0, 0, 0, 0], [3, 3, 5, 1, 0, 0, 0, 0, 0], [3, 4, 7, 0, 0, 0, 0, 0, 0], [4, 0, 1, 3, 0, 0, 0, 0, 0], [4, 0, 2, 1, 1, 0, 0, 0, 0], [4, 0, 3, 0, 0, 1, 0, 0, 0], [4, 1, 3, 2, 0, 0, 0, 0, 0], [4, 1, 4, 0, 1, 0, 0, 0, 0], [4, 2, 5, 1, 0, 0, 0, 0, 0], [4, 3, 7, 0, 0, 0, 0, 0, 0], [5, 0, 3, 2, 0, 0, 0, 0, 0], [5, 0, 4, 0, 1, 0, 0, 0, 0], [5, 1, 5, 1, 0, 0, 0, 0, 0], [5, 2, 7, 0, 0, 0, 0, 0, 0], [6, 0, 5, 1, 0, 0, 0, 0, 0], [6, 1, 7, 0, 0, 0, 0, 0, 0], [7, 0, 7, 0, 0, 0, 0, 0, 0]]
    elif p == 9:
        return [[0, 1, 0, 0, 0, 0, 0, 0, 0, 1], [0, 2, 0, 0, 0, 2, 0, 0, 0, 0], [0, 2, 0, 0, 1, 0, 1, 0, 0, 0], [0, 2, 0, 1, 0, 0, 0, 1, 0, 0], [0, 2, 1, 0, 0, 0, 0, 0, 1, 0], [0, 3, 0, 1, 2, 0, 0, 0, 0, 0], [0, 3, 0, 2, 0, 1, 0, 0, 0, 0], [0, 3, 1, 0, 1, 1, 0, 0, 0, 0], [0, 3, 1, 1, 0, 0, 1, 0, 0, 0], [0, 3, 2, 0, 0, 0, 0, 1, 0, 0], [0, 4, 0, 4, 0, 0, 0, 0, 0, 0], [0, 4, 1, 2, 1, 0, 0, 0, 0, 0], [0, 4, 2, 0, 2, 0, 0, 0, 0, 0], [0, 4, 2, 1, 0, 1, 0, 0, 0, 0], [0, 4, 3, 0, 0, 0, 1, 0, 0, 0], [0, 5, 2, 3, 0, 0, 0, 0, 0, 0], [0, 5, 3, 1, 1, 0, 0, 0, 0, 0], [0, 5, 4, 0, 0, 1, 0, 0, 0, 0], [0, 6, 4, 2, 0, 0, 0, 0, 0, 0], [0, 6, 5, 0, 1, 0, 0, 0, 0, 0], [0, 7, 6, 1, 0, 0, 0, 0, 0, 0], [0, 8, 8, 0, 0, 0, 0, 0, 0, 0], [1, 0, 0, 0, 0, 0, 0, 0, 0, 1], [1, 1, 0, 0, 0, 2, 0, 0, 0, 0], [1, 1, 0, 0, 1, 0, 1, 0, 0, 0], [1, 1, 0, 1, 0, 0, 0, 1, 0, 0], [1, 1, 1, 0, 0, 0, 0, 0, 1, 0], [1, 2, 0, 1, 2, 0, 0, 0, 0, 0], [1, 2, 0, 2, 0, 1, 0, 0, 0, 0], [1, 2, 1, 0, 1, 1, 0, 0, 0, 0], [1, 2, 1, 1, 0, 0, 1, 0, 0, 0], [1, 2, 2, 0, 0, 0, 0, 1, 0, 0], [1, 3, 0, 4, 0, 0, 0, 0, 0, 0], [1, 3, 1, 2, 1, 0, 0, 0, 0, 0], [1, 3, 2, 0, 2, 0, 0, 0, 0, 0], [1, 3, 2, 1, 0, 1, 0, 0, 0, 0], [1, 3, 3, 0, 0, 0, 1, 0, 0, 0], [1, 4, 2, 3, 0, 0, 0, 0, 0, 0], [1, 4, 3, 1, 1, 0, 0, 0, 0, 0], [1, 4, 4, 0, 0, 1, 0, 0, 0, 0], [1, 5, 4, 2, 0, 0, 0, 0, 0, 0], [1, 5, 5, 0, 1, 0, 0, 0, 0, 0], [1, 6, 6, 1, 0, 0, 0, 0, 0, 0], [1, 7, 8, 0, 0, 0, 0, 0, 0, 0], [2, 0, 0, 0, 0, 2, 0, 0, 0, 0], [2, 0, 0, 0, 1, 0, 1, 0, 0, 0], [2, 0, 0, 1, 0, 0, 0, 1, 0, 0], [2, 0, 1, 0, 0, 0, 0, 0, 1, 0], [2, 1, 0, 1, 2, 0, 0, 0, 0, 0], [2, 1, 0, 2, 0, 1, 0, 0, 0, 0], [2, 1, 1, 0, 1, 1, 0, 0, 0, 0], [2, 1, 1, 1, 0, 0, 1, 0, 0, 0], [2, 1, 2, 0, 0, 0, 0, 1, 0, 0], [2, 2, 0, 4, 0, 0, 0, 0, 0, 0], [2, 2, 1, 2, 1, 0, 0, 0, 0, 0], [2, 2, 2, 0, 2, 0, 0, 0, 0, 0], [2, 2, 2, 1, 0, 1, 0, 0, 0, 0], [2, 2, 3, 0, 0, 0, 1, 0, 0, 0], [2, 3, 2, 3, 0, 0, 0, 0, 0, 0], [2, 3, 3, 1, 1, 0, 0, 0, 0, 0], [2, 3, 4, 0, 0, 1, 0, 0, 0, 0], [2, 4, 4, 2, 0, 0, 0, 0, 0, 0], [2, 4, 5, 0, 1, 0, 0, 0, 0, 0], [2, 5, 6, 1, 0, 0, 0, 0, 0, 0], [2, 6, 8, 0, 0, 0, 0, 0, 0, 0], [3, 0, 0, 1, 2, 0, 0, 0, 0, 0], [3, 0, 0, 2, 0, 1, 0, 0, 0, 0], [3, 0, 1, 0, 1, 1, 0, 0, 0, 0], [3, 0, 1, 1, 0, 0, 1, 0, 0, 0], [3, 0, 2, 0, 0, 0, 0, 1, 0, 0], [3, 1, 0, 4, 0, 0, 0, 0, 0, 0], [3, 1, 1, 2, 1, 0, 0, 0, 0, 0], [3, 1, 2, 0, 2, 0, 0, 0, 0, 0], [3, 1, 2, 1, 0, 1, 0, 0, 0, 0], [3, 1, 3, 0, 0, 0, 1, 0, 0, 0], [3, 2, 2, 3, 0, 0, 0, 0, 0, 0], [3, 2, 3, 1, 1, 0, 0, 0, 0, 0], [3, 2, 4, 0, 0, 1, 0, 0, 0, 0], [3, 3, 4, 2, 0, 0, 0, 0, 0, 0], [3, 3, 5, 0, 1, 0, 0, 0, 0, 0], [3, 4, 6, 1, 0, 0, 0, 0, 0, 0], [3, 5, 8, 0, 0, 0, 0, 0, 0, 0], [4, 0, 0, 4, 0, 0, 0, 0, 0, 0], [4, 0, 1, 2, 1, 0, 0, 0, 0, 0], [4, 0, 2, 0, 2, 0, 0, 0, 0, 0], [4, 0, 2, 1, 0, 1, 0, 0, 0, 0], [4, 0, 3, 0, 0, 0, 1, 0, 0, 0], [4, 1, 2, 3, 0, 0, 0, 0, 0, 0], [4, 1, 3, 1, 1, 0, 0, 0, 0, 0], [4, 1, 4, 0, 0, 1, 0, 0, 0, 0], [4, 2, 4, 2, 0, 0, 0, 0, 0, 0], [4, 2, 5, 0, 1, 0, 0, 0, 0, 0], [4, 3, 6, 1, 0, 0, 0, 0, 0, 0], [4, 4, 8, 0, 0, 0, 0, 0, 0, 0], [5, 0, 2, 3, 0, 0, 0, 0, 0, 0], [5, 0, 3, 1, 1, 0, 0, 0, 0, 0], [5, 0, 4, 0, 0, 1, 0, 0, 0, 0], [5, 1, 4, 2, 0, 0, 0, 0, 0, 0], [5, 1, 5, 0, 1, 0, 0, 0, 0, 0], [5, 2, 6, 1, 0, 0, 0, 0, 0, 0], [5, 3, 8, 0, 0, 0, 0, 0, 0, 0], [6, 0, 4, 2, 0, 0, 0, 0, 0, 0], [6, 0, 5, 0, 1, 0, 0, 0, 0, 0], [6, 1, 6, 1, 0, 0, 0, 0, 0, 0], [6, 2, 8, 0, 0, 0, 0, 0, 0, 0], [7, 0, 6, 1, 0, 0, 0, 0, 0, 0], [7, 1, 8, 0, 0, 0, 0, 0, 0, 0], [8, 0, 8, 0, 0, 0, 0, 0, 0, 0]]
    else:
        raise ValueError("p = "+str(p)+" is not supported!")

def _Diophantine_3D(p):
    """
    Find all (3+p-1)-tuples of non-negative integers which fulfill:
    k_1 + ... + k_(p-1) = p-1
    k_1 + 2*k_2 + ... + (p-1)*k_(p-1) = mu_x + mu_y + mu_z

    Return an array of arrays: [[mu_x, mu_y, mu_z, k_1, ..., k_(p-1)], ..., [mu_x, mu_y, mu_z, k_1, ..., k_(p-1)]]
    """
    if p == 2:
        return [[1, 0, 0, 1], [0, 1, 0, 1], [0, 0, 1, 1]]
    elif p == 3:
        return [[0, 0, 1, 0, 1], [0, 0, 2, 2, 0], [0, 1, 0, 0, 1], [0, 1, 1, 2, 0], [0, 2, 0, 2, 0], [1, 0, 0, 0, 1], [1, 0, 1, 2, 0], [1, 1, 0, 2, 0], [2, 0, 0, 2, 0]]
    elif p == 4:
        return [[0, 0, 1, 0, 0, 1], [0, 0, 2, 1, 1, 0], [0, 0, 3, 3, 0, 0], [0, 1, 0, 0, 0, 1], [0, 1, 1, 1, 1, 0], [0, 1, 2, 3, 0, 0], [0, 2, 0, 1, 1, 0], [0, 2, 1, 3, 0, 0], [0, 3, 0, 3, 0, 0], [1, 0, 0, 0, 0, 1], [1, 0, 1, 1, 1, 0], [1, 0, 2, 3, 0, 0], [1, 1, 0, 1, 1, 0], [1, 1, 1, 3, 0, 0], [1, 2, 0, 3, 0, 0], [2, 0, 0, 1, 1, 0], [2, 0, 1, 3, 0, 0], [2, 1, 0, 3, 0, 0], [3, 0, 0, 3, 0, 0]]
    elif p == 5:
        return [[0, 0, 1, 0, 0, 0, 1], [0, 0, 2, 0, 2, 0, 0], [0, 0, 2, 1, 0, 1, 0], [0, 0, 3, 2, 1, 0, 0], [0, 0, 4, 4, 0, 0, 0], [0, 1, 0, 0, 0, 0, 1], [0, 1, 1, 0, 2, 0, 0], [0, 1, 1, 1, 0, 1, 0], [0, 1, 2, 2, 1, 0, 0], [0, 1, 3, 4, 0, 0, 0], [0, 2, 0, 0, 2, 0, 0], [0, 2, 0, 1, 0, 1, 0], [0, 2, 1, 2, 1, 0, 0], [0, 2, 2, 4, 0, 0, 0], [0, 3, 0, 2, 1, 0, 0], [0, 3, 1, 4, 0, 0, 0], [0, 4, 0, 4, 0, 0, 0], [1, 0, 0, 0, 0, 0, 1], [1, 0, 1, 0, 2, 0, 0], [1, 0, 1, 1, 0, 1, 0], [1, 0, 2, 2, 1, 0, 0], [1, 0, 3, 4, 0, 0, 0], [1, 1, 0, 0, 2, 0, 0], [1, 1, 0, 1, 0, 1, 0], [1, 1, 1, 2, 1, 0, 0], [1, 1, 2, 4, 0, 0, 0], [1, 2, 0, 2, 1, 0, 0], [1, 2, 1, 4, 0, 0, 0], [1, 3, 0, 4, 0, 0, 0], [2, 0, 0, 0, 2, 0, 0], [2, 0, 0, 1, 0, 1, 0], [2, 0, 1, 2, 1, 0, 0], [2, 0, 2, 4, 0, 0, 0], [2, 1, 0, 2, 1, 0, 0], [2, 1, 1, 4, 0, 0, 0], [2, 2, 0, 4, 0, 0, 0], [3, 0, 0, 2, 1, 0, 0], [3, 0, 1, 4, 0, 0, 0], [3, 1, 0, 4, 0, 0, 0], [4, 0, 0, 4, 0, 0, 0]]
    elif p == 6:
        return [[0, 0, 1, 0, 0, 0, 0, 1], [0, 0, 2, 0, 1, 1, 0, 0], [0, 0, 2, 1, 0, 0, 1, 0], [0, 0, 3, 1, 2, 0, 0, 0], [0, 0, 3, 2, 0, 1, 0, 0], [0, 0, 4, 3, 1, 0, 0, 0], [0, 0, 5, 5, 0, 0, 0, 0], [0, 1, 0, 0, 0, 0, 0, 1], [0, 1, 1, 0, 1, 1, 0, 0], [0, 1, 1, 1, 0, 0, 1, 0], [0, 1, 2, 1, 2, 0, 0, 0], [0, 1, 2, 2, 0, 1, 0, 0], [0, 1, 3, 3, 1, 0, 0, 0], [0, 1, 4, 5, 0, 0, 0, 0], [0, 2, 0, 0, 1, 1, 0, 0], [0, 2, 0, 1, 0, 0, 1, 0], [0, 2, 1, 1, 2, 0, 0, 0], [0, 2, 1, 2, 0, 1, 0, 0], [0, 2, 2, 3, 1, 0, 0, 0], [0, 2, 3, 5, 0, 0, 0, 0], [0, 3, 0, 1, 2, 0, 0, 0], [0, 3, 0, 2, 0, 1, 0, 0], [0, 3, 1, 3, 1, 0, 0, 0], [0, 3, 2, 5, 0, 0, 0, 0], [0, 4, 0, 3, 1, 0, 0, 0], [0, 4, 1, 5, 0, 0, 0, 0], [0, 5, 0, 5, 0, 0, 0, 0], [1, 0, 0, 0, 0, 0, 0, 1], [1, 0, 1, 0, 1, 1, 0, 0], [1, 0, 1, 1, 0, 0, 1, 0], [1, 0, 2, 1, 2, 0, 0, 0], [1, 0, 2, 2, 0, 1, 0, 0], [1, 0, 3, 3, 1, 0, 0, 0], [1, 0, 4, 5, 0, 0, 0, 0], [1, 1, 0, 0, 1, 1, 0, 0], [1, 1, 0, 1, 0, 0, 1, 0], [1, 1, 1, 1, 2, 0, 0, 0], [1, 1, 1, 2, 0, 1, 0, 0], [1, 1, 2, 3, 1, 0, 0, 0], [1, 1, 3, 5, 0, 0, 0, 0], [1, 2, 0, 1, 2, 0, 0, 0], [1, 2, 0, 2, 0, 1, 0, 0], [1, 2, 1, 3, 1, 0, 0, 0], [1, 2, 2, 5, 0, 0, 0, 0], [1, 3, 0, 3, 1, 0, 0, 0], [1, 3, 1, 5, 0, 0, 0, 0], [1, 4, 0, 5, 0, 0, 0, 0], [2, 0, 0, 0, 1, 1, 0, 0], [2, 0, 0, 1, 0, 0, 1, 0], [2, 0, 1, 1, 2, 0, 0, 0], [2, 0, 1, 2, 0, 1, 0, 0], [2, 0, 2, 3, 1, 0, 0, 0], [2, 0, 3, 5, 0, 0, 0, 0], [2, 1, 0, 1, 2, 0, 0, 0], [2, 1, 0, 2, 0, 1, 0, 0], [2, 1, 1, 3, 1, 0, 0, 0], [2, 1, 2, 5, 0, 0, 0, 0], [2, 2, 0, 3, 1, 0, 0, 0], [2, 2, 1, 5, 0, 0, 0, 0], [2, 3, 0, 5, 0, 0, 0, 0], [3, 0, 0, 1, 2, 0, 0, 0], [3, 0, 0, 2, 0, 1, 0, 0], [3, 0, 1, 3, 1, 0, 0, 0], [3, 0, 2, 5, 0, 0, 0, 0], [3, 1, 0, 3, 1, 0, 0, 0], [3, 1, 1, 5, 0, 0, 0, 0], [3, 2, 0, 5, 0, 0, 0, 0], [4, 0, 0, 3, 1, 0, 0, 0], [4, 0, 1, 5, 0, 0, 0, 0], [4, 1, 0, 5, 0, 0, 0, 0], [5, 0, 0, 5, 0, 0, 0, 0]]
    elif p == 7:
        return [[0, 0, 1, 0, 0, 0, 0, 0, 1], [0, 0, 2, 0, 0, 2, 0, 0, 0], [0, 0, 2, 0, 1, 0, 1, 0, 0], [0, 0, 2, 1, 0, 0, 0, 1, 0], [0, 0, 3, 0, 3, 0, 0, 0, 0], [0, 0, 3, 1, 1, 1, 0, 0, 0], [0, 0, 3, 2, 0, 0, 1, 0, 0], [0, 0, 4, 2, 2, 0, 0, 0, 0], [0, 0, 4, 3, 0, 1, 0, 0, 0], [0, 0, 5, 4, 1, 0, 0, 0, 0], [0, 0, 6, 6, 0, 0, 0, 0, 0], [0, 1, 0, 0, 0, 0, 0, 0, 1], [0, 1, 1, 0, 0, 2, 0, 0, 0], [0, 1, 1, 0, 1, 0, 1, 0, 0], [0, 1, 1, 1, 0, 0, 0, 1, 0], [0, 1, 2, 0, 3, 0, 0, 0, 0], [0, 1, 2, 1, 1, 1, 0, 0, 0], [0, 1, 2, 2, 0, 0, 1, 0, 0], [0, 1, 3, 2, 2, 0, 0, 0, 0], [0, 1, 3, 3, 0, 1, 0, 0, 0], [0, 1, 4, 4, 1, 0, 0, 0, 0], [0, 1, 5, 6, 0, 0, 0, 0, 0], [0, 2, 0, 0, 0, 2, 0, 0, 0], [0, 2, 0, 0, 1, 0, 1, 0, 0], [0, 2, 0, 1, 0, 0, 0, 1, 0], [0, 2, 1, 0, 3, 0, 0, 0, 0], [0, 2, 1, 1, 1, 1, 0, 0, 0], [0, 2, 1, 2, 0, 0, 1, 0, 0], [0, 2, 2, 2, 2, 0, 0, 0, 0], [0, 2, 2, 3, 0, 1, 0, 0, 0], [0, 2, 3, 4, 1, 0, 0, 0, 0], [0, 2, 4, 6, 0, 0, 0, 0, 0], [0, 3, 0, 0, 3, 0, 0, 0, 0], [0, 3, 0, 1, 1, 1, 0, 0, 0], [0, 3, 0, 2, 0, 0, 1, 0, 0], [0, 3, 1, 2, 2, 0, 0, 0, 0], [0, 3, 1, 3, 0, 1, 0, 0, 0], [0, 3, 2, 4, 1, 0, 0, 0, 0], [0, 3, 3, 6, 0, 0, 0, 0, 0], [0, 4, 0, 2, 2, 0, 0, 0, 0], [0, 4, 0, 3, 0, 1, 0, 0, 0], [0, 4, 1, 4, 1, 0, 0, 0, 0], [0, 4, 2, 6, 0, 0, 0, 0, 0], [0, 5, 0, 4, 1, 0, 0, 0, 0], [0, 5, 1, 6, 0, 0, 0, 0, 0], [0, 6, 0, 6, 0, 0, 0, 0, 0], [1, 0, 0, 0, 0, 0, 0, 0, 1], [1, 0, 1, 0, 0, 2, 0, 0, 0], [1, 0, 1, 0, 1, 0, 1, 0, 0], [1, 0, 1, 1, 0, 0, 0, 1, 0], [1, 0, 2, 0, 3, 0, 0, 0, 0], [1, 0, 2, 1, 1, 1, 0, 0, 0], [1, 0, 2, 2, 0, 0, 1, 0, 0], [1, 0, 3, 2, 2, 0, 0, 0, 0], [1, 0, 3, 3, 0, 1, 0, 0, 0], [1, 0, 4, 4, 1, 0, 0, 0, 0], [1, 0, 5, 6, 0, 0, 0, 0, 0], [1, 1, 0, 0, 0, 2, 0, 0, 0], [1, 1, 0, 0, 1, 0, 1, 0, 0], [1, 1, 0, 1, 0, 0, 0, 1, 0], [1, 1, 1, 0, 3, 0, 0, 0, 0], [1, 1, 1, 1, 1, 1, 0, 0, 0], [1, 1, 1, 2, 0, 0, 1, 0, 0], [1, 1, 2, 2, 2, 0, 0, 0, 0], [1, 1, 2, 3, 0, 1, 0, 0, 0], [1, 1, 3, 4, 1, 0, 0, 0, 0], [1, 1, 4, 6, 0, 0, 0, 0, 0], [1, 2, 0, 0, 3, 0, 0, 0, 0], [1, 2, 0, 1, 1, 1, 0, 0, 0], [1, 2, 0, 2, 0, 0, 1, 0, 0], [1, 2, 1, 2, 2, 0, 0, 0, 0], [1, 2, 1, 3, 0, 1, 0, 0, 0], [1, 2, 2, 4, 1, 0, 0, 0, 0], [1, 2, 3, 6, 0, 0, 0, 0, 0], [1, 3, 0, 2, 2, 0, 0, 0, 0], [1, 3, 0, 3, 0, 1, 0, 0, 0], [1, 3, 1, 4, 1, 0, 0, 0, 0], [1, 3, 2, 6, 0, 0, 0, 0, 0], [1, 4, 0, 4, 1, 0, 0, 0, 0], [1, 4, 1, 6, 0, 0, 0, 0, 0], [1, 5, 0, 6, 0, 0, 0, 0, 0], [2, 0, 0, 0, 0, 2, 0, 0, 0], [2, 0, 0, 0, 1, 0, 1, 0, 0], [2, 0, 0, 1, 0, 0, 0, 1, 0], [2, 0, 1, 0, 3, 0, 0, 0, 0], [2, 0, 1, 1, 1, 1, 0, 0, 0], [2, 0, 1, 2, 0, 0, 1, 0, 0], [2, 0, 2, 2, 2, 0, 0, 0, 0], [2, 0, 2, 3, 0, 1, 0, 0, 0], [2, 0, 3, 4, 1, 0, 0, 0, 0], [2, 0, 4, 6, 0, 0, 0, 0, 0], [2, 1, 0, 0, 3, 0, 0, 0, 0], [2, 1, 0, 1, 1, 1, 0, 0, 0], [2, 1, 0, 2, 0, 0, 1, 0, 0], [2, 1, 1, 2, 2, 0, 0, 0, 0], [2, 1, 1, 3, 0, 1, 0, 0, 0], [2, 1, 2, 4, 1, 0, 0, 0, 0], [2, 1, 3, 6, 0, 0, 0, 0, 0], [2, 2, 0, 2, 2, 0, 0, 0, 0], [2, 2, 0, 3, 0, 1, 0, 0, 0], [2, 2, 1, 4, 1, 0, 0, 0, 0], [2, 2, 2, 6, 0, 0, 0, 0, 0], [2, 3, 0, 4, 1, 0, 0, 0, 0], [2, 3, 1, 6, 0, 0, 0, 0, 0], [2, 4, 0, 6, 0, 0, 0, 0, 0], [3, 0, 0, 0, 3, 0, 0, 0, 0], [3, 0, 0, 1, 1, 1, 0, 0, 0], [3, 0, 0, 2, 0, 0, 1, 0, 0], [3, 0, 1, 2, 2, 0, 0, 0, 0], [3, 0, 1, 3, 0, 1, 0, 0, 0], [3, 0, 2, 4, 1, 0, 0, 0, 0], [3, 0, 3, 6, 0, 0, 0, 0, 0], [3, 1, 0, 2, 2, 0, 0, 0, 0], [3, 1, 0, 3, 0, 1, 0, 0, 0], [3, 1, 1, 4, 1, 0, 0, 0, 0], [3, 1, 2, 6, 0, 0, 0, 0, 0], [3, 2, 0, 4, 1, 0, 0, 0, 0], [3, 2, 1, 6, 0, 0, 0, 0, 0], [3, 3, 0, 6, 0, 0, 0, 0, 0], [4, 0, 0, 2, 2, 0, 0, 0, 0], [4, 0, 0, 3, 0, 1, 0, 0, 0], [4, 0, 1, 4, 1, 0, 0, 0, 0], [4, 0, 2, 6, 0, 0, 0, 0, 0], [4, 1, 0, 4, 1, 0, 0, 0, 0], [4, 1, 1, 6, 0, 0, 0, 0, 0], [4, 2, 0, 6, 0, 0, 0, 0, 0], [5, 0, 0, 4, 1, 0, 0, 0, 0], [5, 0, 1, 6, 0, 0, 0, 0, 0], [5, 1, 0, 6, 0, 0, 0, 0, 0], [6, 0, 0, 6, 0, 0, 0, 0, 0]]
    elif p == 8:
        return [[0, 0, 1, 0, 0, 0, 0, 0, 0, 1], [0, 0, 2, 0, 0, 1, 1, 0, 0, 0], [0, 0, 2, 0, 1, 0, 0, 1, 0, 0], [0, 0, 2, 1, 0, 0, 0, 0, 1, 0], [0, 0, 3, 0, 2, 1, 0, 0, 0, 0], [0, 0, 3, 1, 0, 2, 0, 0, 0, 0], [0, 0, 3, 1, 1, 0, 1, 0, 0, 0], [0, 0, 3, 2, 0, 0, 0, 1, 0, 0], [0, 0, 4, 1, 3, 0, 0, 0, 0, 0], [0, 0, 4, 2, 1, 1, 0, 0, 0, 0], [0, 0, 4, 3, 0, 0, 1, 0, 0, 0], [0, 0, 5, 3, 2, 0, 0, 0, 0, 0], [0, 0, 5, 4, 0, 1, 0, 0, 0, 0], [0, 0, 6, 5, 1, 0, 0, 0, 0, 0], [0, 0, 7, 7, 0, 0, 0, 0, 0, 0], [0, 1, 0, 0, 0, 0, 0, 0, 0, 1], [0, 1, 1, 0, 0, 1, 1, 0, 0, 0], [0, 1, 1, 0, 1, 0, 0, 1, 0, 0], [0, 1, 1, 1, 0, 0, 0, 0, 1, 0], [0, 1, 2, 0, 2, 1, 0, 0, 0, 0], [0, 1, 2, 1, 0, 2, 0, 0, 0, 0], [0, 1, 2, 1, 1, 0, 1, 0, 0, 0], [0, 1, 2, 2, 0, 0, 0, 1, 0, 0], [0, 1, 3, 1, 3, 0, 0, 0, 0, 0], [0, 1, 3, 2, 1, 1, 0, 0, 0, 0], [0, 1, 3, 3, 0, 0, 1, 0, 0, 0], [0, 1, 4, 3, 2, 0, 0, 0, 0, 0], [0, 1, 4, 4, 0, 1, 0, 0, 0, 0], [0, 1, 5, 5, 1, 0, 0, 0, 0, 0], [0, 1, 6, 7, 0, 0, 0, 0, 0, 0], [0, 2, 0, 0, 0, 1, 1, 0, 0, 0], [0, 2, 0, 0, 1, 0, 0, 1, 0, 0], [0, 2, 0, 1, 0, 0, 0, 0, 1, 0], [0, 2, 1, 0, 2, 1, 0, 0, 0, 0], [0, 2, 1, 1, 0, 2, 0, 0, 0, 0], [0, 2, 1, 1, 1, 0, 1, 0, 0, 0], [0, 2, 1, 2, 0, 0, 0, 1, 0, 0], [0, 2, 2, 1, 3, 0, 0, 0, 0, 0], [0, 2, 2, 2, 1, 1, 0, 0, 0, 0], [0, 2, 2, 3, 0, 0, 1, 0, 0, 0], [0, 2, 3, 3, 2, 0, 0, 0, 0, 0], [0, 2, 3, 4, 0, 1, 0, 0, 0, 0], [0, 2, 4, 5, 1, 0, 0, 0, 0, 0], [0, 2, 5, 7, 0, 0, 0, 0, 0, 0], [0, 3, 0, 0, 2, 1, 0, 0, 0, 0], [0, 3, 0, 1, 0, 2, 0, 0, 0, 0], [0, 3, 0, 1, 1, 0, 1, 0, 0, 0], [0, 3, 0, 2, 0, 0, 0, 1, 0, 0], [0, 3, 1, 1, 3, 0, 0, 0, 0, 0], [0, 3, 1, 2, 1, 1, 0, 0, 0, 0], [0, 3, 1, 3, 0, 0, 1, 0, 0, 0], [0, 3, 2, 3, 2, 0, 0, 0, 0, 0], [0, 3, 2, 4, 0, 1, 0, 0, 0, 0], [0, 3, 3, 5, 1, 0, 0, 0, 0, 0], [0, 3, 4, 7, 0, 0, 0, 0, 0, 0], [0, 4, 0, 1, 3, 0, 0, 0, 0, 0], [0, 4, 0, 2, 1, 1, 0, 0, 0, 0], [0, 4, 0, 3, 0, 0, 1, 0, 0, 0], [0, 4, 1, 3, 2, 0, 0, 0, 0, 0], [0, 4, 1, 4, 0, 1, 0, 0, 0, 0], [0, 4, 2, 5, 1, 0, 0, 0, 0, 0], [0, 4, 3, 7, 0, 0, 0, 0, 0, 0], [0, 5, 0, 3, 2, 0, 0, 0, 0, 0], [0, 5, 0, 4, 0, 1, 0, 0, 0, 0], [0, 5, 1, 5, 1, 0, 0, 0, 0, 0], [0, 5, 2, 7, 0, 0, 0, 0, 0, 0], [0, 6, 0, 5, 1, 0, 0, 0, 0, 0], [0, 6, 1, 7, 0, 0, 0, 0, 0, 0], [0, 7, 0, 7, 0, 0, 0, 0, 0, 0], [1, 0, 0, 0, 0, 0, 0, 0, 0, 1], [1, 0, 1, 0, 0, 1, 1, 0, 0, 0], [1, 0, 1, 0, 1, 0, 0, 1, 0, 0], [1, 0, 1, 1, 0, 0, 0, 0, 1, 0], [1, 0, 2, 0, 2, 1, 0, 0, 0, 0], [1, 0, 2, 1, 0, 2, 0, 0, 0, 0], [1, 0, 2, 1, 1, 0, 1, 0, 0, 0], [1, 0, 2, 2, 0, 0, 0, 1, 0, 0], [1, 0, 3, 1, 3, 0, 0, 0, 0, 0], [1, 0, 3, 2, 1, 1, 0, 0, 0, 0], [1, 0, 3, 3, 0, 0, 1, 0, 0, 0], [1, 0, 4, 3, 2, 0, 0, 0, 0, 0], [1, 0, 4, 4, 0, 1, 0, 0, 0, 0], [1, 0, 5, 5, 1, 0, 0, 0, 0, 0], [1, 0, 6, 7, 0, 0, 0, 0, 0, 0], [1, 1, 0, 0, 0, 1, 1, 0, 0, 0], [1, 1, 0, 0, 1, 0, 0, 1, 0, 0], [1, 1, 0, 1, 0, 0, 0, 0, 1, 0], [1, 1, 1, 0, 2, 1, 0, 0, 0, 0], [1, 1, 1, 1, 0, 2, 0, 0, 0, 0], [1, 1, 1, 1, 1, 0, 1, 0, 0, 0], [1, 1, 1, 2, 0, 0, 0, 1, 0, 0], [1, 1, 2, 1, 3, 0, 0, 0, 0, 0], [1, 1, 2, 2, 1, 1, 0, 0, 0, 0], [1, 1, 2, 3, 0, 0, 1, 0, 0, 0], [1, 1, 3, 3, 2, 0, 0, 0, 0, 0], [1, 1, 3, 4, 0, 1, 0, 0, 0, 0], [1, 1, 4, 5, 1, 0, 0, 0, 0, 0], [1, 1, 5, 7, 0, 0, 0, 0, 0, 0], [1, 2, 0, 0, 2, 1, 0, 0, 0, 0], [1, 2, 0, 1, 0, 2, 0, 0, 0, 0], [1, 2, 0, 1, 1, 0, 1, 0, 0, 0], [1, 2, 0, 2, 0, 0, 0, 1, 0, 0], [1, 2, 1, 1, 3, 0, 0, 0, 0, 0], [1, 2, 1, 2, 1, 1, 0, 0, 0, 0], [1, 2, 1, 3, 0, 0, 1, 0, 0, 0], [1, 2, 2, 3, 2, 0, 0, 0, 0, 0], [1, 2, 2, 4, 0, 1, 0, 0, 0, 0], [1, 2, 3, 5, 1, 0, 0, 0, 0, 0], [1, 2, 4, 7, 0, 0, 0, 0, 0, 0], [1, 3, 0, 1, 3, 0, 0, 0, 0, 0], [1, 3, 0, 2, 1, 1, 0, 0, 0, 0], [1, 3, 0, 3, 0, 0, 1, 0, 0, 0], [1, 3, 1, 3, 2, 0, 0, 0, 0, 0], [1, 3, 1, 4, 0, 1, 0, 0, 0, 0], [1, 3, 2, 5, 1, 0, 0, 0, 0, 0], [1, 3, 3, 7, 0, 0, 0, 0, 0, 0], [1, 4, 0, 3, 2, 0, 0, 0, 0, 0], [1, 4, 0, 4, 0, 1, 0, 0, 0, 0], [1, 4, 1, 5, 1, 0, 0, 0, 0, 0], [1, 4, 2, 7, 0, 0, 0, 0, 0, 0], [1, 5, 0, 5, 1, 0, 0, 0, 0, 0], [1, 5, 1, 7, 0, 0, 0, 0, 0, 0], [1, 6, 0, 7, 0, 0, 0, 0, 0, 0], [2, 0, 0, 0, 0, 1, 1, 0, 0, 0], [2, 0, 0, 0, 1, 0, 0, 1, 0, 0], [2, 0, 0, 1, 0, 0, 0, 0, 1, 0], [2, 0, 1, 0, 2, 1, 0, 0, 0, 0], [2, 0, 1, 1, 0, 2, 0, 0, 0, 0], [2, 0, 1, 1, 1, 0, 1, 0, 0, 0], [2, 0, 1, 2, 0, 0, 0, 1, 0, 0], [2, 0, 2, 1, 3, 0, 0, 0, 0, 0], [2, 0, 2, 2, 1, 1, 0, 0, 0, 0], [2, 0, 2, 3, 0, 0, 1, 0, 0, 0], [2, 0, 3, 3, 2, 0, 0, 0, 0, 0], [2, 0, 3, 4, 0, 1, 0, 0, 0, 0], [2, 0, 4, 5, 1, 0, 0, 0, 0, 0], [2, 0, 5, 7, 0, 0, 0, 0, 0, 0], [2, 1, 0, 0, 2, 1, 0, 0, 0, 0], [2, 1, 0, 1, 0, 2, 0, 0, 0, 0], [2, 1, 0, 1, 1, 0, 1, 0, 0, 0], [2, 1, 0, 2, 0, 0, 0, 1, 0, 0], [2, 1, 1, 1, 3, 0, 0, 0, 0, 0], [2, 1, 1, 2, 1, 1, 0, 0, 0, 0], [2, 1, 1, 3, 0, 0, 1, 0, 0, 0], [2, 1, 2, 3, 2, 0, 0, 0, 0, 0], [2, 1, 2, 4, 0, 1, 0, 0, 0, 0], [2, 1, 3, 5, 1, 0, 0, 0, 0, 0], [2, 1, 4, 7, 0, 0, 0, 0, 0, 0], [2, 2, 0, 1, 3, 0, 0, 0, 0, 0], [2, 2, 0, 2, 1, 1, 0, 0, 0, 0], [2, 2, 0, 3, 0, 0, 1, 0, 0, 0], [2, 2, 1, 3, 2, 0, 0, 0, 0, 0], [2, 2, 1, 4, 0, 1, 0, 0, 0, 0], [2, 2, 2, 5, 1, 0, 0, 0, 0, 0], [2, 2, 3, 7, 0, 0, 0, 0, 0, 0], [2, 3, 0, 3, 2, 0, 0, 0, 0, 0], [2, 3, 0, 4, 0, 1, 0, 0, 0, 0], [2, 3, 1, 5, 1, 0, 0, 0, 0, 0], [2, 3, 2, 7, 0, 0, 0, 0, 0, 0], [2, 4, 0, 5, 1, 0, 0, 0, 0, 0], [2, 4, 1, 7, 0, 0, 0, 0, 0, 0], [2, 5, 0, 7, 0, 0, 0, 0, 0, 0], [3, 0, 0, 0, 2, 1, 0, 0, 0, 0], [3, 0, 0, 1, 0, 2, 0, 0, 0, 0], [3, 0, 0, 1, 1, 0, 1, 0, 0, 0], [3, 0, 0, 2, 0, 0, 0, 1, 0, 0], [3, 0, 1, 1, 3, 0, 0, 0, 0, 0], [3, 0, 1, 2, 1, 1, 0, 0, 0, 0], [3, 0, 1, 3, 0, 0, 1, 0, 0, 0], [3, 0, 2, 3, 2, 0, 0, 0, 0, 0], [3, 0, 2, 4, 0, 1, 0, 0, 0, 0], [3, 0, 3, 5, 1, 0, 0, 0, 0, 0], [3, 0, 4, 7, 0, 0, 0, 0, 0, 0], [3, 1, 0, 1, 3, 0, 0, 0, 0, 0], [3, 1, 0, 2, 1, 1, 0, 0, 0, 0], [3, 1, 0, 3, 0, 0, 1, 0, 0, 0], [3, 1, 1, 3, 2, 0, 0, 0, 0, 0], [3, 1, 1, 4, 0, 1, 0, 0, 0, 0], [3, 1, 2, 5, 1, 0, 0, 0, 0, 0], [3, 1, 3, 7, 0, 0, 0, 0, 0, 0], [3, 2, 0, 3, 2, 0, 0, 0, 0, 0], [3, 2, 0, 4, 0, 1, 0, 0, 0, 0], [3, 2, 1, 5, 1, 0, 0, 0, 0, 0], [3, 2, 2, 7, 0, 0, 0, 0, 0, 0], [3, 3, 0, 5, 1, 0, 0, 0, 0, 0], [3, 3, 1, 7, 0, 0, 0, 0, 0, 0], [3, 4, 0, 7, 0, 0, 0, 0, 0, 0], [4, 0, 0, 1, 3, 0, 0, 0, 0, 0], [4, 0, 0, 2, 1, 1, 0, 0, 0, 0], [4, 0, 0, 3, 0, 0, 1, 0, 0, 0], [4, 0, 1, 3, 2, 0, 0, 0, 0, 0], [4, 0, 1, 4, 0, 1, 0, 0, 0, 0], [4, 0, 2, 5, 1, 0, 0, 0, 0, 0], [4, 0, 3, 7, 0, 0, 0, 0, 0, 0], [4, 1, 0, 3, 2, 0, 0, 0, 0, 0], [4, 1, 0, 4, 0, 1, 0, 0, 0, 0], [4, 1, 1, 5, 1, 0, 0, 0, 0, 0], [4, 1, 2, 7, 0, 0, 0, 0, 0, 0], [4, 2, 0, 5, 1, 0, 0, 0, 0, 0], [4, 2, 1, 7, 0, 0, 0, 0, 0, 0], [4, 3, 0, 7, 0, 0, 0, 0, 0, 0], [5, 0, 0, 3, 2, 0, 0, 0, 0, 0], [5, 0, 0, 4, 0, 1, 0, 0, 0, 0], [5, 0, 1, 5, 1, 0, 0, 0, 0, 0], [5, 0, 2, 7, 0, 0, 0, 0, 0, 0], [5, 1, 0, 5, 1, 0, 0, 0, 0, 0], [5, 1, 1, 7, 0, 0, 0, 0, 0, 0], [5, 2, 0, 7, 0, 0, 0, 0, 0, 0], [6, 0, 0, 5, 1, 0, 0, 0, 0, 0], [6, 0, 1, 7, 0, 0, 0, 0, 0, 0], [6, 1, 0, 7, 0, 0, 0, 0, 0, 0], [7, 0, 0, 7, 0, 0, 0, 0, 0, 0]]
    elif p == 9:
        return [[0, 0, 1, 0, 0, 0, 0, 0, 0, 1, 0], [0, 0, 2, 0, 0, 1, 1, 0, 0, 0, 0], [0, 0, 2, 0, 1, 0, 0, 1, 0, 0, 0], [0, 0, 2, 1, 0, 0, 0, 0, 1, 0, 0], [0, 0, 3, 0, 2, 1, 0, 0, 0, 0, 0], [0, 0, 3, 1, 0, 2, 0, 0, 0, 0, 0], [0, 0, 3, 1, 1, 0, 1, 0, 0, 0, 0], [0, 0, 3, 2, 0, 0, 0, 1, 0, 0, 0], [0, 0, 4, 1, 3, 0, 0, 0, 0, 0, 0], [0, 0, 4, 2, 1, 1, 0, 0, 0, 0, 0], [0, 0, 4, 3, 0, 0, 1, 0, 0, 0, 0], [0, 0, 5, 3, 2, 0, 0, 0, 0, 0, 0], [0, 0, 5, 4, 0, 1, 0, 0, 0, 0, 0], [0, 0, 6, 5, 1, 0, 0, 0, 0, 0, 0], [0, 0, 7, 7, 0, 0, 0, 0, 0, 0, 0], [0, 1, 0, 0, 0, 0, 0, 0, 0, 1, 0], [0, 1, 1, 0, 0, 1, 1, 0, 0, 0, 0], [0, 1, 1, 0, 1, 0, 0, 1, 0, 0, 0], [0, 1, 1, 1, 0, 0, 0, 0, 1, 0, 0], [0, 1, 2, 0, 2, 1, 0, 0, 0, 0, 0], [0, 1, 2, 1, 0, 2, 0, 0, 0, 0, 0], [0, 1, 2, 1, 1, 0, 1, 0, 0, 0, 0], [0, 1, 2, 2, 0, 0, 0, 1, 0, 0, 0], [0, 1, 3, 1, 3, 0, 0, 0, 0, 0, 0], [0, 1, 3, 2, 1, 1, 0, 0, 0, 0, 0], [0, 1, 3, 3, 0, 0, 1, 0, 0, 0, 0], [0, 1, 4, 3, 2, 0, 0, 0, 0, 0, 0], [0, 1, 4, 4, 0, 1, 0, 0, 0, 0, 0], [0, 1, 5, 5, 1, 0, 0, 0, 0, 0, 0], [0, 1, 6, 7, 0, 0, 0, 0, 0, 0, 0], [0, 2, 0, 0, 0, 1, 1, 0, 0, 0, 0], [0, 2, 0, 0, 1, 0, 0, 1, 0, 0, 0], [0, 2, 0, 1, 0, 0, 0, 0, 1, 0, 0], [0, 2, 1, 0, 2, 1, 0, 0, 0, 0, 0], [0, 2, 1, 1, 0, 2, 0, 0, 0, 0, 0], [0, 2, 1, 1, 1, 0, 1, 0, 0, 0, 0], [0, 2, 1, 2, 0, 0, 0, 1, 0, 0, 0], [0, 2, 2, 1, 3, 0, 0, 0, 0, 0, 0], [0, 2, 2, 2, 1, 1, 0, 0, 0, 0, 0], [0, 2, 2, 3, 0, 0, 1, 0, 0, 0, 0], [0, 2, 3, 3, 2, 0, 0, 0, 0, 0, 0], [0, 2, 3, 4, 0, 1, 0, 0, 0, 0, 0], [0, 2, 4, 5, 1, 0, 0, 0, 0, 0, 0], [0, 2, 5, 7, 0, 0, 0, 0, 0, 0, 0], [0, 3, 0, 0, 2, 1, 0, 0, 0, 0, 0], [0, 3, 0, 1, 0, 2, 0, 0, 0, 0, 0], [0, 3, 0, 1, 1, 0, 1, 0, 0, 0, 0], [0, 3, 0, 2, 0, 0, 0, 1, 0, 0, 0], [0, 3, 1, 1, 3, 0, 0, 0, 0, 0, 0], [0, 3, 1, 2, 1, 1, 0, 0, 0, 0, 0], [0, 3, 1, 3, 0, 0, 1, 0, 0, 0, 0], [0, 3, 2, 3, 2, 0, 0, 0, 0, 0, 0], [0, 3, 2, 4, 0, 1, 0, 0, 0, 0, 0], [0, 3, 3, 5, 1, 0, 0, 0, 0, 0, 0], [0, 3, 4, 7, 0, 0, 0, 0, 0, 0, 0], [0, 4, 0, 1, 3, 0, 0, 0, 0, 0, 0], [0, 4, 0, 2, 1, 1, 0, 0, 0, 0, 0], [0, 4, 0, 3, 0, 0, 1, 0, 0, 0, 0], [0, 4, 1, 3, 2, 0, 0, 0, 0, 0, 0], [0, 4, 1, 4, 0, 1, 0, 0, 0, 0, 0], [0, 4, 2, 5, 1, 0, 0, 0, 0, 0, 0], [0, 4, 3, 7, 0, 0, 0, 0, 0, 0, 0], [0, 5, 0, 3, 2, 0, 0, 0, 0, 0, 0], [0, 5, 0, 4, 0, 1, 0, 0, 0, 0, 0], [0, 5, 1, 5, 1, 0, 0, 0, 0, 0, 0], [0, 5, 2, 7, 0, 0, 0, 0, 0, 0, 0], [0, 6, 0, 5, 1, 0, 0, 0, 0, 0, 0], [0, 6, 1, 7, 0, 0, 0, 0, 0, 0, 0], [0, 7, 0, 7, 0, 0, 0, 0, 0, 0, 0], [1, 0, 0, 0, 0, 0, 0, 0, 0, 1, 0], [1, 0, 1, 0, 0, 1, 1, 0, 0, 0, 0], [1, 0, 1, 0, 1, 0, 0, 1, 0, 0, 0], [1, 0, 1, 1, 0, 0, 0, 0, 1, 0, 0], [1, 0, 2, 0, 2, 1, 0, 0, 0, 0, 0], [1, 0, 2, 1, 0, 2, 0, 0, 0, 0, 0], [1, 0, 2, 1, 1, 0, 1, 0, 0, 0, 0], [1, 0, 2, 2, 0, 0, 0, 1, 0, 0, 0], [1, 0, 3, 1, 3, 0, 0, 0, 0, 0, 0], [1, 0, 3, 2, 1, 1, 0, 0, 0, 0, 0], [1, 0, 3, 3, 0, 0, 1, 0, 0, 0, 0], [1, 0, 4, 3, 2, 0, 0, 0, 0, 0, 0], [1, 0, 4, 4, 0, 1, 0, 0, 0, 0, 0], [1, 0, 5, 5, 1, 0, 0, 0, 0, 0, 0], [1, 0, 6, 7, 0, 0, 0, 0, 0, 0, 0], [1, 1, 0, 0, 0, 1, 1, 0, 0, 0, 0], [1, 1, 0, 0, 1, 0, 0, 1, 0, 0, 0], [1, 1, 0, 1, 0, 0, 0, 0, 1, 0, 0], [1, 1, 1, 0, 2, 1, 0, 0, 0, 0, 0], [1, 1, 1, 1, 0, 2, 0, 0, 0, 0, 0], [1, 1, 1, 1, 1, 0, 1, 0, 0, 0, 0], [1, 1, 1, 2, 0, 0, 0, 1, 0, 0, 0], [1, 1, 2, 1, 3, 0, 0, 0, 0, 0, 0], [1, 1, 2, 2, 1, 1, 0, 0, 0, 0, 0], [1, 1, 2, 3, 0, 0, 1, 0, 0, 0, 0], [1, 1, 3, 3, 2, 0, 0, 0, 0, 0, 0], [1, 1, 3, 4, 0, 1, 0, 0, 0, 0, 0], [1, 1, 4, 5, 1, 0, 0, 0, 0, 0, 0], [1, 1, 5, 7, 0, 0, 0, 0, 0, 0, 0], [1, 2, 0, 0, 2, 1, 0, 0, 0, 0, 0], [1, 2, 0, 1, 0, 2, 0, 0, 0, 0, 0], [1, 2, 0, 1, 1, 0, 1, 0, 0, 0, 0], [1, 2, 0, 2, 0, 0, 0, 1, 0, 0, 0], [1, 2, 1, 1, 3, 0, 0, 0, 0, 0, 0], [1, 2, 1, 2, 1, 1, 0, 0, 0, 0, 0], [1, 2, 1, 3, 0, 0, 1, 0, 0, 0, 0], [1, 2, 2, 3, 2, 0, 0, 0, 0, 0, 0], [1, 2, 2, 4, 0, 1, 0, 0, 0, 0, 0], [1, 2, 3, 5, 1, 0, 0, 0, 0, 0, 0], [1, 2, 4, 7, 0, 0, 0, 0, 0, 0, 0], [1, 3, 0, 1, 3, 0, 0, 0, 0, 0, 0], [1, 3, 0, 2, 1, 1, 0, 0, 0, 0, 0], [1, 3, 0, 3, 0, 0, 1, 0, 0, 0, 0], [1, 3, 1, 3, 2, 0, 0, 0, 0, 0, 0], [1, 3, 1, 4, 0, 1, 0, 0, 0, 0, 0], [1, 3, 2, 5, 1, 0, 0, 0, 0, 0, 0], [1, 3, 3, 7, 0, 0, 0, 0, 0, 0, 0], [1, 4, 0, 3, 2, 0, 0, 0, 0, 0, 0], [1, 4, 0, 4, 0, 1, 0, 0, 0, 0, 0], [1, 4, 1, 5, 1, 0, 0, 0, 0, 0, 0], [1, 4, 2, 7, 0, 0, 0, 0, 0, 0, 0], [1, 5, 0, 5, 1, 0, 0, 0, 0, 0, 0], [1, 5, 1, 7, 0, 0, 0, 0, 0, 0, 0], [1, 6, 0, 7, 0, 0, 0, 0, 0, 0, 0], [2, 0, 0, 0, 0, 1, 1, 0, 0, 0, 0], [2, 0, 0, 0, 1, 0, 0, 1, 0, 0, 0], [2, 0, 0, 1, 0, 0, 0, 0, 1, 0, 0], [2, 0, 1, 0, 2, 1, 0, 0, 0, 0, 0], [2, 0, 1, 1, 0, 2, 0, 0, 0, 0, 0], [2, 0, 1, 1, 1, 0, 1, 0, 0, 0, 0], [2, 0, 1, 2, 0, 0, 0, 1, 0, 0, 0], [2, 0, 2, 1, 3, 0, 0, 0, 0, 0, 0], [2, 0, 2, 2, 1, 1, 0, 0, 0, 0, 0], [2, 0, 2, 3, 0, 0, 1, 0, 0, 0, 0], [2, 0, 3, 3, 2, 0, 0, 0, 0, 0, 0], [2, 0, 3, 4, 0, 1, 0, 0, 0, 0, 0], [2, 0, 4, 5, 1, 0, 0, 0, 0, 0, 0], [2, 0, 5, 7, 0, 0, 0, 0, 0, 0, 0], [2, 1, 0, 0, 2, 1, 0, 0, 0, 0, 0], [2, 1, 0, 1, 0, 2, 0, 0, 0, 0, 0], [2, 1, 0, 1, 1, 0, 1, 0, 0, 0, 0], [2, 1, 0, 2, 0, 0, 0, 1, 0, 0, 0], [2, 1, 1, 1, 3, 0, 0, 0, 0, 0, 0], [2, 1, 1, 2, 1, 1, 0, 0, 0, 0, 0], [2, 1, 1, 3, 0, 0, 1, 0, 0, 0, 0], [2, 1, 2, 3, 2, 0, 0, 0, 0, 0, 0], [2, 1, 2, 4, 0, 1, 0, 0, 0, 0, 0], [2, 1, 3, 5, 1, 0, 0, 0, 0, 0, 0], [2, 1, 4, 7, 0, 0, 0, 0, 0, 0, 0], [2, 2, 0, 1, 3, 0, 0, 0, 0, 0, 0], [2, 2, 0, 2, 1, 1, 0, 0, 0, 0, 0], [2, 2, 0, 3, 0, 0, 1, 0, 0, 0, 0], [2, 2, 1, 3, 2, 0, 0, 0, 0, 0, 0], [2, 2, 1, 4, 0, 1, 0, 0, 0, 0, 0], [2, 2, 2, 5, 1, 0, 0, 0, 0, 0, 0], [2, 2, 3, 7, 0, 0, 0, 0, 0, 0, 0], [2, 3, 0, 3, 2, 0, 0, 0, 0, 0, 0], [2, 3, 0, 4, 0, 1, 0, 0, 0, 0, 0], [2, 3, 1, 5, 1, 0, 0, 0, 0, 0, 0], [2, 3, 2, 7, 0, 0, 0, 0, 0, 0, 0], [2, 4, 0, 5, 1, 0, 0, 0, 0, 0, 0], [2, 4, 1, 7, 0, 0, 0, 0, 0, 0, 0], [2, 5, 0, 7, 0, 0, 0, 0, 0, 0, 0], [3, 0, 0, 0, 2, 1, 0, 0, 0, 0, 0], [3, 0, 0, 1, 0, 2, 0, 0, 0, 0, 0], [3, 0, 0, 1, 1, 0, 1, 0, 0, 0, 0], [3, 0, 0, 2, 0, 0, 0, 1, 0, 0, 0], [3, 0, 1, 1, 3, 0, 0, 0, 0, 0, 0], [3, 0, 1, 2, 1, 1, 0, 0, 0, 0, 0], [3, 0, 1, 3, 0, 0, 1, 0, 0, 0, 0], [3, 0, 2, 3, 2, 0, 0, 0, 0, 0, 0], [3, 0, 2, 4, 0, 1, 0, 0, 0, 0, 0], [3, 0, 3, 5, 1, 0, 0, 0, 0, 0, 0], [3, 0, 4, 7, 0, 0, 0, 0, 0, 0, 0], [3, 1, 0, 1, 3, 0, 0, 0, 0, 0, 0], [3, 1, 0, 2, 1, 1, 0, 0, 0, 0, 0], [3, 1, 0, 3, 0, 0, 1, 0, 0, 0, 0], [3, 1, 1, 3, 2, 0, 0, 0, 0, 0, 0], [3, 1, 1, 4, 0, 1, 0, 0, 0, 0, 0], [3, 1, 2, 5, 1, 0, 0, 0, 0, 0, 0], [3, 1, 3, 7, 0, 0, 0, 0, 0, 0, 0], [3, 2, 0, 3, 2, 0, 0, 0, 0, 0, 0], [3, 2, 0, 4, 0, 1, 0, 0, 0, 0, 0], [3, 2, 1, 5, 1, 0, 0, 0, 0, 0, 0], [3, 2, 2, 7, 0, 0, 0, 0, 0, 0, 0], [3, 3, 0, 5, 1, 0, 0, 0, 0, 0, 0], [3, 3, 1, 7, 0, 0, 0, 0, 0, 0, 0], [3, 4, 0, 7, 0, 0, 0, 0, 0, 0, 0], [4, 0, 0, 1, 3, 0, 0, 0, 0, 0, 0], [4, 0, 0, 2, 1, 1, 0, 0, 0, 0, 0], [4, 0, 0, 3, 0, 0, 1, 0, 0, 0, 0], [4, 0, 1, 3, 2, 0, 0, 0, 0, 0, 0], [4, 0, 1, 4, 0, 1, 0, 0, 0, 0, 0], [4, 0, 2, 5, 1, 0, 0, 0, 0, 0, 0], [4, 0, 3, 7, 0, 0, 0, 0, 0, 0, 0], [4, 1, 0, 3, 2, 0, 0, 0, 0, 0, 0], [4, 1, 0, 4, 0, 1, 0, 0, 0, 0, 0], [4, 1, 1, 5, 1, 0, 0, 0, 0, 0, 0], [4, 1, 2, 7, 0, 0, 0, 0, 0, 0, 0], [4, 2, 0, 5, 1, 0, 0, 0, 0, 0, 0], [4, 2, 1, 7, 0, 0, 0, 0, 0, 0, 0], [4, 3, 0, 7, 0, 0, 0, 0, 0, 0, 0], [5, 0, 0, 3, 2, 0, 0, 0, 0, 0, 0], [5, 0, 0, 4, 0, 1, 0, 0, 0, 0, 0], [5, 0, 1, 5, 1, 0, 0, 0, 0, 0, 0], [5, 0, 2, 7, 0, 0, 0, 0, 0, 0, 0], [5, 1, 0, 5, 1, 0, 0, 0, 0, 0, 0], [5, 1, 1, 7, 0, 0, 0, 0, 0, 0, 0], [5, 2, 0, 7, 0, 0, 0, 0, 0, 0, 0], [6, 0, 0, 5, 1, 0, 0, 0, 0, 0, 0], [6, 0, 1, 7, 0, 0, 0, 0, 0, 0, 0], [6, 1, 0, 7, 0, 0, 0, 0, 0, 0, 0], [7, 0, 0, 7, 0, 0, 0, 0, 0, 0, 0]]
    else:
        raise ValueError("p = "+str(p)+" is not supported!")

# ------------------------------------------------------------------------------
# partial_v_B and partial_v_A are callables with arguments:
# mu_x-th derivative, mu_y-th derivative, mu_z-th derivative, x, y, z,
# depending on the dimensionality of the problem (1,2,3)
# ------------------------------------------------------------------------------
def _single_kernel_1D(p, partial_v_A, partial_v_B, x, verbose = False):
    """
    Calculate the kernel expression at order p and point x for the transmutation
    of v_A to v_B
    """
    if p not in [1, 2, 3, 4, 5, 6, 7, 8, 9]:
        raise ValueError("p = " + str(p) + " is not supported!")
    if p == 1:
        return partial_v_B(0,x) - partial_v_A(0,x)
    else:
        final_sum = 0
        Dio = _Diophantine_1D(p)
        for index in range(len(Dio)):
            mu_x = Dio[index][0]
            deriv = partial_v_B(mu_x,x) - partial_v_A(mu_x,x)
            product_i = x**mu_x
            for i in range(p-1):
                k_i = Dio[index][1+i]
                product_i /= _fc(k_i)
            final_sum += deriv*product_i
        ratio = (partial_v_B(0,x) - _reg)/(partial_v_A(0,x) - _reg)
        if verbose:
            if abs(ratio) >= 1:
                print('Warning: naive convergence criterion | 1 - v_B/v_A | < 1 violated at point ' + str(x) + ' !')
        return final_sum*((1 - ratio)**(p-1))/p


def _single_kernel_2D(p, partial_v_A, partial_v_B, x,y, verbose = False):
    """
    Calculate the kernel expression at order p and point x,y for the transmutation
    of v_A to v_B
    """
    if p not in [1, 2, 3, 4, 5, 6, 7, 8, 9]:
        raise ValueError("p = " + str(p) + " is not supported!")
    if p == 1:
        return partial_v_B(0,0,x,y) - partial_v_A(0,0,x,y)
    else:
        final_sum = 0
        Dio = _Diophantine_2D(p)
        for index in range(len(Dio)):
            mu_x = Dio[index][0]
            mu_y = Dio[index][1]
            deriv = partial_v_B(mu_x,mu_y,x,y) - partial_v_A(mu_x,mu_y,x,y)
            k = mu_x + mu_y
            product_i = (x + y)**k
            for i in range(p-1):
                k_i = Dio[index][2+i]
                product_i /= _fc(k_i)
            final_sum += deriv*product_i
        ratio = (partial_v_B(0,0,x,y) - _reg)/(partial_v_A(0,0,x,y) - _reg)
        if verbose:
            if abs(ratio) >= 1:
                print('Warning: naive convergence criterion | 1 - v_B/v_A | < 1 violated at point (' + str(x) + ', ' + str(y) + ') !')
        return final_sum*((1 - ratio)**(p-1))/p


def _single_kernel_3D(p, partial_v_A, partial_v_B, x,y,z, verbose = False):
    """
    Calculate the kernel expression at order p and point x,y,z for the transmutation
    of v_A to v_B
    """
    if p not in [1, 2, 3, 4, 5, 6, 7, 8, 9]:
        raise ValueError("p = " + str(p) + " is not supported!")
    if p == 1:
        return partial_v_B(0,0,0,x,y,z) - partial_v_A(0,0,0,x,y,z)
    else:
        final_sum = 0
        Dio = _Diophantine_3D(p)
        for index in range(len(Dio)):
            mu_x = Dio[index][0]
            mu_y = Dio[index][1]
            mu_z = Dio[index][2]
            deriv = partial_v_B(mu_x,mu_y,mu_z,x,y,z) - partial_v_A(mu_x,mu_y,mu_z,x,y,z)
            k = mu_x + mu_y + mu_z
            product_i = (x + y + z)**k
            for i in range(p-1):
                k_i = Dio[index][3+i]
                product_i /= _fc(k_i)
            final_sum += deriv*product_i
        ratio = (partial_v_B(0,0,0,x,y,z) - _reg)/(partial_v_A(0,0,0,x,y,z) - _reg)
        if verbose:
            if abs(ratio) >= 1:
                print('Warning: naive convergence criterion | 1 - v_B/v_A | < 1 violated at point (' + str(x) + ', ' + str(y) + str(z) + ') !')
        return final_sum*((1 - ratio)**(p-1))/p



# ------------------------------------------------------------------------------
# Calculate the kernel expression and summing over all p in orders.
# ------------------------------------------------------------------------------
def kernel_1D(partial_v_A, partial_v_B, x, orders = [1, 2, 3], verbose = False):
    """
    1D kernel of the Alchemical Integral Transform

    Parameters
    ----------
    partial_v_A : callable
        A function of the initial system's external potential in 1D. It expects two arguments,
        `n_x` and `x`, such that `partial_v_A(n_x, x)`
        :math:`= \frac{\partial^{n_x} }{\partial x^{n_x}} v_A(x)`
    partial_v_B : callable
        A function of the final system's external potential in 1D. It expects two arguments,
        `n_x` and `x`, such that `partial_v_B(n_x, x)`
        :math:`= \frac{\partial^{n_x} }{\partial x^{n_x}} v_B(x)`
    x : float
        coordinate
    orders : list, optional
        A list of the orders :math:`p` in the kernel to be summed over. Recommended are at least `[1,2,3]`,
        precise is `[1,2,3,4,5]`. :math:`p` is implemented up to and including 9-th order
    verbose : bool, optional
        When `True`, prints a warning if the naive convergence criterion :math:`|1 - v_B(x)/v_A(x)| < 1` is violated.
        This does not imply divergence of the series but may hint towards
        too large differences between initial and final system.

    Returns
    -------
    float
        The 1D kernel of AIT between systems A and B at `x` for all orders in `orders`.


    """
    if len(orders) == 0:
        print('Warning: No orders defined!')
        return 0
    if not all([n in [1, 2, 3, 4, 5, 6, 7, 8, 9] for n in orders]):
        raise ValueError('Only orders p = {1,2,3,4,5,6,7,8,9} are supported!')
        return 0
    sum = 0
    for i in orders:
        sum += _single_kernel_1D(i, partial_v_A, partial_v_B, x, verbose = verbose)
    return sum


def kernel_2D(partial_v_A, partial_v_B, x,y, orders = [1, 2, 3], verbose = False):
    """
    2D kernel of the Alchemical Integral Transform

    Parameters
    ----------
    partial_v_A : callable
        A function of the initial system's external potential in 2D. It expects four arguments,
        `n_x, n_y` and `x, y`, such that `partial_v_A(n_x, n_y, x, y)`
        :math:`= \frac{\partial^{n_x + n_y} }{\partial x^{n_x} \partial y^{n_y} } v_A(x,y) `
    partial_v_B : callable
        A function of the final system's external potential in 2D. It expects four arguments,
        `n_x, n_y` and `x, y`, such that `partial_v_B(n_x, n_y, x, y)`
        :math:`= \frac{\partial^{n_x + n_y} }{\partial x^{n_x} \partial y^{n_y} } v_B(x,y) `
    x, y : float
        coordinates
    orders : list, optional
        A list of the orders :math:`p` in the kernel to be summed over. Recommended are at least `[1,2,3]`,
        precise is `[1,2,3,4,5]`. :math:`p` is implemented up to and including 9-th order
    verbose : bool, optional
        When `True`, prints a warning if the naive convergence criterion :math:`|1 - v_B(x,y)/v_A(x,y)| < 1` is violated.
        This does not imply divergence of the series but may hint towards
        too large differences between initial and final system.

    Returns
    -------
    float
        The 2D kernel of AIT between systems A and B at `x, y` for all orders in `orders`.


    """
    if len(orders) == 0:
        print('Warning: No orders defined!')
        return 0
    if not all([n in [1, 2, 3, 4, 5, 6, 7, 8, 9] for n in orders]):
        raise ValueError('Only orders p = {1,2,3,4,5,6,7,8,9} are supported!')
        return 0
    sum = 0
    for i in orders:
        sum += _single_kernel_2D(i, partial_v_A, partial_v_B, x,y, verbose = verbose)
    return sum


def kernel_3D(partial_v_A, partial_v_B, x,y,z, orders = [1, 2, 3], verbose = False):
    """
    3D kernel of the Alchemical Integral Transform

    Parameters
    ----------
    partial_v_A : callable
        A function of the initial system's external potential in 3D. It expects six arguments,
        `n_x, n_y, n_z` and `x, y, z`, such that `partial_v_A(n_x, n_y, n_z, x, y, z)`
        :math:`= \frac{\partial^{n_x + n_y + n_z} }{\partial x^{n_x} \partial y^{n_z} \partial y^{n_z} } v_A(x,y,z) `
    partial_v_B : callable
        A function of the final system's external potential in 3D. It expects six arguments,
        `n_x, n_y, n_z` and `x, y, z`, such that `partial_v_B(n_x, n_y, n_z, x, y, z)`
        :math:`= \frac{\partial^{n_x + n_y + n_z} }{\partial x^{n_x} \partial y^{n_z} \partial y^{n_z} } v_B(x,y,z) `
    x, y, z : float
        coordinates
    orders : list, optional
        A list of the orders :math:`p` in the kernel to be summed over. Recommended are at least `[1,2,3]`,
        precise is `[1,2,3,4,5]`. :math:`p` is implemented up to and including 9-th order
    verbose : bool, optional
        When `True`, prints a warning if the naive convergence criterion :math:`|1 - v_B(x,y,z)/v_A(x,y,z)| < 1` is violated.
        This does not imply divergence of the series but may hint towards
        too large differences between initial and final system.

    Returns
    -------
    float
        The 3D kernel of AIT between systems A and B at `x, y, z` for all orders in `orders`.


    """
    if len(orders) == 0:
        print('Warning: No orders defined!')
        return 0
    if not all([n in [1, 2, 3, 4, 5, 6, 7, 8, 9] for n in orders]):
        raise ValueError('Only orders p = {1,2,3,4,5,6,7,8,9} are supported!')
        return 0
    sum = 0
    for i in orders:
        sum += _single_kernel_3D(i, partial_v_A, partial_v_B, x,y,z, verbose = verbose)
    return sum
